/* resampling module 
 *
 * The audio data has been read. Here are the
 * functions to ensure a correct continuation
 * of the output stream and to convert to a
 * lower sample rate.
 *
 */

#define SYNC_OFF	150L
#undef    DEBUG_SHIFTS		/* simulate bad cdrom drives */
#undef SHOW_JITTER

#include <stdio.h>
#include <stdlib.h>
#if defined (HAVE_UNISTD_H) && (HAVE_UNISTD_H == 1)
#include <sys/types.h>
#include <unistd.h>
#endif
#include <stddef.h>
#include <string.h>
#include <limits.h>
#include <assert.h>

#include "mycdrom.h"
#include "cdda2wav.h"
#include "interface.h"
#include "byteorder.h"
#include "ringbuff.h"
#include "resample.h"
#include "global.h"


long waitforsignal = 0;	/* flag: wait for any audio response */

short undersampling;	/* conversion factor */
short samples_to_do;	/* loop variable for conversion */
int Halved;		/* interpolate due to non integral divider */

static long lsum = 0, rsum = 0;	       /* accumulator for left/right channel */
static long ls2 = 0, rs2 = 0, ls3 = 0, rs3 = 0, auxl = 0, auxr = 0;

#if !defined (__USE_GNU)
#undef USE_GNUMEMMEM
#endif
#ifndef USE_GNUMEMMEM
#if 0
/* smarter but slower than brute force :-( */
static void 
init_search(const unsigned char * NEEDLE, size_t NEEDLE_LEN, size_t shift[])
{
  size_t *sp;
  size_t c;

  NEEDLE_LEN++;
  sp = shift;

  /* initialize jump table for characters non existent in NEEDLE */
  for ( c = 1 + (size_t)UCHAR_MAX; c-- > 0; ) *sp++ = NEEDLE_LEN;

  /* initialize jump table for characters existent in NEEDLE */
  for (; --NEEDLE_LEN != 0; shift[(int)*(unsigned char *)(NEEDLE++)] = NEEDLE_LEN );
}

static const unsigned char *
my_memmem (const unsigned char * const HAYSTACK, const size_t HAYSTACK_LEN,
	   const unsigned char * const NEEDLE, const size_t NEEDLE_LEN)
{
  const unsigned char * const UPPER_LIMIT = HAYSTACK + HAYSTACK_LEN;

  static size_t shift[1 + (int)UCHAR_MAX];
  const unsigned char *cp;
  
  if (HAYSTACK_LEN < NEEDLE_LEN) return NULL;
  if (HAYSTACK == NULL || HAYSTACK_LEN == 0) return NULL;
  if (NEEDLE == NULL || NEEDLE_LEN == 0) return HAYSTACK;

  init_search(NEEDLE, NEEDLE_LEN, shift);

  cp = HAYSTACK + NEEDLE_LEN;
  while (cp <= (const unsigned char *) UPPER_LIMIT) {
    const unsigned char *p = NEEDLE;
    const unsigned char *t = cp - NEEDLE_LEN;

    do {
      if ((NEEDLE + NEEDLE_LEN) == p) {
	  return (void *) (cp - NEEDLE_LEN);
      }
    } while ( *p++ == *t++ );
    cp += shift[(int)*cp];
  }
  return NULL;
}

#else
/* A simple brute force approach */
/* A bit faster than gnu memmem and much faster than the smart memmem above */
static const unsigned char *
my_memmem (const unsigned char * HAYSTACK, const size_t HAYSTACK_LEN,
	   const unsigned char * const NEEDLE, const size_t NEEDLE_LEN)
{
  const unsigned char * const UPPER_LIMIT = HAYSTACK + HAYSTACK_LEN - NEEDLE_LEN;

  if (HAYSTACK_LEN < NEEDLE_LEN) return NULL;
  if (HAYSTACK == NULL || HAYSTACK_LEN == 0) return NULL;
  if (NEEDLE == NULL || NEEDLE_LEN == 0) return HAYSTACK;

  while (HAYSTACK <= UPPER_LIMIT) {
    if (memcmp(NEEDLE, HAYSTACK, NEEDLE_LEN) == 0) {
      return HAYSTACK;
    } else {
      HAYSTACK++;
    }
  }
  return NULL;
}
#endif
#else
#define my_memmem memmem
#endif

/* find continuation in new buffer */
static const unsigned char *
sync_buffers(const unsigned char * const newbuf)
{
    const unsigned char *retval = newbuf;

    if (global.overlap != 0) {
      /* find position of SYNC_SIZE bytes 
	 of the old buffer in the new buffer */
      size_t haystack_len;
      size_t needle_len   = SYNC_SIZE;
      const unsigned char * const oldbuf = (const unsigned char *) (get_previous_read_buffer()->data);
      const unsigned char * needle = oldbuf + global.nsectors*CD_FRAMESIZE_RAW - SYNC_SIZE - SYNC_OFF;
      const unsigned char * haystack;

      /* compare the end of the previous buffer with the beginning
       * of the new one
       *
       * At first we try from the exact position.
       * When no match occurs, we search in the head part of the new buffer.
       * When no match occurs, we search in the tail part of the new buffer.
       * When no match occurs, the sequence cannot be found.
       */

      haystack = newbuf + global.overlap*CD_FRAMESIZE_RAW - SYNC_SIZE - SYNC_OFF;

#ifdef DEBUG_SHIFTS
#define SIM_SHIFTBYTES (-2)
      retval =  haystack + SYNC_SIZE + SYNC_OFF + SIM_SHIFTBYTES;
#else
      if (memcmp(haystack, needle, needle_len) == 0) {
	retval = haystack + SYNC_SIZE + SYNC_OFF;
      } else {
	/* not exact */
	haystack = newbuf;
	haystack_len = global.nsectors*CD_FRAMESIZE_RAW;
	retval = my_memmem(haystack, haystack_len, needle, needle_len);

	if (retval) {
          retval += SYNC_SIZE + SYNC_OFF;
	}
      }

#endif

#ifdef SHOW_JITTER
      if (retval) {
	fprintf(stderr,"matched\tj1=%d\t",
		retval-(newbuf+global.overlap*CD_FRAMESIZE_RAW));
      } else {
	fprintf(stderr,"no match\n");
      }
#endif
    }

    return retval;
}

/* quadratic interpolation
 * p1, p3 span the interval 0 - 2. give interpolated value for 1/2 */
static long int 
interpolate( long int p1, long int p2, long int p3)
{
  return (3L*p1 + 6L*p2 - p3)/8L;
}

static int any_signal = 0;
static unsigned char *pStart;	/* running ptr defining end of output buffer */
static unsigned char *pDst;	/* start of output buffer */
/*
 * Write the filtered sample into the output buffer.
 */
static void 
emit_sample( long lsumval, long rsumval, long channels )
{
    if (global.findminmax) {
       if (lsumval < global.minamp) global.minamp = lsumval;
       if (lsumval > global.maxamp) global.maxamp = lsumval;
       if (rsumval > global.maxamp) global.maxamp = rsumval;
       if (rsumval < global.minamp) global.minamp = rsumval;
    }
    /* convert to output format */
    if ( channels == 1 ) {
	short sum;       /* mono section */
	sum = ( lsumval + rsumval ) >> (global.sh_bits + 1);
	if ( global.sh_bits == 8 ) {
	    if ( ( (char) sum) != '\0' ) {
		if ( any_signal == 0 ) {
		    pStart = (unsigned char *) pDst;
		    any_signal = 1;
		}
	    }
	    *pDst++ = ( unsigned char ) sum + ( 1 << 7 );
	} else {
	    short * myptr = (short *) pDst;
	    if ( sum != 0 ) {
		if ( any_signal == 0 ) {
		    pStart = (unsigned char *) pDst;
		    any_signal = 1;
		}
	    }
	    *myptr = sum;
	    pDst += sizeof( short );
	}
    } else {
	/* stereo section */
	lsumval >>= global.sh_bits;
	rsumval >>= global.sh_bits;
	if ( global.sh_bits == 8 ) {
	    if ( (( char ) lsumval != '\0') || (( char ) rsumval != '\0')) {
		if ( any_signal == 0 ) {
		    pStart = (unsigned char *) pDst;
		    any_signal = 1;
		}
	    }
	    *pDst++ = ( unsigned char )( short ) lsumval + ( 1 << 7 );
	    *pDst++ = ( unsigned char )( short ) rsumval + ( 1 << 7 );
	} else {
	    short * myptr = (short *) pDst;
	    if ( (( short ) lsumval != 0) || (( short ) rsumval != 0)) {
		if ( any_signal == 0 ) {
		    pStart = (unsigned char *) pDst;
		    any_signal = 1;
		}
	    }
	    *myptr++ = ( short ) lsumval;
	    *myptr   = ( short ) rsumval;
	    pDst += 2*sizeof( short );
	}
    }
}

static void
change_endianness(UINT4 *pSam, unsigned int Samples)
{
  UINT4 *pend = (pSam + Samples);

  /* type UINT4 may not be greater than the assumed biggest type */
#if (SIZEOF_LONG_UNSIGNED_INT < 4)
#error type unsigned long is too small
#endif

#if (SIZEOF_LONG_UNSIGNED_INT == 4)

#if !(defined __GNUC__ && defined i386) || defined PORTABLE
  unsigned long *plong = (unsigned long *)pSam;
#endif

#if !(defined __GNUC__ && defined i386) || defined PORTABLE
  for (; plong < pend;)
    *plong++ = ((*plong >> 8L) & 0x00ff00ffUL) |
               ((*plong << 8L) & 0xff00ff00UL);
#else
  __asm__ ("   cld\n\t"
           "1:"
#if 0 /* for 486 or newer */
	   "   movl (%%edi),%%eax\n\t"
	   "   bswap     %%eax\n\t"
	   "   rorl $16, %%eax\n\t"
	   "   stosl\n\t"
#else
	   "   movl (%%edi),%%eax\n\t"
	   "   rorw $8,  %%ax\n\t"
	   "   rorl $16, %%eax\n\t"
	   "   rorw $8,  %%ax\n\t"
	   "   rorl $16, %%eax\n\t"
	   "   stosl\n\t"
#endif
	   "   cmpl %%edx,%%edi\n\t"
	   "   jb 1b\n\t":
	   :"D" ((unsigned long) pSam),
	    "d" ((unsigned long) pend));
#endif
#else  /* sizeof long unsigned > 4 bytes */
#if (SIZEOF_LONG_UNSIGNED_INT == 8)
#define INTEGRAL_LONGS (SIZEOF_LONG_UNSIGNED_INT-1UL)
  register unsigned long *plong;
  unsigned long *pend0 = (unsigned long *) (((unsigned long) pend) & ~ INTEGRAL_LONGS);

  if (((unsigned long) pSam) & INTEGRAL_LONGS)
    *pSam++ = ((*pSam >> 8L) & 0x00ff00ffUL) |
               ((*pSam << 8L) & 0xff00ff00UL);

  plong = (unsigned long *)pSam;

  for (; plong < pend0;)
    *plong++ = ((*plong >> 8L) & 0x00ff00ff00ff00ffUL) |
               ((*plong << 8L) & 0xff00ff00ff00ff00UL);

  if (((unsigned long *) pend) != pend0) {
    UINT4 *pint = (UINT4 *) pend0;

    for (;pint < pend;)
      *pint++ = ((*pint >> 8) & 0x00ff00ffU) |
                ((*pint << 8) & 0xff00ff00U);
  }
#else  /* sizeof long unsigned > 4 bytes but not 8 */
  {
    UINT4 *pint = pSam;

    for (;pint < pend;)
      *pint++ = ((*pint >> 8) & 0x00ff00ffU) |
                ((*pint << 8) & 0xff00ff00U);
  }
#endif
#endif
}

static long ReSampleBuffer( unsigned char *p, unsigned char *newp, long samples)
{
       double idx=0;
       long    di=0,si=0;

       while( si < samples ){
               memcpy( newp+(di*4), p+(si*4), 4 );
               idx += (double)(global.playback_rate/100.0);
               si = (long)idx;
               di++;
       }
       return di;
}

#define DEBUG_VOTE_ENDIANESS
static int guess_endianess(UINT4 *p, short *p2, unsigned SamplesToDo)
{
    /* analyse samples */
    int vote_for_little = 0;
    int vote_for_big = 0;
    int total_votes;

    while (((UINT4 *)p2 - p) + 1U < SamplesToDo) {
      unsigned char *p3 = (unsigned char *)p2;
      int diff_lowl = *(p2+0) - *(p2+2);
      int diff_lowr = *(p2+1) - *(p2+3);
      int diff_bigl = ((*(p3  ) << 8) + *(p3+1)) - ((*(p3+4) << 8) + *(p3+5));
      int diff_bigr = ((*(p3+2) << 8) + *(p3+3)) - ((*(p3+6) << 8) + *(p3+7));

      if ((abs(diff_lowl) + abs(diff_lowr)) <
	  (abs(diff_bigl) + abs(diff_bigr))) {
	vote_for_little++;
      } else {
	if ((abs(diff_lowl) + abs(diff_lowr)) >
	    (abs(diff_bigl) + abs(diff_bigr))) {
	  vote_for_big++;
	}
      }
      p2 += 2;
   }
#ifdef DEBUG_VOTE_ENDIANESS
   if (global.quiet != 1)
     fprintf(stderr, "votes for little: %4d,  votes for big: %4d\n", 
		vote_for_little, vote_for_big);
#endif
   total_votes = vote_for_big + vote_for_little;
   if (total_votes == 0
       || abs(vote_for_big - vote_for_little) < total_votes/3) {
     return -1;
   } else {
	if (vote_for_big > vote_for_little)
		return 1;
	else
		return 0;
   }
}

int jitterShift = 0; 

unsigned char *synchronize(UINT4 *p, unsigned SamplesToDo, unsigned TotSamplesDone)
{
  static int jitter = 0;
  char *pSrc;                   /* start of cdrom buffer */

  /* if endianess is unknown, guess endianess based on 
     differences between succesive samples. If endianess
     is correct, the differences are smaller than with the
     opposite byte order.
   */
  if (global.littleendian < 0) {
    short *p2 = (short *)p;

    /* skip constant samples */
    while ((((UINT4 *)p2 - p) + 1U < SamplesToDo)
           && *p2 == *(p2+2)) p2++;

    if (((UINT4 *)p2 - p) + 1U < SamplesToDo) {
      switch (guess_endianess(p, p2, SamplesToDo)) {
        case -1: break;
        case  1: global.littleendian = 0;
	         if (global.quiet != 1)
		   fprintf(stderr, "big endian detected\n");
	break;
        case  0: global.littleendian = 1;
	         if (global.quiet != 1)
		   fprintf(stderr, "little endian detected\n");
	break;
      }
    }
  }

  /* ENDIAN ISSUES:
   * the individual endianess of cdrom/cd-writer, cpu, 
   * sound card and audio output format need a careful treatment.
   *
   * For possible sample processing (rate conversion) we need
   * the samples in cpu byte order. This is the first conversion.
   *
   * After processing it depends on the endianness of the output
   * format, whether a second conversion is needed.
   *
   */

  if (global.littleendian != MY_LITTLE_ENDIAN) {
    /* change endianess of delivered samples to native cpu order */
    change_endianness(p, SamplesToDo);
  }

  /* synchronisation code */
  if (TotSamplesDone != 0 && global.overlap != 0 && SamplesToDo > CD_FRAMESAMPLES) {

    pSrc = (char *) sync_buffers((unsigned char *)p);
    if (!pSrc ) {
#if 0

#define OLD_READ "__buff_old"
#define NEW_READ "__buff_new"
      { FILE *b1 = fopen(OLD_READ, "wb");
	FILE *b2 = fopen(NEW_READ, "wb");
	unsigned char *oldbuf = get_previous_read_buffer()->data;
	if (b1 != NULL && b2 != NULL) {
	  fprintf(b1,"old: version "VERSION", interface %5.5s, nsectors %2u, "
		      "overlap %1u, SYNC_SIZE %4ld, BytesToDo %6u\n",
		      interface == GENERIC_SCSI ? "SCSI" : "ATAPI",
		      global.nsectors, global.overlap, SYNC_SIZE, global.nsectors*CD_FRAMESIZE_RAW);
	  if (fwrite(oldbuf, 1, global.nsectors*CD_FRAMESIZE_RAW, b1) !=
	                        global.nsectors*CD_FRAMESIZE_RAW)
	    perror("write error dump1:");
	  (void) fclose(b1);

	  fprintf(b2,"new: version "VERSION", interface %5.5s, nsectors %2u, "
		     "overlap %1u, SYNC_SIZE %3ld, BytesToDo %6ld\n",
		  interface == GENERIC_SCSI ? "SCSI" : "ATAPI",
		  global.nsectors, global.overlap, SYNC_SIZE, SamplesToDo*4);
	  if (fwrite(p, 1, (size_t) SamplesToDo*4, b2) !=
	                   (size_t) SamplesToDo*4)
	    perror("write error dump2:");
	  (void) fclose(b2);

	  fprintf(stderr, "Could not synchronize successive reads.\n"
		          "Please email the files "OLD_READ" and "NEW_READ"\n"
                          "in uuencoded form to heiko@colossus.escape.de\n");
	  return NULL;
	}
      }
#else
#if 0
      fprintf(stderr, "no match: increase overlap (%u) or decrease SYNC_SIZE (%ld).\n", global.overlap, SYNC_SIZE);
      fprintf(stderr, "no match: at position %ld\n",
	      TotSamplesDone);
#else
      global.nosynccount++;
#endif
#endif
      return NULL;
    }
    if (pSrc) {
      jitter = ((unsigned char *)pSrc - (((unsigned char *)p) + global.overlap*CD_FRAMESIZE_RAW))/4;
      jitterShift += jitter;
      SamplesToDo -= jitter + global.overlap*CD_FRAMESAMPLES;
#if 0
      fprintf(stderr,
	    "Length: pre %d, diff1 %ld, diff2 %ld, min %ld\n", SamplesToDo,
	   (TotSamplesWanted - TotSamplesDone),
	   SamplesNeeded((TotSamplesWanted - TotSamplesDone), undersampling),
	   min(SamplesToDo, SamplesNeeded((TotSamplesWanted - TotSamplesDone), undersampling)));
#endif
    }
  } else {
    pSrc = ( char * ) p;
  }
  return (unsigned char *) pSrc;
}

/* convert cdda data to required output format
 * sync code for unreliable cdroms included
 * 
 */
long 
SaveBuffer (UINT4 *p,
	    unsigned long SamplesToDo, 
	    unsigned long *TotSamplesDone)
{
  UINT4 *pSrc;                   /* start of cdrom buffer */
  UINT4 *pSrcStop;               /* end of cdrom buffer */

  /* in case of different endianness between host and output format,
     copy in a seperate buffer and modify the local copy */
  if (global.need_big_endian != MY_BIG_ENDIAN && global.OutSampleSize > 1) {
     static UINT4 *localoutputbuffer;
     if (localoutputbuffer == NULL) {
       localoutputbuffer = malloc(global.nsectors*CD_FRAMESIZE_RAW);
       if (localoutputbuffer == NULL) {
         perror("cannot allocate local buffer");
         return 1;
       }
     }
     memcpy(localoutputbuffer, p, SamplesToDo*4);
     p = localoutputbuffer;
  }

  pSrc = p;
  pDst = (unsigned char *) p;
  pStart = ( unsigned char * ) pSrc;
  pSrcStop = pSrc + SamplesToDo;
#ifdef SHOW_JITTER
  if (global.overlap)
    fprintf(stderr, "Total: %d, Jitter: %d\n", jitterShift, jitter);
#endif

  /* code for subsampling and output stage */

  if (global.ismono && global.findmono) {
    short *pmm;
    for (pmm = (short *)pStart; (UINT4 *) pmm < pSrcStop; pmm += 2) {
      if (*pmm != *(pmm+1)) {
        global.ismono = 0;
        break;
      }
    }
  }
  /* optimize the case of no conversion */
  if ( undersampling == 1 && samples_to_do == 1 &&
       global.channels == 2 && global.OutSampleSize == 2 && Halved == 0) {
    /* output format is the original cdda format ->
     * just forward the buffer 
     */
      
    if ( waitforsignal != 0 && any_signal == 0) {
      int *myptr = (int *)pStart;
      while ((UINT4 *)myptr < pSrcStop && *myptr == 0) myptr++;
      pStart = (unsigned char *) myptr;
      /* scan for first signal */
      if ( (UINT4 *)pStart != pSrcStop ) {
	/* first non null amplitude is found in buffer */
	any_signal = 1;
      }
    }
    pDst = (unsigned char *) pSrcStop;		/* set pDst to end */
    if (global.findminmax) {
      short *pmm;
      for (pmm = (short *)pStart; pmm < (short *)pDst; pmm++) {
        if (*pmm < global.minamp) global.minamp = *pmm;
        if (*pmm > global.maxamp) global.maxamp = *pmm;
        pmm++;
        if (*pmm > global.maxamp) global.maxamp = *pmm;
        if (*pmm < global.minamp) global.minamp = *pmm;
      }
    }
  } else {

#define none_missing	0
#define one_missing	1
#define two_missing	2
#define collecting	3

    static int sample_state = collecting;
    static int Toggle_on = 0;

    /* conversion required */
    while ( pSrc < pSrcStop ) {
	  
	long l,r;

	long iSamples_left = (pSrcStop - pSrc) / sizeof(signed short) / 2;
	short *myptr = (short *) pSrc;

	/* LSB l, MSB l */
	l = *myptr++;	/* left channel */
	r = *myptr++;	/* right channel */
	pSrc = (UINT4 *) myptr;

	switch (sample_state) {
	case two_missing:
two__missing:
	    ls2 += l; rs2 += r;
	    if (undersampling > 1) {
		ls3 += l; rs3 += r;
	    }
	    sample_state = one_missing;
	    break;
	case one_missing:
	    auxl = l; auxr = r;

	    ls3 += l; rs3 += r;
	    sample_state = none_missing;

	    /* FALLTHROUGH */
none__missing:
	case none_missing:
	    /* Filtered samples are complete. Now interpolate and scale. */

	    if (Halved != 0 && Toggle_on == 0) {
                lsum = interpolate(lsum, ls2, ls3)/(int) undersampling;
	        rsum = interpolate(rsum, rs2, rs3)/(int) undersampling;
            } else {
		lsum /= (int) undersampling;
		rsum /= (int) undersampling;
            }
	    emit_sample(lsum, rsum, global.channels);
	    /* reload counter */
	    samples_to_do = undersampling - 1;
	    lsum = auxl;
	    rsum = auxr;
	    /* reset sample register */
	    auxl = ls2 = ls3 = 0;
	    auxr = rs2 = rs3 = 0;
	    Toggle_on ^= 1;
	    sample_state = collecting;
	    break;
	case collecting:
	    if ( samples_to_do > 0) {
		samples_to_do--;
		if (Halved != 0 && Toggle_on == 0) {
		    /* Divider x.5 : we need data for quadratic interpolation */
		    iSamples_left--;

		    lsum += l; rsum += r;
		    if ( samples_to_do < undersampling - 1) {
			ls2 += l; rs2 += r;
		    }
		    if ( samples_to_do < undersampling - 2) {
			ls3 += l; rs3 += r;
		    }
		} else {
		    /* integral divider */
		    lsum += l;
		    rsum += r;
		    iSamples_left--;
		}
	    } else {
	        if (Halved != 0 && Toggle_on == 0) {
		    sample_state = two_missing;
		    goto two__missing;
		} else {
		    auxl = l;
		    auxr = r;
		    sample_state = none_missing;
		    goto none__missing;
		}
	    }
	    break;
	} /* switch state */

    } /* while */

    /* flush_buffer */
    if ((samples_to_do == 0 && Halved == 0))
    {
	if (Halved != 0 && Toggle_on == 0) {
	    lsum = interpolate(lsum, ls2, ls3)/(int) undersampling;
	    rsum = interpolate(rsum, rs2, rs3)/(int) undersampling;
	} else {
	    lsum /= (int) undersampling;
	    rsum /= (int) undersampling;
	}
	emit_sample(lsum, rsum, global.channels);
	
	/* reload counter */
	samples_to_do = undersampling;
	
	/* reset sample register */
	lsum = auxl = ls2 = ls3 = 0;
	rsum = auxr = rs2 = rs3 = 0;
	Toggle_on ^= 1;
	sample_state = collecting;
    }

  } /* if optimize else */

  if ( waitforsignal == 0 ) pStart = (unsigned char *)p;
  else if (any_signal != 0) global.SkippedSamples += ((UINT4 *)pStart - p);
  else global.SkippedSamples += (pSrcStop - p);

  if ( waitforsignal == 0 || any_signal != 0) {
    int retval = 0;
    unsigned outlen;

    assert(pDst >= pStart);
    outlen = (size_t) (pDst - pStart);

    if (outlen <= 0) return 0;

#ifdef	ECHO_TO_SOUNDCARD

    /* this assumes the soundcard needs samples in native cpu byte order */
    if (global.echo != 0) {
#include <sys/time.h>

               static unsigned char *newp;
               unsigned    newlen;

               newlen = (100*(outlen/4))/global.playback_rate;
               newlen = (newlen*4);
               if ( (newp != NULL) || (newp = malloc( newlen+32 )) ) {
			newlen = 4*ReSampleBuffer( pStart, newp, outlen/4 );
			{
				unsigned todo = newlen;
				unsigned char *newp2 = newp;
				int retval2;
				do {
					fd_set writefds[1];
					struct timespec timeout = { 0, 120000000 };
					struct timeval timeout2 = { 0, 120000 };
					int wrote;

					FD_ZERO(writefds);
					FD_SET(global.soundcard_fd, writefds);
					retval2 = select(global.soundcard_fd + 1, NULL, writefds, NULL, &timeout2);
					switch (retval2) {
					default:
					case -1: perror ("select failed");
						/* fall through */
					case 0: /* timeout */
						goto outside_loop;
						break;
					case 1: break;
					}
					wrote = write(global.soundcard_fd, newp2, todo);
					if (wrote <= 0) {
						perror( "cant write audio");
						goto outside_loop;
					} else {
						todo -= wrote;
						newp2 += wrote;
					}
				} while (todo > 0);
outside_loop:
			;
			}
               }
    }
#endif

    if ( global.no_file != 0 ) {
        *TotSamplesDone += SamplesToDo;
        return 0;
    }
    if ( global.need_big_endian != MY_BIG_ENDIAN) {
      if ( global.OutSampleSize > 1) {
        /* change endianness from native cpu order 
           to required output endianness */
        change_endianness((UINT4 *)pStart, outlen/4);
      }
    }
    if ((unsigned)(retval = write ( global.audio, pStart, outlen )) == outlen) {
        *TotSamplesDone += SamplesToDo;
	return 0;
    } else {
        fprintf(stderr, "write(audio, 0x%p, %u) = %d\n",pStart,outlen,retval);
        perror("Probably disk space exhausted");
        return 1;
    }
  } else return 0;
}


