/* gradio-main.c
 * Contains functions to build and display the GRadio interface,
 * and for reading/writing the .gradiorc file and initializing the program
 */

/* (c) 1998 by Keith Wesolowski (wesolows@cs.unr.edu) 

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. 

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details. 

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 */

#include <gtk/gtk.h>
#include "gradio.h"

radio_status gr_stat;

void gradio_init (void)
{
    /* This function could use some work */

    FILE *rcfile;
    char rcfilename[256];
    char *homedir = getenv ("HOME");
    int i, temp;
    
    /* Let there br POWER */
    gr_stat.power = TRUE;
    gr_stat.memset = -1;

    /* Read in (or create) the rcfile */
    /* The rcfile format is as follows:
     * {ioport}<newline>
     * {current frequency}<newline>
     * {volume level}<tab>{muted}<newline>
     * {memory freq 1}<newline>
     * .
     * .
     * .
     * {memory freq 8}<newline>
     */

    /* Yes, the volume to zero and back up is annoying if the radio is on.
     * However, it is also the only way to ensure a consistent volume
     * reading.
     */
    sprintf (rcfilename, "%s/.gradiorc", (homedir != NULL) ? homedir : ".");
    if ((rcfile = fopen (rcfilename, "r")) == NULL) {
	/* do the lowlevel stuff */
	gr_stat.port = 0;
	gradio_get_interface ();
	if (gr_stat.interface == GR_IF_NONE) {
	    printf ("No interface found\n");
	    exit (1);
	}
	if ((rcfile = fopen (rcfilename, "w")) == NULL)
	    return;
	fprintf (rcfile, "%x\n", (gr_stat.interface == GR_IF_KERNEL) ? 0 : gr_stat.port);
	fprintf (rcfile, "%d\n", GR_FREQ_DEFAULT);
	fprintf (rcfile, "0 0\n");
	for (i=0; i<8; i++) {
	    fprintf (rcfile, "%d\n", GR_FREQ_DEFAULT);
	    gr_stat.mem[i] = GR_FREQ_DEFAULT;
	}
	fclose (rcfile);
	gr_stat.vol = GR_VOL_MAX;
	gradio_setvol (0);
	gr_stat.vol = 0;
	gradio_setvol (5);
	gradio_setfreq (GR_FREQ_DEFAULT);
	gradio_mute (FALSE); 
	return;
    }
    fscanf (rcfile, "%x", &(gr_stat.port));
    /* do the lowlevel stuff */
    gradio_get_interface ();
    if (gr_stat.interface == GR_IF_NONE) {
	printf ("No interface found!\n");
	exit (1);
    }
    
    gr_stat.vol = GR_VOL_MAX;
    gradio_setvol (1);
    fscanf (rcfile, "%d", &(gr_stat.freq));
    fscanf (rcfile, "%d", &temp);
    fscanf (rcfile, "%d", &(gr_stat.muted));
    for (i=0; i<8; i++)
	fscanf (rcfile, "%d", &(gr_stat.mem[i]));
    gradio_setfreq (gr_stat.freq);
    gradio_mute (gr_stat.muted);
    gradio_setvol (temp);
}

void gradio_save_rcfile (void)
{
    FILE *rcfile;
    char rcfilename[256];
    char *homedir = getenv ("HOME");
    int i;

    sprintf (rcfilename, "%s/.gradiorc", (homedir != NULL) ? homedir : ".");
    if ((rcfile = fopen (rcfilename, "w")) == NULL)
	return;
    fprintf (rcfile, "%x\n", (gr_stat.interface == GR_IF_KERNEL) ? 0 : gr_stat.port);
    fprintf (rcfile, "%d\n", gr_stat.freq);
    fprintf (rcfile, "%d %d\n", gr_stat.vol, gr_stat.muted);
    for (i=0; i<8; i++)
	fprintf (rcfile, "%d\n", gr_stat.mem[i]);
    fclose (rcfile);
}

int gradio_check_cmdline (int argc, char *argv[])
{
    int i;

    if (argc == 1)
	return FALSE;
    if (strcmp (argv[1], "-nw") || argc == 2) {
	gradio_usage (argv[0]);
	return TRUE;
    }
    for (i=1; i<argc; ) {
	switch (argv[i++][0]) {
	case 'v':
	    gradio_setvol (atoi (argv[i++]));
	    break;
	case 'f':
	    gradio_setfreq ((int)(atof (argv[i++]) * 1000. + 0.5));
	    break;
	case '-':
	    gradio_mute (TRUE);
	    break;
	case '+':
	    gradio_mute (FALSE);
	    break;
	default:
	    gradio_usage (argv[0]);
	    return TRUE;
	}
    }
    return TRUE;
}

void gradio_usage (char *name)
{
    fprintf (stderr,
	     "Usage: %s [-nw [v n] [f n] [+|-]]\n"
	     "Options:\n\n"
	     "-nw\t\tDo not use the X interface\n"
	     " v n\t\tSet volume to n, where n is an integer 0-11\n"
	     " f n\t\tSet frequency to n MHz\n"
	     " +\t\tTurn on/unmute the radio\n"
	     " -\t\tTurn off/mute the radio\n", name);
}

int main (int argc, char *argv[])
{
    GtkWidget *main_window;
    GtkWidget *main_box;
    GtkWidget *main_box_left;
    GtkWidget *main_box_right;
    GtkWidget *main_box_bottom;
    GtkWidget *volume_box;
    GtkWidget *tuner_box;
    GtkWidget *memory_box;
    GtkWidget *control_box;
    GtkWidget *display;
    /*    int i; */

    gradio_init ();
    
    /* This returns TRUE if the gtk interface is not being used,
     * and if that is the case, it does whatever is needed, so we can quit
     * here.
     */
    if (gradio_check_cmdline (argc, argv)) {
	gradio_save_rcfile ();
	return 0;
    }

    gtk_init (&argc, &argv);

    /* main_window inits */
    main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (main_window), "GRadio");
    gtk_window_position (GTK_WINDOW (main_window), GTK_WIN_POS_NONE);
    gtk_window_set_policy (GTK_WINDOW (main_window), FALSE, FALSE, TRUE);
    gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
			GTK_SIGNAL_FUNC (gradio_delete_event_cb), NULL);
    gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
			GTK_SIGNAL_FUNC (gradio_destroy_cb), NULL);
    /* ends (main_window inits) */

    main_box = gtk_hbox_new (FALSE, 2);
    main_box_left = gtk_vbox_new (FALSE, 2);
    main_box_right = gtk_vbox_new (FALSE, 2);
    main_box_bottom = gtk_hbox_new (FALSE, 2);

    gtk_widget_realize (main_window);

    /* let other functions build the rest */
    display = gradio_get_display (main_window); /* Must be first for inits */
    volume_box = gradio_get_volume_box (main_window);
    tuner_box = gradio_get_tuner_box (main_window);
    memory_box = gradio_get_memory_box (main_window);
    control_box = gradio_get_control_box (main_window);
    
    gtk_widget_show (display);
    gtk_widget_show (tuner_box); 
    gtk_widget_show (control_box);
    gtk_widget_show (memory_box);
    gtk_widget_show (volume_box); 
    
    gtk_container_border_width (GTK_CONTAINER (main_window), 6);

    gtk_box_pack_start (GTK_BOX (main_box_bottom), control_box, TRUE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_box_bottom), memory_box, TRUE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_box_left), display, TRUE, FALSE, 5);
    gtk_box_pack_start (GTK_BOX (main_box_left), main_box_bottom, TRUE, FALSE, 5);
    gtk_box_pack_start (GTK_BOX (main_box_right), tuner_box, FALSE, FALSE, 14);
    gtk_box_pack_end (GTK_BOX (main_box_right), volume_box, FALSE, FALSE, 5);

    gtk_widget_show (main_box_bottom);
    gtk_widget_show (main_box_left);
    gtk_widget_show (main_box_right);

    gtk_box_pack_start (GTK_BOX (main_box), main_box_left, TRUE, FALSE, 4);
    gtk_box_pack_start (GTK_BOX (main_box), main_box_right, TRUE, FALSE, 4);
    gtk_widget_show (main_box);

    gtk_container_add (GTK_CONTAINER (main_window), main_box);

    gtk_widget_show (main_window);

    gtk_main();

    return 0;
}
