/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>

#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "sample_settings.h"
#include "filebrowse.h"
#include "misc.h"
#include "menu.h"
#include "audio_file.h"
#include "status.h"
#include "edit.h"
#include "graphics.h"
#include "fileop.h"

extern Main_Data *MD; 
extern AppResources app_resources;

static int saveas_file(char *fname,int type,int comp);
static int save_as_file(char *absname,int type,int comp);
static int saveas_close_file(char *fname,int type,int comp);
static void new_return(Sample_Return *sr);
static void release_canvas(Next_Wave *nw);
static void load_raw(Sample_Return *sr);
static void save_close_file(Widget w,Main_Data *md);
static void im_close_file(Widget w,Main_Data *md);
static int saveas_close_quit(char *fname,int type,int comp);
static void save_close_quit(Widget w,Main_Data *md);
static void close_quit(Widget w,Main_Data *md);


/* to load raw files */
static char *rawname=NULL;

int load_file(char *fname,int type,int comp)
{
    Audio_File af;
    Wave_Data *wd;
    Next_Wave *nw;
    int af_type;
    af_type=af_open(fname,&af,AF_OPEN);
    if (af_type==AF_RAW) {
	af_close(af);
	XtFree(rawname);
	rawname=NULL;
	/* can't load raw files from cmdline */
	if (type==AF_NORAW) return(AF_RAW);
	rawname=get_abs_name(fname);
	if (rawname==NULL) rawname=XtNewString(fname);
	sample_dialog (load_raw,NULL);
	return(0);
    }
    
    if (af_type==AF_NOTSUPPORTED) {
	if (type==AF_NORAW) return(AF_NOTSUPPORTED);
	sprintf(MD->mw->messages,"%s <%s>\n",
		app_resources.err_notsupported,fname);
	warn_popup(MD->mw->form,MD->mw->messages);
	return(AF_NOTSUPPORTED);
    }
    
    
    sprintf(MD->mw->messages,"%s <%s>\n",app_resources.err_openfile,fname);
    
    if (af_type==AF_ERROR) {
	warn_popup(MD->mw->form,MD->mw->messages);
	return(-1);
    }
    
    if ((wd=(Wave_Data*) XtCalloc(1,sizeof(Wave_Data)))==NULL) {
	warn_popup(MD->mw->form,MD->mw->messages);
	close(af.fd);
	return(-1);
    }
    
    watch_cursor(True);
    
    if (af.length<MD->maxmem) {
	if ((wd->buffer = (byte *) malloc(af.length)) == NULL) {
	    watch_cursor(False);
	    warn_popup(MD->mw->form,MD->mw->messages);
	    close(af.fd);
	    XtFree((char*) wd);
	    return(-1);
	}
	if (af_read(af,(char*) wd->buffer,af.length)==AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(MD->mw->form,MD->mw->messages);
	    close(af.fd);
	    free(wd->buffer);
	    XtFree((char*) wd);
	    return(-1);
	}
	close(af.fd);
    } else {
	af_rewind(af);
	wd->actual_name=NULL;
	wd->actual_no=0;
    }
    
    af2wd(af,wd);
    wd->name=get_abs_name(fname);
    if (wd->name==NULL) wd->name=XtNewString(fname);
    
    wd->markbeg=0;
    wd->marklength=0;
    wd->lines_l=NULL;
    wd->lines_r=NULL;
    
    /* now we can enter all the values in the Global Main_Data variable */
    if (MD->no>0) {
	nw=MD->nw;
	while (nw->nw!=NULL) nw=nw->nw;
	nw->nw=XtNew(Next_Wave);   
	nw=nw->nw;
    } else {
	MD->nw=XtNew(Next_Wave);
	nw=MD->nw;
    }
    
    nw->wd=wd;
    nw->cw=XtNew(Canvas_Widget);
    nw->cb=XtNew(Canvas_Bool);
    nw->cg=XtNew(Canvas_Graphic);
    nw->nw=NULL;
    
    if (nw->wd->length>MD->maxmem) nw->wd->inmem=FALSE;
    else nw->wd->inmem=TRUE;
    nw->wd->ismark=False;
    nw->wd->isplay=False;
    nw->cb->canwrite=can_write(nw->wd->name);
    nw->cb->modified=False;
    nw->cb->update=False;
    nw->cg->step=1;
    nw->cg->pos=0;
    
    MD->no++;
    MD->wd=nw->wd;
    MD->cb=nw->cb;
    MD->cw=nw->cw;
    MD->cg=nw->cg;
    
    new_canvas(MD->mw->form,nw);
    update_canvas(MD,NULL);
    update_display(MD);
    watch_cursor(False);
    return(af_type);
}

void load_raw(Sample_Return *sr)
{
    Wave_Data *wd;
    Next_Wave *nw;
    Audio_File af;
    
    if (af_open(rawname,&af,AF_OPEN)==AF_ERROR) return;
    
    if ((wd=(Wave_Data*) XtCalloc(1,sizeof(Wave_Data)))==NULL) {
	af_close(af);
	return;
    }
    
    wd->name=rawname;
    wd->res=sr->res;
    wd->freq=sr->freq;
    wd->channels=sr->channels;
    wd->bpspl=(wd->res*wd->channels)/8;
    wd->headoffs=af.headoffs;
    wd->type=af.type;
    wd->comp=af.comp;
    wd->length=af.length;
    /* length modulo byte per sample */
    wd->length+=(wd->length % wd->bpspl);
    wd->tlength=wd->length/wd->bpspl;
    wd->markbeg=0;
    wd->marklength=0;
    
    if (wd->length<MD->maxmem) {
	if ((wd->buffer = (byte *) malloc(wd->length)) == NULL) {
	    XtFree((char*) wd);
	    af_close(af);
	    return;
	}
	if (af_read(af,(char*) wd->buffer,af.length)==AF_ERROR) {
	    af_close(af);
	    free(wd->buffer);
	    XtFree((char*) wd);
	    return;
	} 
	close(af.fd);
    } else {
	wd->buffer=NULL;
	wd->fd=af.fd;
	wd->actual_name=NULL;
	wd->actual_no=0;
    }
    
    /* now we can enter all the values in the Global Main_Data variable */
    if (MD->no>0) {
	nw=MD->nw;
	while (nw->nw!=NULL) nw=nw->nw;
	nw->nw=XtNew(Next_Wave);   
	nw=nw->nw;
    } else {
	MD->nw=XtNew(Next_Wave);
	nw=MD->nw;
    }
    nw->wd=wd;
    nw->cw=XtNew(Canvas_Widget);
    nw->cb=XtNew(Canvas_Bool);
    nw->cg=XtNew(Canvas_Graphic);
    nw->nw=NULL;
    
    if (nw->wd->length>MD->maxmem) nw->wd->inmem=FALSE;
    else nw->wd->inmem=TRUE;
    nw->wd->ismark=FALSE;
    nw->wd->isplay=FALSE;
    nw->cb->canwrite=can_write(nw->wd->name);
    nw->cb->modified=False;
    nw->cb->update=False;
    nw->cg->step=1;
    nw->cg->pos=0;
    
    MD->no++;
    MD->wd=nw->wd;
    MD->cb=nw->cb;
    MD->cw=nw->cw;
    MD->cg=nw->cg;
    
    new_canvas(MD->mw->form,nw);
    update_canvas(MD,NULL);
    update_display(MD);
    
    XtFree((char*)sr);
    
    /* global variable rawname, md->wd->name points to mem, 
     * so don't release it 
     */
    rawname=NULL;
    return;
}


void close_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *)client_data;
    
    if (md->no==0) return;
    
    if (md->cb->modified) {
	sprintf(md->mw->messages,"<%s> %s\n",md->wd->name,
		app_resources.err_modified);
	bool_popup(save_close_file,(XtPointer) w,(XtPointer) md,NULL,
		   im_close_file,(XtPointer) w,(XtPointer) md,NULL,
		   XtParent(md->mw->form),md->mw->messages);
	return;
    }
    close_file(md);
    update_display(md);
}


void save_close_file(Widget w,Main_Data *md) 
{
    if (!md->cb->canwrite) {
	file_dialog(saveas_close_file,app_resources.default_ext,SAVE);
	return;
    }
    save_call(w, (XtPointer) md, (XtPointer) NULL);
    close_file(md);
    update_display(md);
}

void im_close_file(Widget w,Main_Data *md) 
{
    close_file(md);
    update_display(md);
}

int saveas_close_file(char *fname,int type,int comp)
{
    if (save_as_file(fname,type,comp)==-1) return(-1);
    close_file(MD);
    update_display(MD);
    return(0);
}

void close_file(Main_Data *md)
{
    Next_Wave *nw=NULL,*lw=NULL;
    
    /* remove temp files */
    if (md->wd->actual_no>0) {
	XtFree(md->wd->actual_name);
	while (md->wd->actual_no>0) { 
	    md->wd->actual_no--;
	    md->wd->actual_name=get_actual_name(md->wd);
	    remove(md->wd->actual_name);
	    XtFree(md->wd->actual_name);
	}
	md->wd->actual_name=NULL;
    }
    
    if (md->no==1) {
	release_canvas(md->nw);
	md->wd=NULL;
	md->nw=NULL;
    } else {      
	nw=md->nw;
	while ((nw->nw!=NULL)&(md->wd!=nw->wd)) {
	    lw=nw;
	    nw=nw->nw;
	}
	if (lw==NULL) {
	    md->nw=nw->nw;
	    md->wd=md->nw->wd;
	    md->cw=md->nw->cw;
	    md->cb=md->nw->cb;
	    md->cg=md->nw->cg;
	    lw=md->nw;
	} else {
	    if (nw->nw!=NULL) lw->nw=nw->nw;
	    else lw->nw=NULL; 
	    md->wd=lw->wd;
	    md->cw=lw->cw;
	    md->cb=lw->cb;
	    md->cg=nw->cg;
	}
	md->mg->step=(float)md->wd->tlength/(float)md->mg->width;
	release_canvas(nw);
    }
    md->no--;
}

void release_canvas(Next_Wave *nw)
{
    remove_menu_entry(nw);
    if (nw->cg->pixmap!=None) 
      XFreePixmap(XtDisplay(nw->cw->popup),nw->cg->pixmap);
    XtDestroyWidget(nw->cw->popup);
    if (nw->wd->buffer!=NULL) free(nw->wd->buffer);
    if (nw->wd->name!=NULL) XtFree(nw->wd->name);
    if (nw->wd->actual_name!=NULL) XtFree(nw->wd->actual_name);
    if (!nw->wd->inmem) close(nw->wd->fd);
    XtFree((char*)nw->wd);
    XtFree((char*)nw->cw);
    XtFree((char*)nw->cb);
    XtFree((char*)nw->cg);
    XtFree((char*)nw);
}

int saveas_file(char *fname,int type,int comp)
{
    Next_Wave *nw=MD->nw;
    char *absname;
    
    absname=get_abs_name(fname);
    if (absname==NULL) absname=XtNewString(fname);
    
    while (nw!=NULL) {
	if (strcmp(nw->wd->name,absname)==0) {
	    warn_popup(MD->mw->form,app_resources.err_fileinuse);
	    XtFree((char*) absname);
	    return(0);
	}
	nw=nw->nw;
    }
    
    if (file_exists(absname)) {
	bool_popup(save_as_file,absname,(XtPointer) type,(XtPointer) comp,
		   NULL,NULL,NULL,NULL,
		   XtParent(MD->mw->form),app_resources.err_overwrite);
	return(0);
    }
    
    return(save_as_file(absname,type,comp));
}

int save_as_file(char *absname,int type,int comp)
{
    char *oldname;
    Audio_File af;
    int length;
    Wave_Data *wd=MD->wd;
    
    oldname=wd->name;
    
    wd2af(wd,&af);
    af.headoffs=-1;
    af.fd=-1;
    af.length=0;
    af.comp=comp;
    af.type=type;
    
    if (af_open(absname,&af,AF_NEW)!=af.type) {
	warn_popup(MD->mw->form,app_resources.err_openfile);
	XtFree((char*) absname);
	return(-1);
    }
    
    watch_cursor(True);
    
    if (wd->inmem) {
	if ((length=af_write(af,(char*) wd->buffer,wd->length)) == AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(MD->mw->form,app_resources.err_openfile);
	    XtFree((char*) absname);
	    af_close(af);
	    return(-1);
	}
	af.length=length;
    } else {
	Audio_File old_af;
	
	wd2af(wd,&old_af);
	
	if (af_rewind(old_af)==AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(MD->mw->form,app_resources.err_openfile);
	    XtFree((char*) absname);
	    af_close(af);
	    return(-1);
	}
	
	while ((length=af_read(old_af,(char*) MD->mg->fbuf,MAXUSHORT))>0) {
	    if (length==AF_ERROR) {
		watch_cursor(False);
		warn_popup(MD->mw->form,app_resources.err_openfile);
		XtFree((char*) absname);
		af_close(af);
		return(-1);
	    }
	    if ((length=af_write(af,(char*) MD->mg->fbuf,length)) == AF_ERROR) {
		watch_cursor(False);
		warn_popup(MD->mw->form,app_resources.err_openfile);
		XtFree((char*) absname);
		af_close(af);
		return(-1);
	    }
	    af.length+=length;
	}
	close(wd->fd);
    }
    af_close(af);
    
    /* remove all temporary files for this sample file */
    if (wd->actual_no>0) {
	
	XtFree(wd->actual_name);
	while (wd->actual_no>0) { 
	    wd->actual_no--;
	    wd->actual_name=get_actual_name(wd);
	    remove(wd->actual_name);
	    XtFree(wd->actual_name);
	}
	wd->actual_name=NULL;
    }
    
    af_open(absname,&af,AF_OPEN);
    af2wd(af,wd);
    wd->name=absname;
    MD->cb->canwrite=can_write(wd->name);
    
    actual_menu_wname(MD,oldname);
    setWmWindowName(MD->cw->popup, wd->name);
    XtFree(oldname);
    MD->cb->modified=False;
    update_status(MD);
    watch_cursor(False);
    
    return(0);
}

int saveas_close_quit(char *fname,int type,int comp)
{
    if (save_as_file(fname,type,comp)==-1) return(-1);
    close_file(MD);
    update_display(MD);
    quit_call(MD->mw->form,(XtPointer)MD,(XtPointer)NULL);
    return(0);
}

void save_close_quit(Widget w,Main_Data *md)
{
    if (!md->cb->canwrite) {
	file_dialog(saveas_close_quit,app_resources.default_ext,SAVE);
	return;
    }
    save_call(w, (XtPointer) md, (XtPointer) NULL);
    close_file(md);
    update_display(md);
    quit_call(w,(XtPointer)md,(XtPointer)NULL);
}

void close_quit(Widget w,Main_Data *md)
{
    close_file(md);
    update_display(md);
    quit_call(w,(XtPointer)md,(XtPointer)NULL);
}

void quit_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *)client_data;
    Next_Wave *nw=md->nw;
    
    while (md->no>0) {
	if (nw->cb->modified) {
	    set_actual_canvas(nw);
	    sprintf(md->mw->messages,"<%s> %s\n",md->wd->name,
		    app_resources.err_modified);
	    bool_popup(save_close_quit,(XtPointer) w,(XtPointer) md,NULL,
		       close_quit,(XtPointer) w,(XtPointer) md,NULL,
		       XtParent(md->mw->form),md->mw->messages);
	    return;
	}
	nw=nw->nw;
	md->no--;
    }
    if (md->mb->isclip) {
	remove(md->cd->wd->name);
    }
    XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
    exit(0);
}

void load_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    file_dialog(load_file,app_resources.default_ext,LOAD);
}

void saveas_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    file_dialog(saveas_file,app_resources.default_ext,SAVE);
}

void save_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *) client_data;
    Audio_File af;
    int i,toread,length;
    Wave_Data *wd=md->wd;
    
    if (!md->cb->modified) return;
    
    if (!md->cb->canwrite) {
	warn_popup(md->mw->form,app_resources.err_permissions);
	return;
    }
    
    sprintf(md->mw->messages,"%s\n\"%s\" !",app_resources.err_openfile,
	    md->wd->name);
    
    if (wd->inmem) {
	
	wd2af(wd,&af);
	af.headoffs=-1;
	af.fd=-1;
	af.length=0;
	/* we have had tmp files, they are raw, but orig. file was probably
	 * something other */
	if (wd->actual_no>0) {
	    af.type=wd->oldtype;
	    af.comp=wd->oldcomp;
	}
	if (af_open(wd->name,&af,AF_NEW)==AF_ERROR) {
	    warn_popup(md->mw->form,md->mw->messages);
	    return;
	}
	
	watch_cursor(True);
	
	if ((length=af_write(af,(char*)wd->buffer,wd->length)) == AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,md->mw->messages);
	    af_close(af);
	    return;
	}
	af.length=length;
	af_close(af);
	
    } else {
	Audio_File new_af;
	
	length=wd->length;
	
	/* old file, probably tmp */
	wd2af(wd,&af);
	af_rewind(af);
	
	wd2af(wd,&new_af);
	new_af.headoffs=-1;
	new_af.fd=-1;
	new_af.length=0;
	new_af.type=wd->oldtype;
	new_af.comp=wd->oldcomp;
	
	if ((af_open(wd->name,&new_af,AF_NEW))==AF_ERROR) {
	    warn_popup(md->mw->form,md->mw->messages);
	    return;
	}
	
	watch_cursor(True);
	
	i=length;
	while (i>0) {
	    if (i>MAXUSHORT) toread=MAXUSHORT;
	    else toread=i;
	    toread=af_read(af,(char*)md->mg->fbuf,toread);
	    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
	    if (toread==-1) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		af_close(new_af);
		return;
		
	    }
	    i-=toread;
	    new_af.length+=toread;
	}
	af_close(af);
	af_close(new_af);
	af_open(wd->name,&af,AF_OPEN);
	af2wd(af,wd);
    }
    
    /* remove all temporary files for this sample file */
    if (wd->actual_no>0) {
	
	XtFree(wd->actual_name);
	while (wd->actual_no>0) { 
	    wd->actual_no--;
	    wd->actual_name=get_actual_name(wd);
	    remove(wd->actual_name);
	    XtFree(wd->actual_name);
	}
	wd->actual_name=NULL;
    }
    
    watch_cursor(False);
    md->cb->modified=False;
    update_status(md);
}

void insert_frombuffer(char *name,char *buffer,int length,byte res,byte mode,
		       int freq)
{
    Next_Wave *nw;
    
    if (MD->no>0) {
	nw=MD->nw;
	while (nw->nw!=NULL) nw=nw->nw;
	nw->nw=XtNew(Next_Wave);   
	nw=nw->nw;
    } else {
	MD->nw=XtNew(Next_Wave);
	nw=MD->nw;
    }
    nw->wd=XtNew(Wave_Data);
    nw->cw=XtNew(Canvas_Widget);
    nw->cb=XtNew(Canvas_Bool);
    nw->cg=XtNew(Canvas_Graphic);
    nw->nw=NULL;
    
    nw->wd->name=name;
    nw->wd->actual_name=NULL;
    nw->wd->freq=freq;
    nw->wd->res=res;
    nw->wd->channels=mode;
    
    nw->wd->length=length;
    nw->wd->buffer=(unsigned char*) buffer;
    nw->wd->inmem=TRUE;
    if (nw->wd->res==8) {
	if (nw->wd->channels==MONO) nw->wd->bpspl=1;
	else nw->wd->bpspl=2;
    } else {
	if (nw->wd->channels==MONO) nw->wd->bpspl=2;
	else nw->wd->bpspl=4;
    }
    nw->wd->tlength=length/nw->wd->bpspl;
    nw->wd->markbeg=0;
    nw->wd->marklength=0;
    nw->wd->ismark=False;
    nw->wd->isplay=False;
    nw->cb->canwrite=can_write(name);
    nw->cb->modified=True;
    nw->cg->step=1;
    nw->cg->pos=0;
    
    MD->no++;
    MD->wd=nw->wd;
    MD->cb=nw->cb;
    MD->cw=nw->cw;
    MD->cg=nw->cg;
    
    new_canvas(MD->mw->form,nw);
    
    update_canvas(MD,NULL);
    update_display(MD);
}

void new_return(Sample_Return *sr)
{
    Next_Wave *nw;
    char *name;
    
    if ((name=new_name())==NULL) {
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }
    
    if (MD->no>0) {
	nw=MD->nw;
	while (nw->nw!=NULL) nw=nw->nw;
	nw->nw=XtNew(Next_Wave);   
	nw=nw->nw;
    } else {
	MD->nw=XtNew(Next_Wave);
	nw=MD->nw;
    }
    nw->wd=XtNew(Wave_Data);
    nw->cw=XtNew(Canvas_Widget);
    nw->cb=XtNew(Canvas_Bool);
    nw->cg=XtNew(Canvas_Graphic);
    nw->nw=NULL;
    
    nw->wd->name=name;
    nw->wd->actual_name=NULL;
    nw->wd->actual_no=0;
    nw->wd->freq=sr->freq;
    nw->wd->res=sr->res;
    nw->wd->channels=sr->channels;
    nw->wd->bpspl=(nw->wd->res*nw->wd->channels)/8;
    XtFree((char*)sr);
    
    nw->wd->type=app_resources.default_ftype;
    nw->wd->comp=app_resources.default_comp;
    nw->wd->length=0;
    nw->wd->buffer=NULL;
    nw->wd->lines_l=NULL;
    nw->wd->lines_r=NULL;
    nw->wd->inmem=True;
    if (nw->wd->res==8) 
      if (nw->wd->channels==MONO) nw->wd->bpspl=1;
    else nw->wd->bpspl=2;
    else
      if (nw->wd->channels==MONO) nw->wd->bpspl=2;
    else nw->wd->bpspl=4;
    nw->wd->tlength=0;
    nw->wd->markbeg=0;
    nw->wd->marklength=0;
    nw->wd->ismark=False;
    nw->wd->isplay=FALSE;
    nw->cb->canwrite=True;
    nw->cb->modified=False;
    nw->cg->step=1;
    nw->cg->pos=0;
    
    MD->no++;
    MD->wd=nw->wd;
    MD->cb=nw->cb;
    MD->cw=nw->cw;
    MD->cg=nw->cg;
    
    new_canvas(MD->mw->form,nw);
    update_display(MD);
}

void new_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    sample_dialog (new_return,NULL);
}

