/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $Source: /homes/cvs/ftape/ftape-tools/src/ftformat/ftfmt-options.c,v $
 * $Revision: 1.5 $
 * $Date: 1998/08/13 21:46:19 $
 *
 *      This program contains the options parsing for the user level
 *      floppy tape formatting stuff for the QIC-40/80/3010/3020
 *      floppy-tape driver "ftape" for Linux.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <getopt.h>
#include <libintl.h>
#include <ctype.h>
#include <unistd.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/zftape.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-tapelib.h"
#include "version.h"

struct opt_parms opt = {
	FTAPE_DEF_LABEL,
	FTAPE_DEF_TAPE,
	FT_BUFF_SIZE * NR_FTAPE_BUFFERS,
	AUTO,
	1, /* erase using phys. forward/reverse */
	1, /* read old header segment */
	1, /* write reference bursts  */
	1, /* really format the cartridge */
	0, /* verbose */
	0, /* debug */
};

static char *progname = NULL;
static const char *short_options = "f:Vdvh?L:m:p:M:";
static const struct option long_options[] =
{
	{"device",             1, 0, 'f'},   /* full path to device */
	{"version",            0, 0, 'V'},   /* version information */
	{"debug",              0, 0, 'd'},   /* switch on debugging, unused */
	{"verbose",            0, 0, 'v'},   /* verbosity level, unused */ 
	{"usage",              0, 0, 'h'},
	{"help",               0, 0, 'h'},
	{"label",              1, 0, 'L'},
	{"verify-only",        0, &opt.do_format, 0},
	{"omit-erase",         0, &opt.erase, 0},
	{"discard-header",     0, &opt.read_hseg, 0},
	{"no-reference-bursts",0, &opt.write_bursts, 0},
	{"mode",               1, 0, 'm'},
	{"format-parameters",  1, 0, 'p'},
	{"dma-memory",         1, 0, 'M'}, /* amount of memory to mmap */
	{"parsable-output",    0, &opt.notty, 1},
	{0, 0, 0, 0}
};

enum {
	qic_none = -1,
	qic40_205ft = 0,
	qic40_307ft,
	qic40_1100ft,
	qic80_205ft,
	qic80_307ft,
	qic80_425ft,
	qic80_1100ft,
	OPT_QICSTD,
	OPT_FMTCODE,
	OPT_FHM,
	OPT_FTM,
	OPT_SPT,
	OPT_TPC,
	OPT_GAP3,
} fixed_format = qic_none;

/* these are all formats of fixed tape length documented in the QIC stds
 */
const struct format_parms fixed_fmts[] = {
	{{QIC_TAPE_QIC40, fmt_normal,  1, 169,  68, 20}, 233},
	{{QIC_TAPE_QIC40, fmt_normal,  1, 254, 102, 20}, 233},
	{{QIC_TAPE_QIC40, fmt_1100ft,  7, 253, 365, 20}, 233},
	{{QIC_TAPE_QIC80, fmt_normal,  4, 149, 100, 28}, 233},
	{{QIC_TAPE_QIC80, fmt_normal,  6, 149, 150, 28}, 233},
	{{QIC_TAPE_QIC80, fmt_425ft,   9, 149, 207, 28}, 233},
	{{QIC_TAPE_QIC80, fmt_1100ft, 14, 253, 537, 28}, 233},
};

const char *format_opts[] = {
	"qic40-205ft",
	"qic40-307ft",
	"qic40-1100ft",
	"qic80-205ft",
	"qic80-307ft",
	"qic80-425ft",
	"qic80-1100ft",
	"qic-standard",
	"format-code" ,
	"floppy-head-max",
	"floppy-track-max",
	"segments-per-track",
	"tracks-per-tape",
	"gap3",
	NULL
};

const char *mode_opts[] = {
	"auto",
	"force",
	"probe",
	NULL
};

static void usage(FILE *file, int exit_val, char *message, ...);

const struct qic_std_tbl {
	char *name;
	u_int8_t value;
} qic_std_tbl[] = {
	{ "QIC-40",        QIC_TAPE_QIC40 },
	{ "QIC-80",        QIC_TAPE_QIC80 },
	{ "QIC-80/WIDE",   QIC_TAPE_QIC80 | QIC_TAPE_WIDE },
	{ "TR-1",          QIC_TAPE_QIC80 | QIC_TAPE_WIDE },
	{ "QIC-3010",      QIC_TAPE_QIC3010 },
	{ "QIC-3010/WIDE", QIC_TAPE_QIC3010 | QIC_TAPE_WIDE},
	{ "QIC-3020",      QIC_TAPE_QIC3020 },
	{ "QIC-3020/WIDE", QIC_TAPE_QIC3020 | QIC_TAPE_WIDE},
	{ "TR-3",          QIC_TAPE_QIC3020 | QIC_TAPE_WIDE},
	{ NULL, 0xff }
};

static u_int8_t parse_qic_std(char *argv0, char *optarg)
{
	char *p = optarg;
	const struct qic_std_tbl *tbl = qic_std_tbl;
	u_int8_t value;

	while (*p) *p++ = toupper(*p);
	while (tbl->name != NULL) {
		if (strcmp(optarg, tbl->name) == 0) {
			return tbl->value;
		}
		tbl ++;
	}
	value = strtol(optarg, NULL, 0);
	if (value == 0x00) {
		usage(stderr, 1, _("Unknown QIC standard: %s\n"), optarg);
	}
	return value;
}

/*  Parse the options and initialize op_parms and fmt_parms
 */
void parse_options(int argc, char *argv[],
				   struct opt_parms *optp, struct format_parms *fmt_parms)
{
	char *subopts, *value;
	int c;
	int fmt_opt;

	progname = argv[0];
	memset(fmt_parms, 0xff, sizeof(struct format_parms));

	while (1) {
		int option_index = 0;
		
		c = getopt_long(argc, argv, short_options, long_options,&option_index);

		if (c == -1) {
            break;
		}

		switch (c) {
		case 0:
#if 0			
			printf ("option %s", long_options[option_index].name);
			if (optarg)
				printf (" with arg %s", optarg);
			printf ("\n");
#endif
			break;
		case 'm':
			if (!strcmp(optarg, mode_opts[AUTO])) {
				opt.mode = AUTO;
			} else if (!strcmp(optarg, mode_opts[FORCE])) {
				opt.mode = FORCE;
			} else if (!strcmp(optarg, mode_opts[PROBING])) {
				opt.mode = PROBING;
			} else {
				usage(stderr, 1, _("wrong mode: %s\n"), optarg);
			}
			break;
		case 'p':
			subopts = optarg;
			while (subopts && *subopts != '\0') {
				fmt_opt = getsubopt(&subopts, format_opts, &value);

				if (fmt_opt <= qic80_1100ft && 
					fmt_opt >= qic40_205ft) {
					if (value != NULL) {
						usage(stderr, 1, _("The option \"%s\" takes no value!\n"),
							  format_opts[fmt_opt]);
					}
				} else {
					if (fmt_opt != -1 && value == NULL) {
						usage(stderr, 1, _("The option \"%s\" needs a value!\n"),
							  format_opts[fmt_opt]);
					}
				}
				
				switch(fmt_opt) {
				case qic40_205ft:
				case qic40_307ft:
				case qic40_1100ft:
				case qic80_205ft:
				case qic80_307ft:
				case qic80_425ft:
				case qic80_1100ft:
					fixed_format = fmt_opt;
					break;
				case OPT_QICSTD:
					fmt_parms->fmtparms.ft_qicstd = parse_qic_std(argv[0], 
																  value);
					break;
				case OPT_FMTCODE:
					fmt_parms->fmtparms.ft_fmtcode  = strtol(value, NULL, 0);
					break;
				case OPT_FHM:
					fmt_parms->fmtparms.ft_fhm = strtol(value, NULL, 0);
					break;
				case OPT_FTM:
					fmt_parms->fmtparms.ft_ftm = strtol(value, NULL, 0);
					break;
				case OPT_SPT:
					fmt_parms->fmtparms.ft_spt = strtol(value, NULL, 0);
					break;
				case OPT_TPC:
					fmt_parms->fmtparms.ft_tpc = strtol(value, NULL, 0);
					break;
				case OPT_GAP3:
					fmt_parms->gap3 = strtol(value, NULL, 0);
					break;
				default:
					usage(stderr, 1, _("Unknown option: %s\n"), value);
					break;
				}
			}
			break;
		case 'L':
			opt.label = optarg;
			break;
		case 'M':
			opt.dma_size = strtol(optarg, NULL, 0);
			opt.dma_size *= 1024;
			if (opt.dma_size % FT_BUFF_SIZE != 0) {
				usage(stderr, 1,
					  _("Size of dma buffer %s must be a multiple of %d\n"),
					  optarg, FT_BUFF_SIZE/1024);
			}
			if (opt.dma_size / FT_BUFF_SIZE < NR_FTAPE_BUFFERS) {
				usage(stderr, 1,
					  _("Size of dma buffer %s must be greater than %d\n"),
					  optarg, NR_FTAPE_BUFFERS*FT_BUFF_SIZE/1024);
			}
			break;
		case 'f':
			opt.tape_dev = optarg;
			break;
		case 'h':
		case '?':
			usage(stdout, 0, NULL);
			break;
		case 'd':
			opt.debug ++;
			break;
		case 'v':
			opt.verbose ++;
			break;
		case 'V':
			printf(version_string);
			exit(0);
			break;
		default:
			fprintf (stderr, _("?? getopt returned character code 0%o ??\n"), c);
			exit(1);
			break;
		}
	}
	if (!opt.notty && !isatty(STDOUT_FILENO)) {
		opt.notty = 1;
	}
	if (optind < argc) {
		fprintf (stderr, _("non-option ARGV-elements: "));
		while (optind < argc)
			fprintf (stderr, "%s ", argv[optind++]);
		fprintf (stderr, "\n");
		usage(stderr, 1, NULL);
	}
	if (!opt.do_format) {
		opt.erase = opt.write_bursts = 0;
	}
	if (fixed_format != qic_none) {
		if (!UNSET(fmt_parms->fmtparms.ft_qicstd)  ||
			!UNSET(fmt_parms->fmtparms.ft_fmtcode) ||
			!UNSET(fmt_parms->fmtparms.ft_fhm)     ||
			!UNSET(fmt_parms->fmtparms.ft_ftm)     ||
			!UNSET(fmt_parms->fmtparms.ft_spt)     ||
			!UNSET(fmt_parms->fmtparms.ft_tpc)     ||
			!UNSET(fmt_parms->gap3)) {
			usage(stderr, 1,
				  _("Can't use one of the fixed length formats \n"
				  "together with other format specifiers.\n"));
		}
		*fmt_parms = fixed_fmts[fixed_format];
	}
	if (opt.mode == FORCE) {
		if (UNSET(fmt_parms->fmtparms.ft_qicstd)  ||
			UNSET(fmt_parms->fmtparms.ft_fmtcode) ||
			UNSET(fmt_parms->fmtparms.ft_fhm)     ||
			UNSET(fmt_parms->fmtparms.ft_ftm)     ||
			UNSET(fmt_parms->fmtparms.ft_spt)     ||
			UNSET(fmt_parms->fmtparms.ft_tpc)     ||
			UNSET(fmt_parms->gap3)) {
			usage(stderr, 1,
				  _("Need ALL format parameters when used in \"force\" mode:\n"
				  "qic standard      : %d\n"
				  "format code       : %d\n"
				  "max. floppy head  : %d\n"
				  "max. floppy track : %d\n"
				  "segments per track: %d\n"
				  "tracks per tape   : %d\n"
				  "gap3              : %d\n\n"),
				  fmt_parms->fmtparms.ft_qicstd,
				  fmt_parms->fmtparms.ft_fmtcode,
				  fmt_parms->fmtparms.ft_fhm,
				  fmt_parms->fmtparms.ft_ftm,
				  fmt_parms->fmtparms.ft_spt,
				  fmt_parms->fmtparms.ft_tpc, fmt_parms->gap3);
		}
		opt.read_hseg = 0;
	}
	*optp = opt;
}

static void usage(FILE *file, int exit_val, char *message, ...)
{
	va_list ap;
	if (message) {
		va_start(ap, message);
		vfprintf(file, message, ap);
		va_end(ap);
	}
	fprintf(file,
_("%s\n"
"\n"
"Usage: %s [OPTIONS]\n"
"Format floppy tape cartridges, for use with\n"
"%s, %s\n"
"\n"
"Mandatory or optional arguments to long options are mandatory or optional\n"
"for short options too. Unique abbreviations for long options are accepted\n"
"\n"
"  -f, --device=FILE     Tape device to use. Default is "
"\"%s\".\n"
"  -h, --help            Print this help.\n"
"  -?                    Same as \'-h\'.\n"
"      --usage           Same as \'-h\'.\n"
"  -V, --version         Print version information.\n"
"  -d, --debug           Unused yet.\n"
"  -v, --verbose         Unused yet.\n"
"      --parsable-ouput  Terminate each message by a newline, don't use\n"
"                        backspace or carriage return characters. Used by\n"
"                        the X11 interface \"ftape-tool\".\n"
"  -L, --label=LABEL     Label to write to the header segment.\n"
"      --omit-erase      Don\'t try to erase the cartridge using\n"
"                        \"physical forward\" and \"physical reverse\".\n"
"                        This affects only already formatted cartridges.\n"
"                        The use of this option is deprecated.\n"
"      --discard-header  Don\'t try to read the old header segment and\n"
"                        discard statistics and time stamps contained\n"
"                        therein.\n"
"                        This affects only already formatted cartridges.\n"
"                        The use of this option is deprecated.\n"
"\n"
"      --no-reference-bursts\n"
"                        Don't write the reference bursts, only format and\n"
"                        verify. Only useful with preformatted cartridges.\n"
"                        The use of this option is deprecated.\n"
"      --verify-only     Only update the bad sector map, don't re-format.\n"
"                        Only possible with already formatted cartridges.\n"
"                        The use of this option is deprecated.\n"
"  -M  --dma-memory=SIZE Amount of dma memory to mmap. Measured in kb.\n"
"                        Normally not needed.\n"
"  -m  --mode=MODE       MODE has to be one out of {auto, force, probe}.\n"
"                        \"auto\" is the default. \"force\" will bypass\n"
"                        almost all consistency checks, assuming THAT YOU\n"
"                        REALLY KNOW WHAT YOU ARE DOING, \"probe\"\n"
"                        performs the auto-detection only and prints a\n"
"                        command line to stdout that is suitable to be\n"
"                        used with the \"force\" mode.\n"
"\n"
"  -p  --format-parameters=PARMS\n"
"                        This option allows to overwrite the auto-detected\n"
"                        values, or help when the format of the inserted\n"
"                        cartridge cannot be auto-detected (e.g. 205ft\n"
"                        QIC-80 cartridges versus 425ft QIC-80 cartridges)\n"
"                        With \"--mode=force\" ALL of the parameters below\n"
"                        must be specified. WITHOUT \"--mode=force\" the\n"
"                        parameters given in PARMS are used as supplements\n"
"                        to the auto-detected values. In this case some of\n"
"                        the values can be omitted. PARMS is a comma\n"
"                        separated list of the following parameters:\n"
"\n"
"      qic-standard=QIC_STD\n"
"      format-code=FORMAT_CODE\n"
"      floppy-head-max=MAX_FLOPPY_HEAD\n"
"      floppy-track-max=MAX_FLOPPY_TRACK\n"
"      segments-per-track=SEGMENTS_PER_TRACK\n"
"      tracks-per-tape=TRACKS_PER_TAPE\n"
"      gap3=GAP3\n"
"\n"
" The meaning of the parameters is:\n"
"\n"
" QIC_STD             one out of:\n"
"                       \"QIC-40\", \"QIC-80\",\n"
"                       \"QIC-80/WIDE\", \"TR-1\",\n"
"                       \"QIC-3010\", \"QIC-3010/WIDE\",\n"
"                       \"QIC-3020\",\n"
"                       \"QIC-3020/WIDE\", \"TR-3\".\n"
"                     You may as well give a number in hexadecimal or\n"
"                     decimal representation.\n"
"                     TR-3 cartridges conform to \"QIC-3020/WIDE\".\n"
"                     TR-3/Extra cartridges are extra length TR-3.\n"
"                     TR-1 cartridges conform to \"QIC-80/WIDE\".\n"
" FORMAT_CODE         A decimal number to be stored in the appropriate\n"
"                     field in the header segments. Normally not needed.\n"
" MAX_FLOPPY_HEAD     A decimal number.\n"
" MAX_FLOPPY_TRACK    A decimal number.\n"
" SEGMENTS_PER_TRACK  The number of tape segments per track.\n"
" TRACKS_PER_TAPE     The number of tape tracks per cartridge\n"
" GAP3                A decimal number\n"
"\n"
"Alternatively, you may use the following short-cuts for various\n"
"fixed length formats. Note that each shortcut is a place holder for a\n"
"full set of \"low level\" options, i.e. use them like\n"
"\"--format-parameters qic80-307ft\":\n"
"\n"
"      qic40-205ft     format a 205ft cartridge to QIC-40 format, 40MB\n"
"      qic40-307ft     likewise for 307ft cartridge, 60MB\n"
"      qic40-1100ft    likewise for 1100ft cartrigde 210MB\n"
"      qic80-205ft     format a 205ft cartridge to QIC-80 format, 80MB\n"
"      qic80-307ft     likewise for 307ft cartridge, 120MB\n"
"      qic80-425ft     likewise for 425ft cartridge, 170MB\n"
"      qic80-1100ft    likewise for 1100ft cartrige, 400MB\n"
"\n"
"Please refer to the development standards\n"
"\n"
"        QIC-40, QIC-80, QIC-3010 and QIC-3020\n"
"\n"
"to be found at \"http://www.qic.org\" for the correct settings of\n"
"QIC_STD, FORMAT_CODE, MAX_FLOPPY_HEAD, MAX_FLOPPY_TRACK,\n"
"SEGMENTS_PER_TRACK, TRACKS_PER_TAPE and GAP3\n"
"\n"
"Normally, you only want to use one of the short cuts, or give the\n"
"desired tape format, i.e. the QIC_STD, or override the number of segments\n"
"per track (use the latter with extreme prejudice).\n"
"\n"
"Examples:\n"
"\n"
"Auto-detect the format parameters of a Travan-3 cartridge:\n"
"\n"
"         ftformat --mode=probe --format-parameters qic-standard=TR-3\n"
"\n"
"Format a 425ft QIC-80 cartridge (ONLY possible with certain tape drives):\n"
"\n"
"         ftformat --mode=auto --format-parameters qic80-425ft\n"
"\n"
"Format an unformatted Travan-3 cartridge:\n"
"\n"
"         ftformat --mode=auto --format-parameters qic-standard=TR-3\n"
"\n"
"Reformat a Travan-3 cartridge with damaged format:\n"
"\n"
"         ftformat --mode=auto --format-parameters qic-standard=TR-3 \\\n"
"                  --omit-erase --discard-header\n"
"\n"),
			version_string,
			progname,
			FTAPE_VERSION,
			ZFTAPE_VERSION,
			FTAPE_DEF_TAPE);
	exit(exit_val);
}


void report_format_parms(FILE *channel, const struct format_parms *fmt_opts)
{
	fprintf(channel,
			"--format-parameters qic-standard=0x%02x,"
			"format-code=%d,"
			"floppy-head-max=%d,"
			"floppy-track-max=%d,"
			"segments-per-track=%d,"
			"tracks-per-tape=%d,"
			"gap3=%d\n",
			fmt_opts->fmtparms.ft_qicstd, fmt_opts->fmtparms.ft_fmtcode,
			fmt_opts->fmtparms.ft_fhm, fmt_opts->fmtparms.ft_ftm,
			fmt_opts->fmtparms.ft_spt, fmt_opts->fmtparms.ft_tpc,
			fmt_opts->gap3);
}

/*  tries to figure out how the cartridge should be formatted, decoding
 *  drive_config and tape_status.
 *
 *  The needed parameters are returned in parms and gap3.
 *
 *  Strategy:
 *  Decode drive_config and tape_status and handle fixed format cases
 *
 *  Problem: distinguish between 425ft from 207ft cartridges
 *  Try "calibrate tape length". If unsupported, try it the "soft" way in
 *  case the tape speed is known.
 *  Indicate failure if "set N format segments" is not supported (let
 *  user decide)
 *
 *  Variable length cartridges can only be formatted if "set N format
 *  segments" and friends are supported
 *
 *  We try to fill in all required fields of ftfmtparms, and determine
 *  whether this is a "wide" tape
 *
 *  If any of the fields of "parms" is left at "-1", i.e. unset, then
 *  the user has to be prompt.
 *
 */
int autodetect(const int tape_fd,
			   const u_int8_t drive_status,
			   const u_int8_t drive_config,
			   const u_int8_t _tape_status,
			   struct opt_parms *opts,
			   struct format_parms *fmt_opts)
{
	struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int need_supported = 0;
	int supported;
	u_int8_t fmt;
	u_int8_t tape_status = _tape_status;

	if (UNSET(fmt_opts->fmtparms.ft_qicstd)   &&
		(!UNSET(fmt_opts->fmtparms.ft_fmtcode) ||
		 !UNSET(fmt_opts->fmtparms.ft_fhm)     ||
		 !UNSET(fmt_opts->fmtparms.ft_ftm)     ||
		 !UNSET(fmt_opts->fmtparms.ft_spt)     ||
		 !UNSET(fmt_opts->fmtparms.ft_tpc)     ||
		 !UNSET(fmt_opts->gap3))) {
		usage(stderr, 1,
_("Cannot format with given parameters without knowing about the QIC standard!\n")
			);
	}
	if (tape_status == 0) { /* old tape drive */
		if (drive_config == 0) { /* this MUST be very old QIC-40 drive */
			*fmt_opts = fixed_fmts[qic40_205ft];
		} else if (drive_config & QIC_CONFIG_LONG) { 
			*fmt_opts = ((drive_config & QIC_CONFIG_80) ?
						 fixed_fmts[qic80_307ft] : fixed_fmts[qic40_307ft]);
		} else 	if (drive_config & QIC_CONFIG_80) {
			if (memcmp(fmt_opts,&fixed_fmts[qic80_205ft],
					   sizeof(struct format_parms))) {
				fprintf(stderr,
_("Unable to auto-detect tape type. If this is a 205ft cartridge,\n"
"try again with the \"qic80-205ft\" option\n"));
				return -1;
			}
		} else {
			*fmt_opts = fixed_fmts[qic40_205ft];
		}
		return 0;
	}
	if (!UNSET(fmt_opts->fmtparms.ft_qicstd) &&
		fmt_opts->fmtparms.ft_qicstd != (tape_status &
										  (QIC_TAPE_STD_MASK|QIC_TAPE_WIDE))) {
		tape_status  &= ~(QIC_TAPE_STD_MASK|QIC_TAPE_WIDE);
		tape_status  |= (fmt_opts->fmtparms.ft_qicstd &
						 (QIC_TAPE_STD_MASK|QIC_TAPE_WIDE));
		need_supported = 1;
	}
	fmt = (((tape_status & QIC_TAPE_STD_MASK) << 2) |
		   ((tape_status & QIC_TAPE_WIDE) ? 3 : 1));
	supported = !qic_simple_command(tape_fd, 0, QIC_SELECT_RATE, fmt, 1, NULL);
	if (need_supported && !supported) {
		fprintf(stderr,
				_("Your tape drive doesn't support the select format command\n"));
		return -1;
	}
	/*  I think that the tape_status bits should not depend on the format
	 *  of the load zone. Do they? Find out. FIXME.
	 *
	 *  Note that tape_status now is set to the user specified value.
	 */
	switch(tape_status & QIC_TAPE_LEN_MASK) { /* tape status is valid */
	case QIC_TAPE_307FT:
		*fmt_opts = (tape_status & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC40 ?
					 fixed_fmts[qic40_307ft] : fixed_fmts[qic80_307ft];
		break;
	case QIC_TAPE_1100FT:
		*fmt_opts = (tape_status & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC40 ?
					 fixed_fmts[qic40_1100ft] : fixed_fmts[qic80_1100ft];
		break;
	case QIC_TAPE_205FT:
		if ((tape_status & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC40) {
			*fmt_opts = fixed_fmts[qic40_205ft];
			break; /* done */
		}
	default: /* need calibrate tape length */
		if (!supported) { /* out of luck */
			if (!memcmp(fmt_opts, &fixed_fmts[qic80_205ft],
						sizeof(struct format_parms))) { /* may work */
				break;
			}
			fprintf(stderr,
_("Unable to auto-detect tape type. If this is a 205ft cartridge,\n"
"try again with the \"qic80-205ft\" option\n"));
			return -1; /* bail out, this really doesn't work */
		}
		if (UNSET(parms->ft_spt)) { /* let user override this */
			parms->ft_spt = qic_calibrate_tape_length(tape_fd);
		}
		parms->ft_qicstd = tape_status & (QIC_TAPE_STD_MASK|QIC_TAPE_WIDE);
		switch(tape_status & QIC_TAPE_LEN_MASK) {
		case QIC_TAPE_205FT:/* override to comply to QIC standard */
			if (parms->ft_spt <= 153) {
				*fmt_opts = fixed_fmts[qic80_205ft];
				break;
			} else if (parms->ft_spt <= 228) {
				*fmt_opts = fixed_fmts[qic80_425ft];
				break;
			}
		case QIC_TAPE_VARIABLE:
			if (tape_status & QIC_TAPE_WIDE) {
				parms->ft_tpc = 36;
			} else {
				parms->ft_tpc = 28;
			}
			parms->ft_fmtcode = fmt_var;
			/* max. floppy head  */
			parms->ft_fhm = (parms->ft_tpc * parms->ft_spt - 1)/1020; 
			parms->ft_ftm = 254;       /* max. floppy track */
			fmt_opts->gap3 = 233;
			break;
		case QIC_TAPE_FLEX:
			if (tape_status & QIC_TAPE_WIDE) {
				parms->ft_tpc = 50;
			} else {
				parms->ft_tpc = 40;
			}
			/* max. floppy head  */
			parms->ft_fhm = (parms->ft_tpc * parms->ft_spt - 1)/1020; 
			parms->ft_ftm = 254;       /* max. floppy track */
			if (parms->ft_tpc * parms->ft_spt >= (1<<16)) {
				parms->ft_fmtcode = fmt_big;
			} else {
				parms->ft_fmtcode = fmt_var;
			}
			fmt_opts->gap3 = 248;
			break;
		}
	}
	return 0;
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
