/*
  Extracts a requested key from an rdbtable along with cursor info.

  Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_KEY_LENGTH          1024
#define EMPTY			""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options] Key Value

Options:
    -h    Print this help info.
    -n    Strip header from output.

Extracts from an rdbtable the record with the key field 'Key' that has
the value 'Value'. The output record is prepended with another six
columns: 'rN, 'oF', 'fKey', 'lKey', 'nKey' and 'pKey'. Their content is :

  rN   = Row No. of the record being requested.
  oF   = Total No. of records in table.

  fKey = Value of the Key field of the first record in table
  lKey =               \"               last         \"
  nKey =               \"               next         \"
  pKey =               \"               previous     \"

where 'next' and 'previous' are relative to the row number of the record
being requested.

The pointers 'wrap-around', i.e. if the current Key points to record 8 of
a table with 8 records, nKey will contain the Key value of the 1st record.
Conversely, if the current Key points to the 1st record, pKey will point to
the last one. If the input table contains only one record, all pointers
will be the same. If the input table contains no records, then the output
table will be empty too.

This operator reads an rdbtable from STDIN and prints the results to
STDOUT.


'$Id: nsq-cursor.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n", my_name, my_name, "\\");
}

int main( int  argc, char *argv[] ) {
  
  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
	a_loop;
  char
	*my_name=argv[0],
	cmd_buf[MAX_COMMAND_LENGTH],
	k_name[MAX_KEY_LENGTH]="",
	k_value[MAX_KEY_LENGTH]="";

  int no_hdr=0, debug=0;

  while ((a_loop = getopt(argc, argv, "xhn")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'n': 
        no_hdr=1;
        break;
      case 'x': 
        debug=1;
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  if( optind < argc )
	snprintf(k_name,MAX_KEY_LENGTH,"%s", argv[optind++]);
  
  if( optind < argc )
	snprintf(k_value,MAX_KEY_LENGTH,"%s", argv[optind++]);
  
  /* Check for mandatory arguments. */
  if( ! strcmp( k_name, EMPTY ) ||
	  ! strcmp( k_value, EMPTY ) ) {
	show_help(my_name);
	exit(2);
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN { NULL=\"\"; FS=\"\\t\"; OFS=FS;
     k_name=\"%s\"; k_value=\"%s\";}
# Table comments.
r==0 && $0 ~ /^ *#/ { next }
# Column names and positions.
r==0 { 
  while( ++p <= NF ) {
	# Make sure we pick the first occurrence of duplicated column
	# names (it may happen after a join).
	if( P[$p] == NULL ) {
	  P[$p]=p
	  N[p]=$p
	}
  }	
  if(!%d)
	print \"rN\", \"oF\", \"f\" k_name, \"l\" k_name, \"n\" k_name,
	  \"p\"  k_name, $0
  r++; next;
}
# Column definitions.
r==1 {
  NR=0
  if(!%d) {
	out_rec = \"6N\" OFS \"6N\"
    p=0
    while( ++p <= NF )
	  if( D[N[p]] == NULL ) D[N[p]] = $p
    for( i=1; i<=4; i++) out_rec = out_rec OFS D[k_name]
    print out_rec, $0
  }
  r++; next;
}
{ 
  cursor[NR] = $P[k_name]
  if ( cursor[NR] == k_value ) {
	needle = $0
	c_pointer = NR
	prev_row = NR - 1
	next_row = NR + 1
  }
}
END {
  # Handle empty table, or record not found.
  if( NR == 0 || c_pointer == NULL ) exit
  if ( prev_row == 0 ) prev_row = NR
  if ( next_row > NR )  next_row = 1
  print c_pointer, NR, cursor[1], cursor[NR], cursor[next_row],
	cursor[prev_row], needle
}",k_name, k_value, no_hdr, no_hdr);

  if( debug )
	fprintf (stderr, "Generated AWK program:
	  ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}

