/*
 Selects specific columns from an rdbtable.

 Author: Carlo Strozzi <carlos@linux.it>
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_SORT_ARGS           32
#define MAX_ARG_LENGTH          1024
#define EMPTY                   ""

void show_help( char *my_name)
{
    printf("
        NoSQL operator: %s

Usage:  %s [options] 'column  [column]  ...'

Options:
    -help    Print this help info.
    -n       Strip header from output.
    -x       Debug option.
    -s \"xyz\" List of any additional options to the Unix \"sort\" program. 
             Acceptable sort options are \"cbdfinr\". See sort(1) for details.
             The list MUST be one single string, or it must be quoted as
             shown if there are intervening blanks. For instance: '-s rdf'.
             Any other characters in the string will be silently ignored.

Sorts an rdbtable on one or more columns. Each column may be sorted in normal
(ascending) or reverse (descending) order. Chars that are special to the UNIX
shell must be quoted. 

Column names are in the form 'column_1 column_2 ...'.
For example, to sort the input rdbtable on columns 'NAME' and 'JOB' the
statement is:

                        'NAME  JOB' 

Note how the list of columns must be quoted, i.e. it must be one
single token. Long lists of column names may be folded over multiple
lines by ending each line with a backslash, i.e.:

             'COL1 COL2 COL3 COL4 COL5 COL6 COL7 %s
              COL8 COL9 COL10'

This operator reads an rdbtable via STDIN and writes an rdbtable
via STDOUT. If only an invalid column is specified, then the whole input
table is written to STDOUT.


'$Id: nsq-fsort.c,v 1.1 1998/05/29 20:43:01 carlos Exp $'

            ----------------------
NoSQL RDBMS, Copyright (C) 1998 Carlo Strozzi.
This program comes with ABSOLUTELY NO WARRANTY; for details
refer to the GNU General Public License.

You should have received a copy of the GNU General Public License
along with this program;  if not, write to the Free Software
Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
USA.
            ----------------------\n", my_name, my_name, "\\");
}

int main( int  argc, char *argv[] ) {

  /* For getopt() */
  extern char* optarg;
  extern int optind;

  /* for the rest of the program. */
  register int
    a_loop;
  char
    *my_name=argv[0],
    cmd_buf[MAX_COMMAND_LENGTH],
    sort_args[MAX_SORT_ARGS] = EMPTY,
    c_names[MAX_ARG_LENGTH] = EMPTY;

  int no_hdr=0, debug=0;

  while ((a_loop = getopt(argc, argv, "xnhs:")) != EOF) {
    switch (a_loop) {
      case 'h':
        show_help(my_name);
        exit(0);
        break;
      case 'n': 
        no_hdr=1;
        break;
      case 'x': 
        debug=1;
        break;
      case 's':
        snprintf(sort_args,MAX_SORT_ARGS,"%s", optarg);
        break;
      default:
        show_help(my_name);
        exit(1);
    }
  }

  if( optind < argc )
    snprintf(c_names,MAX_ARG_LENGTH,"%s", argv[optind++]);
  else {
    show_help(my_name);       /* No columns where specified */
    exit(1);
  }

  snprintf(cmd_buf,MAX_COMMAND_LENGTH,"#
#
BEGIN {
  NULL = \"\"
  FS = \"\\t\"; OFS = FS;

  # Get requested column names.
  split( \"%s\", c_names_tmp, \" \" )

  # Honour the 'debug' switch.
  if( %d ) {
    arg_vec = \"# ARGC = \" ARGC

    for( arg in ARGV )
      arg_vec = arg_vec \"\\n# ARGV[\" arg \"] = \" ARGV[arg]

    print arg_vec > \"/dev/stderr\"
  }
}
# Table comments.
r == 0 && $0 ~ /^ *#/ {
  if( ! %d ) print
  next
}
# Column names and positions.
r == 0 {
  while( ++p <= NF ) {
    # Make sure we pick the first occurrence of duplicated column
    # names (it may happen after a join).
    if( P[$p] == NULL ) {
      P[$p]=p
      N[p]=$p
    }
  }
  # Remove references to invalid column names.
  while( c_names_tmp[++i] != NULL )
    if( c_names_tmp[i] in P )  c_names[++k] = c_names_tmp[i]
  # Build sort arg list.
  c = 1
  sort_args = \"%s\"
  # Remove forbidden sort options.
  gsub( /[^cbdfinru]/, NULL, sort_args ) 
  # Add leading '-'.
  sort_args = \"-\" sort_args
  if( P[c_names[c]] != NULL )
    sort_args = sort_args \"k \" P[c_names[c]]
  while( c_names[++c] != NULL )
    sort_args = sort_args \",\" P[c_names[c]]
  sort_cmd = \"sort -t \\\"\\011\\\" \" sort_args
  if( %d )  print \"# sort_cmd = \" sort_cmd > \"/dev/stderr\"
  if( ! %d )  print 
  r++
  next
}
# Column definitions.
r == 1 {
  if( ! %d )  print
  r++
  NR = 0
  next
}
# Table body.
{
  print | sort_cmd
}", c_names, debug, no_hdr, sort_args, debug, no_hdr, no_hdr);

  if( debug )
    fprintf (stderr, "Generated AWK program:
      ----------\n%s\n----------\n",cmd_buf);

  execlp(AWK,"awk",cmd_buf,NULL);
  exit(0);
}
