/* -*-Mode: C;-*-
 * $Id: serializeio.h 1.15 Thu, 01 Oct 1998 04:56:35 -0700 jmacd $
 *
 * Copyright (C) 1998, Josh MacDonald.
 * All Rights Reserved.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 */

#ifndef _SERIALIZEIO_H_
#define _SERIALIZEIO_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>

#if HAVE_SSLEAY
#include <stdio.h>
/* zlib typedefs free_func! */
#define free_func ssl_free_func

#include <ssl.h>

#else

#define SSL_ERROR_NONE			0
#define SSL_ERROR_SSL			1
#define SSL_ERROR_WANT_READ		2
#define SSL_ERROR_WANT_WRITE		3
#define SSL_ERROR_WANT_X509_LOOKUP	4
#define SSL_ERROR_SYSCALL		5 /* look at errno */
#define SSL_ERROR_ZERO_RETURN		6
#define SSL_ERROR_WANT_CONNECT		7

#endif

typedef struct _SerialSource    SerialSource;
typedef struct _SerialSink      SerialSink;
typedef enum   _SerialStatus    SerialStatus;
typedef guint32                 SerialType;
typedef struct _HandleFuncTable HandleFuncTable;

#if !defined(HAVE_FILE_HANDLE_TYPEDEF)
typedef struct _FileHandle    FileHandle;
#define HAVE_FILE_HANDLE_TYPEDEF
#endif

#define ST_Error 0
#define ST_NotFound 1

#define SER_LIBRARY_OFFSET_BITS 8
#define SER_LIBRARY_OFFSET_MASK 0xff

#define ALIGN_8(v) if (((v) % 8) != 0) { (v) += 8; (v) &= ~7; }

enum _SerialStatus {
  SerialSuccess    = SSL_ERROR_NONE,
  SerialWantWrite  = SSL_ERROR_WANT_WRITE,
  SerialWantRead   = SSL_ERROR_WANT_READ,
  SerialWantLookup = SSL_ERROR_WANT_X509_LOOKUP,
  SerialZero       = SSL_ERROR_ZERO_RETURN,
  SerialSyscall    = SSL_ERROR_SYSCALL,
  SerialSSL        = SSL_ERROR_SSL,
  SerialFailure    = -1,
  SerialIncorrectType  = -2
};

/* This serves as a virtual table for I/O to the FileHandle */

struct _HandleFuncTable
{
  gssize            (* table_handle_length)       (FileHandle *fh);
  gssize            (* table_handle_pages)        (FileHandle *fh);
  gssize            (* table_handle_pagesize)     (FileHandle *fh);
  gssize            (* table_handle_map_page)     (FileHandle *fh, guint pgno, const guint8** mem);
  gboolean          (* table_handle_unmap_page)   (FileHandle *fh, guint pgno, const guint8** mem);
  const guint8*     (* table_handle_checksum_md5) (FileHandle *fh);
  gboolean          (* table_handle_close)        (FileHandle *fh, gint flags);
  gboolean          (* table_handle_write)        (FileHandle *fh, const guint8 *buf, gsize nbyte);
  gboolean          (* table_handle_copy)         (FileHandle *from, FileHandle *to, guint off, guint len);
  gboolean          (* table_handle_getui)        (FileHandle *fh, guint32* i);
  gboolean          (* table_handle_putui)        (FileHandle *fh, guint32 i);
  gssize            (* table_handle_read)         (FileHandle *fh, guint8 *buf, gsize nbyte);
};

struct _SerialSource {
  SerialStatus status;

  const HandleFuncTable* table;

  guint total_alloc;
  guint alloc_pos;
  void* alloc;

  SerialType (* source_type)           (SerialSource* source);
  gboolean   (* source_close)          (SerialSource* source);
  gboolean   (* source_read)           (SerialSource* source, guint8 *ptr, guint32 len);
  void       (* source_free)           (SerialSource* source);
  void       (* source_reset)          (SerialSource* source);

  /* more or less standard, use the functions below, these call source_read.
   * memory is allocated using alloc(). */
  void*      (* source_alloc)        (SerialSource* source, guint32        len);
  gboolean   (* next_bytes_known)    (SerialSource* source, guint8        *ptr, guint32  len);
  gboolean   (* next_bytes)          (SerialSource* source, const guint8 **ptr, guint32 *len);
  gboolean   (* next_uint)           (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint32)         (SerialSource* source, guint32       *ptr);
  gboolean   (* next_uint16)         (SerialSource* source, guint16       *ptr);
  gboolean   (* next_uint8)          (SerialSource* source, guint8        *ptr);
  gboolean   (* next_bool)           (SerialSource* source, gboolean      *ptr);
  gboolean   (* next_string)         (SerialSource* source, const char   **ptr);
};

struct _SerialSink {
  SerialStatus status;

  const HandleFuncTable* table;

  gboolean   (* sink_type)          (SerialSink* sink, SerialType type, guint mem_size);
  gboolean   (* sink_close)         (SerialSink* sink);
  gboolean   (* sink_write)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  void       (* sink_free)          (SerialSink* sink);

  SerialStatus (* sink_quantum)     (SerialSink* sink);

  /* more or less standard, use the functions below, these call sink_write */
  gboolean   (* next_bytes_known)   (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_bytes)         (SerialSink* sink, const guint8 *ptr, guint32 len);
  gboolean   (* next_uint)          (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint32)        (SerialSink* sink, guint32       ptr);
  gboolean   (* next_uint16)        (SerialSink* sink, guint16       ptr);
  gboolean   (* next_uint8)         (SerialSink* sink, guint8        ptr);
  gboolean   (* next_bool)          (SerialSink* sink, gboolean      ptr);
  gboolean   (* next_string)        (SerialSink* sink, const char   *ptr);
};

void     source_default_reset               (SerialSource* source);
void*    source_default_alloc               (SerialSource* source, guint len);
gboolean source_next_uint16                 (SerialSource* source, guint16 *ptr);
gboolean source_next_uint32                 (SerialSource* source, guint32 *ptr);
gboolean source_next_uint8                  (SerialSource* source, guint8 *ptr);
gboolean source_next_uint                   (SerialSource* source, guint32 *ptr);
gboolean source_next_bool                   (SerialSource* source, gboolean *ptr);
gboolean source_next_string                 (SerialSource* source, const char   **ptr);
gboolean source_next_bytes                  (SerialSource* source, const guint8 **ptr, guint32 *len);
gboolean source_next_bytes_known            (SerialSource* source, guint8 *ptr, guint32 len);

gboolean sink_next_uint                     (SerialSink* sink, guint32       ptr);
gboolean sink_next_uint32                   (SerialSink* sink, guint32       ptr);
gboolean sink_next_uint16                   (SerialSink* sink, guint16       ptr);
gboolean sink_next_uint8                    (SerialSink* sink, guint8        ptr);
gboolean sink_next_bool                     (SerialSink* sink, gboolean      ptr);
gboolean sink_next_string                   (SerialSink* sink, const char   *ptr);
gboolean sink_next_bytes                    (SerialSink* sink, const guint8   *ptr, guint32 len);
gboolean sink_next_bytes_known              (SerialSink* sink, const guint8 *ptr, guint32 len);

void         serializeio_initialize_type    (guint32 val, SerialStatus (*func) ());
SerialStatus unserialize_generic            (SerialSource* source, SerialType* object_type, void** object);
SerialStatus unserialize_generic_acceptable (SerialSource* source, guint32 acceptable, SerialType* object_type, void** object);

void ser_sink_init   (SerialSink* sink);
void ser_source_init (SerialSource* source);

SerialSink*   serializeio_handle_sink       (FileHandle *fh, const HandleFuncTable* table);
SerialSource* serializeio_handle_source     (FileHandle *fh, const HandleFuncTable* table);

#endif /* _SERIALIZEIO_H_ */
