/*
 * Handle directories & files
 *
 * Copyright (c) 1998 Chris Ridd <c.ridd@isode.com>
 * Copyright (c) 1997 Guylhem AZNAR <guylhem@oeil.qc.ca>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <dirent.h>
#include <sys/stat.h>
#include "../../configure.h"
#include "../afterstep.h"
#include "aftersteplib.h"

int HomeCreate(char *filename)
{
    int c, error = 0;
    char copyaction[255], target[255], source[255];
    FILE *TARGETFILE, *SOURCEFILE;

    strcpy(copyaction, filename);

    sprintf(target, "%s/%s", AFTER_DIR, copyaction);

    sprintf(source, "%s/%s", AFTER_SHAREDIR, copyaction);

    TARGETFILE = fopen(PutHome(target), "w");

    if (TARGETFILE == (FILE *) NULL) {
	fprintf(stderr, "can't open %s !", target);
	error = -1;
    }
    SOURCEFILE = fopen(PutHome(source), "r");

    if (SOURCEFILE == (FILE *) NULL) {
	fprintf(stderr, "can't open %s !", source);
	error = -1;
    }
    while ((c = getc(SOURCEFILE)) != EOF)
	putc(c, TARGETFILE);

    fclose(TARGETFILE);
    fclose(SOURCEFILE);

    return error;
}

int CheckFile(char *file)
{
    struct stat st;

    if ((stat(file, &st) == -1) || (st.st_mode & S_IFMT) != S_IFREG)
	return (-1);
    else
	return (0);
}

int CheckDir(char *directory)
{
    struct stat st;

    if ((stat(directory, &st) == -1) || (st.st_mode & S_IFMT) != S_IFDIR)
	return (-1);
    else
	return (0);
}

char *CheckOrShare(char *directory, char *homedir, char *sharedir, int share)
{
    char tmp[255], *returned;

    sprintf(tmp, "%s/%s", homedir, directory);

    if (CheckDir(tmp) == -1) {
/*      printf("\nNo directory '%s'\n", tmp); */
	if (share == 0) {
	    sprintf(tmp, "%s/%s", sharedir, directory);
/*          printf("\nTrying with system directory '%s'\n", tmp); */
	    if (CheckDir(tmp) == -1) {
/*              printf("\nWas system directory ! Exiting\n"); */
		return (NULL);
	    } else {
		returned = tmp;
		return (returned);
	    }
	} else {
/*          printf("\nWas system directory ! Exiting\n"); */
	    return (NULL);
	}
    }
    returned = tmp;
    return (returned);
}

char *PutHome(char path_with_home[])
{
    char *Home;			/* the HOME environment variable */
    char *realpath;

/* home dir ? */
    if (strncmp(path_with_home, "~/", 2) == 0) {
/* get home */
	Home = (char *) getenv("HOME");
	if (Home == NULL)
	    Home = "./";
/* alloc it */
	realpath = (char *) safemalloc(strlen(Home) + strlen(&path_with_home[2]) + 3);
/* put it */
	strcpy(realpath, Home);
	strcat(realpath, "/");
	strcat(realpath, &path_with_home[2]);
    } else {
	realpath = (char *) safemalloc(strlen(path_with_home) + 1);
	strcpy(realpath, path_with_home);
    }

    return realpath;
}

void CheckOrCreate(char *WHAT)
{
    char *checkdir;
    mode_t perms = 0755;

    checkdir = PutHome(WHAT);

    if (CheckDir(checkdir) == -1) {
	fprintf(stderr, "Creating %s ... ", WHAT);
	if (mkdir(checkdir, perms) != 0)
	    fprintf(stderr, "ERROR !\n AfterStep depends on %s directory !\nPlease check permissions or contact your sysadmin !\n", WHAT);
	else
	    fprintf(stderr, "done (%s)\n", WHAT);

    }
}

void CheckOrCreateFile(char *WHAT)
{
    char *checkfile;
    FILE *touch;

    checkfile = PutHome(WHAT);

    if (CheckFile(checkfile) == -1) {
	fprintf(stderr, "Creating %s ... ", WHAT);
	if ((touch = fopen(checkfile, "w")) == (FILE *) NULL)
	    fprintf(stderr, "ERROR !\n"
		    " AfterStep depends on %s directory !\n"
		    " Please check permissions or contact your sysadmin !\n", WHAT);
	else {
	    fclose(touch);
	    fprintf(stderr, "done (%s)\n", WHAT);
	}
    }
}

/*
 * Non-NULL select and dcomp pointers are *NOT* tested, but should be OK.
 * They are not used by afterstep however, so this implementation should
 * be good enough.
 *
 * c.ridd@isode.com
 */
int my_scandir(char *dirname, struct direntry *(*namelist[]),
	       int (*select) (struct dirent *),
	       int (*dcomp) (struct direntry **, struct direntry **))
{
    DIR *d;
    struct dirent *e;		/* Pointer to static struct inside readdir() */
    struct direntry **nl;	/* Array of pointers to dirents */
    struct direntry **nnl;
    int n;			/* Count of nl used so far */
    int sizenl;			/* Number of entries in nl array */
    int j;
    size_t realsize;
    char *filename;		/* For building filename to pass to stat */
    char *p;			/* Place where filename starts */
    struct stat buf;


    d = opendir(dirname);

    if (d == NULL)
	return -1;

    filename = malloc(strlen(dirname) + PATH_MAX + 2);
    if (filename == NULL) {
	closedir(d);
	return -1;
    }
    strcpy(filename, dirname);
    p = filename + strlen(filename);
    *p++ = '/';
    *p = 0;			/* Just in case... */

    nl = NULL;
    n = 0;
    sizenl = 0;

    while ((e = readdir(d)) != NULL) {
	if ((select == NULL) || select(e)) {
	    /* add */
	    if (sizenl == n) {
		/* Grow array */
		sizenl += 32;	/* arbitrary delta */
		nnl = realloc(nl, sizenl * sizeof(struct direntry *));
		if (nnl == NULL) {
		    /* Free the old array */
		    for (j = 0; j < n; j++)
			free(nl[j]);
		    free(nl);
		    free(filename);
		    closedir(d);
		    return -1;
		}
		nl = nnl;
	    }
	    realsize = offsetof(struct direntry, d_name) +strlen(e->d_name) + 1;
	    nl[n] = malloc(realsize);
	    if (nl[n] == NULL) {
		for (j = 0; j < n; j++)
		    free(nl[j]);
		free(nl);
		free(filename);
		closedir(d);
		return -1;
	    }
	    /* Fill in the fields using stat() */
	    strcpy(p, e->d_name);
	    if (stat(filename, &buf) == -1) {
		for (j = 0; j <= n; j++)
		    free(nl[j]);
		free(nl);
		free(filename);
		closedir(d);
		return -1;
	    }
	    nl[n]->d_mode = buf.st_mode;
	    nl[n]->d_mtime = buf.st_mtime;
	    strcpy(nl[n]->d_name, e->d_name);
	    n++;
	}
    }
    free(filename);

    if (closedir(d) == -1) {
	free(nl);
	return -1;
    }
    *namelist = realloc(nl, n * sizeof(struct direntry *));

    if (n == 0)
/* OK, but not point sorting or freeing anything */
	return 0;

    if (*namelist == NULL) {
	for (j = 0; j < n; j++)
	    free(nl[j]);
	free(nl);
	return -1;
    }
    /* Optionally sort the list */
    if (dcomp)
	qsort(*namelist, n, sizeof(struct direntry *), (int (*)()) dcomp);

    /* Return the count of the entries */
    return n;
}

/* Sort entries based on their names. A comes before Z. */
int my_alphasort(struct direntry **d1, struct direntry **d2)
{
    return strcmp((*d1)->d_name, (*d2)->d_name);
}				/*
				 * Sort entries based on their mtimes. Old entries come before new entries,
				 * entries with the same times get sorted alphabetically.
				 */
int my_datesort(struct direntry **d1, struct direntry **d2)
{
    int diff = (*d1)->d_mtime - (*d2)->d_mtime;
    if (diff == 0)
	diff = my_alphasort(d1, d2);
    return diff;
}				/*
				 * Use this function as the select argument to my_scandir to make it ignore
				 * all files and directories starting with "."
				 */
int ignore_dots(struct dirent *e)
{
    return (e->d_name[0] != '.');
}
