/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#pragma implementation "icewm.h"
#include "icewm.h"

YTimer *YFrameWindow::fAutoRaiseTimer = 0;
YTimer *YFrameWindow::fDelayFocusTimer = 0;

YFrameWindow::YFrameWindow(YWindow *parent, YFrameClient *client): YFocusWindow(parent) {
    fClient = 0;
    fFocused = false;
    fNextFrame = fPrevFrame = 0;
    fPopupActive = 0;

    normalX = 0;
    normalY = 0;
    normalWidth = 1;
    normalHeight = 1;
    iconX = -1;
    iconY = -1;
    movingWindow = 0;
    sizingWindow = 0;
    indicatorsVisible = 0;
    fFrameFunctions = 0;
    fFrameDecors = 0;
    fFrameOptions = 0;
    fFrameIcon = 0;
#ifdef CONFIG_TASKBAR
    fTaskBarApp = 0;
#endif
#ifdef CONFIG_WINLIST
    fWinListItem = 0;
#endif
    fMiniIcon = 0;
    fTransient = 0;
    fNextTransient = 0;
    fOwner = 0;

    setStyle(wsOverrideRedirect);
     
    assert(client != 0);
    fClient = client;

    fWinWorkspace = manager->activeWorkspace();
    fWinLayer = WinLayerNormal;
    fWinState = 0;

    createPointerWindows();

    fClientContainer = new YClientContainer(this, this);
    fClientContainer->show();
 
    fTitleBar = new YFrameTitleBar(this, this);
    fTitleBar->show();
    fMaximizeButton = new YFrameButton(fTitleBar, this, cmdMaximize, cmdMaximizeVert);
    fMaximizeButton->setWinGravity(NorthEastGravity);
    fMaximizeButton->show();
    fMaximizeButton->setToolTip("Maximize");

    fMinimizeButton = new YFrameButton(fTitleBar, this, cmdMinimize, cmdHide);
    fMinimizeButton->setWinGravity(NorthEastGravity);
    fMinimizeButton->setToolTip("Minimize");
    fMinimizeButton->show();

    fCloseButton = new YFrameButton(fTitleBar, this, cmdClose, cmdClose);
    fCloseButton->setWinGravity(NorthEastGravity);
    fCloseButton->setToolTip("Close");
    if (useXButton)
        fCloseButton->show();
    fMenuButton = new YFrameButton(fTitleBar, this, cmdSysMenu);
    fMenuButton->show();

    getFrameHints();
    updateIcon();

    manage(client);
    insertFrame();

    getDefaultOptions();
    if (frameOptions() & foAllWorkspaces)
        setSticky(true);

    addAsTransient();
    addTransients();

    if (!(frameOptions() & foFullKeys))
        grabKeys();
    fClientContainer->grabButtons();

    if (minimizeToDesktop)
        fMiniIcon = new MiniIcon(this, this);
#ifdef CONFIG_WINLIST
    if (windowList && !(frameOptions() & foIgnoreWinList))
        fWinListItem = windowList->addWindowListApp(this);
#endif
    manager->restackWindows(this, 1);
    if (getLayer() == WinLayerDock)
        manager->updateWorkArea();
#ifdef CONFIG_GUIEVENTS
    wmapp->signalGuiEvent(geWindowOpened);
#endif
}

YFrameWindow::~YFrameWindow() {
#ifdef CONFIG_GUIEVENTS
    wmapp->signalGuiEvent(geWindowClosed);
#endif
    if (fAutoRaiseTimer && fAutoRaiseTimer->getListener() == this) {
        fAutoRaiseTimer->stopTimer();
        fAutoRaiseTimer->setListener(0);
    }
    if (fDelayFocusTimer && fDelayFocusTimer->getListener() == this) {
        fDelayFocusTimer->stopTimer();
        fDelayFocusTimer->setListener(0);
    }
    if (movingWindow || sizingWindow)
        endMoveSize();
    if (fPopupActive)
        fPopupActive->cancelPopup();
    if (manager->colormapWindow() == this)
        manager->setColormapWindow(0);

#ifdef CONFIG_TASKBAR
    if (fTaskBarApp) {
        taskBar->removeApp(this);
        fTaskBarApp = 0;
    }
#endif
#ifdef CONFIG_WINLIST
    if (fWinListItem) {
        if (windowList)
            windowList->removeWindowListApp(fWinListItem);
        fWinListItem = 0;
    }
#endif
    if (fMiniIcon) {
        delete fMiniIcon;
        fMiniIcon = 0;
    }
    fClientContainer->releaseButtons();
    // perhaps should be done another way
    switchWindow->destroyedFrame(this);
    
    if (fClient != 0) {
        XRemoveFromSaveSet(app->display(), client()->clientWindow());
        XDeleteContext(app->display(), client()->clientWindow(), frameContext);
    }

    removeTransients();
    removeAsTransient();
    removeFrame();

    if (getLayer() == WinLayerDock)
        manager->updateWorkArea();

    delete fClient; fClient = 0;
    delete fClientContainer; fClientContainer = 0;
    delete fMenuButton; fMenuButton = 0;
    delete fCloseButton; fCloseButton = 0;
    delete fMaximizeButton; fMaximizeButton = 0;
    delete fMinimizeButton; fMinimizeButton = 0;
    delete fTitleBar; fTitleBar = 0;

    XDestroyWindow(app->display(), topSide);
    XDestroyWindow(app->display(), leftSide);
    XDestroyWindow(app->display(), rightSide);
    XDestroyWindow(app->display(), bottomSide);
    XDestroyWindow(app->display(), topLeftCorner);
    XDestroyWindow(app->display(), topRightCorner);
    XDestroyWindow(app->display(), bottomLeftCorner);
    XDestroyWindow(app->display(), bottomRightCorner);
}


// create 8 windows that are used to show the proper pointer
// on frame (for resize)
void YFrameWindow::createPointerWindows() {
    XSetWindowAttributes attributes;
    unsigned int klass = InputOnly;

    attributes.event_mask = 0;

    attributes.cursor = sizeTopPointer;
    topSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeLeftPointer;
    leftSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeRightPointer;
    rightSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeBottomPointer;
    bottomSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeTopLeftPointer;
    topLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                  CopyFromParent, klass, CopyFromParent,
                                  CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeTopRightPointer;
    topRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                   CopyFromParent, klass, CopyFromParent,
                                   CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomLeftPointer;
    bottomLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                     CopyFromParent, klass, CopyFromParent,
                                     CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomRightPointer;
    bottomRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                      CopyFromParent, klass, CopyFromParent,
                                      CWCursor | CWEventMask, &attributes);

    XMapSubwindows(app->display(), handle());
    indicatorsVisible = 1;
}

void YFrameWindow::grabKeys() {
    if (app->AltMask) {
        grabKey(XK_F1, app->AltMask);
        grabKey(XK_F2, app->AltMask);
        grabKey(XK_F3, app->AltMask);
        grabKey(XK_F4, app->AltMask);
        grabKey(XK_F5, app->AltMask);
        grabKey(XK_F6, app->AltMask);
        grabKey(XK_F7, app->AltMask);
        grabKey(XK_F8, app->AltMask);
        grabKey(XK_F9, app->AltMask);
        grabKey(XK_F10, app->AltMask);
        grabKey(XK_F10, app->AltMask | ShiftMask);
        grabKey(XK_F11, app->AltMask);
        grabKey(XK_F12, app->AltMask);
        //grabKey(' ', app->AltMask);
    }
}

void YFrameWindow::manage(YFrameClient *client) {
    assert(client != 0);
    fClient = client;
    
    XSetWindowBorderWidth(app->display(),
                          client->handle(),
                          0);
    
    XAddToSaveSet(app->display(), client->clientWindow());

    client->reparent(fClientContainer, 0, 0);

    client->setFrame(this);

    sendConfigure();
}

void YFrameWindow::unmanage() {
    assert(fClient != 0);
     
    if (!fClient->destroyed()) {
        int gx, gy;
#ifdef STRICT_POSITIONING
        gx = gy = 0;
#else
        client()->gravityOffsets(gx, gy);
#endif

        XSetWindowBorderWidth(app->display(),
                              client()->handle(),
                              client()->border());
        int posX, posY;

        if (isIconic()) { // !!!! check
            posX = normalX; // - borderX() * gx, // - app->minX(getLayer()),
            posY = normalY; // - borderY() * gy  // - app->minY(getLayer())
        } else {
            posX = x() + borderX();// - borderX() * gx; // - app->minX(getLayer()),
            posY = y() + borderX();// - borderY() * gy;  // - app->minY(getLayer())
        }
        if (gx > 0)
            posX += 2 * borderX() - 1 - client()->border();
        if (gy > 0)
            posY += 2 * borderY() + titleY() - 1 - client()->border();

        client()->reparent(manager, posX, posY);

        if (phase != phaseRestart)
            client()->setFrameState(WithdrawnState);

        XRemoveFromSaveSet(app->display(), client()->clientWindow());
    }

    client()->setFrame(0);

    fClient = 0;
}

void YFrameWindow::configureClient(const XConfigureRequestEvent &configureRequest) {
    client()->setBorder((configureRequest.value_mask & CWBorderWidth) ? configureRequest.border_width : client()->border());
    int cx = (configureRequest.value_mask & CWX) ? configureRequest.x + DO_BORDER(client()->border()) : x() + borderX();
    int cy = (configureRequest.value_mask & CWY) ? configureRequest.y + DO_BORDER(client()->border()) : y() + titleY() + borderY();
    int cwidth = (configureRequest.value_mask & CWWidth) ? configureRequest.width : client()->width();
    int cheight = (configureRequest.value_mask & CWHeight) ? configureRequest.height : client()->height();

    cx -= borderX();
    cy -= borderY() + titleY();
    cwidth += 2 * borderX();
    cheight += 2 * borderY() + titleY();

#if 0
    int gx, gy;
    client()->gravityOffsets(gx, gy);

    cx += borderX() * gx;
    cy += borderY() * gx;
#endif

    if (cx != x() || cy != y() ||
        (unsigned int)cwidth != width() || (unsigned int)cheight != height())
        if (isMaximized()) {
            fWinState &= ~(WinStateMaximizedVert | WinStateMaximizedHoriz); 
            fMaximizeButton->setCommand(cmdMaximize, cmdMaximizeVert);
        }

    MSG(("setting geometry (%d:%d %dx%d)", cx, cy, cwidth, cheight));

    if (isIconic()) {
        cx += borderX();
        cy += borderY();
        cwidth -= 2 * borderX();
        cheight -= 2 * borderY() + titleY();

        client()->setGeometry(0, 0, cwidth, cheight);

        int nx = cx;
        int ny = cy;
        int nw = cwidth;
        int nh = cheight;
        XSizeHints *sh = client()->sizeHints();
        bool cxw = true;
        bool cy = true;
        bool ch = true;

        if (isMaximizedHoriz())
            cxw = false;
        if (isMaximizedVert())
            cy = ch = false;
        if (isRollup())
            ch = false;

        if (cxw) {
            normalX = nx;
            normalWidth = sh ? (nw - sh->base_width) / sh->width_inc : nw;
        }
        if (cy)
            normalY = ny;
        if (ch)
            normalHeight = sh ? (nh - sh->base_height) / sh->height_inc : nh;
    } else
        setGeometry(cx, cy, cwidth, cheight);

    if (configureRequest.value_mask & CWStackMode) {
        YFrameWindow *sibling = 0;
        XWindowChanges xwc;

        if ((configureRequest.value_mask & CWSibling) &&
            XFindContext(app->display(),
                         configureRequest.above,
                         frameContext,
                         (XPointer *)&sibling) == 0)
            xwc.sibling = sibling->handle();
        else
            xwc.sibling = configureRequest.above;

        xwc.stack_mode = configureRequest.detail;

        /* !!! implement the rest, and possibly fix these: */

        if (sibling && xwc.sibling != None) { /* ICCCM suggests sibling==None */
            switch (xwc.stack_mode) {
            case Above:
                setAbove(sibling);
                break;
            case Below:
                setBelow(sibling);
                break;
            default:
                return ;
            }
            XConfigureWindow (app->display(),
                              handle(),
                              configureRequest.value_mask & (CWSibling | CWStackMode),
                              &xwc);
        } else if (xwc.sibling == None && manager->top(getLayer()) != 0) {
            switch (xwc.stack_mode) {
            case Above:
                if (canRaise()) {
                    wmRaise();
                    if (frameOptions() & foNoFocusOnAppRaise)
                        activate();
                }
                break;
            case Below:
                wmLower();
                break;
            default:
                return ;
            }
        } else
            return ;
    }
}

void YFrameWindow::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int /*count*/) {
    if (up.button == 3) {
        popupSystemMenu(up.x_root, up.y_root, -1, -1,
                        YPopupWindow::pfCanFlipVertical |
                        YPopupWindow::pfCanFlipHorizontal |
                        YPopupWindow::pfPopupMenu);
    }
}


void YFrameWindow::handleCrossing(const XCrossingEvent &crossing) {
    if (crossing.type == EnterNotify && crossing.mode == NotifyNormal) {
        if (!clickFocus && visible()) {
            if (!delayPointerFocus)
                focus(false);
            else {
                if (fDelayFocusTimer == 0)
                    fDelayFocusTimer = new YTimer();
                if (fDelayFocusTimer) {
                    fDelayFocusTimer->setInterval(pointerFocusDelay);
                    fDelayFocusTimer->setListener(this);
                    fDelayFocusTimer->startTimer();
                }
            }
        }
        if (autoRaise) {
            if (fAutoRaiseTimer == 0)
                fAutoRaiseTimer = new YTimer();
            if (fAutoRaiseTimer) {
                fAutoRaiseTimer->setInterval(autoRaiseDelay);
                fAutoRaiseTimer->setListener(this);
                fAutoRaiseTimer->startTimer();
            }
        }
    } else if (crossing.type == LeaveNotify && fFocused && focusRootWindow) {
        if (crossing.detail != NotifyInferior &&
            crossing.mode == NotifyNormal)
        {
            if (fDelayFocusTimer && fDelayFocusTimer->getListener() == this) {
                fDelayFocusTimer->stopTimer();
                fDelayFocusTimer->setListener(0);
            }
            if (!clickFocus) {
                deactivate();
            }
            if (autoRaise) {
                if (fAutoRaiseTimer && fAutoRaiseTimer->getListener() == this) {
                    fAutoRaiseTimer->stopTimer();
                    fAutoRaiseTimer->setListener(0);
                }
            }
        }
    }
}

bool YFrameWindow::handleTimer(YTimer *t) {
    if (t == fAutoRaiseTimer) {
        if (canRaise())
            wmRaise();
    }
    if (t == fDelayFocusTimer)
        focus(false);
    return false;
}

void YFrameWindow::raise() {
    if (this != manager->top(getLayer())) {
        YWindow::raise();
        setAbove(manager->top(getLayer()));
    }
}

void YFrameWindow::lower() {
    if (this != manager->bottom(getLayer())) {
        YWindow::lower();
        setAbove(0);
    }
}

void YFrameWindow::removeFrame() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before removing", this);
#endif
    if (prev())
        prev()->setNext(next());
    else
        manager->setTop(getLayer(), next());

    if (next())
        next()->setPrev(prev());
    else
        manager->setBottom(getLayer(), prev());

    YFrameWindow *p = prev(), *n = next();
    
    setPrev(0);
    setNext(0);

    if (this == manager->focus())
        manager->loseFocus(this, n, p);
 
#ifdef DEBUG
    if (debug_z) dumpZorder("after removing", this);
#endif
}

void YFrameWindow::insertFrame() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before inserting", this);
#endif
    setNext(manager->top(getLayer()));
    setPrev(0);
    if (next())
        next()->setPrev(this);
    else
        manager->setBottom(getLayer(), this);
    manager->setTop(getLayer(), this);
#ifdef DEBUG
    if (debug_z) dumpZorder("after inserting", this);
#endif
}

void YFrameWindow::setAbove(YFrameWindow *aboveFrame) {
#ifdef DEBUG
    if (debug_z) dumpZorder("before setAbove", this, aboveFrame);
#endif
    if (aboveFrame != next() && aboveFrame != this) {
        if (prev())
            prev()->setNext(next());
        else
            manager->setTop(getLayer(), next());

        if (next())
            next()->setPrev(prev());
        else
            manager->setBottom(getLayer(), prev());
        
        setNext(aboveFrame);
        if (next()) {
            setPrev(next()->prev());
            next()->setPrev(this);
        } else {
            setPrev(manager->bottom(getLayer()));
            manager->setBottom(getLayer(), this);
        }
        if (prev())
            prev()->setNext(this);
        else
            manager->setTop(getLayer(), this);
#ifdef DEBUG
        if (debug_z) dumpZorder("after setAbove", this, aboveFrame);
#endif
    }
}

void YFrameWindow::setBelow(YFrameWindow *belowFrame) {
    if (belowFrame != next())
        setAbove(belowFrame->next());
}

YFrameWindow *YFrameWindow::findWindow(int flags) {
    YFrameWindow *p = this;

     if (flags & fwfNext) 
         goto next;
     
     do {
         if ((flags & fwfMinimized) && !p->isMinimized())
             goto next;
         if ((flags & fwfUnminimized) && !(p->visible() && !p->isMinimized()))
             goto next;
         if ((flags & fwfVisible) && !p->visible())
             goto next;
         if ((flags & fwfFocusable) && !p->isFocusable())
             goto next;
         if ((flags & fwfWorkspace) && !p->visibleOn(manager->activeWorkspace()))
             goto next;
         if ((flags & fwfSwitchable) && (p->frameOptions() & foIgnoreQSwitch))
             goto next;
         
         return p;

     next:
         if (flags & fwfBackward) 
             p = (flags & fwfLayers) ? p->prevLayer() : p->prev();
         else
             p = (flags & fwfLayers) ? p->nextLayer() : p->next();
         if (p == 0)
             if (!(flags & fwfCycle))
                 return 0;
             else if (flags & fwfBackward)
                 p = (flags & fwfLayers) ? manager->bottomLayer() : manager->bottom(getLayer());
             else 
                 p = (flags & fwfLayers) ? manager->topLayer() : manager->top(getLayer());
     } while (p != this);

     if (!(flags & fwfSame))
         return 0;
     if ((flags & fwfVisible) && !p->visible())
         return 0;
     if ((flags & fwfFocusable) && !p->isFocusable())
         return 0;
     if ((flags & fwfWorkspace) && !p->visibleOn(manager->activeWorkspace()))
         return 0;
     
     return this;
}

void YFrameWindow::handleConfigure(const XConfigureEvent &configure) {
}

void YFrameWindow::sendConfigure() {
    XEvent xev;
    
    xev.xconfigure.type = ConfigureNotify;
    xev.xconfigure.display = app->display();
    xev.xconfigure.event = client()->handle();
    xev.xconfigure.window = client()->handle();
    xev.xconfigure.x = x() + borderX() - DO_BORDER(client()->border());
    xev.xconfigure.y = y() + borderY()
#ifndef TITLEBAR_BOTTOM
        + titleY()
#endif
        - DO_BORDER(client()->border());
    xev.xconfigure.width = client()->width();
    xev.xconfigure.height = client()->height();
    xev.xconfigure.border_width = client()->border();

    assert(titlebar() != 0);
    xev.xconfigure.above = None; //titlebar()->handle();
    xev.xconfigure.override_redirect = False;

#ifdef DEBUG_C
    Status rc = 
#endif
        XSendEvent(app->display(),
               client()->clientWindow(),
               False,
               StructureNotifyMask,
               &xev);

#ifdef DEBUG_C
    MSG(("sent %d: x=%d, y=%d, width=%d, height=%d",
         rc,
         x(),
         y(),
         client()->width(),
         client()->height()));
#endif
}

void YFrameWindow::handleCommand(WMCommand command, void *context, unsigned int modifiers) {
    switch (command) {
    case cmdActivate: activate(); break;
    case cmdRestore: wmRestore(); break;
    case cmdMinimize: wmMinimize(); break;
    case cmdMaximize: wmMaximize(); break;
    case cmdMaximizeVert: wmMaximizeVert(); break;
    case cmdLower: wmLower(); break;
    case cmdRaise: wmRaise(); break;
    case cmdRollup: wmRollup(); break;
    case cmdClose: wmClose(); break;
    case cmdKill: wmKill(); break;
    case cmdHide: wmHide(); break;
    case cmdShow: wmShow(); break;
    case cmdMove: wmMove(); break;
    case cmdSize: wmSize(); break;
    case cmdOccupyAll: wmOccupyAll(); break;
    case cmdOccupyAllOrCurrent: wmOccupyAllOrCurrent(); break;
    case cmdOccupyWorkspace: wmOccupyWorkspace((int)context); break;
    case cmdOccupyOnlyWorkspace: wmOccupyOnlyWorkspace((int)context); break;
    case cmdMoveToWorkspace: wmMoveToWorkspace((int)context); break;
    case cmdSetLayer: wmSetLayer((CARD32)context); break;
    default:
	wmapp->handleCommand(command, context, modifiers);
        break;
    }
}

void YFrameWindow::wmSetLayer(CARD32 layer) {
    setLayer(layer);
}

void YFrameWindow::wmMove() {
    startMoveSize(1, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmSize() {
    startMoveSize(0, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmRestore() {
#ifdef CONFIG_GUIEVENTS
    wmapp->signalGuiEvent(geWindowRestore);
#endif
    setState(WinStateMaximizedVert | WinStateMaximizedHoriz |
             WinStateMinimized |
             WinStateHidden |
             WinStateRollup, 0);
}

void YFrameWindow::wmMinimize() {
#ifdef DEBUG_S
    MSG(("wmMinimize - Frame: %d", visible()));
    MSG(("wmMinimize - Client: %d", client()->visible()));
#endif
    if (isMinimized()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateMinimized, 0);
    } else {
        if (!canMinimize())
            return ;
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowMin);
#endif
        setState(WinStateMinimized, WinStateMinimized);
        wmLower();
    }
    manager->focusTopWindow();
}

void YFrameWindow::DoMaximize(CARD32 flags) {
    setState(WinStateRollup, 0);

    if (isMaximized()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateMaximizedVert |
                 WinStateMaximizedHoriz |
                 WinStateMinimized, 0);
    } else {
        if (!canMaximize())
            return ;

#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowMax);
#endif
        setState(WinStateMaximizedVert |
                 WinStateMaximizedHoriz |
                 WinStateMinimized, flags);
    }
}

void YFrameWindow::wmMaximize() {
    DoMaximize(WinStateMaximizedVert | WinStateMaximizedHoriz);
}

void YFrameWindow::wmMaximizeVert() {
    if (isMaximizedVert()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateMaximizedVert, 0);
    } else {
        if (!canMaximize())
            return ;
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowMax);
#endif
        setState(WinStateMaximizedVert, WinStateMaximizedVert);
    }
}

void YFrameWindow::wmMaximizeHorz() {
    if (isMaximizedHoriz()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateMaximizedHoriz, 0);
    } else {
        if (!canMaximize())
            return ;
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowMax);
#endif
        setState(WinStateMaximizedHoriz, WinStateMaximizedHoriz);
    }
}

void YFrameWindow::wmRollup() {
    if (isRollup()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateRollup, 0);
    } else {
        if (!canRollup())
            return ;
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRollup);
#endif
        setState(WinStateRollup, WinStateRollup);
    }
}

void YFrameWindow::wmHide() {
    if (isHidden()) {
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowRestore);
#endif
        setState(WinStateHidden, 0);
    } else {
        if (!canHide())
            return ;

#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowHide);
#endif
        setState(WinStateHidden, WinStateHidden);
    }
    manager->focusTopWindow();
}

void YFrameWindow::wmLower() {
    if (this != manager->bottom(getLayer())) {
        YFrameWindow *w = this;

#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWindowLower);
#endif
        while (w) {
            w->doLower();
            w = w->owner();
        }
        manager->restackWindows(this, -1);
        if (clickFocus)
            manager->focusTopWindow();
    }
    if (raiseOnFocus)
        manager->focusTopWindow();
}

void YFrameWindow::doLower() {
    setAbove(0);
}

void YFrameWindow::wmRaise() {
    doRaise();
    manager->restackWindows(this, 1);
}

void YFrameWindow::doRaise() {
#ifdef DEBUG
    if (debug_z) dumpZorder("wmRaise: ", this);
#endif
    if (this != manager->top(getLayer())) {
        setAbove(manager->top(getLayer()));
        {
            YFrameWindow *w = transient();
            while (w) {
                w->doRaise();
                w = w->nextTransient();
            }
        }
#ifdef DEBUG
        if (debug_z) dumpZorder("wmRaise after raise: ", this);
#endif
    }
}

void YFrameWindow::wmClose() {
    if (!canClose())
        return ;

    XGrabServer(app->display());
    client()->getProtocols();

    if (client()->protocols() & YFrameClient::wpDeleteWindow)
        client()->sendMessage(_XA_WM_DELETE_WINDOW);
    else
        wmKill();
    XUngrabServer(app->display());
}

void YFrameWindow::wmKill() {
    if (!canClose())
        return ;
#ifdef DEBUG
    msg("No WM_DELETE_WINDOW protocol");
#endif
    XBell(app->display(), 100);
    XKillClient(app->display(), client()->clientWindow());
}

void YFrameWindow::wmPrevWindow() {
    if (next() != this) {
        YFrameWindow *f = findWindow(fwfNext | fwfBackward | fwfVisible | fwfCycle | fwfFocusable | fwfLayers | fwfWorkspace);
        if (f) {
            f->wmRaise();
            manager->setFocus(f, true);
        }
    }
}

void YFrameWindow::wmNextWindow() {
    if (next() != this) {
        wmLower();
        manager->setFocus(findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfLayers | fwfWorkspace), true);
    }
}

void YFrameWindow::wmLastWindow() {
    if (next() != this) {
        YFrameWindow *f = findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfLayers | fwfWorkspace);
        if (f) {
            f->wmRaise();
            manager->setFocus(f, true);
        }
    }
}

void YFrameWindow::loseFocus() {
    if (fFocused) {
        fFocused = false;
        if (focusOnClickClient || raiseOnClickClient)
            if (fClientContainer)
                fClientContainer->grabButtons();
        if (isIconic())
            fMiniIcon->repaint();
        else {
            repaint();
            titlebar()->deactivate();
        }
#ifdef CONFIG_TASKBAR
        if (fTaskBarApp)
            fTaskBarApp->repaint();
#endif
    }
}

void YFrameWindow::setFocus(bool canWarp) {
    if (!fFocused) {
        fFocused = true;

        if (isIconic())
            fMiniIcon->repaint();
        else {
            titlebar()->activate();
            repaint();
        }
#ifdef CONFIG_TASKBAR
        if (fTaskBarApp)
            fTaskBarApp->repaint();
#endif
        
        if (canWarp && warpPointer && !clickFocus && phase == phaseRunning) {
            /* warp pointer sucks */
            XWarpPointer(app->display(), None, manager->handle(),
                         0, 0, 0, 0, x() + borderX(), y() + borderY() + titleY());
        }

        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);

        if (focusOnClickClient &&
            !(raiseOnClickClient && (this != manager->top(getLayer()))))
            fClientContainer->releaseButtons();

        if (client()->protocols() & YFrameClient::wpTakeFocus)
            client()->sendMessage(_XA_WM_TAKE_FOCUS);
    }
}

void YFrameWindow::focusOnMap() {
    if (owner() != 0) {
        if (focusOnMapTransient)
            activate();
    } else {
        if (::focusOnMap)
            activate();
    }
}

void YFrameWindow::wmShow() {
   // recover lost (offscreen) windows !!! (unify with code below)
    if (x() >= int(manager->width()) ||
        y() >= int(manager->height()) ||
        x() <= - int(width()) ||
        y() <= - int(height()))
    {
        int newX = x();
        int newY = y();
        
        if (x() >= int(manager->width()))
            newX = int(manager->width() - width() + borderX());
        if (y() >= int(manager->height()))
            newY = int(manager->height() - height() + borderY());
        
        if (newX < int(- borderX()))
            newX = int(- borderX());
        if (newY < int(- borderY()))
            newY = int(- borderY());
        setPosition(newX, newY);
    }

    setState(WinStateHidden | WinStateMinimized, 0);
}

void YFrameWindow::focus(bool canWarp) {
    if (!visibleOn(manager->activeWorkspace()))
        manager->activateWorkspace(getWorkspace());
    // recover lost (offscreen) windows !!!
    if (x() >= int(manager->width()) ||
        y() >= int(manager->height()) ||
        x() <= - int(width()) ||
        y() <= - int(height()))
    {
        int newX = x();
        int newY = y();
        
        if (x() >= int(manager->width()))
            newX = int(manager->width() - width() + borderX());
        if (y() >= int(manager->height()))
            newY = int(manager->height() - height() + borderY());
        
        if (newX < int(- borderX()))
            newX = int(- borderX());
        if (newY < int(- borderY()))
            newY = int(- borderY());
        setPosition(newX, newY);
    }

    if (raiseOnFocus)
        wmRaise();
    if (isFocusable())
        manager->setFocus(this, canWarp);
}

void YFrameWindow::activate(bool canWarp) {
    if (fWinState & (WinStateHidden | WinStateMinimized))
        setState(WinStateHidden | WinStateMinimized, 0);
    focus(canWarp);
}

void YFrameWindow::deactivate() {
    if (manager->focus() == this)
        manager->loseFocus(this);
}

void YFrameWindow::paint(Graphics &g, int , int , unsigned int , unsigned int ) {
    YColor *bg;

    if (!(frameDecors() & fdResize))
        return ;

    if (focused())
        bg = activeBorderBg;
    else
        bg = inactiveBorderBg;

    g.setColor(bg);
    switch (wmLook) {
#if defined(CONFIG_LOOK_WIN95) || defined(CONFIG_LOOK_WARP4) || defined(CONFIG_LOOK_NICE)
#ifdef CONFIG_LOOK_WIN95
    case lookWin95:
#endif
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4:
#endif
#ifdef CONFIG_LOOK_NICE
    case lookNice:
#endif
        g.fillRect(1, 1, width() - 3, height() - 3);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
        break;
#endif
#if defined(CONFIG_LOOK_MOTIF) || defined(CONFIG_LOOK_WARP3)
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif:
#endif
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3:
#endif
        g.fillRect(1, 1, width() - 2, height() - 2);
        g.draw3DRect(0, 0, width() - 1, height() - 1, true);
        g.draw3DRect(borderX() - 1, borderY() - 1,
                     width() - 2 * borderX() + 1, height() - 2 * borderY() + 1,
                     false);

#ifdef CONFIG_LOOK_MOTIF
        if (wmLook == lookMotif && canSize()) {
            YColor *b = bg->brighter();
            YColor *d = bg->darker();


            g.setColor(d);
            g.drawLine(wsCornerX - 1, 0, wsCornerX - 1, height() - 1);
            g.drawLine(width() - wsCornerX - 1, 0, width() - wsCornerX - 1, height() - 1);
            g.drawLine(0, wsCornerY - 1, width(),wsCornerY - 1);
            g.drawLine(0, height() - wsCornerY - 1, width(), height() - wsCornerY - 1);
            g.setColor(b);
            g.drawLine(wsCornerX, 0, wsCornerX, height() - 1);
            g.drawLine(width() - wsCornerX, 0, width() - wsCornerX, height() - 1);
            g.drawLine(0, wsCornerY, width(), wsCornerY);
            g.drawLine(0, height() - wsCornerY, width(), height() - wsCornerY);
        }
        break;
#endif
#endif
#ifdef CONFIG_LOOK_PIXMAP
    case lookPixmap:
    case lookMetal:
    case lookGtk:
        {
            int n = focused() ? 1 : 0;
            int t = (frameDecors() & fdResize) ? 0 : 1;

            g.drawPixmap(frameTL[t][n], 0, 0);
            g.repHorz(frameT[t][n], wsCornerX, 0, width() - 2 * wsCornerX);
            g.drawPixmap(frameTR[t][n], width() - wsCornerX, 0);
            g.repVert(frameL[t][n], 0, wsCornerY, height() - 2 * wsCornerY);
            g.repVert(frameR[t][n], width() - borderX(), wsCornerY, height() - 2 * wsCornerY);
            g.drawPixmap(frameBL[t][n], 0, height() - wsCornerY);
            g.repHorz(frameB[t][n], wsCornerX, height() - borderY(), width() - 2 * wsCornerX);
            g.drawPixmap(frameBR[t][n], width() - wsCornerX, height() - wsCornerY);
        }
        break;
#endif
    default:
        break;
    }
}

void YFrameWindow::handlePopDown(YPopupWindow *popup) {
    MSG(("popdown %ld up %ld", popup, fPopupActive));
    if (fPopupActive == popup)
        fPopupActive = 0;
}

void YFrameWindow::popupSystemMenu() {
    if (fPopupActive == 0) {
        if (fMenuButton && fMenuButton->visible() &&
            fTitleBar && fTitleBar->visible())
            fMenuButton->popupMenu();
        else {
            int ax = x() + container()->x();
            int ay = y() + container()->y();
            if (isIconic()) {
                ax = x();
                ay = y() + height();
            }
            popupSystemMenu(ax, ay,
                            -1, -1, //width(), height(),
                            YPopupWindow::pfCanFlipVertical);
        }
    }
}

void YFrameWindow::popupSystemMenu(int x, int y,
                                   int x_delta, int y_delta,
                                   unsigned int flags,
                                   YWindow *forWindow)
{
    if (fPopupActive == 0) {
        updateMenu();
        if (windowMenu()->popup(forWindow, this, this,
                                x, y, x_delta, y_delta, flags))
            fPopupActive = windowMenu();
    }
}

void YFrameWindow::updateTitle() {
    titlebar()->repaint();
    updateIconTitle();
}

void YFrameWindow::updateIconTitle() {
#ifdef CONFIG_TASKBAR
    if (fTaskBarApp) {
        fTaskBarApp->repaint();
        fTaskBarApp->setToolTip((char *)client()->windowTitle());
    }
#endif
    if (isIconic()) {
        fMiniIcon->repaint();
    }
}

void YFrameWindow::wmOccupyAllOrCurrent() {
    if (isSticky()) {
        setWorkspace(manager->activeWorkspace());
        setSticky(false);
    } else {
        setSticky(true);
    }
#ifdef CONFIG_TASKBAR
    taskBar->relayout();
#endif
}

void YFrameWindow::wmOccupyAll() {
    setSticky(!isSticky());
#ifdef CONFIG_TASKBAR
    taskBar->relayout();
#endif
}

void YFrameWindow::wmOccupyWorkspace(CARD32 workspace) {
    assert(workspace < workspaceCount);
    setWorkspace(workspace);
#ifdef CONFIG_TASKBAR
    taskBar->relayout();
#endif
}

void YFrameWindow::wmOccupyOnlyWorkspace(CARD32 workspace) {
    assert(workspace < workspaceCount);
    setWorkspace(workspace);
    setSticky(false);
#ifdef CONFIG_TASKBAR
    taskBar->relayout();
#endif
}

void YFrameWindow::wmMoveToWorkspace(CARD32 workspace) {
    wmOccupyOnlyWorkspace(workspace);
}

void YFrameWindow::addToMenu(YMenu *menu) {
    YMenu::YMenuItem *item =
        new YMenu::YMenuItem((char *)client()->windowTitle(), -1, 0,
                             cmdActivateWindow, 0, this);
    if (item) {
        if (clientIcon())
            item->setPixmap(clientIcon()->small());
        menu->add(item);
    }
}

void YFrameWindow::getFrameHints() {
#ifndef NO_MWM_HINTS
    CARD32 decors = client()->mwmDecors();
    CARD32 functions = client()->mwmFunctions();

    fFrameFunctions = ffRollup;
    fFrameDecors = 0;
    fFrameOptions = 0;
    
    if (decors & MWM_DECOR_BORDER)      fFrameDecors |= fdBorder;
    if (decors & MWM_DECOR_RESIZEH)     fFrameDecors |= fdResize;
    if (decors & MWM_DECOR_TITLE)       fFrameDecors |= fdTitleBar;
    if (decors & MWM_DECOR_MENU)        fFrameDecors |= fdSysMenu;
    if (decors & MWM_DECOR_MAXIMIZE)    fFrameDecors |= fdMaximize;
    if (decors & MWM_DECOR_MINIMIZE)    fFrameDecors |= fdMinimize;

    if (functions & MWM_FUNC_MOVE)      fFrameFunctions |= ffMove;
    if (functions & MWM_FUNC_RESIZE)    fFrameFunctions |= ffResize;
    if (functions & MWM_FUNC_MAXIMIZE)  fFrameFunctions |= ffMaximize;
    if (functions & MWM_FUNC_MINIMIZE)
        fFrameFunctions |= ffMinimize | ffHide;
    if (functions & MWM_FUNC_CLOSE) {
        fFrameFunctions |= ffClose;
        fFrameDecors |= fdClose; /* hack */
    }
#else
    fFrameFunctions =
        ffMove | ffResize | ffClose |
        ffMinimize | ffMaximize | ffHide | ffRollup;
    fFrameDecors =
        fdTitleBar | fdSysMenu | fdBorder | fdResize |
        fdClose | fdMinimize | fdMaximize;
    fFrameOptions = 0;
#endif

#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;

    getWindowOptions(wo, false);

    /*fprintf(stderr, "decor: %lX %lX %lX %lX %lX %lX",
            wo.function_mask, wo.functions,
            wo.decor_mask, wo.decors,
            wo.option_mask, wo.options);*/
     
    fFrameFunctions &= ~wo.function_mask;
    fFrameFunctions |= wo.functions;
    fFrameDecors &= ~wo.decor_mask;
    fFrameDecors |= wo.decors;
    fFrameOptions &= ~wo.option_mask;
    fFrameOptions |= wo.options;
#endif
}

#ifndef NO_WINDOW_OPTIONS

void YFrameWindow::getWindowOptions(WindowOption &opt, bool remove) {
    memset((void *)&opt, 0, sizeof(opt));
    opt.workspace = WinWorkspaceInvalid;
    opt.layer = WinLayerInvalid;

    getWindowOptions(defOptions, opt, false);
    getWindowOptions(hintOptions, opt, remove);
}

void YFrameWindow::getWindowOptions(WindowOptions *list, WindowOption &opt, bool remove) {
    XClassHint *h = client()->classHint();
    WindowOption *wo;
    
    if (!h)
        return;

    if (h->res_name && h->res_class) {
        char *both = (char *) MALLOC(strlen(h->res_name) + 1 +
                                     strlen(h->res_class) + 1);
        if (both) {
            strcpy(both, h->res_class);
            strcat(both, ".");
            strcat(both, h->res_name);
        }
        wo = both ? list->getWindowOption(both, false, remove) : 0;
        if (wo) WindowOptions::combineOptions(opt, *wo);
        FREE(both);
    }
    if (h->res_class) {
        wo = list->getWindowOption(h->res_class, false, remove);
        if (wo) WindowOptions::combineOptions(opt, *wo);
    }
    if (h->res_name) {
        wo = list->getWindowOption(h->res_name, false, remove);
        if (wo) WindowOptions::combineOptions(opt, *wo);
    }
    wo = list->getWindowOption(0, false, remove);
    if (wo) WindowOptions::combineOptions(opt, *wo);
}
#endif

void YFrameWindow::getDefaultOptions() {
#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;
    getWindowOptions(wo, true);
    
    if (wo.icon) {
        if (fFrameIcon)
            delete fFrameIcon;
        fFrameIcon = getIcon(wo.icon);
    }
    if (wo.workspace != WinWorkspaceInvalid && wo.workspace < workspaceCount)
        setWorkspace(wo.workspace);
    if (wo.layer != WinLayerInvalid && wo.layer < WinLayerCount)
        setLayer(wo.layer);
#endif
}

void YFrameWindow::updateIcon() {
    int count;
    CARD32 *elem;
    Pixmap *pixmap;
    Atom type;

    if (client()->getWinIcons(&type, &count, &elem)) {
        if (type == _XA_WIN_ICONS)
            fFrameIcon = new YIcon(elem[0], elem[1], elem + 2);
        else // compatibility
            fFrameIcon = new YIcon(count, 2, elem);
        XFree(elem);
    } else if (client()->getKwmIcon(&count, &pixmap) && count == 2) {
        XWMHints *h = client()->hints();
        if (h && (h->flags & IconPixmapHint)) {
            CARD32 pix[4];
            pix[0] = pixmap[0];
            pix[1] = pixmap[1];
            pix[2] = h->icon_pixmap;
            pix[3] = (h->flags & IconMaskHint) ? h->icon_mask : None;
            fFrameIcon = new YIcon(2, 2, pix);
        } else {
            CARD32 pix[2];
            for (int i = 0; i < count; i++) {
                pix[i] = pixmap[i];
            }
            pix[0] = pixmap[0];
            pix[1] = pixmap[1];
            fFrameIcon = new YIcon(count / 2, 2, pix);
        }
        XFree(pixmap);
    } else {
        XWMHints *h = client()->hints();
        if (h && (h->flags & IconPixmapHint) && (h->flags & IconMaskHint)) {
            CARD32 pix[2];
            pix[0] = h->icon_pixmap;
            pix[1] = (h->flags & IconMaskHint) ? h->icon_mask : None;
            fFrameIcon = new YIcon(1, 2, pix);
        }
    }
    if (fFrameIcon) {
        if (fFrameIcon->small() == 0 &&
            fFrameIcon->large() == 0) 
        {
            delete fFrameIcon;
            fFrameIcon = 0;
        }
    }
}

YFrameWindow *YFrameWindow::nextLayer() {
    if (fNextFrame)
        return fNextFrame;
    CARD32 l = getLayer();
    while (l-- > 0)
        if (manager->top(l))
            return manager->top(l);
    return 0;
}

YFrameWindow *YFrameWindow::prevLayer() {
    if (fPrevFrame)
        return fPrevFrame;
    CARD32 l = getLayer();
    while (++l < WinLayerCount)
        if (manager->bottom(l))
            return manager->bottom(l);
    return 0;
}

YMenu *YFrameWindow::windowMenu() {
    //if (frameOptions() & foFullKeys)
    //    return windowMenuNoKeys;
    //else
    return ::windowMenu;
}

void YFrameWindow::addAsTransient() {
    Window fTransientFor = client()->ownerWindow();
    if (fTransientFor) {
        fOwner = manager->getFrame(fTransientFor);
        if (fOwner != 0) {
            MSG(("transient for 0x%lX: 0x%lX", fTransientFor, fOwner));
            if (fOwner) {
                fNextTransient = fOwner->transient();
                fOwner->setTransient(this);
            } else {
                fTransientFor = 0; // ?

                fNextTransient = 0;
                fOwner = 0;
            }
        }
    }
}

void YFrameWindow::removeAsTransient() {
    if (fOwner) {
        MSG(("removeAsTransient"));
        
        YFrameWindow *w = fOwner->transient(), *cp = 0;

        while (w) {
            if (w == this) {
                if (cp)
                    cp->setNextTransient(nextTransient());
                else
                    fOwner->setTransient(nextTransient());
            }
            w = w->nextTransient();
        }
        fOwner = 0;
        fNextTransient = 0;
    }
}

void YFrameWindow::addTransients() {
    YFrameWindow *w = manager->bottomLayer();

    while (w) {
        if (w->owner() == 0)
            w->addAsTransient();
        w = w->prevLayer();
    }
}

void YFrameWindow::removeTransients() {
    if (transient()) {
        MSG(("removeTransients"));
        YFrameWindow *w = transient(), *n;

        while (w) {
            n = w->nextTransient();
            w->setNextTransient(0);
            w->setOwner(0);
            w = n;
        }
        fTransient = 0;
    }
}

bool YFrameWindow::isModal() {
    if (!client())
        return false;

    MwmHints *mwmHints = client()->mwmHints();
    if (mwmHints && (mwmHints->flags & MWM_HINTS_INPUT_MODE))
        if (mwmHints->input_mode != MWM_INPUT_MODELESS)
            return true;

    if (hasModal())
        return true;
    
    return false;
}

bool YFrameWindow::hasModal() {
    YFrameWindow *w = transient();
    while (w) {
        if (w->isModal())
            return true;
        w = w->nextTransient();
    }
    /* !!! missing code for app modal dialogs */
    return false;
}

bool YFrameWindow::isFocusable() {
    if (hasModal())
        return false;

    if (!client())
        return false;

    XWMHints *hints = client()->hints();
    
    if (!hints)
        return true;
#if 1
    if (frameOptions() & foIgnoreNoFocusHint)
        return true;
    if (!(hints->flags & InputHint))
        return true;
    if (hints->input)
        return true;
    if (client()->protocols() & YFrameClient::wpTakeFocus)
        return true;
    return false;
#else
    return true;
#endif
}

void YFrameWindow::updateWorkspace() {
    if (isSticky() || visibleOn(manager->activeWorkspace()))
        setState(WinStateHidWorkspace, 0);
    else
        setState(WinStateHidWorkspace, WinStateHidWorkspace);
}

void YFrameWindow::setWorkspace(CARD32 workspace) {
    if (workspace >= workspaceCount)
        return ;
    if (workspace != fWinWorkspace) {
        fWinWorkspace = workspace;
        client()->setWinWorkspaceHint(fWinWorkspace);
        updateWorkspace();
    }
}

void YFrameWindow::setLayer(CARD32 layer) {
    if (layer >= WinLayerCount)
        return ;
    if (layer != fWinLayer) {
        CARD32 oldLayer = fWinLayer;

        removeFrame();
        fWinLayer = layer;
        insertFrame();
        client()->setWinLayerHint(fWinLayer);
        manager->restackWindows(this, 1);
        if (getLayer() == WinLayerDock || oldLayer == WinLayerDock)
            manager->updateWorkArea();
    }
}

void YFrameWindow::updateState() {
    client()->setWinStateHint(fWinState, fWinState);

    FrameState newState = NormalState;
    bool show_frame = true;
    bool show_client = true;

    // some commented out code it against the ICCCM.
    // some applications misbehave either way.
    // (some hide windows on iconize, this is bad when switching workspaces
    // or rolling up the window).
    //
    
    if (isHidden()) {
        show_frame = false;
        show_client = false;
        newState = IconicState;
    } else if (isHidWorkspace() || isHidTransient()) { /// !!! ICCCM ?
        show_frame = false;
        //if (isMinimized() || isRollup()) {
        show_client = false;
        newState = IconicState;
        //} else
        //    show_client = true;
    } else if (isMinimized()) {
        if (minimizeToDesktop)
            show_frame = true;
        else
            show_frame = false;
        show_client = false;
        newState = IconicState;
    } else if (isRollup()) {
        show_frame = true;
        show_client = false;
        newState = IconicState; // ICCCM
    } else {
        show_frame = true;
        show_client = true;
    }

    MSG(("updateState: winState=%lX, frame=%d, client=%d",
         fWinState, show_frame, show_client));

    if (show_client) {
        client()->show();
        fClientContainer->show();
    }
    if (show_frame)
        show();
    else
        hide();
    if (!show_client) {
        fClientContainer->hide();
        client()->hide();
    }

    client()->setFrameState(newState);
}

#if 0
void YFrameWindow::getNormalSize(int *x, int *y, int *w, int *h) {
    XSizeHints *sh = client()->sizeHints();
    bool cxw = true;
    bool cy = true;
    bool ch = true;
    
    //fprintf(stderr, "current: (%d:%d %dx%d)\n", *x, *y, *w, *h);

    if (isIconic())
        cxw = cy = ch = false;
    else {
        if (isMaximizedHoriz())
            cxw = false;
        if (isMaximizedVert())
            cy = ch = false;
        if (isRollup())
            ch = false;
    }
    if (!cxw) {
        if (x) *x = normalX;
        if (w) *w = sh ? normalWidth * sh->width_inc + sh->base_width : normalWidth;
    }
    if (!cy)
        if (y) *y = normalY;
    if (!ch)
        if (h) *h = sh ? normalHeight * sh->height_inc + sh->base_height : normalHeight;

    //fprintf(stderr, "getNormal: (%d:%d %dx%d)\n", normalX, normalY, normalWidth, normalHeight);
}
#endif

void YFrameWindow::updateNormalSize() {

    if (isIconic()) {
        iconX = this->x();
        iconY = this->y();
    } else {
        int nx = this->x() + borderX();
        int ny = this->y() + borderY();
        int nw = client()->width();
        int nh = client()->height();
        XSizeHints *sh = client()->sizeHints();
        bool cxw = true;
        bool cy = true;
        bool ch = true;

        if (isMaximizedHoriz())
            cxw = false;
        if (isMaximizedVert())
            cy = ch = false;
        if (isRollup())
            ch = false;

        if (cxw) {
            normalX = nx;
            normalWidth = sh ? (nw - sh->base_width) / sh->width_inc : nw;
        }
        if (cy)
            normalY = ny;
        if (ch)
            normalHeight = sh ? (nh - sh->base_height) / sh->height_inc : nh;
    }

    //fprintf(stderr, "setNormal: (%d:%d %dx%d) icon (%d:%d)\n", normalX, normalY, normalWidth, normalHeight, iconX, iconY);
}

void YFrameWindow::updateLayout() {
    XSizeHints *sh = client()->sizeHints();
    int nx = normalX;
    int ny = normalY;
    int nw = sh ? normalWidth * sh->width_inc + sh->base_width : normalWidth;
    int nh = sh ? normalHeight * sh->height_inc + sh->base_height : normalHeight;

    if (isIconic()) {
        if (iconX == -1 && iconY == -1)
            manager->getIconPosition(this, &iconX, &iconY);
        nx = iconX;
        ny = iconY;
        nw = fMiniIcon->width();
        nh = fMiniIcon->height();
    } else {
        if (isMaximizedHoriz()) {
            nw = manager->maxWidth(getLayer());

            if (nx + nw > int(manager->maxX(getLayer())))
                nx = manager->minX(getLayer());
            if (nx < manager->minX(getLayer()))
                nx = manager->minX(getLayer());
        }
        if (isMaximizedVert()) {
            nh = manager->maxHeight(getLayer()) - titleY();

            if (ny + nh + int(wsTitleBar) > int(manager->maxY(getLayer())))
                ny = manager->minY(getLayer());
            if (ny < manager->minY(getLayer()))
                ny = manager->minY(getLayer());
        }
        client()->constrainSize(nw, nh, getLayer());

        if (isRollup())
            nh = 0;

        nx -= borderX();
        ny -= borderY();
        nw += 2 * borderX();
        nh += 2 * borderY() + titleY();
    }
    setGeometry(nx, ny, nw, nh);
}

void YFrameWindow::setState(CARD32 mask, CARD32 state) {
    updateNormalSize();

    CARD32 fOldState = fWinState;
    CARD32 fNewState = fWinState = (fWinState & ~mask) | (state & mask);

    if ((fOldState ^ fNewState) & WinStateAllWorkspaces) {
        MSG(("WinStateAllWorkspaces: %d", isSticky()));
        updateWorkspace();
    }
    MSG(("setState: oldState: %lX, newState: %lX, mask: %lX, state: %lX", 
         fOldState, fNewState, mask, state));
    //fprintf(stderr, "normal1: (%d:%d %dx%d)\n", normalX, normalY, normalWidth, normalHeight);
    if ((fOldState ^ fNewState) & (WinStateMaximizedVert |
                                   WinStateMaximizedHoriz))
    {
        MSG(("WinStateMaximized: %d", isMaximized()));

        if (fMaximizeButton)
            if (isMaximized())
                fMaximizeButton->setCommand(cmdRestore, cmdRestore);
            else
                fMaximizeButton->setCommand(cmdMaximize, cmdMaximizeVert);
    }
    if ((fOldState ^ fNewState) & WinStateMinimized) {
        MSG(("WinStateMinimized: %d", isMaximized()));
        
        if (minimizeToDesktop && fMiniIcon) {
            if (isIconic()) {
                fMiniIcon->raise();
                fMiniIcon->show();
            } else {
                fMiniIcon->hide();
                iconX = x();
                iconY = y();
            }
        }
        manager->focusTopWindow();
    }
    if ((fOldState ^ fNewState) & WinStateRollup) {
        MSG(("WinStateRollup: %d", isRollup()));
    }
    if ((fOldState ^ fNewState) & WinStateHidden) {
        MSG(("WinStateHidden: %d", isHidden()));
#ifdef CONFIG_TASKBAR
        updateTaskBar();
#endif
    }
    if ((fOldState ^ fNewState) & WinStateHidWorkspace) {
        MSG(("WinStateHidWorkspace: %d", isHidWorkspace()));
        if (isHidWorkspace()) {
#ifdef CONFIG_TASKBAR
            updateTaskBar();
#endif
        } else {
#ifdef CONFIG_TASKBAR
            updateTaskBar();
#endif
        }
    }
    if ((fOldState ^ fNewState) & WinStateDockHorizontal) {
        if (getLayer() == WinLayerDock)
            manager->updateWorkArea();
    }

    updateLayout();
    updateState();

    if (this == manager->focus())
	//if ((fOldState ^ fNewState) & WinStateRollup)
        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);
    //fprintf(stderr, "normal2: (%d:%d %dx%d)\n", normalX, normalY, normalWidth, normalHeight);
}

void YFrameWindow::setSticky(bool sticky) {
    setState(WinStateAllWorkspaces, sticky ? WinStateAllWorkspaces : 0);
}

void YFrameWindow::updateMwmHints() {
    int bx = borderX();
    int by = borderY();
    int ty = titleY(), tt;
    
    getFrameHints();

    int gx, gy;
    client()->gravityOffsets(gx, gy);

#ifdef TITLEBAR_BOTTOM
    if (gy == -1)
#else
    if (gy == 1)
#endif
        tt = titleY() - ty;
    else
        tt = 0;
    
    setGeometry(x() + bx - borderX(),
                y() + by - borderY() + tt,
                client()->width() + 2 * borderX(),
                client()->height() + 2 * borderY() + titleY());
}

YIcon *YFrameWindow::clientIcon() {
    YFrameWindow *f = this;
    while (f) {
        if (f->getClientIcon())
            return f->getClientIcon();
        f = f->owner();
    }
    return defaultAppIcon;
}

void YFrameWindow::updateProperties() {
    client()->setWinWorkspaceHint(fWinWorkspace);
    client()->setWinLayerHint(fWinLayer);
    client()->setWinStateHint(fWinState,
                              fWinState);
}

#ifdef CONFIG_TASKBAR
void YFrameWindow::updateTaskBar() {
    bool needTaskBarApp = false;

    if (taskBar) {
        if (!isHidden() &&
            !(frameOptions() & foIgnoreTaskBar))
            if (taskBarShowAllWindows || visibleOn(manager->activeWorkspace()))
                needTaskBarApp = true;

        if (needTaskBarApp && fTaskBarApp == 0)
            fTaskBarApp = taskBar->addApp(this);

        if (fTaskBarApp)
            fTaskBarApp->setShown(needTaskBarApp);
        taskBar->relayout();
    }
}
#endif
