/*
 * IceWM
 *
 * Copyright (C) 1997,1988 Marko Macek
 */

#include "icewm.h"

YWindowManager::YWindowManager(YWindow *parent, Window win): YDesktop(parent, win) {
    setStyle(wsManager);
    setPointer(leftPointer);

    if (app->AltMask != 0) {
        grabKey(XK_Tab, app->AltMask);
        grabKey(XK_Tab, app->AltMask | ShiftMask);
        grabKey(XK_Escape, ControlMask | app->AltMask | ShiftMask);
        grabKey(XK_Escape, app->AltMask);
        grabKey(XK_Escape, app->AltMask | ShiftMask);
        grabKey(XK_Delete, ControlMask | app->AltMask);
        grabKey(XK_Delete, ControlMask | app->AltMask | ShiftMask); // if c-a-d broken

        grabKey(XK_Left, app->AltMask | ControlMask);
        grabKey(XK_Right, app->AltMask | ControlMask);
        grabKey(XK_Left, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_Right, app->AltMask | ControlMask | ShiftMask);

        grabKey(XK_1, app->AltMask | ControlMask);
        grabKey(XK_2, app->AltMask | ControlMask);
        grabKey(XK_3, app->AltMask | ControlMask);
        grabKey(XK_4, app->AltMask | ControlMask);
        grabKey(XK_5, app->AltMask | ControlMask);
        grabKey(XK_6, app->AltMask | ControlMask);
        grabKey(XK_7, app->AltMask | ControlMask);
        grabKey(XK_8, app->AltMask | ControlMask);
        grabKey(XK_9, app->AltMask | ControlMask);
        grabKey(XK_0, app->AltMask | ControlMask);

        grabKey(XK_1, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_2, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_3, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_4, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_5, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_6, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_7, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_8, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_9, app->AltMask | ControlMask | ShiftMask);
        grabKey(XK_0, app->AltMask | ControlMask | ShiftMask);
    }
    grabKey(XK_Escape, ShiftMask);
    grabKey(XK_Escape, ControlMask);
    grabKey(XK_Escape, ControlMask | ShiftMask);

    metaDown = false;
    buttonsPressed = 0;
    if (win95keys) {
        grabKey(XK_Menu, app->AltMask);

        grabKey(XK_Left, app->MetaMask);
        grabKey(XK_Right, app->MetaMask);
        grabKey(XK_Left, app->MetaMask | ShiftMask);
        grabKey(XK_Right, app->MetaMask | ShiftMask);
        

        //fix
        grabKey(XK_Meta_L, 0);
        grabKey(XK_Meta_R, 0);
        //grabKey(AnyKey, app->MetaMask);
    }

    if (useMouseWheel) {
        grabButton(4, app->MetaMask);
        grabButton(5, app->MetaMask);
        
        grabButton(4, ControlMask | app->AltMask);
        grabButton(5, ControlMask | app->AltMask);
    }
    
    YWindow::setFocus();

    fFocus = 0;
    for (int l = 0; l < WinLayerCount; l++)
        fTop[l] = fBottom[l] = 0;
    fColormapWindow = 0;
    fActiveWorkspace = WinWorkspaceInvalid;
}

YWindowManager::~YWindowManager() {
}

bool YWindowManager::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        KeySym k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        unsigned int m = KEY_MODMASK(key.state);

        MSG(("down key: %d, mod: %d", k, m));

        if (metaDown) {
            MSG(("meta down"));
            if (k == XK_Left && (win95keys && m == app->MetaMask)) {
                activateWorkspace((activeWorkspace() + workspaceCount() - 1) % workspaceCount());
            } else if (k == XK_Right && (win95keys && m == app->MetaMask)) {
                activateWorkspace((activeWorkspace() + 1) % workspaceCount());
            } else {
                XAllowEvents(app->display(), ReplayKeyboard, CurrentTime);//!!!???FIX
            }
            app->releaseEvents();
            metaDown = false;
        } else {
            if (k == XK_Left &&
                (app->AltMask && (m == (app->AltMask | ControlMask)) ||
                (win95keys && m == app->MetaMask && app->MetaMask)))
            {
                activateWorkspace((activeWorkspace() + workspaceCount() - 1) % workspaceCount());
            } else if (k == XK_Right &&
                       (app->AltMask && (m == (app->AltMask | ControlMask)) ||
                       (win95keys && m == app->MetaMask && app->MetaMask)))
            {
                activateWorkspace((activeWorkspace() + 1) % workspaceCount());
            } else if (m == (app->AltMask | ControlMask) && app->AltMask &&
                       (k >= XK_0 && k <= XK_9))
            {
                CARD32 nw = (k == XK_0) ? 9 : (k - XK_1);

                if (nw >= 0 && nw < workspaceCount())
                    activateWorkspace(nw);
            } else if (k == XK_Escape && m == (app->AltMask | ControlMask | ShiftMask))
                app->exit(0);
            else if (k == XK_Delete && m == (app->AltMask | ControlMask)) {
                ctrlAltDelete->activate();
            } else if (k == XK_Delete && m == (app->AltMask | ControlMask | ShiftMask)) {
                app->exit(0);
#ifdef CONFIG_TASKBAR
            } else if (k == XK_Escape && m == ControlMask) {
                if (taskBar->visible())
                    taskBar->raise();
                //app->popupWindowListMenu(key.x_root, key.y_root);
                popupStartMenu();
#endif
#ifdef CONFIG_WINLIST
            } else if (k == XK_Escape && m == ControlMask + ShiftMask) {
                if (windowList)
                    windowList->showFocused();
#endif
            }
            else if (k == XK_Tab && m == app->AltMask + ShiftMask)
                switchWindow->begin(0);
            else if (k == XK_Tab && m == app->AltMask)
                switchWindow->begin(1);
            else if (win95keys && k == XK_Menu && m == app->AltMask)
                popupWindowListMenu();
            /// !!! fix
            else if (win95keys && m == 0 && (k == XK_Meta_L || k == XK_Meta_R)) {
                if (app->grabEvents(this, None, ButtonPressMask)) {
                    metaDown = true;
                    MSG(("meta pressed"));
                }
            } else if (focus()) {
                if (k == XK_Escape && app->AltMask && m == (app->AltMask | ShiftMask))
                    focus()->wmPrevWindow();
                else if (k == XK_Escape && m == ShiftMask)
                    focus()->popupSystemMenu();
                else if (k == XK_Left &&
                         ((m == (app->AltMask | ControlMask | ShiftMask)) ||
                         (win95keys && m == (app->MetaMask | ShiftMask))))
                {
                    CARD32 nw = (activeWorkspace() +
                                 workspaceCount() - 1) % workspaceCount();
                    
                    focus()->wmOccupyWorkspace(nw);
                    activateWorkspace(nw);
                    focus()->wmOccupyOnlyWorkspace(nw);
                } else if (k == XK_Right &&
                           ((m == (app->AltMask | ControlMask | ShiftMask)) ||
                           (win95keys && m == (app->MetaMask | ShiftMask))))
                {
                    CARD32 nw = (activeWorkspace() + 1) % workspaceCount();
                    
                    focus()->wmOccupyWorkspace(nw);
                    activateWorkspace(nw);
                    focus()->wmOccupyOnlyWorkspace(nw);
                } else if (m == (app->AltMask | ControlMask | ShiftMask) && app->AltMask &&
                           (k >= XK_0 && k <= XK_9))
                {
                    CARD32 nw = (k == XK_0) ? 9 : (k - XK_1);

                    if (nw >= 0 && nw < workspaceCount()) {
                        focus()->wmOccupyWorkspace(nw);
                        activateWorkspace(nw);
                        focus()->wmOccupyOnlyWorkspace(nw);
                    }
                } else if (m == app->AltMask) {
                    if (k == XK_Escape)
                        focus()->wmNextWindow();
                }
            }
        }
    } else if (key.type == KeyRelease) {
        KeySym k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        unsigned int m = KEY_MODMASK(key.state);

        MSG(("up key: %d, mod: %d", k, m));
        
        if (metaDown) {
            MSG(("meta is down"));
            metaDown = false;

            if (win95keys && (k == XK_Meta_L)) {
                app->releaseEvents();
                popupStartMenu();
#ifdef CONFIG_WINLIST
            } else if (win95keys && (k == XK_Meta_R)) {
                app->releaseEvents();
                if (windowList)
                    windowList->showFocused();
#endif
            }
        }
    }
    return true;
}

void YWindowManager::handleButton(const XButtonEvent &button) {
    if (metaDown) {
        app->releaseEvents();
        metaDown = false;
    }
    if (button.type == ButtonPress) {
        if (BUTTON_MODMASK(button.state) == 0)
            buttonsPressed = 0;
        if (button.button == 1)
            buttonsPressed |= 1;
        if (button.button == 2)
            buttonsPressed |= 2;
        if (button.button == 3)
            buttonsPressed |= 4;
    }
    if ((button.type == ButtonRelease) &&
        (button.button == 1 || button.button == 3) &&
        (BUTTON_MODMASK(button.state) == Button1Mask + Button3Mask))
    {
    }
    if (useMouseWheel && focus() && button.type == ButtonPress &&
        ((KEY_MODMASK(button.state) == app->MetaMask && app->MetaMask) ||
         (KEY_MODMASK(button.state) == ControlMask + app->AltMask && app->AltMask)))
    {
        if (button.button == 4)
            focus()->wmNextWindow();
        else if (button.button == 5)
            focus()->wmPrevWindow();
    } else if (button.type == ButtonRelease) {
        if (button.button == 3 && buttonsPressed == 4 && BUTTON_MODMASK(button.state) == Button3Mask) {
            rootMenu->popup(0, wmapp, 0, button.x, button.y, -1, -1,
                        YPopupWindow::pfCanFlipVertical |
                        YPopupWindow::pfCanFlipHorizontal |
                        YPopupWindow::pfPopupMenu);
#ifdef CONFIG_WINLIST
        } else if (button.button == 1 || button.button == 3) {
            if ((BUTTON_MODMASK(button.state) == Button3Mask) ||
                (BUTTON_MODMASK(button.state) == Button1Mask))
            {
                if (buttonsPressed == (1 | 4) && windowList)
                    windowList->showFocused();
            }
#endif
        }
    }
    YWindow::handleButton(button);
}

void YWindowManager::handleClick(const XButtonEvent &down, const XButtonEvent &up, int /*count*/) {
    if (up.button == 1 && BUTTON_MODMASK(down.state) == 0)
        popupWindowListMenu(up.x, up.y);
//    else if (up.button == 3 && BUTTON_MODMASK(down.state) == 0)

}

void YWindowManager::handleConfigureRequest(const XConfigureRequestEvent &configureRequest) {
    YFrameWindow *frame;

    frame = getFrame(configureRequest.window);
    if (frame == 0) {
        XWindowChanges xwc;
        
        MSG(("root configure request -- client"));

	xwc.x = configureRequest.x;
	xwc.y = configureRequest.y;
	xwc.width = configureRequest.width;
	xwc.height = configureRequest.height;
	xwc.border_width = configureRequest.border_width;
	xwc.stack_mode = configureRequest.detail;
	xwc.sibling = configureRequest.above;
        XConfigureWindow(app->display(), configureRequest.window,
                         configureRequest.value_mask, &xwc);

        {
            Window rootw;
            unsigned int bw, depth;
            int fX, fY;
            unsigned int fWidth, fHeight;

            XGetGeometry(app->display(), configureRequest.window, &rootw,
                         &fX, &fY, &fWidth, &fHeight, &bw, &depth);
            /*fX = attributes.x;
             fY = attributes.y;
             fWidth = attributes.width;
             fHeight = attributes.height;*/

            MSG(("changed geometry (%d:%d %dx%d)",
                 fX, fY, fWidth, fHeight));
        }
    } else {
        MSG(("root configure request -- frame"));
        frame->configureClient(configureRequest);
    } 
}

void YWindowManager::handleMapRequest(const XMapRequestEvent &mapRequest) {
    mapWindow(mapRequest.window);
}

void YWindowManager::handleUnmap(const XUnmapEvent &unmap) {
    if (unmap.send_event)
        unmanageWindow(unmap.window, false);
}

void YWindowManager::handleDestroyWindow(const XDestroyWindowEvent &destroyWindow) {
    if (destroyWindow.window == handle())
        YWindow::handleDestroyWindow(destroyWindow);
}

void YWindowManager::handleClientMessage(const XClientMessageEvent &message) {
    if (message.message_type == _XA_WIN_WORKSPACE) {
        setWinWorkspace(message.data.l[0]);
    }
}

YFrameWindow *YWindowManager::getFrame(Window win) {
    YFrameWindow *frame;

    if (XFindContext(app->display(),
                     win,
                     frameContext,
                     (XPointer *)&frame) == 0)
        return frame;
    else
        return 0;
}

YFrameWindow *YWindowManager::createFrame(YFrameClient *client) {
    return new YFrameWindow(this, client);
}

YFrameClient *YWindowManager::getClient(Window win) {
    YFrameClient *client;

    if (XFindContext(app->display(),
                     win,
                     clientContext,
                     (XPointer *)&client) == 0)
        return client;
    else
        return 0;
}

YFrameClient *YWindowManager::createClient(Window win) {
    return new YFrameClient(this, 0, win);
}

void YWindowManager::setFocus(YFrameWindow *window, bool canWarp) {
    assert((YWindow *)window != this);

    if (window == fFocus)
        return ;
    
    if (focus())
        focus()->loseFocus();
    fFocus = window;
    if (focus())
        focus()->setFocus(canWarp);
    else
        XSetInputFocus(app->display(), PointerRoot, RevertToPointerRoot, app->getEventTime());
    if (!pointerColormap && colormapWindow() != focus())
        setColormapWindow(focus());
}

void YWindowManager::loseFocus(YFrameWindow *window) {
    assert(window != 0);
#ifdef DEBUG
    if (debug_z) dumpZorder("losing focus: ", window);
#endif
    YFrameWindow *w = window->findWindow(YFrameWindow::fwfNext |
                                         YFrameWindow::fwfVisible |
                                         YFrameWindow::fwfCycle |
                                         YFrameWindow::fwfFocusable |
                                         YFrameWindow::fwfLayers |
                                         YFrameWindow::fwfWorkspace);

    assert (w != window);
    setFocus(w, false);
}

void YWindowManager::loseFocus(YFrameWindow *window,
                             YFrameWindow *next,
                             YFrameWindow *prev)
{
    assert(window != 0);
#ifdef DEBUG
    if (debug_z) dumpZorder("close: losing focus: ", window);
#endif

    YFrameWindow *w = 0;
    
    if (next)
        w = next->findWindow(YFrameWindow::fwfVisible |
                             YFrameWindow::fwfCycle |
                             YFrameWindow::fwfFocusable |
                             YFrameWindow::fwfLayers |
                             YFrameWindow::fwfWorkspace);
    else if (prev)
        w = prev->findWindow(YFrameWindow::fwfNext |
                             YFrameWindow::fwfVisible |
                             YFrameWindow::fwfCycle |
                             YFrameWindow::fwfFocusable |
                             YFrameWindow::fwfLayers |
                             YFrameWindow::fwfWorkspace);

    assert (w != window);
    setFocus(w, false);
}

void YWindowManager::activate(YFrameWindow *window, bool canWarp) {
    if (window) {
        window->wmRaise();
        window->activate(canWarp);
    }
//    else
//        XBell(display(), 100);
}

void YWindowManager::setTop(CARD32 layer, YFrameWindow *top) {
    if (fTop[layer]) {
        if (raiseOnClickClient)
            fTop[layer]->container()->grabButtons();
    }
    fTop[layer] = top;
    if (fTop[layer]) {
        if (raiseOnClickClient &&
           !(focusOnClickClient && !fTop[layer]->focused()))
            fTop[layer]->container()->releaseButtons();
    }
}

void YWindowManager::installColormap(Colormap cmap) {
    //MSG(("installing colormap 0x%lX", cmap));
    if (app->grabWindow() == 0) {
        if (cmap == None) {
            XInstallColormap(app->display(), defaultColormap);
        } else {
            XInstallColormap(app->display(), cmap);
        }
    }
}

void YWindowManager::setColormapWindow(YFrameWindow *frame) {
    if (fColormapWindow != frame) {
        fColormapWindow = frame;

        if (colormapWindow())
            installColormap(colormapWindow()->client()->colormap());
        else
            installColormap(None);
    }
}

void YWindowManager::manageClients() {
    unsigned int clientCount;
    Window winRoot, winParent, *winClients;
 
    XGrabServer(app->display());
    XSync(app->display(), 0);
    XQueryTree(app->display(), handle(),
               &winRoot, &winParent, &winClients, &clientCount);

    if (winClients)
        for (unsigned int i = 0; i < clientCount; i++)
            if (findClient(winClients[i]) == 0) {
                XWindowAttributes attributes;

                /// !!! optimize calling of XGetWindowAttributes (currently 3 times)
                XGetWindowAttributes(app->display(),
                                     winClients[i],
                                     &attributes);

                if (attributes.map_state != IsUnmapped)
                    manageClient(winClients[i]);
            }

    XUngrabServer(app->display());
    if (winClients)
        XFree(winClients);
    updateWorkArea();
    focusTopWindow();
}

void YWindowManager::unmanageClients() {
    Window w;

    setFocus(0);
    XGrabServer(app->display());
    for (unsigned int l = 0; l < WinLayerCount; l++) {
        while (bottom(l)) {
            w = bottom(l)->client()->handle();
            unmanageWindow(w, true);
        }
    }
    XUngrabServer(app->display());
}

YFrameClient *YWindowManager::manageClient(Window win) {
    YFrameClient *client = manageWindow(win);
    if (client && client->visible()) {
        mapWindow(win, 0);
    }

    return client;
}

YFrameClient *YWindowManager::findClient(Window win) {
    YFrameClient *client = getClient(win);
    if (client == 0) {
        YFrameWindow *frame = getFrame(win);
        if (frame)
            client = frame->client();
        else
            client = 0;
    }
    return client;
}

YFrameClient *YWindowManager::manageWindow(Window win) {
    YFrameClient *client = 0;

    MSG(("managing window 0x%lX", win));

    XEvent xev;
#ifdef NEED_GRAB
    XGrabServer(app->display());
    XSync(app->display(), 0);
#endif
    if (XCheckTypedWindowEvent(app->display(), win, DestroyNotify, &xev))
        goto end;

    client = getClient(win);

    if (client == 0) {
        XWindowAttributes attributes;
    
        if (!XGetWindowAttributes(app->display(),
                                  win,
                                  &attributes))
            goto end;

        if (attributes.override_redirect)
            goto end;

        client = createClient(win);

        client->setBorder(attributes.border_width);

        client->setColormap(attributes.colormap);
    }
end:
#ifdef NEED_GRAB
    XUngrabServer(display());
#endif
    return client;
}

void YWindowManager::placeWindow(YFrameWindow *frame, int x, int y, int newClient) {
    YFrameClient *client = frame->client();

    int posWidth = client->width() + 2 * frame->borderX();
    int posHeight = client->height() + 2 * frame->borderY() + frame->titleY();

    if (newClient && client->sizeHints() &&
        !(client->sizeHints()->flags & (USPosition | PPosition))
#ifdef CONFIG_WINLIST
        && !(client == windowList)
#endif
       )
    {
        static int lastX = 0;
        static int lastY = 0;

        /// !!! make auto placement cleaner and (optionally) smarter
        if (lastX < minX(frame->getLayer())) lastX = minX(frame->getLayer());
        if (lastY < minY(frame->getLayer())) lastY = minY(frame->getLayer());

        x = lastX;
        y = lastY;

        lastX += wsTitleBar;
        lastY += wsTitleBar;

        if (int(y + posHeight) >= maxY(frame->getLayer())) {
            y = minY(frame->getLayer());
            lastY = wsTitleBar;
        }
        if (int(x + posWidth) >= maxX(frame->getLayer())) {
            x = minX(frame->getLayer());
            lastX = wsTitleBar;
        }
        if (frame->getLayer() == WinLayerDock) {
            if (frame->getState() & WinStateDockHorizontal)
                y = 0;
            else
                x = 0;
        }
        
        newClient = 0;
        //x -= frame->borderX();
        //y -= frame->borderY();
    } else {
    }

    int posX = x - DO_BORDER(frame->client()->border());
    int posY = y - DO_BORDER(frame->client()->border());

    if (1) {
        int gx, gy;
        
        client->gravityOffsets(gx, gy);
       
        // !!! why is -1 needed ?
        if (gx > 0)
            posX -= 2 * frame->borderX() - 1 - client->border();
        if (gy > 0)
            posY -= 2 * frame->borderY() + frame->titleY() - 1 - client->border();

#ifndef STRICT_POSITIONING
        /* !!! possibly problem with ICCCM ? */
        posX += frame->borderX() * gx;
        posY += frame->borderY() * gy;

        /* position on screen, should really be configurable */
        posX += frame->borderX(); 
        posY += frame->borderY(); 
        posWidth -= 2 * frame->borderX();
        posHeight -= 2 * frame->borderY();

        /// !!! make constrainPosition function
        if (posX + posWidth > int(maxX(frame->getLayer())))
            posX = maxX(frame->getLayer()) - posWidth;
        if (posY + posHeight > int(maxY(frame->getLayer())))
            posY = maxY(frame->getLayer()) - posHeight;
        if (posX < minX(frame->getLayer()))
            posX = minX(frame->getLayer());
        if (posY < minY(frame->getLayer()))
            posY = minY(frame->getLayer());
        
        posX -= frame->borderX(); 
        posY -= frame->borderY(); 
        posWidth += 2 * frame->borderX();
        posHeight += 2 * frame->borderY();
#endif
    }

    MSG(("mapping geometry (%d:%d %dx%d)", posX, posY, posWidth, posHeight));
    frame->setGeometry(posX,
                       posY,
                       posWidth,
                       posHeight);
}

YFrameWindow *YWindowManager::mapWindow(Window win, int newClient) {
    YFrameWindow *frame = getFrame(win);
    bool canManualPlace = false;

    MSG(("mapping window 0x%lX", win));

    if (frame == 0) {
        
        YFrameClient *client = getClient(win);
        if (client == 0) {
            client = manageWindow(win);
        }

        if (client == 0)
            return 0;

        MSG(("initial geometry (%d:%d %dx%d)",
             client->x(), client->y(), client->width(), client->height()));
        
        int x = client->x();
        int y = client->y();

        frame = createFrame(client);
        placeWindow(frame, x, y, newClient);

#ifdef SHAPE
        frame->setShape();
#endif

        if (manualPlacement && phase == phaseRunning &&
#ifdef CONFIG_WINLIST
            client != windowList &&
#endif
            !frame->owner() &&
            (!client->sizeHints() ||
             !(client->sizeHints()->flags & (USPosition | PPosition))))
            canManualPlace = true;
    }

    MSG(("Map - Frame: %d", frame->visible()));
    MSG(("Map - Client: %d", frame->client()->visible()));

    CARD32 workspace, layer, state_mask, state = 0;
    
    if (frame->client()->getWinLayerHint(&layer))
        frame->setLayer(layer);

    if (frame->client()->getWinStateHint(&state_mask, &state)) {
#if 0
        /// !!!
	FrameState st = frame->client()->getFrameState();
        MSG(("map: mask=%lX, state=%lX, framestate=%d\n",
             (unsigned long)state_mask,
             (unsigned long)state,
             st);
#endif
        frame->setState(state_mask, state);
    }
    else {
        FrameState st = frame->client()->getFrameState();
        
        if (st == WithdrawnState) {
            XWMHints *h = frame->client()->hints();
            if (h && (h->flags & StateHint))
                st = h->initial_state;
            else
                st = NormalState;
        }
        
        switch (st) {
        case IconicState:
            frame->setState(WinStateMinimized, WinStateMinimized);
            break;
            
        case NormalState:
        case WithdrawnState:
            if (canManualPlace && !opaqueMove)
                frame->manualPlace();
            
            frame->updateState();
            //frame->changeState(NormalState);
            
            if (phase == phaseRunning/* || frame->visibleOn(activeWorkspace())*/)
                frame->focusOnMap();
            if (canManualPlace && opaqueMove)
                frame->wmMove();
            break;
        }
        frame->updateState();
    }
    if (frame->client()->getWinWorkspaceHint(&workspace))
        frame->setWorkspace(workspace);
    frame->updateProperties();
    frame->updateTaskBar();

    if (frame->getLayer() == WinLayerDock)
        updateWorkArea();
    return frame;
}

YFrameClient *YWindowManager::unmapWindow(Window win) {
    YFrameWindow *frame = getFrame(win);
    
    MSG(("unmapping window 0x%lX", win));

    if (frame) {
        YFrameClient *client = frame->client();

        client->hide();
        frame->hide();
        frame->unmanage();
        delete frame;
        return client;
    } else {
        YFrameClient *client = getClient(win);
        
        return client;
    }
}

void YWindowManager::unmanageWindow(Window win, bool map) {
    YFrameClient *client = unmapWindow(win);

    MSG(("unmanaging window 0x%lX", win));

    if (client) {
        if (map)
            client->show();
        delete client;
    }
}

void YWindowManager::destroyedWindow(Window win) {
    YFrameWindow *frame = getFrame(win);
    if (frame) {
        delete frame;
    } else {
        YFrameClient *client = getClient(win);
        if (client) 
            delete client;
        else
            MSG(("destroyed? unknown window: 0x%lX", win));
    }
}

void YWindowManager::focusTopWindow() {
    if (topLayer())
        setFocus(topLayer()->findWindow(YFrameWindow::fwfVisible |
                                        YFrameWindow::fwfFocusable |
                                        YFrameWindow::fwfWorkspace |
                                        YFrameWindow::fwfSame |
                                        YFrameWindow::fwfLayers));
}

YFrameWindow *YWindowManager::topLayer() {
    for (int l = WinLayerCount - 1; l >= 0; l--)
        if (fTop[l] != 0)
            return fTop[l];
    return 0;
}

YFrameWindow *YWindowManager::bottomLayer() {
    for (int l = 0; l < WinLayerCount; l++)
        if (fBottom[l] != 0)
            return fBottom[l];
    return 0;
}

void YWindowManager::restackWindows(YFrameWindow *win, int oper) {
    int count = 0;
    YFrameWindow *f;

    if (oper == 0) {
        f = top(win->getLayer());
        for (; f; f = f->next()) count++;
    } else if (oper == 1) {
        f = win;
        for (; f; f = f->prev()) count++;
    } else if (oper == -1) {
        f = win;
        for (; f; f = f->next()) count++;
    }

    if (oper == 1) {

        for (CARD32 ll = win->getLayer() + 1; ll < WinLayerCount; ll++) {
            f = bottom(ll);
            for (; f; f = f->prev()) count++;
        }

#ifdef CONFIG_TASKBAR
        if (taskBar->visible())
            count++; // taskBar
#endif

#ifdef CONFIG_WINLIST
        if (windowList && windowList->frame())
            count++;
#endif
        
        YPopupWindow *p = app->popup();
        while (p) {
            count++;
            p = p->prevPopup();
        }
        if (statusMoveSize->visible())
            count++;

        if (ctrlAltDelete->visible())
            count++;
    }

    Window *w = (Window *)MALLOC(sizeof(Window *) * count);
    if (w == 0)
        return ;
    
    int i = 0;

    if (oper == 1) {
        for (CARD32 ll = WinLayerCount - 1; ll > win->getLayer() + 1; ll--) {
            f = top(ll);
            for (; f; f = f->next())
                w[i++] = f->handle();
        }
        
        if (statusMoveSize->visible())
            w[i++] = statusMoveSize->handle();

        YPopupWindow *p = app->popup();
        while (p) {
            w[i++] = p->handle();
            p = p->prevPopup();
        }

#ifdef CONFIG_TASKBAR
        if (taskBar->visible())
            w[i++] = taskBar->handle();
#endif
#ifdef CONFIG_WINLIST
        if (windowList && windowList->frame())
            w[i++] = windowList->frame()->handle();
#endif

        if (ctrlAltDelete->visible())
            w[i++] = ctrlAltDelete->handle();
    }
    
    if (oper == -1) {
        f = win;
        for (; i < count; i++, f = f->next()) {
            w[i] = f->handle();
        }
    } else {
        f = top(win->getLayer());
        for (; i < count; i++, f = f->next()) {
            w[i] = f->handle();
        }
    }
    
    if (count > 0) {
        if (oper == 1) {
#if 1
            XRaiseWindow(app->display(), w[0]);
#else
            /* !!! must determine correct top window */
            if (win->next()) {
                XWindowChanges xwc;

                xwc.sibling = win->next()->handle();
                xwc.stack_mode = Above;
                XConfigureWindow(app->display(), w[0], CWSibling | CWStackMode, &xwc);
            }
#endif
        } else if (oper == -1) {
#if 1
            XLowerWindow(app->display(), w[0]);
#else
            if (win->prev()) {
                XWindowChanges xwc;

                xwc.sibling = win->prev()->handle();
                xwc.stack_mode = Below;
                XConfigureWindow(app->display(), w[0], CWSibling | CWStackMode, &xwc);
            }
#endif
        }
        if (count > 1)
            XRestackWindows(app->display(), w, count);
    }
    FREE(w);
}

int YWindowManager::minX(CARD32 layer) {
    if (layer >= WinLayerDock)
        return 0;
    else
        return fMinX;
}
int YWindowManager::minY(CARD32 layer) {
    if (layer >= WinLayerDock)
        return 0;
    else
        return fMinY;
}

int YWindowManager::maxX(CARD32 layer) {
    if (layer >= WinLayerDock)
        return width();
    else
        return fMaxX;
}

int YWindowManager::maxY(CARD32 layer) {
    if (layer >= WinLayerDock)
        return height();
    else
        return fMaxY;
}

void YWindowManager::updateWorkArea() {
    int nx1, ny1, nx2, ny2;
    
    nx1 = 0;
    ny1 = 0;
    nx2 = width();
    ny2 = height();

    int midX = (nx1 + nx2) / 2;
    int midY = (ny1 + ny2) / 2;

    /// !!! FIX: taskbar is separate from WORKAREA mechanism for now

#ifdef CONFIG_TASKBAR
    if (taskBar->visible() && !taskBarAutoHide) {
        if (taskBar->y() <= 0)
            ny1 = taskBar->height() + taskBar->y();
        else
            ny2 = taskBar->y();
    }
#endif

    for (YFrameWindow *w = fTop[WinLayerDock]; w; w = w->next()) {
        int anx1 = nx1, any1 = ny1, anx2 = nx2, any2 = ny2;
        // !!! FIX: WORKAREA windows must currently be sticky
        if (w->isHidden() || !w->visibleNow() || !w->isSticky())
            continue;
        if (!(w->getState() & WinStateDockHorizontal)) {
            
            if (w->x() + int(w->width()) < midX)
                anx1 = w->x() + w->width();
            else if (w->x() > midX)
                anx2 = w->x();
        } else {
            
            if (w->y() + int(w->height()) < midY)
                any1 = w->y() + w->height();
            else if (w->y() > midY)
                any2 = w->y();
        }
        if (anx1 > nx1) nx1 = anx1;
        if (any1 > ny1) ny1 = any1;
        if (anx2 < nx2) nx2 = anx2;
        if (any2 < ny2) ny2 = any2;
    }
    if (fMinX != nx1 ||
        fMinY != ny1 ||
        fMaxX != nx2 ||
        fMaxY != ny2)
    {
        int oldMinX = fMinX;
        int oldMinY = fMinY;
        
        fMinX = nx1;
        fMinY = ny1;
        fMaxX = nx2;
        fMaxY = ny2;
        relocateWindows(fMinX - oldMinX, fMinY - oldMinY);
        putWorkArea();
    }
}

void YWindowManager::putWorkArea() {
    INT32 workArea[4];

    workArea[0] = minX(WinLayerNormal);
    workArea[1] = minY(WinLayerNormal);
    workArea[2] = maxX(WinLayerNormal);
    workArea[3] = maxY(WinLayerNormal);
        
    XChangeProperty(app->display(), handle(),
                    _XA_WIN_WORKAREA,
                    XA_CARDINAL,
                    32, PropModeReplace,
                    (unsigned char *)workArea, 4);
}

void YWindowManager::relocateWindows(int dx, int dy) {
    YFrameWindow *f = 0;
    for (CARD32 l = WinLayerDock - 1; l > 0; l--)
        if (fTop[l] != 0) {
            f = fTop[l];
            break;
        }
    if (f == 0)
        return ;
    while (f) {
        f->setPosition(f->x() + dx, f->y() + dy);
        f = f->nextLayer();
    }
}

void YWindowManager::activateWorkspace(CARD32 workspace) {
    if (workspace != fActiveWorkspace) {
#ifdef CONFIG_TASKBAR
        if (taskBarShowWorkspaces) {
            if (fActiveWorkspace != WinWorkspaceInvalid)
                taskBar->workspaceButton(fActiveWorkspace)->setPressed(0);
        }
#endif
        fActiveWorkspace = workspace;
#ifdef CONFIG_TASKBAR
        if (taskBarShowWorkspaces) {
            taskBar->workspaceButton(fActiveWorkspace)->setPressed(1);
        }
#endif

        CARD32 ws = fActiveWorkspace;
        
        XChangeProperty(app->display(), handle(),
                        _XA_WIN_WORKSPACE,
                        XA_CARDINAL,
                        32, PropModeReplace,
                        (unsigned char *)&ws, 1);

        //msg("activating: %d", workspace);

        updateWorkArea();

        YFrameWindow *w;

        w = topLayer();
        while (w) {
            if (!w->visibleNow())
                w->updateWorkspace();
            w = w->nextLayer();
        }
        w = bottomLayer();
        while (w) {
            if (w->visibleNow())
                w->updateWorkspace();
            w = w->prevLayer();
        }

        if (clickFocus) {
            if (!focus() || !focus()->visibleNow())
                focusTopWindow();
            if (focus())
                focus()->wmRaise();
        }
#ifdef CONFIG_TASKBAR
        taskBar->relayout();
#endif
#ifdef CONFIG_GUIEVENTS
        wmapp->signalGuiEvent(geWorkspaceChange);
#endif
    }
}

void YWindowManager::setWinWorkspace(CARD32 workspace) {
    if (workspace >= (CARD32)workspaceCount()) {
        MSG(("invalid workspace switch %ld", (long)workspace));
        return ;
    }
    activateWorkspace(workspace);
}

void YWindowManager::wmCloseSession() {
    YFrameWindow *f = topLayer();

    /* shutdown started */
    while (f) {
        f->wmClose();
        f = f->nextLayer();
    }
}

void YWindowManager::getIconPosition(YFrameWindow *frame, int *iconX, int *iconY) {
    static int x = 0, y = 0;
    MiniIcon *iw = frame->getMiniIcon();

    if (x < minX(0))
        x = minX(0);
    if (y < minY(0))
        y = minY(0);

    *iconX = x;
    *iconY = y;
    
    y += iw->height();
    if (y >= maxY(0)) {
        x += iw->width();
        y = minX(0);
        if (x >= maxX(0)) {
            x = 0;
            y = 0;
        }
    }
}

YMenu *YWindowManager::createWindowMenu(YMenu *menu, CARD32 workspace) {
    if (!menu)
        menu = new YMenu();
    
    int level, levelCount, windowLevel, layerCount;
    YFrameWindow *frame;
    bool needSeparator = false;

    for (int layer = 0 ; layer < WinLayerCount; layer++) {
        layerCount = 0;
        for (level = 0; level < 4; level++) {
            levelCount = 0;
            for (frame = top(layer); frame; frame = frame->next()) {
#ifdef CONFIG_WINLIST
                if (frame->client() == windowList)
                    continue;
#endif
                if (!frame->visibleOn(workspace))
                    continue;
                if (frame->frameOptions() & YFrameWindow::foIgnoreWinList)
                    continue;
                if (workspace != activeWorkspace() &&
                    frame->visibleOn(activeWorkspace()))
                    continue;
                
                windowLevel = 0;
                if (frame->isHidden())
                    windowLevel = 3;
                else if (frame->isMinimized())
                    windowLevel = 2;
                else if (frame->isRollup())
                    windowLevel = 1;
                
                if (level != windowLevel)
                    continue;
                
                if ((levelCount == 0 && level > 0) || (layerCount == 0 && layer > 0)
                    && needSeparator)
                    menu->addSeparator();
                
                frame->addToMenu(menu);
                levelCount++;
                layerCount++;
                needSeparator = true;
            }
        }
    }
    return menu;
}

int YWindowManager::windowCount(CARD32 workspace) {
    int count = 0;
    
    for (int layer = 0 ; layer <= WinLayerCount; layer++) {
        for (YFrameWindow *frame = top(layer); frame; frame = frame->next()) {
            if (!frame->visibleOn(workspace))
                continue;
            if (frame->frameOptions() & YFrameWindow::foIgnoreWinList)
                continue;
            if (workspace != activeWorkspace() &&
                frame->visibleOn(activeWorkspace()))
                continue;
            count++;
        }
    }
    return count;
}

WindowListMenu::WindowListMenu(YWindow *parent): YMenu(parent) {
}

void WindowListMenu::updatePopup() {
    removeAll();

    assert(this == manager->createWindowMenu(this, manager->activeWorkspace()));

    int first = 1;
    
    for (CARD32 d = 0; d < manager->workspaceCount(); d++) {
        if (d == manager->activeWorkspace())
            continue;
        if (first) {
            addSeparator();
            first = 0;
        }
        char s[50];
        sprintf(s, "%lu. %-.32s", (unsigned long)(d + 1), manager->workspaceName(d));

        YMenu *sub = 0;
        if (manager->windowCount(d) > 0)
            sub = manager->createWindowMenu(0, d);
        
        YMenu::YMenuItem *item = new YMenuItem(s,
                                               (d < 10) ? 0 : -1, 0,
                                               cmdActivateWorkspace,
                                               sub, (void *)d);
        add(item);
                                               
        /*if (app->windowCount(d) == 0) {
            addItem(s, (d < 10) ? 1 : -1, 0,
                    cmdActivateWorkspace, (void *)d);
        } else {
            if (sub)
                addSubmenu(s, (d < 10) ? 1 : -1, sub);
        }*/
    }
#ifdef CONFIG_WINLIST
    addSeparator();
    addItem("Window list", 0, "Ctrl+Shift+Esc", cmdWindowList);
#endif
}

void YWindowManager::popupWindowListMenu(int x, int y) {
    windowListMenu->popup(0, wmapp, 0, x, y, -1, -1,
                          YPopupWindow::pfCanFlipVertical |
                          YPopupWindow::pfCanFlipHorizontal |
                          YPopupWindow::pfPopupMenu);
}

void YWindowManager::popupStartMenu() {
#ifdef CONFIG_TASKBAR
    if (showTaskBar && taskBar)
        taskBar->popupStartMenu();
    else
#endif
        rootMenu->popup(0, wmapp, 0, 0, 0, -1, -1,
                        YPopupWindow::pfCanFlipVertical |
                        YPopupWindow::pfCanFlipHorizontal |
                        YPopupWindow::pfPopupMenu);
}

void YWindowManager::popupWindowListMenu() {
#ifdef CONFIG_TASKBAR
    if (showTaskBar && taskBar)
        taskBar->popupWindowListMenu();
    else
#endif
        popupWindowListMenu(0, 0);
}

void YWindowManager::resetColormap(bool active) {
    if (active) {
        if (manager->colormapWindow() && manager->colormapWindow()->client())
            manager->installColormap(manager->colormapWindow()->client()->colormap());
    } else {
        manager->installColormap(None);
    }
}

void YWindowManager::resetInputFocus() {
    XSetInputFocus(app->display(),
                   (manager->focus() && manager->focus()->client()) ?
                   manager->focus()->client()->clientWindow() : PointerRoot,
                   RevertToPointerRoot,
                   CurrentTime);
}

void YWindowManager::handleProperty(const XPropertyEvent &property) {
    if (property.atom == XA_IcewmWinOptHint) {
        Atom type;
        int format;
        unsigned long nitems, lbytes;
        unsigned char *propdata;
        
        if (XGetWindowProperty(app->display(), handle(),
                               XA_IcewmWinOptHint, 0, 8192, True, XA_IcewmWinOptHint,
                               &type, &format, &nitems, &lbytes,
                               &propdata) == Success && propdata)
        {
                char *p = (char *)propdata;
                char *e = (char *)propdata + nitems;

                while (p < e) {
                    char *clsin;
                    char *option;
                    char *arg;

                    clsin = p;
                    while (p < e && *p) p++;
                    if (p == e)
                        break;
                    p++;
                    
                    option = p;
                    while (p < e && *p) p++;
                    if (p == e)
                        break;
                    p++;
                    
                    arg = p;
                    while (p < e && *p) p++;
                    if (p == e)
                        break;
                    p++;
                    
                    if (p != e)
                        break;

                    char *n = strdup(clsin);
                    if (n)
                        hintOptions->setWinOption(n, option, arg);
                }
                XFree(propdata);
        }
    }
}
