/*
 *  previews.c:		Pixmap preview browser	
 *
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/07/10 10:24:53 $
 *  $Author: hafner $
 *  $Revision: 1.14 $
 *  $State: Exp $
 */

#include "config.h"

#ifdef HAVE_SYS_TYPES_H
#	include <sys/types.h>
#endif
#include <dirent.h>

#include <gtk/gtk.h>
#include <proplist.h>

#include "previews.h"
#include "window.h"
#include "misc.h"
#include "tileofday.h"
#include "texture.h"

#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

extern proplist_t  *pixmap_path;
extern GtkTooltips *tooltips;

/*******************************************************************************

			     local variables
  
*******************************************************************************/

static GList	*preview_list = NULL;	/* list of all pixmap previews */
static gint	idle_tag      = 0;	/* tag of idle function */
GtkWidget	*progress_bar = NULL;	/* progress bar of pixmap display */

/*******************************************************************************

				prototypes
  
*******************************************************************************/

static GtkWidget *
make_preview (proplist_t *image, GtkWidget *entry);
static void
update_pixmap (GtkWidget *window);
static void
set_selection (GtkWidget *widget, gpointer ptr);
static void
accept_selection (GtkWidget *widget, gpointer ptr);

/*******************************************************************************

				public code
  
*******************************************************************************/

void
image_browser (GtkWidget *widget, gpointer ptr)
/*
 *  Generate/open/close pixmap preview image browser dialog window.
 *  Every preview is allocated with the BulletHole pixmap and replaced
 *  with the correct preview during idle time.
 *  If 'ptr' == NULL then generate window, start idle function but do not
 *  display window.
 *  If window is visible then hide window and vice versa.
 *
 *  No return value.
 */
{
   static GtkWidget *window    = NULL;
   static GtkWidget *ok_button = NULL;

   if (!window)				/* window already created */
   {
      GtkWidget		*scrolled;	/* scrolled window */
      GtkWidget		*table;		/* two-dimensional entry */
      unsigned		n;		/* counter */
      GtkWidget		*entry;		/* not editable text entry */
      proplist_t	*filenames;	/* array of filenames */

      /*
       *  Generate list of available pixmaps
       */
      {
	 DIR		*dir;
	 struct dirent	*file;
	 proplist_t	*images;
	 proplist_t	*dummy = PLMakeString ("dummy");

	 images = PLMakeDictionaryFromEntries (NULL, NULL, NULL);
	 for (n = 0; n < PLGetNumberOfElements (pixmap_path); n++)
	 {
	    char *path = expand_tilde (PLGetString (PLGetArrayElement (pixmap_path, n)));
	    
	    dir = opendir (path);
	    if (!dir)
	    {
	       warning ("Can't open directory `%s' "
			"although its in your PixmapPath.", path);
	       Free (path);
	       continue;
	    }
	    while ((file = readdir (dir)))
	    {
	       if (streq (file->d_name, ".") || streq (file->d_name, ".."))
		  continue;

	       /*
		*  Check whether file is a directory
		*/
	       {
		  char	*tmp;
		  DIR	*tmpdir;

		  tmp = Calloc (strlen (path) + strlen (file->d_name) + 2,
				sizeof (char));
		  strcpy (tmp, path);
		  strcat (tmp, "/");
		  strcat (tmp, file->d_name);
		   
		  tmpdir = opendir (tmp);
		  Free (tmp);
		  if (tmpdir)
		  {
		     closedir (tmpdir);
		     continue;
		  }
	       }
	       /*
		*  Check whether file has valid extension
		*/
	       if (!(strstr (file->d_name, ".xpm")
		     || strstr (file->d_name, ".jpg")
		     || strstr (file->d_name, ".jpeg")
		     || strstr (file->d_name, ".png")
		     || strstr (file->d_name, ".tif")
		     || strstr (file->d_name, ".XPM")
		     || strstr (file->d_name, ".JPG")
		     || strstr (file->d_name, ".JPEG")
		     || strstr (file->d_name, ".PNG")
		     || strstr (file->d_name, ".TIF")))
		   continue;
	       /*
		*  Insert pixmap name into hash table
		*/
	       {
		  proplist_t *key;

		  key = PLMakeString (file->d_name);
		  PLInsertDictionaryEntry (images, key, dummy);
		  PLRelease (key);
	       }
	    }
	    Free (path);
	    closedir (dir);
	 }
	 PLRelease (dummy);
	 filenames = PLGetAllDictionaryKeys (images);
	 PLRelease (images);
      }
      
      /*
       *  Make new dialog window
       */
      window = gtk_dialog_new ();
      gtk_window_set_title (GTK_WINDOW (window), "Pixmap Browser");
      gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
      gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			  GTK_SIGNAL_FUNC (gtk_true), NULL);
      gtk_signal_connect_object (GTK_OBJECT (window), "delete_event",
				 GTK_SIGNAL_FUNC (gtk_widget_hide),
				 GTK_OBJECT (window));

      {
	 GtkWidget *confirm_area;
	 GtkWidget *button;
	 GtkWidget *hbox;
	 
	 hbox = gtk_hbox_new (FALSE, 5);
	 gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area), hbox,
			     FALSE, TRUE, 0);

	 gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new ("Selection: "),
			     FALSE, TRUE, 0);

	 entry = gtk_entry_new ();
	 gtk_widget_set_usize (entry, 200, 0);
	 gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
	 gtk_box_pack_start (GTK_BOX (hbox), entry, FALSE, TRUE, 0);

	 progress_bar = gtk_progress_bar_new ();
	 gtk_widget_show (progress_bar);
	 gtk_box_pack_start (GTK_BOX (hbox), progress_bar, TRUE, TRUE, 0);
	 gtk_widget_set_usize (progress_bar, 100, 0);
	 
	 confirm_area = gtk_hbutton_box_new ();
	 gtk_box_pack_end (GTK_BOX (hbox), confirm_area, FALSE, TRUE, 10);

	 ok_button = gtk_button_new_with_label ("OK");
	 gtk_box_pack_start (GTK_BOX (confirm_area), ok_button, TRUE, TRUE, 0);
	 gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			     GTK_SIGNAL_FUNC (accept_selection), entry);
	 gtk_entry_set_text (GTK_ENTRY (entry), "BulletHole.xpm");
	 gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_hide),
				    GTK_OBJECT (window));

	 button = gtk_button_new_with_label ("Cancel");
	 gtk_box_pack_start (GTK_BOX (confirm_area), button, TRUE, TRUE, 0);
	 gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_hide),
				    GTK_OBJECT (window));
      }
      
      scrolled = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_widget_set_usize (scrolled, 650, 430);
      gtk_container_border_width (GTK_CONTAINER (scrolled), 5);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), scrolled,
			  TRUE, TRUE, 5);

      table = gtk_table_new (1, 1, TRUE);
      gtk_container_border_width (GTK_CONTAINER (table), 5);
      gtk_table_set_row_spacings (GTK_TABLE (table), 2);
      gtk_table_set_col_spacings (GTK_TABLE (table), 2);
      gtk_widget_show (table);
      gtk_container_add (GTK_CONTAINER (scrolled), table);
      
      {
	 unsigned row = 0, column = 0;

	 for (n = 0; n < PLGetNumberOfElements (filenames); n++)
	 {
	    GtkWidget *preview = make_preview (PLGetArrayElement (filenames, n),
					       entry);
	    
	    if (preview)
	    {
	       gtk_table_attach_defaults (GTK_TABLE (table), preview,
					  column, column + 1, row, row + 1);
	       column++;
	       if (column == 8)
	       {
		  column = 0;
		  row++;
	       }
	    }
	 }
      }
      idle_tag = gtk_idle_add ((GtkFunction) update_pixmap, window);
      PLRelease (filenames);
   }

   gtk_object_set_user_data (GTK_OBJECT (window), ptr);
   gtk_object_set_user_data (GTK_OBJECT (ok_button), ptr);
   
   if (!GTK_WIDGET_VISIBLE (window) && ptr)
      gtk_widget_show_all (window);
   else
      gtk_widget_hide (window);
}

static void
update_pixmap (GtkWidget *window)
/*
 *  Idle time function: read a pixmpap and replace BulletHole
 *  with corresponding preview.
 *
 *  No return value.
 */
{
   static bool_t finished = NO;

   if (finished)			/* nothing to do */
      return;
   
   if (preview_list)
   {
      GtkWidget *preview;
      char	*txt;
      static int melem = -1;
      
      preview_list = g_list_first (preview_list);
      preview      = preview_list->data;
      preview_list = g_list_remove (preview_list, preview_list->data);
      txt 	   = gtk_object_get_user_data (GTK_OBJECT (preview));
      make_pixmap (txt, 64, 64, preview);
      if (melem < 0)
	 melem = g_list_length (preview_list);
      gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar),
			       1 - g_list_length (preview_list) / (double) melem);
   }
   else
   {
      gtk_idle_remove (idle_tag);
      gtk_widget_destroy (progress_bar);
      finished = YES;
   }
}

static GtkWidget *
make_preview (proplist_t *image, GtkWidget *entry)
/*
 *  Make button widget displaying a pixmap with tooltips showing the
 *  pixmap filename.
 *  'image' is the filename of the preview.
 *  'entry' is the file text entry.
 *
 *  No return value.
 */
{
   GtkWidget	*button;
   GtkWidget	*pixmap;
   char		*name = PLGetString (image);
   
   button = gtk_button_new ();
   gtk_signal_connect (GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (set_selection), entry);
   pixmap = make_pixmap ("BulletHole.xpm", 64, 64, NULL);
   gtk_object_set_user_data (GTK_OBJECT (pixmap), strdup (name));
   preview_list = g_list_append (preview_list, pixmap);
   gtk_tooltips_set_tip (tooltips, button, name, NULL);
   gtk_container_border_width (GTK_CONTAINER (button), 2);
   gtk_container_add (GTK_CONTAINER (button), pixmap);
   gtk_widget_show_all (button);
   gtk_object_set_user_data (GTK_OBJECT (button), pixmap);

   return button;
}

static void
set_selection (GtkWidget *widget, gpointer ptr)
/*
 *  Change text entry if pixmap button is pressed.
 *
 *  No return value.
 */
{
   GtkWidget *entry  = (GtkWidget *) ptr;
   GtkWidget *pixmap = gtk_object_get_user_data (GTK_OBJECT (widget));
   char      *name   = gtk_object_get_user_data (GTK_OBJECT (pixmap));

   gtk_entry_set_text (GTK_ENTRY (entry), name);
}

static void
accept_selection (GtkWidget *widget, gpointer ptr)
/*
 *  OK button is activated: set pixmap and text entry of texture dialog.
 *
 *  No return value.
 */
{
   GtkWidget    *entry 	  = (GtkWidget *) ptr;
   wtexture_t   *wtexture = gtk_object_get_user_data (GTK_OBJECT (widget));
   char		*name     = gtk_entry_get_text (GTK_ENTRY (entry));
   
   make_pixmap (name, 128, 128, wtexture->wpixmap [PIXMAP_PIXMAP]);
   gtk_entry_set_text (GTK_ENTRY (wtexture->wpixmap [PIXMAP_NAME]), name);

#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */
}
