/*
 *  window.c:		GTK stuff
 *
 *  Written by:		Ullrich Hafner
 *  
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */


/*
 *  $Date: 1998/10/08 22:04:58 $
 *  $Author: hafner $
 *  $Revision: 1.68 $
 *  $State: Exp $
 */

#include "config.h"

#if HAVE_STDLIB_H
#	include <stdlib.h>
#endif /* not HAVE_STDLIB_H */
#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */
#if HAVE_SYS_STAT_H
#	include <sys/stat.h>
#endif /* HAVE_SYS_STAT_H */

#include <stdio.h>
#include <gtk/gtk.h>
#include <proplist.h>

#if HAVE_GDK_IMLIB_H
#include <gdk_imlib.h>
#endif /* HAVE_GDK_IMLIB_H */

#include "gtkfontsel.h"
#include "window.h"
#include "icons.h"
#include "misc.h"
#include "dialog.h"
#include "previews.h"
#include "texture.h"
#include "keys.h"
#include "menu.h"
#include "tileofday.h"

#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

bool_t		 changed      = NO;
GtkWidget	 *main_window = NULL;
GtkTooltips	 *tooltips    = NULL;
proplist_t	 *pixmap_path = NULL;
proplist_t	 *windowmaker = NULL;
proplist_t	 *wmconfig    = NULL;

extern proplist_t *plist_changed;
extern proplist_t *pl_yes;

/*******************************************************************************

			       prototypes
  
*******************************************************************************/

static GtkWidget *
create_top_page (GtkNotebook *notebook, const char *name, pixmap_t *pm);
static void
set_path (GtkWidget *widget, gpointer filesel);
static void
path_remove (GtkWidget *widget, GtkWidget *list);
static void
path_insert (GtkWidget *widget, GtkWidget *list);
static GtkWidget *
path_dialog (proplist_t *key);
static void
page_switch (GtkWidget *widget, GtkNotebookPage *page);
static GtkWidget *
create_page (GtkNotebook *notebook, const char *name);
static void
set_unsigned (GtkWidget *widget, gpointer ptr);
static void
set_entry_text (GtkWidget *widget, gpointer ptr);
static void
set_string (GtkWidget *widget, gpointer ptr);
static void
set_bool (GtkWidget *widget, gpointer ptr);
static bool_t
convert_bool (proplist_t *p);
static void
set_fontname (GtkWidget *widget, gpointer ptr);
static void
fontsel_dialog (GtkWidget *widget, gpointer ptr);

/*******************************************************************************

			       public code
  
*******************************************************************************/

void
root_window (int argc, char **argv, bool_t preview_browser, const char *font,
	     proplist_t *wmc, proplist_t *wmlist)
/*
 *  Create root window and call the event loop of GTK
 *
 *  No return value.
 */
{
   GtkWidget  *box1;
   GtkWidget  *notebook;
   GtkWidget  *progress_window;
   GtkWidget  *progress_label;
   GtkWidget  *progress_bar;
   GtkWidget  *toolbar;
   proplist_t *groups = NULL;
   
   gtk_init (&argc, &argv);

#ifdef HAVE_IMLIB
   gdk_imlib_init ();
   gtk_widget_push_visual (gdk_imlib_get_visual ());
   gtk_widget_push_colormap (gdk_imlib_get_colormap ());
#endif /* HAVE_IMLIB */

#if defined(HAVE_POPEN) && defined(TILE_OF_DAY) && defined(CRONTAB)   
   read_crontab ();
#endif /* defined(HAVE_POPEN) && defined(TILE_OF_DAY) && defined(CRONTAB) */
   
   windowmaker = wmlist;
   wmconfig    = wmc;
   
   /*
    *  Progress bar window
    */
   {
      GtkWidget *vbox;
      
      progress_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
      gtk_window_set_title (GTK_WINDOW (progress_window),
			    PACKAGE " " VERSION " init ...");
      gtk_container_border_width (GTK_CONTAINER (progress_window), 10);
      gtk_signal_connect (GTK_OBJECT (progress_window), "destroy",
                          GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &progress_window);
      gtk_widget_set_usize (progress_window, 250, -1);
      
      vbox = gtk_vbox_new (FALSE, 5);
      gtk_container_add (GTK_CONTAINER (progress_window), vbox);

      progress_label = gtk_label_new ("");
      gtk_box_pack_start (GTK_BOX (vbox), progress_label, FALSE, TRUE, 0);

      progress_bar = gtk_progress_bar_new ();
      gtk_box_pack_start (GTK_BOX (vbox), progress_bar, FALSE, TRUE, 0);

      gtk_widget_show_all (progress_window);
   }
   
   /*
    *  Main window
    */
   main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title (GTK_WINDOW (main_window), "WindowMaker Configuration");
   gtk_container_border_width (GTK_CONTAINER (main_window), 0);
   gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
		       GTK_SIGNAL_FUNC (quit), NULL);
   gtk_widget_realize (main_window);

   init_pixmaps ();

   box1 = gtk_vbox_new (FALSE, 0);
   gtk_container_add (GTK_CONTAINER (main_window), box1);
   tooltips = gtk_tooltips_new ();
   gtk_widget_show (box1);

   /*
    *  Menu and toolbox
    */
   gtk_box_pack_start (GTK_BOX(box1), make_menubar (), FALSE, TRUE, 0);
   gtk_box_pack_start (GTK_BOX(box1), toolbar = make_toolbar (), FALSE, TRUE, 0);
   
   /*
    *  Group notebook
    */
   notebook = gtk_notebook_new ();
   gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
   gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
   gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
   gtk_notebook_popup_enable (GTK_NOTEBOOK (notebook));
   gtk_box_pack_start (GTK_BOX (box1), notebook, TRUE, TRUE, 0);
   gtk_container_border_width (GTK_CONTAINER (notebook), 10);
   gtk_widget_show_all (notebook);

   /*
    *  Generate notebook page for every group
    */
   {
      proplist_t *all_keys = PLGetAllDictionaryKeys (wmconfig);
      proplist_t *plgrp    = PLMakeString ("Group");
      proplist_t *pltype   = PLMakeString ("Type");
      proplist_t *plinfo   = PLMakeString ("Info");
      proplist_t *plrange  = PLMakeString ("Range");
      unsigned   melem     = PLGetNumberOfElements (all_keys);
      unsigned	 n;

      for (n = 0; n < melem; n++)
      {
 	 proplist_t *key    = PLGetArrayElement (all_keys, n);
	 proplist_t *keydef = PLGetDictionaryEntry (wmconfig, key);
	 proplist_t *group  = PLGetDictionaryEntry (keydef, plgrp);
	 char *name         = PLGetString (key);
	 char *type;

	 gtk_label_set (GTK_LABEL (progress_label), PLGetString (key));
	 gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar),
				  n / (double) melem / 2.0);
	 gtk_widget_draw (progress_window, NULL);
	 while (gtk_events_pending())
	    gtk_main_iteration();
	 
	 type = PLGetString (PLGetDictionaryEntry (keydef, pltype));
	 
	 if (strcaseeq ("PixmapPath", name))
	    pixmap_path = PLGetDictionaryEntry (windowmaker, key);
	 if (!groups || !PLGetDictionaryEntry (groups, group))
	 {
	    GtkWidget  *page;
	    GtkWidget  *box;
	    GtkWidget  *scrolled;
	    proplist_t *data;
	    unsigned   *row = NULL;

	    if (strcaseeq ("Bool", type) || strcaseeq ("Key", type)
		|| strcaseeq ("String", type) || strcaseeq ("Int", type)
		|| strcaseeq ("Color", type) || strcaseeq ("Font", type)
		|| strcaseeq ("Text", type))
	    {
	       /*
		*  Two-column table, variable number of rows 
		*/
	       box = gtk_table_new (1, 4, FALSE);
	       gtk_table_set_row_spacings (GTK_TABLE (box), 10);
	       gtk_table_set_col_spacings (GTK_TABLE (box), 10);
	       gtk_container_border_width (GTK_CONTAINER (box), 5);
	       gtk_widget_show (box);
	       row  = Calloc (1, sizeof (unsigned));
	       *row = 1;
	    }
	    else if (strcaseeq ("Texture", type))
	    {
	       /*
		*  Sub-notebook - one for each texture type
		*/
	       box = gtk_notebook_new ();
	       gtk_signal_connect (GTK_OBJECT (box), "switch_page",
				   GTK_SIGNAL_FUNC (page_switch), NULL);
	       gtk_notebook_set_show_tabs (GTK_NOTEBOOK (box), TRUE);
	       gtk_notebook_set_scrollable (GTK_NOTEBOOK (box), TRUE);
	       gtk_notebook_set_tab_pos (GTK_NOTEBOOK (box), GTK_POS_LEFT);
	       gtk_notebook_popup_enable (GTK_NOTEBOOK (box));
	       gtk_container_border_width (GTK_CONTAINER (box), 10);
	       gtk_widget_show (box);
	    }
	    else
	    {
	       /*
		*  Just a simple vertical box
		*/
	       box = gtk_vbox_new (FALSE, 5);
	       gtk_widget_show (box);
	    }
	    gtk_object_set_user_data (GTK_OBJECT (box), row);

	    /*
	     *  Select a pixmap identifier for each notebook group
	     */
	    if (strcaseeq (PLGetString (group), "Fonts"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_FONT);
	    else if (strcaseeq (PLGetString (group), "Textures"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_PAINT);
	    else if (strcaseeq (PLGetString (group), "Misc"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_MISC);
	    else if (strcaseeq (PLGetString (group), "F/X"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_GEAR);
	    else if (strcaseeq (PLGetString (group), "Colors"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_COLOR);
	    else if (strcaseeq (PLGetString (group), "Paths"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_PATH);
	    else if (strcaseeq (PLGetString (group), "Shortcuts"))
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group), p_array + P_KEY);
	    else
	       page = create_top_page (GTK_NOTEBOOK (notebook),
				       PLGetString (group),
				       p_array + P_MISC);
	    gtk_widget_show (page);
	    /*
	     *  Every notebook page is embedded in a scrolled window
	     */
	    scrolled = gtk_scrolled_window_new (NULL, NULL);
	    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					    GTK_POLICY_AUTOMATIC,
					    GTK_POLICY_AUTOMATIC);
	    gtk_widget_set_usize (scrolled, 600, 400);
	    gtk_container_border_width (GTK_CONTAINER (scrolled), 5);
	    gtk_container_add (GTK_CONTAINER (scrolled), box);
	    gtk_box_pack_start (GTK_BOX (page), scrolled, TRUE, TRUE, 0);
	    gtk_widget_show (scrolled);
	    
	    data = PLMakeData ((unsigned char *) &box, sizeof (GtkWidget *));
	    if (!groups)		/* Dictionary already built? */
	       groups = PLMakeDictionaryFromEntries (group, data, NULL);
	    else
	       PLInsertDictionaryEntry (groups, group, data);
	 }
      }
      /*
       *  Generate a widget for every attribute
       */
      for (n = 0; n < melem; n++)
      {
	 proplist_t *key      = PLGetArrayElement (all_keys, n);
	 proplist_t *value    = PLGetDictionaryEntry (windowmaker, key);
	 proplist_t *keydef   = PLGetDictionaryEntry (wmconfig, key);
	 proplist_t *group    = PLGetDictionaryEntry (keydef, plgrp);
	 proplist_t *range    = PLGetDictionaryEntry (keydef, plrange);
	 proplist_t *info     = PLGetDictionaryEntry (keydef, plinfo);
	 proplist_t *pl       = PLGetDictionaryEntry (groups, group);
	 GtkWidget  *page_box = * (GtkWidget **) PLGetDataBytes (pl);
	 char *name           = PLGetString (key);
	 char *type           = PLGetString (PLGetDictionaryEntry (keydef,
								   pltype));

	 /*
	  *  Update of progress bar window
	  */
	 gtk_label_set (GTK_LABEL (progress_label), PLGetString (key));
	 gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_bar),
				  n / (double) melem / 2.0 + 0.5);
	 gtk_widget_draw (progress_window, NULL);
	 while (gtk_events_pending())
	    gtk_main_iteration();

	 if (strcaseeq ("Bool", type)) 
	 {
	    GtkWidget	*button;
	    GtkWidget	*label;
	    unsigned	rows = 0;
	    unsigned    col  = 0;
	    unsigned	*ptr;
	    
	    ptr   = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows  = ++*ptr;
	    col   = (rows % 2) * 2;
	    rows /= 2;

	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label,
			      col, col + 1, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
	    button = gtk_check_button_new ();
	    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button),
					 convert_bool (value));
	    gtk_signal_connect (GTK_OBJECT (button), "toggled",
				GTK_SIGNAL_FUNC (set_bool), (gpointer) key);
	    gtk_table_attach (GTK_TABLE (page_box), button,
			      col + 1, col + 2, rows, rows + 1,
			      GTK_SHRINK, GTK_SHRINK, 0, 0);
	    gtk_tooltips_set_tip (tooltips, button, PLGetString (info),
				  NULL);
	    gtk_widget_show (button);
	 }
	 else if (strcaseeq ("String", type)) 
	 {
	    GtkWidget	*option_menu;
	    GtkWidget	*label;
	    const char	**values;
	    const char	*active;
	    unsigned	current = 0;
	    unsigned	rows = 0;
	    unsigned    col  = 0;
	    unsigned	*ptr;
	    unsigned	n;
	    
	    ptr   = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows  = ++*ptr;
	    col   = (rows % 2) * 2;
	    rows /= 2;

	    values = Calloc (PLGetNumberOfElements (range) + 1,
			     sizeof (char *));
	    for (n = 0; n < PLGetNumberOfElements (range); n++)
	    {
	       proplist_t *element = PLGetArrayElement (range, n);
	       
	       if (strcaseeq (PLGetString (element), PLGetString (value)))
		  current = n;
	       values [n] = PLGetString (element);
	    }
	    values [n] = NULL;
	    active     = values [current];

	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label,
			      col, col + 1, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
		  
	    option_menu = generate_option_menu (NULL, values, active,
						set_string, key);
	    gtk_table_attach (GTK_TABLE (page_box), option_menu,
			      col + 1, col + 2, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK,
			      GTK_SHRINK, 0, 0);
	    gtk_tooltips_set_tip (tooltips, option_menu, PLGetString (info),
				  NULL);
	 }
	 else if (strcaseeq ("Int", type)) /* Spinbutton */
	 {
	    unsigned	rows = 0;
	    unsigned    col  = 0;
	    unsigned	*ptr;
	    GtkWidget   *label;
	    GtkWidget   *hbox;
	    unsigned	n;
	    
	    ptr   = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows  = ++*ptr;
	    col   = (rows % 2) * 2;
	    rows /= 2;
	    
	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label,
			      col, col + 1, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);

	    hbox = gtk_hbox_new (FALSE, 5);
	    gtk_table_attach (GTK_TABLE (page_box), hbox,
			      col + 1, col + 2, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK,
			      GTK_FILL | GTK_SHRINK, 0, 0);
	    gtk_widget_show (hbox);
	    
	    for (n = 0; n < PLGetNumberOfElements (range); n += 2)
	    {
	       GtkWidget *spinner;
	       GtkObject *adj;
	       char	 *m = PLGetString (PLGetArrayElement (range, n));
	       char	 *M = PLGetString (PLGetArrayElement (range, n + 1));
	       char 	 *v;
	       
	       if (PLIsArray (value))
		  v = PLGetString (PLGetArrayElement (value, n / 2));
	       else
		  v = PLGetString (value);
	       
	       adj = gtk_adjustment_new (strtod (v, NULL),
					 strtod (m, NULL), strtod (M, NULL),
					 1.0, 5.0, 0.0);
	       gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
				   GTK_SIGNAL_FUNC (set_unsigned),
				   (gpointer) key);
	       spinner = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0, 0);
	       gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinner), TRUE);
	       gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
	       gtk_widget_show_all (spinner);
	       gtk_object_set_user_data (GTK_OBJECT (adj), (gpointer) spinner);
	       gtk_tooltips_set_tip (tooltips, spinner,
				     PLGetString (info), NULL);
	       gtk_box_pack_start (GTK_BOX (hbox), spinner, TRUE, TRUE, 0);
	       gtk_object_set_user_data (GTK_OBJECT (spinner),
					 (gpointer) (spinner + n / 2));

	    }
	 }
	 else if (strcaseeq ("Font", type) || strcaseeq ("Text", type)) 
	 {
	    GtkWidget *entry;
	    GtkWidget *label;
	    unsigned  rows = 0;
	    unsigned  col  = 0;
	    unsigned  *ptr;

	    ptr = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows = ++*ptr;

	    if (!strcaseeq ("Font", type))
	    {
	       col   = (rows % 2) * 2;
	       rows /= 2;
	    }
		  
	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label, col, col + 1,
			      rows, rows + 1, GTK_FILL | GTK_SHRINK,
			      GTK_FILL | GTK_SHRINK, 0, 0);
		  
	    entry = gtk_entry_new ();
	    gtk_entry_set_text (GTK_ENTRY (entry), PLGetString (value));
	    if (strcaseeq ("Font", type))
	    {
	       gtk_table_attach (GTK_TABLE (page_box), entry,
				 col + 1, col + 2, rows, rows + 1,
				 GTK_FILL | GTK_SHRINK | GTK_EXPAND,
				 GTK_FILL | GTK_SHRINK, 0, 0);
	       {
		  GtkWidget	*hbox;
		  GtkWidget	*button;
	       
		  hbox = gtk_hbutton_box_new ();
		  gtk_widget_show (hbox);
		  gtk_table_attach (GTK_TABLE (page_box), hbox,
				    col + 2, col + 3, rows, rows + 1,
				    GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK,
				    0, 0);

		  button = gtk_button_new_with_label ("Browse");
		  gtk_widget_show (button);
		  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 5);
		  gtk_signal_connect (GTK_OBJECT (button), "pressed",
				      GTK_SIGNAL_FUNC (fontsel_dialog),
				      (gpointer) entry);
		  gtk_tooltips_set_tip (tooltips, button,
					"Browse through the list of "
					"available fonts.", NULL);  
	       }
	    }
	    else
	       gtk_table_attach (GTK_TABLE (page_box), entry,
				 col + 1, col + 2, rows, rows + 1,
				 GTK_FILL | GTK_SHRINK,
				 GTK_FILL | GTK_SHRINK, 0, 0);
		  
	    gtk_signal_connect (GTK_OBJECT (entry), "changed",
				GTK_SIGNAL_FUNC (set_entry_text), key);
	    gtk_widget_set_usize (entry, 100, 0);
	    gtk_widget_show (entry);
	    gtk_tooltips_set_tip (tooltips, entry, PLGetString (info), NULL);
	 }
	 else if (strcaseeq ("Key", type)) 
	 {
	    GtkWidget *label;
	    unsigned  rows = 0;
	    unsigned  col  = 0;
	    unsigned  *ptr;

	    ptr = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows = ++*ptr;

	    col   = (rows % 2) * 2;
	    rows /= 2;
		  
	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label, col, col + 1,
			      rows, rows + 1, GTK_FILL | GTK_SHRINK,
			      GTK_FILL | GTK_SHRINK, 0, 0);
	    gtk_table_attach (GTK_TABLE (page_box), key_dialog (key, info),
			      col + 1, col + 2, rows, rows + 1,
			      GTK_FILL | GTK_SHRINK,
			      GTK_FILL | GTK_SHRINK, 0, 0);
	    gtk_table_set_row_spacings (GTK_TABLE (page_box), 10);
	 }
	 else if (strcaseeq ("Pathlist", type)) /* Text entry */
	 {
	    GtkWidget *hbox;
       
	    hbox = gtk_hbox_new (FALSE, 5);
	    gtk_widget_show (hbox);
	    gtk_box_pack_start (GTK_BOX (page_box), hbox, TRUE, TRUE, 5);
	       
	    gtk_box_pack_start (GTK_BOX (hbox), path_dialog (key), TRUE, TRUE, 5);
	 }
	 else if (strcaseeq ("Color", type)) /* Color preview */
	 {
	    GtkWidget *preview;
	    GtkWidget *frame;
	    GtkWidget *button;
	    GtkWidget *hbox;
	    GtkWidget *label;
	    unsigned  rows = 0;
	    unsigned  col = 0;
	    unsigned  *ptr;

	    ptr   = (unsigned *) gtk_object_get_user_data (GTK_OBJECT (page_box));
	    rows  = ++*ptr;
	    col   = (rows % 2) * 2;
	    rows /= 2;
			
	    label = gtk_label_new (name);
	    gtk_widget_show (label);
	    gtk_table_attach (GTK_TABLE (page_box), label, col, col + 1,
			      rows, rows + 1, GTK_FILL | GTK_SHRINK,
			      GTK_FILL | GTK_SHRINK, 0, 0);
		  
	    hbox = gtk_hbox_new (FALSE, 5);
	    gtk_widget_show (hbox);
	    gtk_table_attach (GTK_TABLE (page_box), hbox, col + 1, col + 2,
			      rows, rows + 1, GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			      GTK_FILL | GTK_SHRINK, 0, 0);

	    frame = gtk_frame_new (NULL);
	    gtk_widget_show (frame);
	    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	    gtk_container_border_width (GTK_CONTAINER (frame), 0);
		  
	    preview = gtk_preview_new (GTK_PREVIEW_COLOR);
	    gtk_preview_size (GTK_PREVIEW (preview), 50, 25);
	    gtk_container_add (GTK_CONTAINER (frame), preview);
	    gtk_object_set_user_data (GTK_OBJECT (preview), (gpointer) value);
	    gtk_widget_show (preview);
	    fill_preview (GTK_PREVIEW (preview), PLGetString (value));

	    button = gtk_button_new ();
	    gtk_tooltips_set_tip (tooltips, button, PLGetString (info), NULL); 
	    gtk_container_add (GTK_CONTAINER (button), frame);
	    gtk_container_border_width (GTK_CONTAINER (button), 0);
	    gtk_object_set_user_data (GTK_OBJECT (button), (gpointer) preview);
	    gtk_widget_show (button);
	    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 5);
	    gtk_signal_connect (GTK_OBJECT (button), "pressed",
				GTK_SIGNAL_FUNC (color_selection_dialog),
				(gpointer) key);
	 }
	 else if (strcaseeq ("Texture", type)) /* Texture dialog */
	 {
	    GtkWidget *page;
	    GtkWidget *label;
		  
	    page = create_page (GTK_NOTEBOOK (page_box), name);
	    texture_dialog (page, key, preview_browser);

	    label = gtk_label_new (PLGetString (info));
	    gtk_widget_show (label);
	    gtk_box_pack_start (GTK_BOX (page), label, FALSE, FALSE, 5);
	 }
      }

      PLRelease (plrange);
      PLRelease (plinfo);
      PLRelease (pltype);
      PLRelease (plgrp);
      PLRelease (all_keys);
   }

   if (preview_browser)
   {
      gtk_label_set (GTK_LABEL (progress_label), "Pixmap previews ...");
      gtk_widget_draw (progress_window, NULL);
      while (gtk_events_pending())
	 gtk_main_iteration();
      image_browser (NULL, NULL);
   }
   
   gtk_widget_show (main_window);

   /*
    *  Initialize tooltips color
    */
   {
      GdkColor		*yellow, *black;		
      GdkColormap	*colormap;
	 
      yellow   = Calloc (1, sizeof (GdkColor));
      black    = Calloc (1, sizeof (GdkColor));
	 
      colormap = gdk_window_get_colormap (main_window->window);

      yellow->red    = 255 << 8;
      yellow->green  = 255 << 8;
      yellow->blue   =   0 << 8;

      if (!gdk_color_alloc (colormap, yellow))
	 gdk_color_white (colormap, yellow);
	 
      black->red   = 0 << 8;
      black->green = 0 << 8;
      black->blue  = 0 << 8;
	 
      if (!gdk_color_alloc (colormap, black))
	 gdk_color_black (colormap, black);

      gtk_tooltips_set_colors (tooltips, yellow, black);
      gtk_tooltips_set_colors (GTK_TOOLBAR (toolbar)->tooltips, yellow, black);

      /*
       * Change font of tooltips
       */
      {
	 GtkStyle *style = gtk_style_new ();
	 gdk_font_unref (style->font);
	 style->font = gdk_font_load (font);
	 if (style->font)
	 {
	    gdk_font_ref (style->font);
	    gtk_widget_set_style (GTK_WIDGET (tooltips->tip_window), style);
	    gtk_style_unref (style);
	 }
	 else
	    warning ("Can't change tooltips font to '%s'", font);
      }
   }

   gtk_widget_destroy (progress_window);
   gtk_main ();
}

char *
get_pixmap_path (const char *name)
/*
 *  Guess filename of pixmap 'name'.
 *  Cycle through list of available paths or use absolute path
 *  if name [0] == '/' or '~'.
 *
 *  Return value:
 *	string with pixmap path on success
 *	NULL if pixmap not found
 */
{
   if (name [0] == '/' || name [0] == '~') /* absolute path ? */
   {
      FILE *file;
      char *path = expand_tilde (name);

      file = fopen (path, "r");
      if (!file)
      {
	 Free (path);
	 return NULL;
      }
      else
      {
	 fclose (file);
	 return path;
      }
   }
   else
   {
      unsigned	n;
      char	*found = NULL;

      for (n = 0; n < PLGetNumberOfElements (pixmap_path) && !found; n++)
      {
	 FILE *file;
	 char *path;
	 char *filename;

	 path = expand_tilde (PLGetString (PLGetArrayElement (pixmap_path, n)));
	 filename = Calloc (strlen (path) + strlen (name) + 2, sizeof (char));
	 sprintf (filename, "%s/%s", path, name);
	 Free (path);
	 
	 file = fopen (filename, "r");
	 if (file)
	 {
	    found = filename;
	    fclose (file);
	 }
	 else
	    Free (filename);
      }
      return found;
   }
}

GtkWidget *
make_pixmap (const char *name, unsigned max_width, unsigned max_height,
	     GtkWidget *pw)
/*
 *  Genearate a new pixmap widget with pixmap 'name' of maximum size
 *  'max_width' x 'max_height'. If 'pw' is already valid (i.e., != NULL)
 *  just replace the pixmap of the widget. 
 *
 *  Return value:
 *	new or changed pixmap widget
 */
{
   char	     *path;			
   GdkBitmap *mask   = NULL;
   GdkPixmap *pixmap = NULL;

   if (streq (name, "BulletHole.xpm"))
   {
      pixmap = p_array [P_HOLE].pixmap;
      mask   = p_array [P_HOLE].mask;
      path   = NULL;
   }
   else
      path = get_pixmap_path (name);
   
#ifdef HAVE_STAT
   /*
    *  Don't load large XPM (may cause trouble with imlib)
    */
   if (path)
   {
      struct stat buffer;
      
      if (strstr (path, ".xpm") && stat (path, &buffer) == 0
	  && buffer.st_size > 65000) /* image too large */
      {
	 Free (path);
	 path   = NULL;
	 pixmap = p_array [P_HOLE].pixmap;
	 mask   = p_array [P_HOLE].mask;
      }
   }
#endif /* HAVE_STAT */
   
   if (path)
   {
#ifdef HAVE_IMLIB
      {
	 GdkImlibImage *img;

	 img = gdk_imlib_load_image (path);
	 if (img)
	 {
	    if (!max_height)
	       max_height = img->rgb_height;
	    if (!max_width)
	       max_width = img->rgb_width;
	    gdk_imlib_render (img,
			      min ((unsigned) img->rgb_width, max_width),
			      min ((unsigned) img->rgb_height, max_height));
	    pixmap = gdk_imlib_move_image (img);
	    mask   = gdk_imlib_move_mask (img);
	    gdk_imlib_kill_image (img);
	 }
      }
#else  /* not HAVE_IMLIB */
      pixmap
	 = gdk_pixmap_create_from_xpm (main_window->window, &mask,
				       &main_window->style->bg [GTK_STATE_NORMAL],
				       path); 
#endif /* not HAVE_IMLIB */
      Free (path);
   }
   
   if (!pixmap)				/* pixmap not found or unknown format */
   {
#ifdef HAVE_IMLIB
      char *message1 = "Unable to display pixmap '%s`\n"
		       "(format not supported by imlib).\n"
		       "See stderr for subsequent messages.";
#else  /* not HAVE_IMLIB */
      char *message1 = "Unable to display pixmap '%s`\n"
		       "(format not supported).\n"
		       "Please install Imlib - available at\n"
		       "http://www.labs.redhat.com/imlib/\n"
		       "See stderr for subsequent messages.";
#endif /* not HAVE_IMLIB */
      char *stderr_msg = "Can't display pixmap '%s`.";
      char *message2 = "Can't find pixmap '%s` in PixmapPath."
		       "\nPlease update your PixmapPath attribute accordingly.";
      char *tmp = Calloc (max (strlen (message1), strlen (message2))
			  + strlen (name), sizeof (char));
      static bool_t error_message = NO;

      sprintf (tmp,
	       path ? (error_message ? stderr_msg : message1)
	       : message2, name);
      if (path)
      {
	 if (!error_message)
	 {
	    dialog_popup (DIALOG_ERROR, tmp, NULL, NULL);
	    error_message = YES;
	 }
	 else
	    warning (tmp);
      }
      else
	 dialog_popup (DIALOG_ERROR, tmp, NULL, NULL);
      
      Free (tmp);
      pixmap = p_array [P_HOLE].pixmap;
      mask   = p_array [P_HOLE].mask;
   }
   if (pw)
   {
      if (pw->window)
	 gdk_window_clear_area (pw->window,
				pw->allocation.x,
				pw->allocation.y,
				pw->allocation.width,
				pw->allocation.height);
      gtk_pixmap_set (GTK_PIXMAP (pw), pixmap, mask);
   }
   else
   {
      pw = gtk_pixmap_new (pixmap, mask);
      gtk_widget_show (pw);
   }
   
   return pw;
}

gint
quit (GtkWidget *widget, gpointer ptr)
/*
 *  Return from gtk_main () loop, i.e. leave the application.
 *  Before quitting ask for confirmation.
 *
 *  No return value.
 */
{
#if defined(HAVE_POPEN) && defined(TILE_OF_DAY) && defined(CRONTAB)   
   extern bool_t cron_changed;
   
   if (cron_changed && !changed)
   {
      quit_and_save_popup ("The crontab file is modified.\n"
			   "Do you want to save it before quitting?",
			   save_config_file, NULL);
      return TRUE;
   }
   else if (cron_changed && changed)
   {
      quit_and_save_popup ("The WindowMaker attributes and \n"
			   "the crontab file are modified.\n"
			   "Do you want to save them before quitting?",
			   save_config_file, NULL);
      return TRUE;
   }
#endif /* HAVE_POPEN && TILE_OF_DAY */
   if (changed)
      quit_and_save_popup ("The WindowMaker attributes are modified.\n"
			   "Do you want to save them before quitting?",
			   save_config_file, NULL);
   else
      gtk_main_quit ();

   return TRUE;				/* do not delete window */
}

void
save_config_file (GtkWidget *widget, gpointer ptr)
/*
 *  Save 'WindowMaker' file if attributes have been changed.
 *
 *  No return value.
 *
 *  Side effects:
 *	'changed' is set to FALSE
 */
{
#if defined(HAVE_POPEN) && defined(TILE_OF_DAY) && defined(CRONTAB)
   save_crontab ();
#endif /* HAVE_POPEN && TILE_OF_DAY && CRONTAB */
   
   if (changed)
   {
      proplist_t *newwm = PLMakeDictionaryFromEntries (NULL, NULL, NULL);
      
      PLSetFilename (newwm, PLRetain (PLGetFilename (windowmaker)));
      /*
       *  Compare user attributes with system attributes
       */
      {
	 proplist_t *all_keys = PLGetAllDictionaryKeys (windowmaker) ;
	 unsigned	 n;
      
	 for (n = 0; n < PLGetNumberOfElements (all_keys); n++)
	 {
	    proplist_t *user;
	    proplist_t *key = PLGetArrayElement (all_keys, n);
	 
	    user   = PLGetDictionaryEntry (windowmaker, key);

	    if (PLGetDictionaryEntry (plist_changed, key))
	       PLInsertDictionaryEntry (newwm, key, PLDeepCopy (user));
	 }
	 PLRelease (all_keys);
      }
   
      if (PLSave (newwm, YES))
      {
	 changed = NO;
	 message ("WindowMaker config file '%s' saved.",
		  PLGetString (PLGetFilename (windowmaker)));
      }
      else
	 error ("Can't save WindowMaker config file '%s'.",
		PLGetString (PLGetFilename (windowmaker)));
      
      PLRelease (newwm);
   }
}

/*******************************************************************************

			       private code
  
*******************************************************************************/

static void
page_switch (GtkWidget *widget, GtkNotebookPage *page)
/*
 *  Change pixmaps if current page != old page.
 *
 *  No return value.
 */
{
   GtkNotebookPage	*oldpage;
   GtkWidget		*pixwid;

   oldpage = GTK_NOTEBOOK (widget)->cur_page;

   if (page == oldpage)
      return;

   pixwid = ((GtkBoxChild*)(GTK_BOX (page->tab_label)->children->data))->widget;
   gtk_pixmap_set (GTK_PIXMAP (pixwid),
		   p_array [P_OPEN].pixmap, p_array [P_OPEN].mask);

   pixwid = ((GtkBoxChild*) (GTK_BOX (page->menu_label)->children->data))->widget;
   gtk_pixmap_set (GTK_PIXMAP (pixwid),
		   p_array [P_OPEN].pixmap, p_array [P_OPEN].mask);

   if (oldpage)
   {
      pixwid = ((GtkBoxChild *) (GTK_BOX (oldpage->tab_label)->children->data))->widget;
      gtk_pixmap_set (GTK_PIXMAP (pixwid),
		      p_array [P_CLOSE].pixmap, p_array [P_CLOSE].mask);
      pixwid = ((GtkBoxChild *) (GTK_BOX (oldpage->menu_label)->children->data))->widget;
      gtk_pixmap_set (GTK_PIXMAP (pixwid),
		      p_array [P_CLOSE].pixmap, p_array [P_CLOSE].mask);
   }
}

static GtkWidget *
create_top_page (GtkNotebook *notebook, const char *name, pixmap_t *pm)
/*
 *  Create a new group notebook page and append it to the given 'notebook'.
 *  'name' is the notebook label and 'pm' is the pixmap identifier.
 *
 *  Return value:
 *	container of label and pixmap
 */
{
   GtkWidget *vbox;
   GtkWidget *label_box;
   GtkWidget *menu_box;
   GtkWidget *pixwid;
   
   vbox = gtk_vbox_new (FALSE, 5);

   label_box = gtk_vbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (label_box), 5);

   gtk_box_pack_start (GTK_BOX (label_box), gtk_pixmap_new (pm->pixmap, pm->mask),
		       FALSE, TRUE, 0);
   gtk_box_pack_start (GTK_BOX (label_box), gtk_label_new (name), FALSE, TRUE, 0);
   gtk_widget_show_all (label_box);
      
   menu_box = gtk_hbox_new (FALSE, 0);
   pixwid = gtk_pixmap_new (pm->pixmap, pm->mask);
   gtk_box_pack_start (GTK_BOX (menu_box), pixwid, FALSE, TRUE, 0);
   gtk_misc_set_padding (GTK_MISC (pixwid), 3, 1);
   gtk_box_pack_start (GTK_BOX (menu_box), gtk_label_new (name), FALSE, TRUE, 0);
   gtk_widget_show_all (menu_box);

   gtk_widget_show_all (vbox);
   gtk_notebook_append_page_menu (notebook, vbox, label_box, menu_box);

   return vbox;
}

static GtkWidget *
create_page (GtkNotebook *notebook, const char *name)
/*
 *  Create a new texture notebook page and append it to the given 'notebook'.
 *  'name' is the notebook label.
 *
 *  Return value:
 *	container of label and
 */
{
   GtkWidget *vbox;
   GtkWidget *label_box;
   GtkWidget *menu_box;
   GtkWidget *pixwid;

   vbox = gtk_vbox_new (FALSE, 5);

   label_box = gtk_hbox_new (FALSE, 0);
   pixwid = gtk_pixmap_new (p_array [P_CLOSE].pixmap, p_array [P_CLOSE].mask);
   gtk_box_pack_start (GTK_BOX (label_box), pixwid, FALSE, TRUE, 0);
   gtk_misc_set_padding (GTK_MISC (pixwid), 3, 1);
   gtk_box_pack_start (GTK_BOX (label_box), gtk_label_new (name), FALSE, TRUE, 0);
   gtk_widget_show_all (label_box);
      
   menu_box = gtk_hbox_new (FALSE, 0);
   pixwid = gtk_pixmap_new (p_array [P_CLOSE].pixmap, p_array [P_CLOSE].mask);
   gtk_box_pack_start (GTK_BOX (menu_box), pixwid, FALSE, TRUE, 0);
   gtk_misc_set_padding (GTK_MISC (pixwid), 3, 1);
   gtk_box_pack_start (GTK_BOX (menu_box), gtk_label_new (name), FALSE, TRUE, 0);
   gtk_widget_show_all (menu_box);

   gtk_widget_show_all (vbox);
   gtk_notebook_append_page_menu (notebook, vbox, label_box, menu_box);

   return vbox;
}

static GtkWidget *
path_dialog (proplist_t *key)
/*
 *  Generate path dialog widget.
 *  Associated wmaker attribute name is given by 'key'.
 *
 *  Return value:
 *	container box
 */ 
{
   GtkWidget *hbox;
   GtkWidget *scrolled;
   GtkWidget *list;
   GtkWidget *frame;
   
   frame = gtk_frame_new (PLGetString (key));
   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_border_width (GTK_CONTAINER (frame), 5);
   
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_container_add (GTK_CONTAINER (frame), hbox);

   /*
    *  Scrolled window for list
    */
   scrolled = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_box_pack_start (GTK_BOX (hbox), scrolled, TRUE, TRUE, 5);
   gtk_widget_set_usize (scrolled, 0, 140);
   gtk_container_border_width (GTK_CONTAINER (scrolled), 5);
   
   /*
    *  Path list
    */
   list = gtk_list_new ();
   gtk_list_set_selection_mode (GTK_LIST (list), GTK_SELECTION_BROWSE);
   gtk_container_add (GTK_CONTAINER (scrolled), list);
   gtk_object_set_user_data (GTK_OBJECT (list), key);
   
   
   /*
    *  Initial list elements
    */
   {
      unsigned		i;
      proplist_t	value  = PLGetDictionaryEntry (windowmaker, key);
      
      for (i = 0; i < PLGetNumberOfElements (value); i++)
      {
	 GtkWidget *list_item;
	 GtkWidget *label;
	 proplist_t *plinfo = PLMakeString ("Info");
	 proplist_t *keydef = PLGetDictionaryEntry (wmconfig, key);
	 proplist_t *info   = PLGetDictionaryEntry (keydef, plinfo);
	 
	 list_item = gtk_list_item_new ();
	 gtk_tooltips_set_tip (tooltips, list_item, PLGetString (info), NULL); 
	 PLRelease (plinfo);

	 gtk_object_set_user_data (GTK_OBJECT (list_item),
				   PLGetArrayElement (value, i));
	 gtk_container_border_width (GTK_CONTAINER (list_item), 2);

	 label = gtk_label_new (PLGetString (PLGetArrayElement (value, i)));
	 gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	 gtk_container_add (GTK_CONTAINER (list_item), label);
	 gtk_container_add (GTK_CONTAINER (list), list_item);
      }
   }   
   gtk_list_select_item (GTK_LIST (list), 0);
   
   /*
    *  List manipulating buttons
    */
   {
      GtkWidget  *bbox, *vbox; 
      const char *text [] = {"Remove", "Insert"};
      const char *info [] = {"Remove selected path from path list.",
			     "Insert a new path before current path."};
      void (*fct []) (GtkWidget *, GtkWidget *list) = {path_remove, path_insert};
      unsigned	i;
      GtkWidget *button [2];
      
      vbox = gtk_vbox_new (FALSE, 5);
      gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 5);

      bbox = gtk_vbutton_box_new ();
      gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 5);

      for (i = 0; i < 2; i++)
      {
	 button [i] = gtk_button_new_with_label (text [i]);
	 gtk_box_pack_start (GTK_BOX (bbox), button [i], FALSE, FALSE, 5);
	 gtk_object_set_user_data (GTK_OBJECT (button [i]), button [0]);
	 gtk_signal_connect (GTK_OBJECT (button [i]), "clicked",
			     GTK_SIGNAL_FUNC (fct [i]), (gpointer) list);
	 gtk_tooltips_set_tip (tooltips, button [i], info [i], NULL);  
      }
   }

   gtk_widget_show_all (frame);
   return frame;
}   

char *last_path = NULL;

static void
path_remove (GtkWidget *widget, GtkWidget *list)
/*
 *  Remove selected path from path list.
 *
 *  No return value.
 */
{
   GList      *tmp_list;
   GList      *clear_list;
   unsigned   i;
   proplist_t *key   = gtk_object_get_user_data (GTK_OBJECT (list));
   proplist_t *array = PLGetDictionaryEntry (windowmaker, key);
   
   if (g_list_length (GTK_LIST (list)->children) <= 1)
      return;				/* can't remove last path */

   tmp_list   = GTK_LIST (list)->selection;
   clear_list = NULL;

   /*
    *  Remove color from proplist array
    */
   for (i = 0; i < PLGetNumberOfElements (array); i++)
   {
      if (PLGetArrayElement (array, i)
	  == gtk_object_get_user_data (GTK_OBJECT (GTK_LIST (list)->selection->data)))
      {
	 if (last_path)
	    Free (last_path);
	 last_path = strdup (PLGetString (PLGetArrayElement (array, i)));
	 PLRemoveArrayElement (array, i);
	 break;
      }
   }
   
   /*
    *  Remove list item
    */
   while (tmp_list)
   {
      clear_list = g_list_prepend (clear_list, tmp_list->data);
      tmp_list = tmp_list->next;
   }

   clear_list = g_list_reverse (clear_list);
   gtk_list_remove_items (GTK_LIST (list), clear_list);
   g_list_free (clear_list);

   if (g_list_length (GTK_LIST (list)->children))
      gtk_list_select_item (GTK_LIST (list), 0);

   if (g_list_length (GTK_LIST (list)->children) == 1)
      gtk_widget_set_sensitive (widget, FALSE);

   changed = YES;
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
}

static void
path_insert (GtkWidget *widget, GtkWidget *list)
/*
 *  Insert new path element into path list.
 *
 *  No return value.
 */
{
   GtkWidget  *list_item;
   int	      position;
   proplist_t *key     = gtk_object_get_user_data (GTK_OBJECT (list));
   proplist_t *array   = PLGetDictionaryEntry (windowmaker, key);
   proplist_t *element = PLMakeString (last_path ? last_path : "");
   GtkWidget  *button  = gtk_object_get_user_data (GTK_OBJECT (widget));
   
   position = gtk_list_child_position (GTK_LIST (list),
				       (GTK_LIST (list)->selection)->data);
   
   PLInsertArrayElement (array, element, position);

   list_item = gtk_list_item_new ();
   gtk_widget_show (list_item);
   gtk_object_set_user_data (GTK_OBJECT (list_item),
			     PLGetArrayElement (array, position));
   gtk_container_border_width (GTK_CONTAINER (list_item), 2);

   {
      proplist_t *plinfo = PLMakeString ("Info");
      proplist_t *keydef = PLGetDictionaryEntry (wmconfig, key);
      proplist_t *info	 = PLGetDictionaryEntry (keydef, plinfo);
      
      gtk_tooltips_set_tip (tooltips, list_item, PLGetString (info), NULL); 
      PLRelease (plinfo);
   }
   
   /*
    *  Insert list item at selection
    */
   {
      GList *gl  = Calloc (1, sizeof (GList));
      GtkList *l = GTK_LIST (list);
      
      gl->next = gl->prev = NULL;
      gl->data = list_item;

      gtk_list_insert_items (l, gl,
			     gtk_list_child_position (l, (l->selection)->data));
   }
   gtk_list_select_item (GTK_LIST (list), position);
   gtk_widget_set_sensitive (button, TRUE);
   
   changed = YES;
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);

   {
      static GtkWidget *filesel = NULL;

      if (!filesel)
      {
	 GtkFileSelection *fs;

	 /*
	  *  Fileselection dialog window
	  */
	 filesel = gtk_file_selection_new ("Pixmap file browser");
	 fs      = GTK_FILE_SELECTION (filesel);
	 gtk_file_selection_hide_fileop_buttons (fs);
	 gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
	 gtk_object_set_user_data (GTK_OBJECT (fs), list_item);
	 gtk_signal_connect (GTK_OBJECT (filesel), "destroy",
			     GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			     &filesel);
	 gtk_signal_connect (GTK_OBJECT (fs->ok_button), "clicked",
			     GTK_SIGNAL_FUNC (set_path),
			     (gpointer) filesel);
	 gtk_signal_connect (GTK_OBJECT (fs->cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (path_remove),
			     (gpointer) list);
	 gtk_object_set_user_data (GTK_OBJECT (fs->ok_button), key);
	 gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_destroy),
				    GTK_OBJECT (filesel));
	 gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
				    GTK_SIGNAL_FUNC (gtk_widget_destroy),
				    GTK_OBJECT (filesel));
	 gtk_file_selection_set_filename (fs, PLGetString (PLGetArrayElement (array, position)));
      }
   
      if (!GTK_WIDGET_VISIBLE (filesel))
	 gtk_widget_show (filesel);
      else
	 gtk_widget_destroy (filesel);
   }
}

static void
set_path (GtkWidget *widget, gpointer filesel)
/*
 *  Accept selected path.
 *
 *  No return value.
 */
{
   char			*path;
   GtkFileSelection	*fs = GTK_FILE_SELECTION (filesel);
   GtkWidget		*list_item = gtk_object_get_user_data (GTK_OBJECT (fs));
   proplist_t		*key = gtk_object_get_user_data (GTK_OBJECT (widget));
   proplist_t		*array = PLGetDictionaryEntry (windowmaker, key);
   proplist_t		*value;
   unsigned		n;
   
   path  = gtk_file_selection_get_filename (fs);
   value = gtk_object_get_user_data (GTK_OBJECT (list_item));
   
   for (n = 0; n < PLGetNumberOfElements (array); n++)
      if (value == PLGetArrayElement (array, n))
      {
	 GtkWidget *label;

	 if (path [strlen (path) - 1] == '/')
	    path [strlen (path) - 1] = 0; 

	 PLRemoveArrayElement (array, n);
	 PLInsertArrayElement (array, PLMakeString ((char *) path), n);
	 
	 label = gtk_label_new (path);
	 gtk_widget_show (label);
	 gtk_container_add (GTK_CONTAINER (list_item), label);
      }
   
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
   changed = YES;
}

static bool_t
convert_bool (proplist_t *p)
/*
 *  Parse boolean values of WindowMaker file.
 *
 *  Return value:
 *	converted value
 */
{
   char *val;

   assert (p && PLIsSimple (p));

   val = PLGetString (p);

   if (strcaseeq (val, "y") || strcaseeq (val, "yes")
       || strcaseeq (val, "t") || strcaseeq (val, "true")
       || strcaseeq (val, "1"))
      return TRUE;
   else if (strcaseeq (val, "n") || strcaseeq (val, "no")
	    || strcaseeq (val, "f") || strcaseeq (val, "false")
	    || strcaseeq (val, "0"))
      return FALSE;
   else
   {
      warning ("Undefined boolean value '%s' converted to true.", val);
      return TRUE;
   }
}

static void
set_string (GtkWidget *widget, gpointer ptr)
/*
 *  Update string value.
 *
 *  No return value.
 */
{
   proplist_t *key  = (proplist_t *) ptr;
   char       *text = gtk_object_get_user_data (GTK_OBJECT (widget));
   
   PLInsertDictionaryEntry (windowmaker, key, PLMakeString (text));
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
   changed = YES;
}

static void
set_bool (GtkWidget *widget, gpointer ptr)
/*
 *  Update boolean value.
 *
 *  No return value.
 */
{
   proplist_t *key  = (proplist_t *) ptr;
   char       *text = GTK_TOGGLE_BUTTON (widget)->active ? "Yes" : "No";
   
   PLInsertDictionaryEntry (windowmaker, key, PLMakeString (text));
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
   changed = YES;
}

static void
set_unsigned (GtkWidget *widget, gpointer ptr)
/*
 *  Update unsigned value.
 *
 *  No return value.
 */
{
   proplist_t	 *key = (proplist_t *) ptr;
   GtkSpinButton *spinner;
   int		 val;
   char		 tmp [MAXSTRLEN];
   
   spinner = GTK_SPIN_BUTTON (gtk_object_get_user_data (GTK_OBJECT (widget)));
   val     = gtk_spin_button_get_value_as_int (spinner);
   
   sprintf (tmp, "%d", val);

   if (PLIsArray (PLGetDictionaryEntry (windowmaker, key)))
   {
      proplist_t *array = PLGetDictionaryEntry (windowmaker, key);
      unsigned	 n;
      
      n = (GtkWidget *) gtk_object_get_user_data (GTK_OBJECT (spinner))
	  - GTK_WIDGET (spinner);
      PLRemoveArrayElement (array, n);
      PLInsertArrayElement (array, PLMakeString (tmp), n);
   }
   else
      PLInsertDictionaryEntry (windowmaker, key, PLMakeString (tmp));
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
   changed = YES;
}

static void
set_entry_text (GtkWidget *widget, gpointer ptr)
/*
 *  Update string value.
 *
 *  No return value.
 */
{
   proplist_t *element;
   proplist_t *key   = (proplist_t *) ptr;
   proplist_t *array = PLGetDictionaryEntry (windowmaker, key);

   element = PLMakeString ((char *) gtk_entry_get_text (GTK_ENTRY (widget)));

   if (PLIsArray (array))
   {
      PLRemoveArrayElement (array, 1);	/* filename */
      PLInsertArrayElement (array, element, 1);
   }
   else
      PLInsertDictionaryEntry (windowmaker, key, element);
   
   PLInsertDictionaryEntry (plist_changed, key, pl_yes);
   changed = YES;
}

static void
fontsel_dialog (GtkWidget *widget, gpointer ptr)
{
  static GtkWidget *fontsel = NULL;

  if (!fontsel)
  {
     GtkWidget			*entry;
     char			*text;
     GtkFontSelectionDialog	*fs;
     
     entry   = GTK_WIDGET (ptr);
     text    = gtk_entry_get_text (GTK_ENTRY (entry));
     fontsel = gtk_font_selection_dialog_new ("Font selection");
     fs	     = GTK_FONT_SELECTION_DIALOG (fontsel);
     
     gtk_window_position (GTK_WINDOW (fontsel), GTK_WIN_POS_MOUSE);
     gtk_object_set_user_data (GTK_OBJECT (fontsel), entry);
     gtk_font_selection_dialog_set_font_name (fs, text);

     gtk_object_set_data (GTK_OBJECT (entry), "fontsel", fontsel);
     gtk_signal_connect (GTK_OBJECT (fontsel), "destroy",
			 GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			 &fontsel);
     gtk_signal_connect  (GTK_OBJECT (fs->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (set_fontname), fontsel);
     gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_destroy),
				GTK_OBJECT (fontsel));
     gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
				GTK_SIGNAL_FUNC (gtk_widget_destroy),
				GTK_OBJECT (fontsel));
  }

  if (!GTK_WIDGET_VISIBLE (fontsel))
     gtk_widget_show (fontsel);
  else
     gtk_widget_destroy (fontsel);
}

static void
set_fontname (GtkWidget *widget, gpointer ptr)
{
   GtkWidget *fontsel = GTK_WIDGET (ptr);
   GtkWidget *entry   = gtk_object_get_user_data (GTK_OBJECT (fontsel));
   gchar     *fontname;

   fontname = gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG (fontsel));
   gtk_entry_set_text (GTK_ENTRY (entry), fontname);
}

