// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: http.cc,v 1.27 1998/07/16 06:45:37 jgg Exp $
/* ######################################################################

   HTTP Aquire Method - This is the HTTP aquire method for APT.
   
   It uses HTTP/1.1 and many of the fancy options there-in, such as
   pipelining, range, if-range and so on. It accepts on the command line
   a list of url destination pairs and writes to stdout the status of the
   operation as defined in the APT method spec.
   
   It is based on a doubly buffered select loop. All the requests are 
   fed into a single output buffer that is constantly fed out the 
   socket. This provides ideal pipelining as in many cases all of the
   requests will fit into a single packet. The input socket is buffered 
   the same way and fed into the fd for the file.
   
   This double buffering provides fairly substantial transfer rates,
   compared to wget the http method is about 4% faster. Most importantly,
   when HTTP is compared with FTP as a protocol the speed difference is
   huge. In tests over the internet from two sites to llug (via ATM) this
   program got 230k/s sustained http transfer rates. FTP on the other 
   hand topped out at 170k/s. That combined with the time to setup the
   FTP connection makes HTTP a vastly superior protocol.
   
   ##################################################################### */
									/*}}}*/
#define _BSD_SOURCE       // For socket stuff
#define _XOPEN_SOURCE     // For timezone
#include <pkglib/md5.h>

#include <string>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <utime.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <signal.h>
#include <strutl.h>

#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>

string Proxy;
string ProxyAuth;

#define MAXLEN 360

struct ServerInfo
{
   unsigned int Major;
   unsigned int Minor;
   unsigned int Result;
   unsigned long Size;
   char Code[MAXLEN];
   signed long StartPos;
   time_t Date;

   enum {Chunked,Stream,Closes} Encoding;
   bool HeaderLine(string Line);
   void Reset() {Major = 0; Minor = 0; Result = 0; Size = 0; StartPos = 0;
                 Encoding = Closes; time(&Date);};
   
   ServerInfo() {Reset();};
};

class Command
{
   void BuildReq();
   
   public:

   string Req;
   string Host;
   string Path;
   const char *Output;
   const char *oURL;
   string URL;
   
   Command(const char *URL,const char *Output);
};

class CircleBuf
{
   unsigned char *Buf;
   unsigned long Size;
   unsigned long InP;
   unsigned long OutP;
   string OutQueue;
   unsigned long StrPos;
   unsigned long MaxGet;
   struct timeval Start;
   
   unsigned long LeftRead()
   {
      unsigned long Sz = Size - (InP - OutP);
      if (Sz > Size - (InP%Size))
	 Sz = Size - (InP%Size);
      return Sz;
   }
   unsigned long LeftWrite()
   {
      unsigned long Sz = InP - OutP;
      if (InP > MaxGet)
	 Sz = MaxGet - OutP;
      if (Sz > Size - (OutP%Size))
	 Sz = Size - (OutP%Size);
      return Sz;
   }
   void FillOut();
   
   public:
   
   MD5Summation *MD5;
   
   // Read data in
   bool Read(int Fd);
   bool Read(string Data);
   
   // Write data out
   bool Write(int Fd);
   bool WriteTillEl(string &Data,bool Single = false);
   
   // Control the write limit
   void Limit(long Max) {if (Max == -1) MaxGet = 0-1; else MaxGet = OutP + Max;}   
   bool IsLimit() {return MaxGet == OutP;};
   void Print() {cout << MaxGet << ',' << OutP << endl;};

   // Test for free space in the buffer
   bool ReadSpace() {return Size - (InP - OutP) > 0;};
   bool WriteSpace() {return InP - OutP > 0;};
   
   void Stats();

   CircleBuf(unsigned long Size);
   ~CircleBuf() {delete [] Buf;};
};

class Loop
{
   // The directions are with respect to the server fd
   CircleBuf In;
   CircleBuf Out;
   
   int FD;
   int Server;
   unsigned int MaxFd;
   ServerInfo SInfo;
   string OFile;
   enum {Header,Data} State;
   bool Fail;
   
   bool SetNonBlocking(int fd,bool On = true);
   bool Go(bool ToFile);
   bool ServerDie();
   
   public:

   bool Restart;

   string Hash() {return (SInfo.StartPos >= 0 && Fail == false)?string(In.MD5->Result()):string();}

   bool SetTime();
   bool OpenRemote(string Host);
   bool OutputTo(string File);
   bool AddRequest(string Header) {Out.Read(Header);};
   bool RunHeaders();
   bool RunData();

   Loop();
   ~Loop();
};

// Case insensitive compare
bool Cmp(const string &S,const char *C)
{
   if (S.length() > strlen(C))
      return strncasecmp(S.begin(),C,S.length()) == 0;
   else
      return strncasecmp(S.begin(),C,strlen(C)) == 0;
}

// RFC1123 - Convert a time_t into RFC1123 format			/*{{{*/
// ---------------------------------------------------------------------
/* The is the format the HTTP spec wants, we always conver to GMT as that
   is simplest for a unix system. */
string RFC1123(const time_t &Date)
{
   struct tm Conv = *gmtime(&Date);
   char Buf[300];

   const char *Day[] = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
   const char *Month[] = {"Jan","Feb","Mar","Apr","May","Jun","Jul",
                          "Aug","Sep","Oct","Nov","Dec"};

   sprintf(Buf,"%s, %02i %s %i %02i:%02i:%02i GMT",Day[Conv.tm_wday],
	   Conv.tm_mday,Month[Conv.tm_mon],Conv.tm_year+1900,Conv.tm_hour,
	   Conv.tm_min,Conv.tm_sec);
   return Buf;
}
									/*}}}*/
// MonthConv - Converts a month string into a number			/*{{{*/
// ---------------------------------------------------------------------
/* This was lifted from the boa webserver which lifted it from 'wn-v1.07'
   Made it a bit more robust with a few touppers though. */
int MonthConv(char *Month)
{
   switch (toupper(*Month)) 
   {
      case 'A':
      return toupper(Month[1]) == 'P'?3:7;
      case 'D':
      return 11;
      case 'F':
      return 1;
      case 'J':
      if (toupper(Month[1]) == 'A')
	 return 0;
      return toupper(Month[2]) == 'N'?5:6;
      case 'M':
      return toupper(Month[2]) == 'R'?2:4;
      case 'N':
      return 10;
      case 'O':
      return 9;
      case 'S':
      return 8;

      // Pretend it is January..
      default:
      return 0;
   }   
}
									/*}}}*/
// GrabHost - Grabs the host portion of a url				/*{{{*/
// ---------------------------------------------------------------------
/* This pulls out the host name from a URL */
string GrabHost(const char *URL,string &Host)
{
   const char *I;
   string Path;

   // Find the letter after the 2nd slash
   for (I = URL; *I != 0 && *I != '/'; I++);
   for (; *I != 0 && *I == '/'; I++);
   
   // Find the next slash
   const char *End;
   for (End = I; *End != 0 && *End != '/'; End++);
   Host = string(I,End - I);
   
   // Set the path
   if (*End == 0)
      Path = '/';
   else
      Path = End;

   return Path;
}
									/*}}}*/
// GrabAuth - Grab authentication information from a URI		/*{{{*/
// ---------------------------------------------------------------------
/* This strips out the authentication information from the URL and returns
   it in a form suitable for base64 encoding. */
string GrabAuth(string &Proxy)
{
   string::size_type At = Proxy.rfind('@');
   string::size_type Slash = Proxy.find("://");
   if (At == string::npos || Slash == string::npos)
      return string();
   Slash += 3;
   
   string Res = string(Proxy,Slash,At - Slash);
   At++;
   Proxy = string(Proxy,0,Slash) + string(Proxy,At,Proxy.length() - At);
   return Res;
}
									/*}}}*/

// Command::Command - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* This strips off the host name and strips off the hostname */
Command::Command(const char *iURL,const char *Output) : Output(Output), 
                  oURL(iURL)
{
   URL = QuoteString(string(iURL),"\\|{}[]<>\"^~_=!@#$%^&*");
   Path = GrabHost(URL.c_str(),Host);
   BuildReq();
}
									/*}}}*/
// Command::BuildReq - Build the request header				/*{{{*/
// ---------------------------------------------------------------------
/* As per the method specification we check the date and size of the file
   in the current directory and try to resume if possible. We also check
   the date of the file in the pervious directory for a IMS query. */
void Command::BuildReq()
{
   // Build the request
   char Buf[300];
   if (Proxy.empty() == true)
      sprintf(Buf,"GET %s HTTP/1.1\r\nHost: %s\r\nConnection: keep-alive\r\n",Path.c_str(),Host.c_str());
   else
      sprintf(Buf,"GET %s HTTP/1.1\r\nHost: %s\r\n",URL.c_str(),Host.c_str());
   Req = Buf;
   
   // Check for a partial file
   struct stat SBuf;
   if (stat(Output,&SBuf) >= 0 && SBuf.st_size > 0)
   {
      // In this case we send an if-range query with a range header
      sprintf(Buf,"Range: bytes=%li-\r\nIf-Range: %s\r\n",SBuf.st_size - 1,
	      RFC1123(SBuf.st_mtime).c_str());
      Req += Buf;
   }
   else
   {
      string Old("../");
      Old += Output;
      if (stat(Old.c_str(),&SBuf) >= 0)
      {
	 sprintf(Buf,"If-Modified-Since: %s\r\n",RFC1123(SBuf.st_mtime).c_str());
	 Req += Buf;
      }
   }

   if (ProxyAuth.empty() == false)
      Req += string("Proxy-Authorization: Basic ") + Base64Encode(ProxyAuth) + "\r\n";

   Req += "User-Agent: Debian APT-HTTP/1.1\r\n\r\n";
   
//   cout << Req << endl;
}
									/*}}}*/

// CircleBuf::CircleBuf - Circular input buffer				/*{{{*/
// ---------------------------------------------------------------------
/* */
CircleBuf::CircleBuf(unsigned long Size) : Size(Size), MD5(0)
{
   Buf = new unsigned char[Size];
   InP = 0;
   OutP = 0;
   StrPos = 0;
   MaxGet = 0-1;
}
									/*}}}*/
// CircleBuf::Read - Read from a FD into the circular buffer		/*{{{*/
// ---------------------------------------------------------------------
/* This fills up the buffer with as much data as is in the FD, assuming it
   is non-blocking.. 
 
   Unfortunately some odd read semantics make this uns*/
bool CircleBuf::Read(int Fd)
{
   while (1)
   {
      // Woops, buffer is full
      if (InP - OutP == Size)
	 return true;
      
      // Write the buffer segment
      int Res;
      Res = read(Fd,Buf + (InP%Size),LeftRead());
      
      if (Res == 0)
	 return false;
      if (Res < 0)
      {
	 if (errno == EAGAIN)
	    return true;
	 return false;
      }

      if (InP == 0)
	 gettimeofday(&Start,0);
      InP += Res;
   }
}
									/*}}}*/
// CircleBuf::Read - Put the string into the buffer			/*{{{*/
// ---------------------------------------------------------------------
/* This will hold the string in and fill the buffer with it as it empties*/
bool CircleBuf::Read(string Data)
{
   OutQueue += Data;
   FillOut();
   return true;
}
									/*}}}*/
// CircleBuf::FillOut - Fill the buffer from the output queue		/*{{{*/
// ---------------------------------------------------------------------
/* */
void CircleBuf::FillOut()
{
   if (OutQueue.empty() == true)
      return;
   while (1)
   {
      // Woops, buffer is full
      if (InP - OutP == Size)
	 return;
      
      // Write the buffer segment
      unsigned long Sz = LeftRead();
      if (OutQueue.length() - StrPos < Sz)
	 Sz = OutQueue.length() - StrPos;
      memcpy(Buf + (InP%Size),OutQueue.begin() + StrPos,Sz);
      
      // Advance
      StrPos += Sz;
      InP += Sz;
      if (OutQueue.length() == StrPos)
      {
	 StrPos = 0;
	 OutQueue = "";
	 return;
      }
   }
}
									/*}}}*/
// CircleBuf::Write - Write from the buffer into a FD			/*{{{*/
// ---------------------------------------------------------------------
/* This empties the buffer into the FD. */
bool CircleBuf::Write(int Fd)
{
   while (1)
   {
      FillOut();
      
      // Woops, buffer is empty
      if (OutP == InP)
	 return true;
      
      if (OutP == MaxGet)
	 return true;
      
      // Write the buffer segment
      int Res;
      Res = write(Fd,Buf + (OutP%Size),LeftWrite());

      if (Res == 0)
	 return false;
      if (Res < 0)
      {
	 if (errno == EAGAIN)
	    return true;
	 
	 return false;
      }
      
      if (MD5 != 0)
	 MD5->Add(Buf + (OutP%Size),Res);
      
      OutP += Res;
   }
}
									/*}}}*/
// CircleBuf::WriteTillEl - Write from the buffer to a string		/*{{{*/
// ---------------------------------------------------------------------
/* This copies till the first empty line */
bool CircleBuf::WriteTillEl(string &Data,bool Single)
{
   // We cheat and assume it is unneeded to have more than one buffer load
   for (unsigned long I = OutP; I < InP; I++)
   {      
      if (Buf[I%Size] != '\n')
	 continue;
      for (I++; I < InP && Buf[I%Size] == '\r'; I++);
      
      if (Single == false)
      {
	 if (Buf[I%Size] != '\n')
	    continue;
	 for (I++; I < InP && Buf[I%Size] == '\r'; I++);
      }
      
      if (I > InP)
	 I = InP;
      
      Data = "";
      while (OutP < I)
      {
	 unsigned long Sz = LeftWrite();
	 if (Sz == 0)
	    return false;
	 if (I - OutP < LeftWrite())
	    Sz = I - OutP;
	 Data += string((char *)(Buf + (OutP%Size)),Sz);
	 OutP += Sz;
      }
      return true;
   }      
   return false;
}
									/*}}}*/
// CircleBuf::Stats - Print out stats information			/*{{{*/
// ---------------------------------------------------------------------
/* */
void CircleBuf::Stats()
{
   if (InP == 0)
      return;
   
   struct timeval Stop;
   gettimeofday(&Stop,0);
/*   float Diff = Stop.tv_sec - Start.tv_sec + 
             (float)(Stop.tv_usec - Start.tv_usec)/1000000;
   clog << "Got " << InP << " in " << Diff << " at " << InP/Diff << endl;*/
}
									/*}}}*/

// Loop::Loop - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Loop::Loop() : In(64*1024), Out(1*1024), FD(-1), Server(-1), Fail(false)
{
}
									/*}}}*/
// Loop::~Loop - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Loop::~Loop()
{
   close(FD);
   close(Server);
   if (FD == -1)
      return;
   
   SetTime();
   In.Stats();
}
									/*}}}*/
// Loop::SetTime - Set the files time					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::SetTime()
{
   if (OFile.empty() == true)
      return false;
       
   // Set the mod time
   struct utimbuf UBuf;
   time(&UBuf.actime);
   UBuf.actime = SInfo.Date;
   UBuf.modtime = SInfo.Date;
   utime(OFile.c_str(),&UBuf);
   return false;
}
									/*}}}*/
// Loop::OpenRemote - Open a FD to a remote server			/*{{{*/
// ---------------------------------------------------------------------
/* */
string LastHost;
in_addr LastHostA;
bool Loop::OpenRemote(string Host)
{
   if (Proxy.empty() == false)
      GrabHost(Proxy.c_str(),Host);
   
   int Port = 80;
   string::size_type I = Host.rfind(':');
   if (I != string::npos)
   {
      Port = atoi(string(Host,I+1).c_str());
      Host = string(Host,0,I);
   }

   if (LastHost != Host)
   {
      cout << "I Connecting to " << Host << endl;

      // Lookup the host
      hostent *Addr = gethostbyname(Host.c_str());
      if (Addr == 0)
      {
	 cout << "E Couldn't lookup host " << Host << endl;
	 return false;
      }
      LastHost = Host;
      LastHostA = *(in_addr *)(Addr->h_addr_list[0]);
   }
   
   cout << "I Connecting to " << Host << " (" <<
      inet_ntoa(LastHostA) << ')' << endl;
   
   // Get a socket
   if ((Server = socket(AF_INET,SOCK_STREAM,0)) < 0)
   {
      cout << "E Couldn't create a socket" << endl;
      return false;
   }
   
   // Connect to the server
   struct sockaddr_in server;
   server.sin_family = AF_INET;
   server.sin_port = htons(Port);
   server.sin_addr = LastHostA;
   if (connect(Server,(sockaddr *)&server,sizeof(server)) < 0)
   {
      cout << "E Couldn't connect to " << Host << ":" << Port << endl;
      close(Server);
      return false;
   }

   MaxFd = Server > FD?Server:FD;
   return SetNonBlocking(Server);
}
									/*}}}*/
// Loop::SetNonBlocking - Make a FD non-blocking			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::SetNonBlocking(int fd,bool On)
{
   int flags;
   int dummy = 0;

   if ((flags = fcntl(fd, F_GETFL, dummy)) < 0)
      return false;
   
   if (On == true)
   {
      if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) < 0)
	 return false;
   }
   else
   {
      if (fcntl(fd, F_SETFL,flags & (~(long)O_NONBLOCK)) < 0)
	 return false;
   }
   return true;
}
									/*}}}*/
// Loop::OutputTo - Set the output file					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::OutputTo(string File)
{
   close(FD);

   // Set the mod time
   if (FD != -1)
      SetTime();
   FD = -1;
   
   // Open the output file
   FD = open(File.c_str(),O_CREAT | O_NONBLOCK | O_RDWR,0664);
   if (FD < 0)
   {
      cout << "E Couldn't open output file " << File << endl;
      return false;
   }

   MaxFd = Server > FD?Server:FD;
   OFile = File;
   
   // Setup the MD5 Hash in the output-to-file buffer
   delete In.MD5;
   In.MD5 = new MD5Summation;
   
   return SetNonBlocking(FD);
}
									/*}}}*/
// Loop::Go - Run a single loop						/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::Go(bool ToFile)
{
   // Server has closed the connection
   if (Server == -1 && In.WriteSpace() == false)
      return false;
   
   fd_set rfds,wfds,efds;
   FD_ZERO(&rfds);
   FD_ZERO(&wfds);
   FD_ZERO(&efds);
   if (Out.WriteSpace() == true && Server != -1) FD_SET(Server,&wfds);
   if (In.WriteSpace() == true && ToFile == true && FD != -1) FD_SET(FD,&wfds);
   if (In.ReadSpace() == true && Server != -1) FD_SET(Server,&rfds);
   if (FD != -1)
      FD_SET(FD,&efds);
   if (Server != -1)
      FD_SET(Server,&efds);

   struct timeval tv;
   tv.tv_sec = 120;
   tv.tv_usec = 0;
   int Res = 0;
   if ((Res = select(MaxFd+1,&rfds,&wfds,&efds,&tv)) < 0)
   {
      cout << "E Select error " << strerror(errno) << endl;
      return false;
   }
   
   if (Res == 0)
   {
      cout << "E Connection timed out" << endl;
      return ServerDie();
   }
   
   // Some kind of exception (error) on the sockets, die
   if ((FD != -1 && FD_ISSET(FD,&efds)) || 
       (Server != -1 && FD_ISSET(Server,&efds)))
   {
      cout << "E Socket Expection" << endl;
      return false;
   }

   if (Server != -1 && FD_ISSET(Server,&rfds))
   {
      errno = 0;
      if (In.Read(Server) == false)
	 return ServerDie();
   }
	 
   if (Server != -1 && FD_ISSET(Server,&wfds))
   {
      errno = 0;
      if (Out.Write(Server) == false)
	 return ServerDie();
   }
      	 
   if (FD != -1 && FD_ISSET(FD,&wfds))
   {
      if (In.Write(FD) == false)
      {
	 cout << "E Error writing to file " << strerror(errno) << endl;
	 return false;
      }
   }

   return true;
}
									/*}}}*/
// Loop::RunHeaders - Get the headers before the data			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::RunHeaders()
{
   cout << "I Waiting for file" << endl;
   SInfo.Reset();
   Restart = true;
   State = Header;
   Fail = false;
   do
   {
      string Data;
      if (In.WriteTillEl(Data) == false)
	 continue;
      Restart = false;
      for (string::const_iterator I = Data.begin(); I < Data.end(); I++)
      {
	 string::const_iterator J = I;
	 for (; J != Data.end() && *J != '\n' && *J != '\r';J++);
	 if (SInfo.HeaderLine(string(I,J-I)) == false)
	    return false;
	 I = J;
      }
      // Not Modified
      if (SInfo.Result == 304)
      {
	 unlink(OFile.c_str());
	 SInfo.Size = 0;
	 SInfo.StartPos = -1;
	 SInfo.Result = 200;
	 SInfo.Encoding = ServerInfo::Stream;
      }

      // We have a bad reply, we can likely still continue..
      if (SInfo.Result < 200 || SInfo.Result >= 300)
      {
	 SInfo.StartPos = 0;
	 unlink(OFile.c_str());
	 Fail = true;
	 cout << "E " << SInfo.Result << ' ' << SInfo.Code << endl;
      }
      
      // Set the expected size
      if (SInfo.StartPos < 0)
	 In.Limit(0);
      else
      {
	 In.Limit(SInfo.Size - SInfo.StartPos);
	 ftruncate(FD,SInfo.StartPos);
      }
      
      // Set the start point
      lseek(FD,0,SEEK_END);

      // Fill the MD5 Hash if the file is non-empty (resume)
      if (SInfo.StartPos > 0 && In.MD5 != 0)
      {
	 SetNonBlocking(FD,false);
	 lseek(FD,0,SEEK_SET);
	 if (In.MD5->AddFD(FD,SInfo.StartPos) == false)
	 {
	    cout << "E MD5 hashing error" << endl;
	    SInfo.Size = 0-1;
	    SInfo.StartPos = 0;
	    Fail = true;
	    unlink(OFile.c_str());
	 }
	 lseek(FD,0,SEEK_END);
	 SetNonBlocking(FD,true);
      }
      
      if ((signed)SInfo.Size > 0)
	 cout << "S " << SInfo.Size << endl;
      return true;
   }
   while (Go(false) == true);
   
   return false;
}
									/*}}}*/
// Loop::RunData - Transfer the data from the socket into the file	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::RunData()
{
   State = Data;
   
   if (SInfo.StartPos >= 0 && (SInfo.Size - SInfo.StartPos > 2 || 
			       SInfo.Encoding != ServerInfo::Stream))
   {
      if (SInfo.StartPos != 0)
	 cout << "I Resuming (" << SInfo.StartPos << ')' << endl;
      else
	 cout << "I Downloading" << endl;
   }
   
   // Chunked transfer encoding is fun..
   if (SInfo.Encoding == ServerInfo::Chunked)
   {
      while (1)
      {
	 // Grab the block size
	 bool Last = true;
	 string Data;
	 In.Limit(-1);
	 do
	 {
	    if (In.WriteTillEl(Data,true) == true)
	       break;
	 }
	 while ((Last = Go(false)) == true);

	 if (Last == false)
	    return false;
	 	 
	 // See if we are done
	 unsigned long Len = strtol(Data.c_str(),0,16);
	 if (Len == 0)
	 {
	    In.Limit(-1);
	    
	    // We have to remove the entity trailer
	    Last = true;
	    do
	    {
	       if (In.WriteTillEl(Data,true) == true && Data.length() <= 2)
		  break;
	    }
	    while ((Last = Go(false)) == true);
	    if (Last == false)
	       return false;
	    return true;
	 }
	 
	 // Transfer the block
	 In.Limit(Len);
	 while (Go(true) == true)
	    if (In.IsLimit() == true)
	       break;
	 
	 // Error
	 if (In.IsLimit() == false)
	    return false;
	 
	 // The server sends an extra new line before the next block specifier..
	 In.Limit(-1);
	 Last = true;
	 do
	 {
	    if (In.WriteTillEl(Data,true) == true)
	       break;
	 }
	 while ((Last = Go(false)) == true);
	 if (Last == false)
	    return false;
      }      
   }
   else
   {
      if (SInfo.Encoding == ServerInfo::Closes)
	 In.Limit(-1);
      
      // Just transfer the whole block.
      do
      {
	 if (In.IsLimit() == false)
	    continue;
	 
	 In.Limit(-1);
	 return true;
      }
      while (Go(true) == true);
   }

   // Flush the file output.
   if (FD != -1)
   {
      SetNonBlocking(FD,false);
      if (In.WriteSpace() == false)
	 return true;
      
      while (In.WriteSpace() == true)
      {
	 if (In.Write(FD) == false)
	 {
	    cout << "E Error writing to file " << strerror(errno) << endl;
	    return false;
	 }
      }

      if (In.IsLimit() == true || SInfo.Encoding == ServerInfo::Closes)
	 return true;
   }
   
   return false;
}
									/*}}}*/
// Loop::ServerDie - The server has closed the connection.		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Loop::ServerDie()
{
   // Dump the buffer to the file
   if (State == Data)
   {
      SetNonBlocking(FD,false);
      while (In.WriteSpace() == true)
      {
	 if (In.Write(FD) == false)
	 {
	    cout << "E Error writing to file " << strerror(errno) << endl;
	    return false;
	 }
      }
   }
   
   // See if this is because the server finished the data stream
   if (In.IsLimit() == false && State != Header && 
       SInfo.Encoding != ServerInfo::Closes)
   {
      /* We are getting the header and the server killed the connection
         this probably means it doesn't support keep-alive */
      if (Restart == true)
	 return false;
      
      if (errno == 0)
	 cout << "E Error reading from server Remote end closed connection" << endl;
      else
	 cout << "E Error reading from server " << strerror(errno) << endl;
      return false;
   }
   else
   {
      In.Limit(-1);

      // Nothing left in the buffer
      if (In.WriteSpace() == false)
	 return false;
      
      // We may have got multiple responses back in one packet..
      close(Server);
      Server = -1;
      return true;
   }
   
   return false;
}
									/*}}}*/

// ServerInfo::HeaderLine - Process a header line			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool ServerInfo::HeaderLine(string Line)
{
   if (Line.empty() == true)
      return true;
   
   // The http server might be trying to do something evil.
   if (Line.length() >= MAXLEN)
   {
      clog << "http: Got a single header line over " << MAXLEN << " characters" << endl;
      return false;
   }

   string::size_type Pos = Line.find(' ');
   if (Pos == string::npos || Pos+1 > Line.length())
   {
      clog << "http: Bad header line" << endl;
      return false;
   }
   
   string Tag = string(Line,0,Pos);
   string Val = string(Line,Pos+1);

   if (strncasecmp(Tag.c_str(),"HTTP",4) == 0)
   {
      // Evil servers return no version
      if (Line[4] == '/')
      {
	 if (sscanf(Line.c_str(),"HTTP/%u.%u %u %[^\n]",&Major,&Minor,
		    &Result,Code) != 4)
	 {
	    clog << "http: The http server sent an invalid reply header" << endl;
	    return false;
	 }
      }
      else
      {
	 Major = 0;
	 Minor = 9;
	 if (sscanf(Line.c_str(),"HTTP %u %[^\n]",&Result,Code) != 2)
	 {
	    clog << "http: The http server sent an invalid reply header" << endl;
	    return false;
	 }
      }
      
      return true;
   }      
      
   if (Cmp(Tag,"Content-Length:"))
   {
      if (Encoding == Closes)
	 Encoding = Stream;
      
      // The length is already set from the Content-Range header
      if (StartPos != 0)
	 return true;
      
      if (sscanf(Val.c_str(),"%lu",&Size) != 1)
      {
	 clog << "http: The http server sent an invalid Content-Length: header" << endl;
	 return false;
      }
      return true;
   }

   if (Cmp(Tag,"Content-Range:"))
   {
      if (sscanf(Val.c_str(),"bytes %lu-%*u/%lu",&StartPos,&Size) != 2)
      {
	 clog << "http: The http server sent an invalid Content-Range: heander" << endl;
	 return false;
      }
      if ((unsigned)StartPos > Size)
      {
	 cout << "E This http server has broken range support" << endl;
	 return false;
      }      
      return true;
   }
   
   if (Cmp(Tag,"Transfer-Encoding:"))
   {
      if (Cmp(Val,"chunked"))
	 Encoding = Chunked;
      return true;
   }

   if (Cmp(Tag,"Last-Modified:"))
   {
      struct tm Tm;
      char Month[MAXLEN];
      const char *I = Val.c_str();
   
      // Skip the day of the week
      for (;*I != 0  && *I != ' '; I++);
   
      // Handle RFC 1123 time
      if (sscanf(I," %d %3s %d %d:%d:%d GMT",&Tm.tm_mday,Month,&Tm.tm_year,
		 &Tm.tm_hour,&Tm.tm_min,&Tm.tm_sec) != 6)
      {
	 // Handle RFC 1036 time
	 if (sscanf(I," %d-%3s-%d %d:%d:%d GMT",&Tm.tm_mday,Month,
		    &Tm.tm_year,&Tm.tm_hour,&Tm.tm_min,&Tm.tm_sec) == 6)
	    Tm.tm_year += 1900;
	 else
	 {
	    // asctime format
	    if (sscanf(I," %3s %d %d:%d:%d %d",Month,&Tm.tm_mday,
		       &Tm.tm_hour,&Tm.tm_min,&Tm.tm_sec,&Tm.tm_year) != 6)
	    {
	       clog << "http: Unknown date format " << Val << endl;
	       return true;
	    }
	 }
      }
      
      Tm.tm_isdst = 0;
      Tm.tm_mon = MonthConv(Month);
      Tm.tm_year -= 1900;
   
      // Convert to local time and then to GMT
      Date = timegm(&Tm);
      return true;
   }

   return true;
}
									/*}}}*/
// SigTerm - Termination signal handler					/*{{{*/
// ---------------------------------------------------------------------
/* To make reget work properly the date on the file must be set to the 
   exact date in the header, this ensures this happens on a term/int
   signal */
Loop *Running = 0;
void SigTerm(int)
{
   if (Running != 0)
      Running->SetTime();
   exit(100);
}
									/*}}}*/

int main(int argc,const char *argv[])
{
   signal(SIGTERM,SigTerm);
   signal(SIGINT,SigTerm);
   signal(SIGPIPE,SIG_IGN);
   
   const char *Env = getenv("http_proxy");
   if (Env != 0)
      Proxy = Env;
   
   ProxyAuth = GrabAuth(Proxy);
									   
   if (argc <= 1)
   {
      cout << "E You need to specify at least one pair" << endl;
      return 100;
   }
   if (argc % 2 != 1)
   {
      cout << "E All URLs must be in pairs" << endl;
      return 100;
   }

   int Count = argc/2;
   Command **List = new Command *[Count];
   for (int I = 0; I < Count; I++)
      List[I] = new Command(argv[I*2+1],argv[I*2+2]);

   Command *Set = 0;
   for (Command **I = List; I < List + argc/2;)
   {
      Loop Run;
      Running = &Run;
      
      // Generate a batch queue for all of the URLs for the host
      Command **J = I;
      for (; J != List + argc/2 && (*J)->Host == (*I)->Host && J - I < 2; J++)
	 Run.AddRequest((*J)->Req);
      
      /* Any errors in this loop cause the batch to be aborted and the current
         item to be errored */
      Command **Start = I;
      if (Set != *I)
	 cout << "F " << (*I)->oURL << endl;
      Set = *I;
      if (Run.OpenRemote((*I)->Host) == false)
	 break;
      
      /* Run each item that goes to the same server at once, this handles
         pipelining */
      for (; I != J;)
      {
	 Command *Cur = *I;
	 I++;
	 
	 if (Set != Cur)
	    cout << "F " << Cur->oURL << endl;

	 Set = Cur;
	 
	 if (Run.OutputTo(Cur->Output) == false)
	    break;

	 if (Run.RunHeaders() == false)
	    break;
	 
	 if (Run.RunData() == false)
	    break;

	 if (Run.Hash().empty() == false)
	    cout << "M " << Run.Hash() << endl;

	 /* At this point we know the remote supports keep-alive, we can
	    safely burst the entire request set. We pipeline a max of 5
	    requests because alot of servers still have problems. Test this
	    with something like 100 requests */
	 for (; J != List + argc/2 && (*J)->Host == (*I)->Host && J - I < 5; J++)
	    Run.AddRequest((*J)->Req);
      }
      
      if (Run.Restart == true && I - 1 != Start)
	 I--;
      else
	 if (Run.Restart == true)
	    cout << "E Connection failed" << endl;
   }

   return 0;
}
