;==============================================================================

; file: "_std.scm"

; Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved.

(##include "header.scm")

;------------------------------------------------------------------------------

; Definition of vector-like data types (i.e. string, vector, u8vector, ...).

(##define-macro (define-vector-procedures name default-elem-value check-elem)

  (define (sym . lst)
    (string->symbol (apply string-append (map symbol->string lst))))

  (let ()
    (define check-vect     (sym 'check- name))

    (define ##vect?        (sym '## name '?))
    (define ##make-vect    (sym '##make- name))
    (define ##vect         (sym '## name))
    (define ##vect-length  (sym '## name '-length))
    (define ##vect-ref     (sym '## name '-ref))
    (define ##vect-set!    (sym '## name '-set!))
    (define ##vect->list   (sym '## name '->list))
    (define ##list->vect   (sym '##list-> name))
    (define ##vect-shrink! (sym '## name '-shrink!))

    (define vect?          (sym name '?))
    (define make-vect      (sym 'make- name))
    (define vect           (sym name))
    (define vect-length    (sym name '-length))
    (define vect-ref       (sym name '-ref))
    (define vect-set!      (sym name '-set!))
    (define vect->list     (sym name '->list))
    (define list->vect     (sym 'list-> name))

    `(begin

       (define-system (,##vect? x))
       (define (,vect? x) (force-vars (x) (,##vect? x)))

       (define-system (,##make-vect x y))

       (define (,make-vect x #!optional (y (absent-obj)))
         (force-vars (x y)
           (let ((init (if (##eq? y (absent-obj)) ,default-elem-value y)))
             (check-exact-int-non-neg x (,make-vect x y)
               (,check-elem init (,make-vect x y)
                 (,##make-vect x init))))))

       (define-system (,##vect . l)
         (,##list->vect l))

       (define (,vect . l)
         (let* ((n (##length l))
                (vect (,##make-vect n ,default-elem-value)))
           (let loop ((x l) (i 0))
             (if (##pair? x)
               (let ((elem (##car x)))
                 (force-vars (elem)
                   (,check-elem elem (,vect . l)
                     (begin
                       (,##vect-set! vect i elem)
                       (loop (##cdr x) (##fixnum.+ i 1))))))
               vect))))

       (define-system (,##vect-length x))

       (define (,vect-length x)
         (force-vars (x)
           (,check-vect x (,vect-length x)
             (,##vect-length x))))

       (define-system (,##vect-ref x y))

       (define (,vect-ref x y)
         (force-vars (x y)
           (,check-vect x (,vect-ref x y)
             (check-exact-int-range y 0 (,##vect-length x) (,vect-ref x y)
               (,##vect-ref x y)))))

       (define-system (,##vect-set! x y z))

       (define (,vect-set! x y z)
         (force-vars (x y)
           (,check-vect x (,vect-set! x y z)
             (check-exact-int-range y 0 (,##vect-length x) (,vect-set! x y z)
               (,check-elem z (,vect-set! x y z)
                 (begin
                   (,##vect-set! x y z)
                   (##void)))))))

       (define-system (,##vect->list vect)
         (let loop ((l '()) (i (##fixnum.- (,##vect-length vect) 1)))
           (if (##fixnum.< i 0)
             l
             (loop (##cons (,##vect-ref vect i) l) (##fixnum.- i 1)))))

       (define (,vect->list vect)
         (force-vars (vect)
           (,check-vect vect (,vect->list vect)
             (let loop ((l '()) (i (##fixnum.- (,##vect-length vect) 1)))
               (if (##fixnum.< i 0)
                 l
                 (loop (##cons (,##vect-ref vect i) l) (##fixnum.- i 1)))))))

       (define-system (,##list->vect l)
         (let* ((n (##length l))
                (vect (,##make-vect n ,default-elem-value)))
           (let loop ((x l) (i 0))
             (if (##pair? x)
               (begin
                 (,##vect-set! vect i (##car x))
                 (loop (##cdr x) (##fixnum.+ i 1)))
               vect))))

       (define (,list->vect l)
         (let loop1 ((x l) (n 0))
           (force-vars (x)
             (if (##pair? x)
               (loop1 (##cdr x) (##fixnum.+ n 1))
               (check-list-end x (,list->vect l)
                 (let ((vect (,##make-vect n ,default-elem-value)))
                   (let loop2 ((x l) (i 0))
                     (force-vars (x)
                       (if (##pair? x)
                         (let ((elem (##car x)))
                           (,check-elem elem (,list->vect l)
                             (begin
                               (,##vect-set! vect i elem)
                               (loop2 (##cdr x) (##fixnum.+ i 1)))))
                         vect)))))))))

       (define-system (,##vect-shrink! x y)))))

(define-vector-procedures string    #\space check-char)
(define-vector-procedures vector    #f      no-check)
(define-vector-procedures u8vector  0       check-exact-int8)
(define-vector-procedures u16vector 0       check-exact-int16)
(define-vector-procedures u32vector 0       check-exact-int32)
(define-vector-procedures f32vector 0.      check-inexact-real)
(define-vector-procedures f64vector 0.      check-inexact-real)

;------------------------------------------------------------------------------

; IEEE Scheme procedures:

(define (not x)
  (force-vars (x)
    (##not x)))

(define (boolean? x)
  (force-vars (x)
    (or (##eq? x #t) (##eq? x #f))))

(define (eqv? x y)
  (force-vars (x y)
    (let ()
      (##declare (generic)) ; do not use fixnum specific ##eqv?
      (##eqv? x y))))

(define (eq? x y)
  (force-vars (x y)
    (##eq? x y)))

(define (equal? x y)
  (##equal? x y (if-forces #t #f)))

(define (pair? x)
  (force-vars (x)
    (##pair? x)))

(define (cons x y)
  (##cons x y))

(define (car x)
  (force-vars (x)
    (check-pair x (car x)
      (##car x))))

(define (cdr x)
  (force-vars (x)
    (check-pair x (cdr x)
      (##cdr x))))

(define (set-car! x y)
  (force-vars (x)
    (check-pair x (set-car! x y)
      (begin
        (##set-car! x y)
        (##void)))))

(define (set-cdr! x y)
  (force-vars (x)
    (check-pair x (set-cdr! x y)
      (begin
        (##set-cdr! x y)
        (##void)))))

(##define-macro (define-c...r name pattern)

  (define (gen name pattern)
    `(check-pair y (,name x)
       ,(if (<= pattern 3)
          (if (= pattern 3) '(##cdr y) '(##car y))
          `(let ((y ,(if (odd? pattern) '(##cdr y) '(##car y))))
             (force-vars (y)
               ,(gen name (quotient pattern 2)))))))

  `(define (,name x)
     (force-vars (x)
       (let ((y x))
         ,(gen name pattern)))))

(define-c...r caar 4)
(define-c...r cadr 5)
(define-c...r cdar 6)
(define-c...r cddr 7)
(define-c...r caaar 8)
(define-c...r caadr 9)
(define-c...r cadar 10)
(define-c...r caddr 11)
(define-c...r cdaar 12)
(define-c...r cdadr 13)
(define-c...r cddar 14)
(define-c...r cdddr 15)
(define-c...r caaaar 16)
(define-c...r caaadr 17)
(define-c...r caadar 18)
(define-c...r caaddr 19)
(define-c...r cadaar 20)
(define-c...r cadadr 21)
(define-c...r caddar 22)
(define-c...r cadddr 23)
(define-c...r cdaaar 24)
(define-c...r cdaadr 25)
(define-c...r cdadar 26)
(define-c...r cdaddr 27)
(define-c...r cddaar 28)
(define-c...r cddadr 29)
(define-c...r cdddar 30)
(define-c...r cddddr 31)

(define (null? x)
  (force-vars (x)
    (##null? x)))

(define (list? lst)
  (let loop ((lst1 lst) (lst2 lst))
    (force-vars (lst1)
      (if (##not (##pair? lst1))
        (##null? lst1)
        (let ((lst1 (##cdr lst1)))
          (force-vars (lst1 lst2)
            (cond ((##eq? lst1 lst2)
                   #f)
                  ((##pair? lst1)
                   (loop (##cdr lst1) (##cdr lst2)))
                  (else
                   (##null? lst1)))))))))

(define (list . lst)
  lst)

(define (length lst)
  (let loop ((lst1 lst) (n 0))
    (force-vars (lst1)
      (if (##pair? lst1)
        (loop (##cdr lst1) (##fixnum.+ n 1))
        (check-list-end lst1 (length lst)
          n)))))

(define (append #!optional (x (absent-obj)) (y (absent-obj)) #!rest z)

  (define (append-multiple head tail)
    (if (##null? tail)
      head
      (force-vars (head)
        (if (##null? head)
          (append-multiple (##car tail) (##cdr tail))
          (list-expected-check (append-multiple-non-null head tail))))))

  (define (append-multiple-non-null lst1 lsts) ; lst1!=(), returns #f on error
    (let ((head (##car lsts))
          (tail (##cdr lsts)))
      (if (##null? tail)
        (append-2-non-null lst1 head)
        (force-vars (head)
          (if (##null? head)
            (append-multiple-non-null lst1 tail)
            (let ((result (append-multiple-non-null head tail)))
              (and result
                   (append-2-non-null lst1 result))))))))

  (define (append-2-non-null lst1 lst2) ; lst1!=(), returns #f on error
    (if (##pair? lst1)
      (let ((head (##car lst1))
            (tail (##cdr lst1)))
        (force-vars (tail)
          (if (##null? tail)
            (##cons head lst2)
            (let ((result (append-2-non-null tail lst2)))
              (and result
                   (##cons head result))))))
      (if-checks
        #f ; error: list expected
        lst2)))

  (define (list-expected-check result)
    (if-checks
      (if (##not result)
        (trap-list (append x y . z))
        result)
      result))

  (cond ((##eq? x (absent-obj))
         '())
        ((##eq? y (absent-obj))
         x)
        ((##null? z)
         (force-vars (x)
           (if (##not (##null? x))
             (list-expected-check (append-2-non-null x y))
             y)))
        (else
         (append-multiple x (##cons y z)))))

(define (reverse lst)
  (let loop ((lst1 lst) (x '()))
    (force-vars (lst1)
      (if (##pair? lst1)
        (loop (##cdr lst1) (##cons (##car lst1) x))
        (check-list-end lst1 (reverse lst)
          x)))))

(define (list-ref lst k)
  (force-vars (k)
    (check-exact-int-non-neg k (list-ref lst k)
      (let loop ((lst1 lst) (i k))
        (force-vars (lst1)
          (check-pair lst1 (list-ref lst k)
            (if (##fixnum.< 0 i)
              (loop (##cdr lst1) (##fixnum.- i 1))
              (##car lst1))))))))

(define (memq x lst)
  (force-vars (x)
    (let loop ((lst1 lst))
      (force-vars (lst1)
        (if (##pair? lst1)
          (let ((y (##car lst1)))
            (force-vars (y)
              (if (##eq? x y)
                lst1
                (loop (##cdr lst1)))))
          (check-list-end lst1 (memq x lst)
            #f))))))

(define (memv x lst)
  (force-vars (x)
    (let loop ((lst1 lst))
      (force-vars (lst1)
        (if (##pair? lst1)
          (let ((y (##car lst1)))
            (force-vars (y)
              (if (let () (##declare (generic)) (##eqv? x y))
                lst1
                (loop (##cdr lst1)))))
          (check-list-end lst1 (memv x lst)
            #f))))))

(define (member x lst)
  (let loop ((lst1 lst))
    (force-vars (lst1)
      (if (##pair? lst1)
        (let ((y (##car lst1)))
          (if (##equal? x y (if-forces #t #f))
            lst1
            (loop (##cdr lst1))))
        (check-list-end lst1 (member x lst)
          #f)))))

(define (assq x lst)
  (force-vars (x)
    (let loop ((lst1 lst))
      (force-vars (lst1)
        (if (##pair? lst1)
          (let ((couple (##car lst1)))
            (force-vars (couple)
              (check-pair couple (assq x lst)
                (let ((y (##car couple)))
                  (force-vars (y)
                    (if (##eq? x y)
                      couple
                      (loop (##cdr lst1))))))))
          (check-list-end lst1 (assq x lst)
            #f))))))

(define (assv x lst)
  (force-vars (x)
    (let loop ((lst1 lst))
      (force-vars (lst1)
        (if (##pair? lst1)
          (let ((couple (##car lst1)))
            (force-vars (couple)
              (check-pair couple (assv x lst)
                (let ((y (##car couple)))
                  (force-vars (y)
                    (if (let () (##declare (generic)) (##eqv? x y))
                      couple
                      (loop (##cdr lst1))))))))
          (check-list-end lst1 (assv x lst)
            #f))))))

(define (assoc x lst)
  (let loop ((lst1 lst))
    (force-vars (lst1)
      (if (##pair? lst1)
        (let ((couple (##car lst1)))
          (force-vars (couple)
            (check-pair couple (assoc x lst)
              (let ((y (##car couple)))
                (if (##equal? x y (if-forces #t #f))
                  couple
                  (loop (##cdr lst1)))))))
        (check-list-end lst1 (assoc x lst)
          #f)))))

(define (symbol? x)
  (force-vars (x)
    (##symbol? x)))

(define (symbol->string sym)
  (force-vars (sym)
    (check-symbol sym (symbol->string sym)
      (##symbol->string sym))))

(define (string->symbol str)
  (force-vars (str)
    (check-string str (string->symbol str)
      (##string->symbol str))))

; Number related procedures are in "_num1.scm" and "_num2.scm"

(##define-macro (char-up-to-down) 32)

(define-system (##char? x))
(define (char? x) (force-vars (x) (##char? x)))

(define-nary-bool (##char=? x y)
  #t
  #t
  (##char=? x y)
  no-force
  no-check)

(define-nary-bool (char=? x y)
  #t
  #t
  (##char=? x y)
  force-vars
  check-char)

(define-nary-bool (##char<? x y)
  #t
  #t
  (##char<? x y)
  no-force
  no-check)

(define-nary-bool (char<? x y)
  #t
  #t
  (##char<? x y)
  force-vars
  check-char)

(define-nary-bool (##char>? x y)
  #t
  #t
  (##char<? y x)
  no-force
  no-check)

(define-nary-bool (char>? x y)
  #t
  #t
  (##char<? y x)
  force-vars
  check-char)

(define-nary-bool (##char<=? x y)
  #t
  #t
  (##not (##char<? y x))
  no-force
  no-check)

(define-nary-bool (char<=? x y)
  #t
  #t
  (##not (##char<? y x))
  force-vars
  check-char)

(define-nary-bool (##char>=? x y)
  #t
  #t
  (##not (##char<? x y))
  no-force
  no-check)

(define-nary-bool (char>=? x y)
  #t
  #t
  (##not (##char<? x y))
  force-vars
  check-char)

(##define-macro (case-independent-char=? x y)
  `(##char=? (##char-downcase ,x) (##char-downcase ,y)))

(##define-macro (case-independent-char<? x y)
  `(##char<? (##char-downcase ,x) (##char-downcase ,y)))

(define-nary-bool (##char-ci=? x y)
  #t
  #t
  (case-independent-char=? x y)
  no-force
  no-check)

(define-nary-bool (char-ci=? x y)
  #t
  #t
  (case-independent-char=? x y)
  force-vars
  check-char)

(define-nary-bool (##char-ci<? x y)
  #t
  #t
  (case-independent-char<? x y)
  no-force
  no-check)

(define-nary-bool (char-ci<? x y)
  #t
  #t
  (case-independent-char<? x y)
  force-vars
  check-char)

(define-nary-bool (##char-ci>? x y)
  #t
  #t
  (case-independent-char<? y x)
  no-force
  no-check)

(define-nary-bool (char-ci>? x y)
  #t
  #t
  (case-independent-char<? y x)
  force-vars
  check-char)

(define-nary-bool (##char-ci<=? x y)
  #t
  #t
  (##not (case-independent-char<? y x))
  no-force
  no-check)

(define-nary-bool (char-ci<=? x y)
  #t
  #t
  (##not (case-independent-char<? y x))
  force-vars
  check-char)

(define-nary-bool (##char-ci>=? x y)
  #t
  #t
  (##not (case-independent-char<? x y))
  no-force
  no-check)

(define-nary-bool (char-ci>=? x y)
  #t
  #t
  (##not (case-independent-char<? x y))
  force-vars
  check-char)

(define-system (##char-alphabetic? c)
  (let ((x (##char-downcase c)))
    (and (##not (##char<? x #\a)) (##not (##char<? #\z x)))))

(define (char-alphabetic? c)
  (force-vars (c)
    (check-char c (char-alphabetic? c)
      (##char-alphabetic? c))))

(define-system (##char-numeric? c)
  (and (##not (##char<? c #\0)) (##not (##char<? #\9 c))))

(define (char-numeric? c)
  (force-vars (c)
    (check-char c (char-numeric? c)
      (##char-numeric? c))))

(define-system (##char-whitespace? c)
  (##char<=? c #\space))

(define (char-whitespace? c)
  (force-vars (c)
    (check-char c (char-whitespace? c)
      (##char-whitespace? c))))

(define-system (##char-upper-case? c)
  (and (##not (##char<? c #\A)) (##not (##char<? #\Z c))))

(define (char-upper-case? c)
  (force-vars (c)
    (check-char c (char-upper-case? c)
      (##char-upper-case? c))))

(define-system (##char-lower-case? c)
  (and (##not (##char<? c #\a)) (##not (##char<? #\z c))))

(define (char-lower-case? c)
  (force-vars (c)
    (check-char c (char-lower-case? c)
      (##char-lower-case? c))))

(define (char->integer c)
  (force-vars (c)
    (check-char c (char->integer c)
      (##fixnum.<-char c))))

(define (integer->char n)
  (force-vars (n)
    (check-exact-int-range-incl n 0 ##max-unicode (integer->char n)
      (##fixnum.->char n))))

(define-system (##char-upcase c)
  (if (and (##not (##char<? c #\a)) (##not (##char<? #\z c)))
    (##fixnum.->char (##fixnum.- (##fixnum.<-char c) (char-up-to-down)))
    c))

(define (char-upcase c)
  (force-vars (c)
    (check-char c (char-upcase c)
      (##char-upcase c))))

(define-system (##char-downcase c)
  (if (and (##not (##char<? c #\A)) (##not (##char<? #\Z c)))
    (##fixnum.->char (##fixnum.+ (##fixnum.<-char c) (char-up-to-down)))
    c))

(define (char-downcase c)
  (force-vars (c)
    (check-char c (char-downcase c)
      (##char-downcase c))))

(define-system (##string=? x y)
  (let ((len (##string-length x)))
    (if (##eq? len (##string-length y))
      (let loop ((i (##fixnum.- len 1)))
        (cond ((##fixnum.< i 0)
               #t)
              ((##char=? (##string-ref x i) (##string-ref y i))
               (loop (##fixnum.- i 1)))
              (else
               #f)))
      #f)))

(define-nary-bool (string=? x y)
  #t
  #t
  (##string=? x y)
  force-vars
  check-string)

(define-system (##string<? x y)
  (let ((lx (##string-length x))
        (ly (##string-length y)))
    (let ((n (if (##fixnum.< lx ly) lx ly)))
      (let loop ((i 0))
        (if (##fixnum.< i n)
          (let ((cx (##string-ref x i))
                (cy (##string-ref y i)))
            (if (##char=? cx cy)
              (loop (##fixnum.+ i 1))
              (##char<? cx cy)))
          (##fixnum.< n ly))))))

(define-nary-bool (string<? x y)
  #t
  #t
  (##string<? x y)
  force-vars
  check-string)

(define-nary-bool (string>? x y)
  #t
  #t
  (##string<? y x)
  force-vars
  check-string)

(define-nary-bool (string<=? x y)
  #t
  #t
  (##not (##string<? y x))
  force-vars
  check-string)

(define-nary-bool (string>=? x y)
  #t
  #t
  (##not (##string<? x y))
  force-vars
  check-string)

(define-system (##string-ci=? x y)
  (let ((len (##string-length x)))
    (if (##eq? len (##string-length y))
      (let loop ((i (##fixnum.- len 1)))
        (cond ((##fixnum.< i 0)
               #t)
              ((##char=? (##char-downcase (##string-ref x i))
                         (##char-downcase (##string-ref y i)))
               (loop (##fixnum.- i 1)))
              (else
               #f)))
      #f)))

(define-nary-bool (string-ci=? x y)
  #t
  #t
  (##string-ci=? x y)
  force-vars
  check-string)

(define-system (##string-ci<? x y)
  (let ((lx (##string-length x))
        (ly (##string-length y)))
    (let ((n (if (##fixnum.< lx ly) lx ly)))
      (let loop ((i 0))
        (if (##fixnum.< i n)
          (let ((cx (##char-downcase (##string-ref x i)))
                (cy (##char-downcase (##string-ref y i))))
            (if (##char=? cx cy)
              (loop (##fixnum.+ i 1))
              (##char<? cx cy)))
          (##fixnum.< n ly))))))

(define-nary-bool (string-ci<? x y)
  #t
  #t
  (##string-ci<? x y)
  force-vars
  check-string)

(define-nary-bool (string-ci>? x y)
  #t
  #t
  (##string-ci<? y x)
  force-vars
  check-string)

(define-nary-bool (string-ci<=? x y)
  #t
  #t
  (##not (##string-ci<? y x))
  force-vars
  check-string)

(define-nary-bool (string-ci>=? x y)
  #t
  #t
  (##not (##string-ci<? x y))
  force-vars
  check-string)

(define-system (##substring x y z)
  (let* ((n (##fixnum.- z y))
         (result (##make-string n #\space)))
    (let loop ((i (##fixnum.- n 1)))
      (if (##not (##fixnum.< i 0))
        (begin
          (##string-set! result i (##string-ref x (##fixnum.+ y i)))
          (loop (##fixnum.- i 1)))))
    result))

(define (substring x y z)
  (force-vars (x y z)
    (check-string x (substring x y z)
      (check-exact-int-range-incl y 0 (##string-length x) (substring x y z)
        (check-exact-int-range-incl z y (##string-length x) (substring x y z)
          (##substring x y z))))))

(define-system (##string-append . lst)
  (##append-strings lst))

(define (##append-strings lst)
  (let loop1 ((n 0) (x lst) (y '()))
    (if (##pair? x)
      (let ((s (##car x)))
        (loop1 (##fixnum.+ n (##string-length s)) (##cdr x) (##cons s y)))
      (let ((result (##make-string n #\space)))
        (let loop2 ((k (##fixnum.- n 1)) (y y))
          (if (##pair? y)
            (let ((s (##car y)))
              (let loop3 ((i k) (j (##fixnum.- (##string-length s) 1)))
                (if (##not (##fixnum.< j 0))
                  (begin
                    (##string-set! result i (##string-ref s j))
                    (loop3 (##fixnum.- i 1) (##fixnum.- j 1)))
                  (loop2 i (##cdr y)))))
            result))))))

(define (string-append . lst)
  (let loop1 ((n 0) (x lst) (y '()))
    (if (##pair? x)
      (let ((s (##car x)))
        (force-vars (s)
          (check-string s (string-append . lst)
            (loop1 (##fixnum.+ n (##string-length s))
                   (##cdr x)
                   (##cons s y)))))
      (let ((result (##make-string n #\space)))
        (let loop2 ((k (##fixnum.- n 1)) (y y))
          (if (##pair? y)
            (let ((s (##car y)))
              (let loop3 ((i k) (j (##fixnum.- (##string-length s) 1)))
                (if (##not (##fixnum.< j 0))
                  (begin
                    (##string-set! result i (##string-ref s j))
                    (loop3 (##fixnum.- i 1) (##fixnum.- j 1)))
                  (loop2 i (##cdr y)))))
            result))))))

(define (procedure? x)
  (force-vars (x)
    (##procedure? x)))

(define (apply p x . y)

  (define (build-arg-list arg1 other-args)
    (if (##pair? other-args)
      (let ((tail (build-arg-list (##car other-args) (##cdr other-args))))
        (if-checks
          (and tail
               (##cons arg1 tail))
          (##cons arg1 tail)))
      (if-forces
        (copy-proper-list arg1)
        (check-proper-list arg1))))

  (define (copy-proper-list lst)
    (force-vars (lst)
      (if (##pair? lst)
        (let ((tail (copy-proper-list (##cdr lst))))
          (if-checks
            (and tail
                 (##cons (##car lst) tail))
            (##cons (##car lst) tail)))
        (if-checks
          (if (##null? lst)
            '()
            #f) ; error: list expected
          '()))))

  (define (check-proper-list lst)
    (if-checks
      (let loop ((lst2 lst))
        (cond ((##pair? lst2)
               (loop (##cdr lst2)))
              ((##null? lst2)
               lst)
              (else
               #f))) ; error: list expected
      lst))

  (force-vars (p)
    (check-procedure p (apply p x . y)
      (let ((lst (build-arg-list x y)))
        (if lst
          (##apply p lst)
          (trap-list (apply p x . y)))))))

(define (map p x . y)
  (force-vars (p)
    (check-procedure p (map p x . y)
      (let ()

        (define (proper-list-length lst)
          (let loop ((lst lst) (n 0))
            (force-vars (lst)
              (cond ((##pair? lst)
                     (loop (##cdr lst) (##fixnum.+ n 1)))
                    ((##null? lst)
                     n)
                    (else
                     #f)))))

        (define (map-1 lst1)
          (force-vars (lst1)
            (if (##pair? lst1)
              (let ((result (p (##car lst1))))
                (##cons result (map-1 (##cdr lst1))))
              '())))

        (define (cars lsts)
          (if (##pair? lsts)
            (let ((lst1 (##car lsts)))
              (force-vars (lst1)
                (let ((head (##car lst1)))
                  (let ((tail (cars (##cdr lsts))))
                    (##cons head tail)))))
            '()))

        (define (cdrs lsts)
          (if (##pair? lsts)
            (let ((lst1 (##car lsts)))
              (force-vars (lst1)
                (let ((head (##cdr lst1)))
                  (if (##pair? head)
                    (let ((tail (cdrs (##cdr lsts))))
                      (and tail
                           (##cons head tail)))
                    #f))))
            '()))

        (define (map-n lsts)
          (if lsts
            (let ((result (##apply p (cars lsts))))
              (##cons result (map-n (cdrs lsts))))
            '()))

        (cond ((##null? y)
               (if-checks
                 (let ((len1 (proper-list-length x)))
                   (if len1
                     (map-1 x)
                     (trap-list (map p x . y))))
                 (map-1 x)))
              (else
               (if-checks
                 (let ((len1 (proper-list-length x)))
                   (if len1
                     (let loop ((lsts y))
                       (if (##null? lsts)
                         (if (##null? x)
                           '()
                           (map-n (##cons x y)))
                         (let ((len2 (proper-list-length (##car lsts))))
                           (if (##eq? len1 len2)
                             (loop (##cdr lsts))
                             (if len2
                               (trap-list-lengths (map p x . y))
                               (trap-list (map p x . y)))))))
                     (trap-list (map p x . y))))
                 (if (##null? x)
                   '()
                   (map-n (##cons x y))))))))))

(define (for-each p x . y)
  (force-vars (p)
    (check-procedure p (for-each p x . y)
      (let ()

        (define (proper-list-length lst)
          (let loop ((lst lst) (n 0))
            (force-vars (lst)
              (cond ((##pair? lst)
                     (loop (##cdr lst) (##fixnum.+ n 1)))
                    ((##null? lst)
                     n)
                    (else
                     #f)))))

        (define (for-each-1 lst1)
          (force-vars (lst1)
            (if (##pair? lst1)
              (let ((result (p (##car lst1))))
                (for-each-1 (##cdr lst1)))
              (##void))))

        (define (cars lsts)
          (if (##pair? lsts)
            (let ((lst1 (##car lsts)))
              (force-vars (lst1)
                (let ((head (##car lst1)))
                  (let ((tail (cars (##cdr lsts))))
                    (##cons head tail)))))
            '()))

        (define (cdrs lsts)
          (if (##pair? lsts)
            (let ((lst1 (##car lsts)))
              (force-vars (lst1)
                (let ((head (##cdr lst1)))
                  (if (##pair? head)
                    (let ((tail (cdrs (##cdr lsts))))
                      (and tail
                           (##cons head tail)))
                    #f))))
            '()))

        (define (for-each-n lsts)
          (if lsts
            (let ((result (##apply p (cars lsts))))
              (for-each-n (cdrs lsts)))
            (##void)))

        (cond ((##null? y)
               (if-checks
                 (let ((len1 (proper-list-length x)))
                   (if len1
                     (for-each-1 x)
                     (trap-list (for-each p x . y))))
                 (for-each-1 x)))
              (else
               (if-checks
                 (let ((len1 (proper-list-length x)))
                   (if len1
                     (let loop ((lsts y))
                       (if (##null? lsts)
                         (if (##null? x)
                           (##void)
                           (for-each-n (##cons x y)))
                         (let ((len2 (proper-list-length (##car lsts))))
                           (if (##eq? len1 len2)
                             (loop (##cdr lsts))
                             (if len2
                               (trap-list-lengths (for-each p x . y))
                               (trap-list (for-each p x . y)))))))
                     (trap-list (for-each p x . y))))
                 (if (##null? x)
                   (##void)
                   (for-each-n (##cons x y))))))))))

(define (call-with-current-continuation p)
  (force-vars (p)
    (check-procedure p (call-with-current-continuation p)
      (##call-with-current-continuation p))))

; Port related procedures are in "_io.scm"

;------------------------------------------------------------------------------

; R4RS Scheme procedures:

(define (list-tail l k)
  (force-vars (k)
    (check-exact-int-non-neg k (list-tail l k)
      (let loop ((x l) (i k))
        (if (##fixnum.< 0 i)
          (force-vars (x)
            (check-pair x (list-tail l k)
              (loop (##cdr x) (##fixnum.- i 1))))
          x)))))

(define-system (##string-copy str)
  (let* ((n (##string-length str))
         (result (##make-string n #\space)))
    (let loop ((i (##fixnum.- n 1)))
      (if (##fixnum.< i 0)
        result
        (begin
          (##string-set! result i (##string-ref str i))
          (loop (##fixnum.- i 1)))))))

(define (string-copy str)
  (force-vars (str)
    (check-string str (string-copy str)
      (let* ((n (##string-length str))
             (result (##make-string n #\space)))
        (let loop ((i (##fixnum.- n 1)))
          (if (##fixnum.< i 0)
            result
            (begin
              (##string-set! result i (##string-ref str i))
              (loop (##fixnum.- i 1)))))))))

(define (string-fill! str c)
  (force-vars (str c)
    (check-string str (string-fill! str c)
      (check-char c (string-fill! str c)
        (let ((n (##string-length str)))
          (let loop ((i (##fixnum.- n 1)))
            (if (##fixnum.< i 0)
              (##void)
              (begin
                (##string-set! str i c)
                (loop (##fixnum.- i 1))))))))))

(define (vector-fill! vect x)
  (force-vars (vect x)
    (check-vector vect (vector-fill! vect x)
      (let ((n (##vector-length vect)))
        (let loop ((i (##fixnum.- n 1)))
          (if (##fixnum.< i 0)
            (##void)
            (begin
              (##vector-set! vect i x)
              (loop (##fixnum.- i 1)))))))))

(define-system (##vector-copy vect)
  (let* ((n (##vector-length vect))
         (result (##make-vector n #f)))
    (let loop ((i (##fixnum.- n 1)))
      (if (##fixnum.< i 0)
        result
        (begin
          (##vector-set! result i (##vector-ref vect i))
          (loop (##fixnum.- i 1)))))))

(define (force x)
  (##force x))

; Port related procedures are in "_io.scm"

;------------------------------------------------------------------------------

; R5RS Scheme procedures:

;(define (values . lst)
;  lst)
;
;(define (call-with-values producer consumer)
;  (apply consumer (producer)))
;
;(dynamic-wind before thunk after)
;(eval expr env)
;(scheme-report-environment version)
;(null-environment version)
;(interaction-environment)

;------------------------------------------------------------------------------

; Multilisp procedures:

(define (touch x)
  (##force x))

;------------------------------------------------------------------------------

; DSSSL procedures:

(define (keyword? x)
  (##keyword? x))

(define (keyword->string key)
  (force-vars (key)
    (check-keyword key (keyword->string key)
      (##keyword->string key))))

(define (string->keyword str)
  (force-vars (str)
    (check-string str (string->keyword str)
      (##string->keyword str))))

;------------------------------------------------------------------------------
