/* file: "c_intf.c" */

/* Copyright (C) 1994-1998 by Marc Feeley, All Rights Reserved. */

/* 
 * This module implements the conversion functions for the C
 * interface.
 */

#define ___INCLUDED_FROM_C_INTF
#define ___VERSION 21
#include "gambit.h"

#include "os.h"
#include "setup.h"
#include "mem.h"
#include "c_intf.h"

#include <limits.h>

___LOCAL ___WORD ___temp; /* needed by some macros in "gambit.h" */

/*---------------------------------------------------------------------------*/

/* Utilities for UTF-8 encoding of Unicode characters. */

/* 
 * '___utf8_bytes (unicode)' returns the number of bytes that are
 * needed to encode the Unicode character 'unicode' using the UTF-8
 * variable-length encoding.  If the character is not a legal Unicode
 * character, 0 is returned.
 */

int ___utf8_bytes ___P((___UCS4 unicode),(unicode)
___UCS4 unicode;)
{
  if (unicode <= 0x7f)       return 1;
  if (unicode <= 0x7ff)      return 2;
  if (unicode <= 0xd7ff)     return 3;
  if (unicode <= 0xdfff)     return 0; /* not a legal character */
  if (unicode <= 0xfffd)     return 3;
  if (unicode <= 0xffff)     return 0; /* not a legal character */
  if (unicode <= 0x1fffff)   return 4;
  if (unicode <= 0x3ffffff)  return 5;
  if (unicode <= 0x7fffffff) return 6;
  return 0;
}


/* 
 * '___utf8_put (ptr, unicode)' converts the Unicode character
 * 'unicode' into its UTF-8 variable-length encoding.  'ptr' is a
 * pointer to a byte pointer designating the start of the UTF-8
 * encoding.  On return the byte pointer points to the first byte
 * following the UTF-8 encoding.  If the character is not a legal
 * Unicode character, the pointer is not updated.
 */

void ___utf8_put ___P((___UTF8STRING *ptr, ___UCS4 unicode),(ptr, unicode)
___UTF8STRING *ptr;
___UCS4 unicode;)
{
  ___UTF8STRING p = *ptr;
  if (unicode <= 0x7f)
    {
      *p++ = unicode;
      *ptr = p;
    }
  else
    {
      int bytes;
      if      (unicode <= 0x7ff)      bytes = 2;
      else if (unicode <= 0xd7ff)     bytes = 3;
      else if (unicode <= 0xdfff)     return; /* not a legal character */
      else if (unicode <= 0xfffd)     bytes = 3;
      else if (unicode <= 0xffff)     return; /* not a legal character */
      else if (unicode <= 0x1fffff)   bytes = 4;
      else if (unicode <= 0x3ffffff)  bytes = 5;
      else if (unicode <= 0x7fffffff) bytes = 6;
      else                            return;
      p += bytes;
      *ptr = p;
      switch (bytes)
        {
          case 6:  *--p = 0x80+(unicode&0x3f); unicode >>= 6;
          case 5:  *--p = 0x80+(unicode&0x3f); unicode >>= 6;
          case 4:  *--p = 0x80+(unicode&0x3f); unicode >>= 6;
          case 3:  *--p = 0x80+(unicode&0x3f); unicode >>= 6;
          default: *--p = 0x80+(unicode&0x3f); unicode >>= 6;
        }
      *--p = 0xff - (0xff>>bytes) + unicode;
    }
}


/* 
 * '___utf8_get (ptr)' converts a UTF-8 variable-length encoding to
 * the Unicode character it encodes.  'ptr' is a pointer to a byte
 * pointer designating the start of the UTF-8 encoding.  If the
 * encoding is legal, the byte pointer will point to the first byte
 * following the UTF-8 encoding and the Unicode character is returned.
 * If the encoding is illegal, the byte pointer is not updated and 0
 * is returned.
 */

___UCS4 ___utf8_get ___P((___UTF8STRING *ptr),(ptr)
___UTF8STRING *ptr;)
{
  ___UTF8STRING p = *ptr;
  unsigned char byte = *p++;
  ___UCS4 unicode;
  int bits;
  if (byte <= 0x7f)
    {
      *ptr = p;
      return byte;
    }
  if (byte <= 0xbf || byte > 0xfd)
    return 0; /* illegal first byte */
  unicode = byte; /* upper bits are removed later */
  bits = 6;
  while (byte & 0x40)
    {
      unsigned char next = *p++;
      if (next <= 0x7f || next > 0xbf)
        return 0; /* faulty byte found after the first byte */
      unicode = (unicode << 6) + (next & 0x3f);
      byte <<= 1;
      bits += 5;
    }
  unicode &= ((unsigned long)1<<bits)-1;
  if ((unicode > 0xd7ff && unicode <= 0xdfff) ||
      (unicode > 0xfffd && unicode <= 0xffff))
    return 0; /* it is not a legal Unicode character */
  if ((unicode & (~(unsigned long)0<<(bits-5))) == 0)
    return 0; /* character was not encoded with the shortest sequence */
  *ptr = p;
  return unicode;
}


/*---------------------------------------------------------------------------*/

/* Utilities for 64 bit arithmetic. */


#ifdef ___U64

/* If the symbol ___U64 is defined it is a builtin integer type */


___U32 ___U64_shift_right ___P((___U64 *dest, int count),(dest, count)
___U64 *dest;
int count;)
{
  ___U32 result = (*dest) & ~(~(___U64)0<<count);
  *dest >>= count;
  return result;
}


#else

/* If the symbol ___U64 is not defined then ___U64 is a structure */


___U64 ___U64_init ___P((___U32 hi32, ___U32 lo32),(hi32, lo32)
___U32 hi32;
___U32 lo32;)
{
  ___U64 r;
  r.lo32 = lo32;
  r.hi32 = hi32;
  return r;
}


___U32 ___U64_to_U32 ___P((___U64 n),(n)
___U64 n;)
{
  return n.lo32;
}


int ___U64_less_than_U64 ___P((___U64 n, ___U64 m),(n, m)
___U64 n;
___U64 m;)
{
  ___U32 x = n.hi32;
  ___U32 y = m.hi32;
  return (x < y) || ((x == y) && (n.lo32 < m.lo32));
}


int ___U64_less_than_U32 ___P((___U64 n, ___U32 m),(n, m)
___U64 n;
___U32 m;)
{
  return (n.hi32 == 0) && (n.lo32 < m);
}


___U64 ___U64_add_U64 ___P((___U64 n, ___U64 m),(n, m)
___U64 n;
___U64 m;)
{
  ___U32 x = n.lo32;
  ___U32 y = n.hi32;
  ___U32 a = x + m.lo32;
  ___U32 b = y + m.hi32;
  if (a < x)
    b++;
  n.lo32 = a;
  n.hi32 = b;
  return n;
}


___U64 ___U64_add_U32 ___P((___U64 n, ___U32 m),(n, m)
___U64 n;
___U32 m;)
{
  ___U32 x = n.lo32;
  ___U32 a = x + m;
  if (a < x)
    n.hi32++;
  n.lo32 = a;
  return n;
}


___U64 ___U64_sub_U64 ___P((___U64 n, ___U64 m),(n, m)
___U64 n;
___U64 m;)
{
  ___U32 x = n.lo32;
  ___U32 y = n.hi32;
  ___U32 a = x - m.lo32;
  ___U32 b = y - m.hi32;
  if (a > x)
    b--;
  n.lo32 = a;
  n.hi32 = b;
  return n;
}


___U64 ___U64_sub_U32 ___P((___U64 n, ___U32 m),(n, m)
___U64 n;
___U32 m;)
{
  ___U32 x = n.lo32;
  ___U32 a = x - m;
  if (a > x)
    n.hi32--;
  n.lo32 = a;
  return n;
}


___U64 ___U64_mul_U32 ___P((___U64 n, ___U32 m),(n, m)
___U64 n;
___U32 m;)
{
  ___U16 xlo16 = n.lo32;
  ___U16 xhi16 = n.lo32>>16;
  ___U16 ylo16 = n.hi32;
  ___U16 yhi16 = n.hi32>>16;
  ___U16 m16 = m;
  ___U32 a = m16 * xlo16;
  ___U32 b = m16 * xhi16 + (a>>16);
  ___U32 c = m16 * ylo16 + (b>>16);
  ___U32 d = m16 * yhi16 + (c>>16);
  m16 = m>>16;
  if (m16 > 0)
    {
      ___U32 e = m16 * xlo16;
      ___U32 f = m16 * xhi16 + (e>>16);
      ___U32 g = m16 * ylo16 + (f>>16);
      b = (b & ~(~0<<16)) + (e & ~(~0<<16));
      c = (c & ~(~0<<16)) + (f & ~(~0<<16)) + (b>>16);
      d = (d & ~(~0<<16)) + (g & ~(~0<<16)) + (c>>16);
    }
  n.lo32 = (a & ~(~0<<16)) + (b<<16);
  n.hi32 = (c & ~(~0<<16)) + (d<<16);
  return n;
}


___U64 ___U64_mul_U64 ___P((___U64 n, ___U64 m),(n, m)
___U64 n;
___U64 m;)
{
  ___U64 r;
  r = ___U64_mul_U32 (n, m.lo32);
  r.hi32 += n.lo32 * m.hi32;
  return r;
}


___U32 ___U64_shift_right ___P((___U64 *dest, int count),(dest, count)
___U64 *dest;
int count;)
{
  /* we assume count < 32 */
  ___U32 x = dest->lo32;
  ___U32 y = dest->hi32;
  dest->lo32 = (x>>count) + (y<<(32-count));
  dest->hi32 = y>>count;
  return x & ~(~(unsigned long)0<<count);
}


#endif


void ___U64_copy_to_scmobj
   ___P((___U64 x, ___WORD *obj),(x, obj)
___U64 x;
___WORD *obj;)
{
  if (___U64_less_than_U32 (x, ___MAX_FIX+1))
    {
      ___U32 n = ___U64_to_U32 (x);
      *obj = ___FIX(n); /* replace bignum by a fixnum */
    }
  else
    {
      ___WORD r = *obj; /* bignum object is preallocated by caller */
      ___U64 temp;
      long i, n = 1;
      temp = x;
      while (!___U64_less_than_U32 (temp, 1)) /* temp > 0 */
        {
          ___U64_shift_right (&temp, ___RADIX_WIDTH);
          n++;
        }
      ___U16VECTORSHRINK(r,___FIX(n))
      ___U16VECTORSET(r,___FIX(0),___FIX(1)) /* set sign to positive*/
      for (i=1; i<n; i++)
        {
          ___U32 d = ___U64_shift_right (&x, ___RADIX_WIDTH);
          ___U16VECTORSET(r,___FIX(i),___FIX(d))
        }
    }
}


/*---------------------------------------------------------------------------*/

/* Scheme to C conversion */

/* 
 * The following Scheme to C conversion functions may allocate memory
 * in the C heap:
 *
 *    ___scmobj_to_function
 *    ___scmobj_to_charstring
 *    ___scmobj_to_latin1string
 *    ___scmobj_to_ucs4string
 *    ___scmobj_to_ucs2string
 *    ___scmobj_to_utf8string
 *
 * The memory allocated must be freed by calling the corresponding
 * freeing function:
 *
 *    ___free_function
 *    ___free_string
 */


/* Convert a Scheme character to a C 'char'. */

___EXP_FUNC(int,___scmobj_to_char)
   ___P((___WORD obj, char *x, int arg_num),(obj, x, arg_num)
___WORD obj;
char *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=unicode_to_uchar(___INT(obj)))>UCHAR_MAX)
    return ___STOC_CHAR_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Convert a Scheme character to a C 'signed char'. */

___EXP_FUNC(int,___scmobj_to_schar)
   ___P((___WORD obj, ___SCHAR *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___SCHAR *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=unicode_to_uchar(___INT(obj)))>UCHAR_MAX)
    return ___STOC_SCHAR_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Convert a Scheme character to a C 'unsigned char'. */

___EXP_FUNC(int,___scmobj_to_uchar)
   ___P((___WORD obj, unsigned char *x, int arg_num),(obj, x, arg_num)
___WORD obj;
unsigned char *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=unicode_to_uchar(___INT(obj)))>UCHAR_MAX)
    return ___STOC_UCHAR_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Convert a Scheme character to a C LATIN-1 encoded Unicode character. */

___EXP_FUNC(int,___scmobj_to_latin1)
   ___P((___WORD obj, ___LATIN1 *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___LATIN1 *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=___INT(obj))>0xff) /* LATIN-1 is 8 bits */
    return ___STOC_LATIN1_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Convert a Scheme character to a C UCS-4 encoded Unicode character. */

___EXP_FUNC(int,___scmobj_to_ucs4)
   ___P((___WORD obj, ___UCS4 *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___UCS4 *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=___INT(obj))>0x7fffffff) /* UCS-4 is 31 bits */
    return ___STOC_UCS4_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Convert a Scheme character to a C UCS-2 encoded Unicode character. */

___EXP_FUNC(int,___scmobj_to_ucs2)
   ___P((___WORD obj, ___UCS2 *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___UCS2 *x;
int arg_num;)
{
  ___UCS4 c;
  if (!___CHARP(obj) || (c=___INT(obj))>0xffff) /* UCS-2 is 16 bits */
    return ___STOC_UCS2_ERR+arg_num;
  *x = c;
  return ___NO_ERR;
}


/* Utility to convert a Scheme bignum to a C 'unsigned long'. */

___HIDDEN int abs_bignum_to_ulong
   ___P((___WORD obj, unsigned long *x, int arg_num),(obj, x, arg_num)
___WORD obj;
unsigned long *x;
int arg_num;)
{
  unsigned long r = 0;
  long n = ___INT(___U16VECTORLENGTH(obj))-1;
  while (n > 0)
    {
      if (r > (ULONG_MAX >> ___RADIX_WIDTH))
        return ___STOC_ULONG_ERR+arg_num; /* doesn't fit in an unsigned long */
      r = (r << ___RADIX_WIDTH) + ___INT(___U16VECTORREF(obj,___FIX(n)));
      n--;
    }
  *x = r;
  return ___NO_ERR;
}


/* Convert a Scheme integer to a C 'long'. */

___EXP_FUNC(int,___scmobj_to_long)
   ___P((___WORD obj, long *x, int arg_num),(obj, x, arg_num)
___WORD obj;
long *x;
int arg_num;)
{
  unsigned long abs_val;
  if (___FIXNUMP(obj))
    {
      *x = ___INT(obj);
      return ___NO_ERR;
    }
  if (!___BIGNUMP(obj) ||
      abs_bignum_to_ulong (obj, &abs_val, arg_num) != ___NO_ERR)
    return ___STOC_LONG_ERR+arg_num;
  if (___U16VECTORREF(obj,___FIX(0)) == 0) /* negative bignum? */
    {
      if (abs_val > -(unsigned long)LONG_MIN) /* cast avoids overflows */
        return ___STOC_LONG_ERR+arg_num;
      *x = -abs_val;
    }
  else
    {
      if (abs_val > (unsigned long)LONG_MAX)
        return ___STOC_LONG_ERR+arg_num;
      *x = abs_val;
    }
  return ___NO_ERR;
}


/* Convert a Scheme integer to a C 'short'. */

___EXP_FUNC(int,___scmobj_to_short)
   ___P((___WORD obj, short *x, int arg_num),(obj, x, arg_num)
___WORD obj;
short *x;
int arg_num;)
{
  long val;
  if (___scmobj_to_long (obj, &val, arg_num) != ___NO_ERR ||
      val < (long)SHRT_MIN ||
      val > (long)SHRT_MAX)
    return ___STOC_SHORT_ERR+arg_num;
  *x = val;
  return ___NO_ERR;
}


/* Convert a Scheme integer to a C 'unsigned long'. */

___EXP_FUNC(int,___scmobj_to_ulong)
   ___P((___WORD obj, unsigned long *x, int arg_num),(obj, x, arg_num)
___WORD obj;
unsigned long *x;
int arg_num;)
{
  if (___FIXNUMP(obj))
    {
      long val = ___INT(obj);
      if (val < 0)
        return ___STOC_ULONG_ERR+arg_num;
      *x = val;
      return ___NO_ERR;
    }
  if (!___BIGNUMP(obj) ||
      ___U16VECTORREF(obj,___FIX(0)) == 0) /* negative bignum? */
    return ___STOC_ULONG_ERR+arg_num;
  return abs_bignum_to_ulong (obj, x, arg_num);
}


/* Convert a Scheme integer to a C 'unsigned short'. */

___EXP_FUNC(int,___scmobj_to_ushort)
   ___P((___WORD obj, unsigned short *x, int arg_num),(obj, x, arg_num)
___WORD obj;
unsigned short *x;
int arg_num;)
{
  unsigned long val;
  if (___scmobj_to_ulong (obj, &val, arg_num) != ___NO_ERR ||
      val > (unsigned long)USHRT_MAX)
    return ___STOC_USHORT_ERR+arg_num;
  *x = val;
  return ___NO_ERR;
}


/* Convert a Scheme integer to a C 'int'. */

___EXP_FUNC(int,___scmobj_to_int)
   ___P((___WORD obj, int *x, int arg_num),(obj, x, arg_num)
___WORD obj;
int *x;
int arg_num;)
{
  long val;
  if (___scmobj_to_long (obj, &val, arg_num) != ___NO_ERR ||
      val < (long)INT_MIN ||
      val > (long)INT_MAX)
    return ___STOC_INT_ERR+arg_num;
  *x = val;
  return ___NO_ERR;
}


/* Convert a Scheme integer to a C 'unsigned int'. */

___EXP_FUNC(int,___scmobj_to_uint)
   ___P((___WORD obj, unsigned int *x, int arg_num),(obj, x, arg_num)
___WORD obj;
unsigned int *x;
int arg_num;)
{
  unsigned long val;
  if (___scmobj_to_ulong (obj, &val, arg_num) != ___NO_ERR ||
      val > (unsigned long)UINT_MAX)
    return ___STOC_UINT_ERR+arg_num;
  *x = val;
  return ___NO_ERR;
}


/* Convert a Scheme flonum to a C 'float'. */

___EXP_FUNC(int,___scmobj_to_float)
   ___P((___WORD obj, float *x, int arg_num),(obj, x, arg_num)
___WORD obj;
float *x;
int arg_num;)
{
  if (!___FLONUMP(obj))
    return ___STOC_FLOAT_ERR+arg_num;
  *x = ___FLONUM_VAL(obj);
  return ___NO_ERR;
}


/* Convert a Scheme flonum to a C 'double'. */

___EXP_FUNC(int,___scmobj_to_double)
   ___P((___WORD obj, double *x, int arg_num),(obj, x, arg_num)
___WORD obj;
double *x;
int arg_num;)
{
  if (!___FLONUMP(obj))
    return ___STOC_DOUBLE_ERR+arg_num;
  *x = ___FLONUM_VAL(obj);
  return ___NO_ERR;
}


/* Convert a Scheme C-pointer object to a C 'void *'. */

___EXP_FUNC(int,___scmobj_to_pointer)
   ___P((___WORD obj, void **x, int arg_num),(obj, x, arg_num)
___WORD obj;
void **x;
int arg_num;)
{
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___TESTSUBTYPE(obj,___sPOINTER))
    return ___STOC_POINTER_ERR+arg_num;
  *x = *(void**)___BODY_AS(obj,___tSUBTYPED);
  return ___NO_ERR;
}


/* Convert a Scheme function to a C function. */

___EXP_FUNC(int,___scmobj_to_function)
   ___P((___WORD obj, void **x, int arg_num),(obj, x, arg_num)
___WORD obj;
void **x;
int arg_num;)
{
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (___TYP(obj) != ___tSUBTYPED ||
      ((___label_struct*)(obj-___tSUBTYPED))->entry != obj ||
      ((___WORD *)(obj-___tSUBTYPED))[-___LS] !=
      ___MAKE_HD((___INTRO_SIZE<<___LWS),___sVECTOR,___PERM))
    /* 
     * At present, arbitrary Scheme functions cannot be converted to C
     * functions.
     */
    return ___STOC_FUNCTION_ERR+arg_num;
  *x = (void*)(((___label_struct*)(obj-___tSUBTYPED))-1)->host;
  return ___NO_ERR;
}


/* Convert a Scheme extended boolean to a C boolean. */

___EXP_FUNC(int,___scmobj_to_bool)
   ___P((___WORD obj, ___BOOL *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___BOOL *x;
int arg_num;)
{
  *x = !___FALSEP(obj); /* #f is false, everything else counts as true */
  return ___NO_ERR;
}


/* Convert a Scheme string to a C 'char *'. */

___EXP_FUNC(int,___scmobj_to_charstring)
   ___P((___WORD obj, char **x, int arg_num),(obj, x, arg_num)
___WORD obj;
char **x;
int arg_num;)
{
  unsigned long i, n;
  char *r;
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___STRINGP(obj))
    return ___STOC_CHARSTRING_ERR+arg_num;
  n = ___INT(___STRINGLENGTH(obj));
  r = (char*)___alloc_mem (n+1);
  if (r == 0)
    return ___STOC_HEAP_OVERFLOW_ERR+arg_num;
  for (i=0; i<n; i++)
    {
      ___UCS4 c = unicode_to_uchar(___INT(___STRINGREF(obj,___FIX(i))));
      if (c > UCHAR_MAX)
        {
          ___free_mem (r);
          return ___STOC_CHARSTRING_ERR+arg_num;
        }
      r[i] = c;
    }
  r[n] = 0;
  *x = r;
  return ___NO_ERR;
}


/* Convert a Scheme string to a C LATIN-1 encoded Unicode character string. */

___EXP_FUNC(int,___scmobj_to_latin1string)
   ___P((___WORD obj, ___LATIN1STRING *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___LATIN1STRING *x;
int arg_num;)
{
  unsigned long i, n;
  ___LATIN1STRING r;
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___STRINGP(obj))
    return ___STOC_CHARSTRING_ERR+arg_num;
  n = ___INT(___STRINGLENGTH(obj));
  r = (___LATIN1STRING)___alloc_mem (n+1);
  if (r == 0)
    return ___STOC_HEAP_OVERFLOW_ERR+arg_num;
  for (i=0; i<n; i++)
    {
      ___UCS4 c = ___INT(___STRINGREF(obj,___FIX(i)));
      if (c > 0xff) /* LATIN-1 is 8 bits */
        {
          ___free_mem (r);
          return ___STOC_LATIN1STRING_ERR+arg_num;
        }
      r[i] = c;
    }
  r[n] = 0;
  *x = r;
  return ___NO_ERR;
}


/* Convert a Scheme string to a C UCS-4 encoded Unicode character string. */

___EXP_FUNC(int,___scmobj_to_ucs4string)
   ___P((___WORD obj, ___UCS4STRING *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___UCS4STRING *x;
int arg_num;)
{
  unsigned long i, n;
  ___UCS4STRING r;
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___STRINGP(obj))
    return ___STOC_UCS4STRING_ERR+arg_num;
  n = ___INT(___STRINGLENGTH(obj));
  r = (___UCS4STRING)___alloc_mem ((n+1)*sizeof(___UCS4));
  if (r == 0)
    return ___STOC_HEAP_OVERFLOW_ERR+arg_num;
  for (i=0; i<n; i++)
    {
      ___UCS4 c = ___INT(___STRINGREF(obj,___FIX(i)));
      if (c > 0x7fffffff) /* UCS-4 is 31 bits */
        {
          ___free_mem (r);
          return ___STOC_UCS4STRING_ERR+arg_num;
        }
      r[i] = c;
    }
  r[n] = 0;
  *x = r;
  return ___NO_ERR;
}


/* Convert a Scheme string to a C UCS-2 encoded Unicode character string. */

___EXP_FUNC(int,___scmobj_to_ucs2string)
   ___P((___WORD obj, ___UCS2STRING *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___UCS2STRING *x;
int arg_num;)
{
  unsigned long i, n;
  ___UCS2STRING r;
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___STRINGP(obj))
    return ___STOC_UCS2STRING_ERR+arg_num;
  n = ___INT(___STRINGLENGTH(obj));
  r = (___UCS2STRING)___alloc_mem ((n+1)*sizeof(___UCS2));
  if (r == 0)
    return ___STOC_HEAP_OVERFLOW_ERR+arg_num;
  for (i=0; i<n; i++)
    {
      ___UCS4 c = ___INT(___STRINGREF(obj,___FIX(i)));
      if (c > 0xffff) /* UCS-2 is 16 bits */
        {
          ___free_mem (r);
          return ___STOC_UCS2STRING_ERR+arg_num;
        }
      r[i] = c;
    }
  r[n] = 0;
  *x = r;
  return ___NO_ERR;
}


/* Convert a Scheme string to a C UTF-8 encoded Unicode character string. */

___EXP_FUNC(int,___scmobj_to_utf8string)
   ___P((___WORD obj, ___UTF8STRING *x, int arg_num),(obj, x, arg_num)
___WORD obj;
___UTF8STRING *x;
int arg_num;)
{
  unsigned long i, bytes, n;
  ___UTF8STRING r;
  ___UTF8STRING p;
  if (___FALSEP(obj)) /* #f counts as NULL */
    {
      *x = 0;
      return ___NO_ERR;
    }
  if (!___STRINGP(obj))
    return ___STOC_UTF8STRING_ERR+arg_num;
  bytes = 0;
  n = ___INT(___STRINGLENGTH(obj));
  for (i=0; i<n; i++)
    {
      ___UCS4 c = ___INT(___STRINGREF(obj,___FIX(i)));
      if (c > 0x7fffffff) /* UCS-4 is 31 bits */
        return ___STOC_UTF8STRING_ERR+arg_num;
      bytes += ___utf8_bytes (c);
    }
  r = (___UTF8STRING)___alloc_mem (bytes+1);
  if (r == 0)
    return ___STOC_HEAP_OVERFLOW_ERR+arg_num;
  p = r;
  for (i=0; i<n; i++)
    ___utf8_put (&p, ___INT(___STRINGREF(obj,___FIX(i))));
  *p = 0;
  *x = r;
  return ___NO_ERR;
}


/* Free storage allocated to a C function created by the C-interface. */

___EXP_FUNC(void,___free_function) ___P((void *x),(x)
void *x;)
{
  /* 
   * At present nothing needs to be done because all C functions
   * created by the C-interface are permanently allocated.
   */
}


/* Free storage of a C string created by the C-interface. */

___EXP_FUNC(void,___free_string) ___P((void *x),(x)
void *x;)
{
  if (x != 0)
    ___free_mem (x);
}


/*---------------------------------------------------------------------------*/

/* C to Scheme conversion */

/* 
 * The C to Scheme conversion functions may allocate memory in the
 * Scheme heap.  However, all objects allocated are still objects with
 * a reference count of 1.  The only special processing that must be
 * performed by the caller of a C to Scheme conversion function is a
 * call to '___release_scmobj' when the caller no longer needs a
 * pointer to the object.  This call is generated automatically by the
 * C-interface.
 */


/* Free storage allocated to a Scheme object created by the C interface. */

___EXP_FUNC(void,___release_scmobj) ___P((___WORD obj),(obj)
___WORD obj;)
{
  if (___MEM_ALLOCATED(obj) &&
      ___TYP(___BODY(obj)[-1]) == ___STILL)
    ___still_obj_refcount_dec (obj);
}


/* Convert a C 'char' to a Scheme character. */

___EXP_FUNC(int,___char_to_scmobj)
   ___P((char x, ___WORD *obj, int arg_num),(x, obj, arg_num)
char x;
___WORD *obj;
int arg_num;)
{
  /*
   * No error possible because a 'char' always fits in
   * a Scheme character.
   */
  *obj = ___CHR(uchar_to_unicode((unsigned char)x));
  return ___NO_ERR;
}


/* Convert a C 'signed char' to a Scheme character. */

___EXP_FUNC(int,___schar_to_scmobj)
   ___P((___SCHAR x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___SCHAR x;
___WORD *obj;
int arg_num;)
{
  /*
   * No error possible because a 'signed char' always fits in
   * a Scheme character.
   */
  *obj = ___CHR(uchar_to_unicode((unsigned char)x));
  return ___NO_ERR;
}


/* Convert a C 'unsigned char' to a Scheme character. */

___EXP_FUNC(int,___uchar_to_scmobj)
   ___P((unsigned char x, ___WORD *obj, int arg_num),(x, obj, arg_num)
unsigned char x;
___WORD *obj;
int arg_num;)
{
  /*
   * No error possible because an 'unsigned char' always fits in
   * a Scheme character.
   */
  *obj = ___CHR(uchar_to_unicode(x));
  return ___NO_ERR;
}


/* Convert a C LATIN-1 encoded Unicode character to a Scheme character. */

___EXP_FUNC(int,___latin1_to_scmobj)
   ___P((___LATIN1 x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___LATIN1 x;
___WORD *obj;
int arg_num;)
{
  /*
   * No error possible because a LATIN-1 character always fits in
   * a Scheme character.
   */
  *obj = ___CHR(x);
  return ___NO_ERR;
}


/* Convert a C UCS-4 encoded Unicode character to a Scheme character. */

___EXP_FUNC(int,___ucs4_to_scmobj)
   ___P((___UCS4 x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___UCS4 x;
___WORD *obj;
int arg_num;)
{
  if (x > ___MAX_CHR) /* check that we are not truncating the character */
    {
      *obj = ___FAL;
      return ___CTOS_UCS4_ERR+arg_num;
    }
  *obj = ___CHR(x);
  return ___NO_ERR;
}


/* Convert a C UCS-2 encoded Unicode character to a Scheme character. */

___EXP_FUNC(int,___ucs2_to_scmobj)
   ___P((___UCS2 x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___UCS2 x;
___WORD *obj;
int arg_num;)
{
  if (x > ___MAX_CHR) /* check that we are not truncating the character */
    {
      *obj = ___FAL;
      return ___CTOS_UCS2_ERR+arg_num;
    }
  *obj = ___CHR(x);
  return ___NO_ERR;
}


/* Utility to convert a C 'unsigned long' to a Scheme bignum. */

___HIDDEN int ulong_to_bignum
   ___P((unsigned long x, ___WORD *obj, int positive, int arg_num),
        (x, obj, positive, arg_num)
unsigned long x;
___WORD *obj;
int positive;
int arg_num;)
{
  ___WORD r;
  unsigned long temp = x;
  long i, n = 1;
  while (temp > 0)
    {
      temp >>= ___RADIX_WIDTH;
      n++;
    }
  r = ___alloc_scmobj (___sBIGNUM, n*sizeof(___U16), ___STILL);
  if (r == ___FAL)
    {
      *obj = ___FAL;
      return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
    }
  ___U16VECTORSET(r,___FIX(0),___FIX(positive)) /* set sign */
  for (i=1; i<n; i++)
    {
      ___U16VECTORSET(r,___FIX(i),___FIX((x&((1<<___RADIX_WIDTH)-1))))
      x >>= ___RADIX_WIDTH;
    }
  *obj = r;
  return ___NO_ERR;
}


/* Convert a C 'long' to a Scheme integer. */

___EXP_FUNC(int,___long_to_scmobj)
   ___P((long x, ___WORD *obj, int arg_num),(x, obj, arg_num)
long x;
___WORD *obj;
int arg_num;)
{
  if (x >= ___MIN_FIX && x <= ___MAX_FIX)
    {
      *obj = ___FIX(x);
      return ___NO_ERR;
    }
  if (x < 0)
    return ulong_to_bignum ((unsigned long)-x, obj, 0, arg_num);
  else
    return ulong_to_bignum ((unsigned long)x, obj, 1, arg_num);
}


/* Convert a C 'unsigned long' to a Scheme integer. */

___EXP_FUNC(int,___ulong_to_scmobj)
   ___P((unsigned long x, ___WORD *obj, int arg_num),(x, obj, arg_num)
unsigned long x;
___WORD *obj;
int arg_num;)
{
  if (x <= ___MAX_FIX)
    {
      *obj = ___FIX(x);
      return ___NO_ERR;
    }
  return ulong_to_bignum (x, obj, 1, arg_num);
}


/* Convert a C 'short' to a Scheme integer. */

___EXP_FUNC(int,___short_to_scmobj)
   ___P((short x, ___WORD *obj, int arg_num),(x, obj, arg_num)
short x;
___WORD *obj;
int arg_num;)
{
  return ___long_to_scmobj ((long)x, obj, arg_num);
}


/* Convert a C 'unsigned short' to a Scheme integer. */

___EXP_FUNC(int,___ushort_to_scmobj)
   ___P((unsigned short x, ___WORD *obj, int arg_num),(x, obj, arg_num)
unsigned short x;
___WORD *obj;
int arg_num;)
{
  return ___ulong_to_scmobj ((unsigned long)x, obj, arg_num);
}


/* Convert a C 'int' to a Scheme integer. */

___EXP_FUNC(int,___int_to_scmobj)
   ___P((int x, ___WORD *obj, int arg_num),(x, obj, arg_num)
int x;
___WORD *obj;
int arg_num;)
{
  return ___long_to_scmobj ((long)x, obj, arg_num);
}


/* Convert a C 'unsigned int' to a Scheme integer. */

___EXP_FUNC(int,___uint_to_scmobj)
   ___P((unsigned int x, ___WORD *obj, int arg_num),(x, obj, arg_num)
unsigned int x;
___WORD *obj;
int arg_num;)
{
  return ___ulong_to_scmobj ((unsigned long)x, obj, arg_num);
}


/* Convert a C 'double' to a Scheme flonum. */

___EXP_FUNC(int,___double_to_scmobj)
   ___P((double x, ___WORD *obj, int arg_num),(x, obj, arg_num)
double x;
___WORD *obj;
int arg_num;)
{
  ___WORD r = ___alloc_scmobj (___sFLONUM, ___FLONUM_SIZE<<___LWS, ___STILL);
  if (r == ___FAL)
    {
      *obj = ___FAL;
      return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
    }
  ___FLONUM_VAL(r) = x;
  *obj = r;
  return ___NO_ERR;
}


/* Convert a C 'float' to a Scheme flonum. */

___EXP_FUNC(int,___float_to_scmobj)
   ___P((float x, ___WORD *obj, int arg_num),(x, obj, arg_num)
float x;
___WORD *obj;
int arg_num;)
{
  return ___double_to_scmobj ((double)x, obj, arg_num);
}


/* Convert a C 'void *' to a Scheme C-pointer object. */

___EXP_FUNC(int,___pointer_to_scmobj)
   ___P((void *x, ___WORD *obj, int arg_num),(x, obj, arg_num)
void *x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r = ___alloc_scmobj (___sPOINTER, sizeof (void*), ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      *(void**)___BODY_AS(r,___tSUBTYPED) = x;
      *obj = r;
    }
  return ___NO_ERR;
}


/* Convert a C function to a Scheme function. */

___EXP_FUNC(int,___function_to_scmobj)
   ___P((void *x, ___WORD *obj, int arg_num),(x, obj, arg_num)
void *x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    {
      *obj = ___FAL; /* #f counts as NULL */
      return ___NO_ERR;
    }
  /* 
   * At present, arbitrary C functions cannot be converted to Scheme
   * functions.
   */
  *obj = ___FAL;
  return ___CTOS_FUNCTION_ERR+arg_num;
}


/* Convert a C extended boolean to a Scheme boolean. */

___EXP_FUNC(int,___bool_to_scmobj)
   ___P((___BOOL x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___BOOL x;
___WORD *obj;
int arg_num;)
{
  *obj = x ? ___TRU : ___FAL;
  return ___NO_ERR;
}


/* Convert a C 'char *' to a Scheme string. */

___EXP_FUNC(int,___charstring_to_scmobj)
   ___P((char *x, ___WORD *obj, int arg_num),(x, obj, arg_num)
char *x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r;
      unsigned long i, n = 0;
      while (x[n] != 0)
        n++;
      r = ___alloc_scmobj (___sSTRING, n<<___LCS, ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      for (i=0; i<n; i++)
        /*
         * No error possible because a 'char' always fits in
         * a Scheme char.
         */
        ___STRINGSET(r,___FIX(i),___CHR(uchar_to_unicode((unsigned char)x[i])))
      *obj = r;
    }
  return ___NO_ERR;
}


/* Convert a C LATIN-1 encoded Unicode character string to a Scheme string. */

___EXP_FUNC(int,___latin1string_to_scmobj)
   ___P((___LATIN1STRING x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___LATIN1STRING x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r;
      unsigned long i, n = 0;
      while (x[n] != 0)
        n++;
      r = ___alloc_scmobj (___sSTRING, n<<___LCS, ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      for (i=0; i<n; i++)
        /*
         * No error possible because a LATIN-1 character always fits in
         * a Scheme char.
         */
        ___STRINGSET(r,___FIX(i),___CHR(x[i]))
      *obj = r;
    }
  return ___NO_ERR;
}


/* Convert a C UCS-4 encoded Unicode character string to a Scheme string. */

___EXP_FUNC(int,___ucs4string_to_scmobj)
   ___P((___UCS4STRING x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___UCS4STRING x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r;
      unsigned long i, n = 0;
      while (x[n] != 0)
        n++;
      r = ___alloc_scmobj (___sSTRING, n<<___LCS, ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      for (i=0; i<n; i++)
        {
          ___UCS4 c = x[i];
          if (c > ___MAX_CHR)
            {
              ___release_scmobj (r);
              *obj = ___FAL;
              return ___CTOS_UCS4STRING_ERR+arg_num;
            }
          ___STRINGSET(r,___FIX(i),___CHR(c))
        }
      *obj = r;
    }
  return ___NO_ERR;
}


/* Convert a C UCS-2 encoded Unicode character string to a Scheme string. */

___EXP_FUNC(int,___ucs2string_to_scmobj)
   ___P((___UCS2STRING x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___UCS2STRING x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r;
      unsigned long i, n = 0;
      while (x[n] != 0)
        n++;
      r = ___alloc_scmobj (___sSTRING, n<<___LCS, ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      for (i=0; i<n; i++)
        {
          ___UCS4 c = x[i];
          if (c > ___MAX_CHR)
            {
              ___release_scmobj (r);
              *obj = ___FAL;
              return ___CTOS_UCS2STRING_ERR+arg_num;
            }
          ___STRINGSET(r,___FIX(i),___CHR(c))
        }
      *obj = r;
    }
  return ___NO_ERR;
}


/* Convert a C UTF-8 encoded Unicode character string to a Scheme string. */

___EXP_FUNC(int,___utf8string_to_scmobj)
   ___P((___UTF8STRING x, ___WORD *obj, int arg_num),(x, obj, arg_num)
___UTF8STRING x;
___WORD *obj;
int arg_num;)
{
  if (x == 0)
    *obj = ___FAL; /* #f counts as NULL */
  else
    {
      ___WORD r;
      unsigned long i, n = 0;
      ___UTF8STRING p = x;
      while (___utf8_get (&p) != 0) /* advance until end or error */
        n++;
      r = ___alloc_scmobj (___sSTRING, n<<___LCS, ___STILL);
      if (r == ___FAL)
        {
          *obj = ___FAL;
          return ___CTOS_HEAP_OVERFLOW_ERR+arg_num;
        }
      p = x;
      for (i=0; i<n; i++)
        {
          ___UTF8STRING start = p;
          ___UCS4 c = ___utf8_get (&p);
          if (p == start || c > ___MAX_CHR)
            {
              ___release_scmobj (r);
              *obj = ___FAL;
              return ___CTOS_UTF8STRING_ERR+arg_num;
            }
          ___STRINGSET(r,___FIX(i),___CHR(c))
        }
      *obj = r;
    }
  return ___NO_ERR;
}


/* Create a "c-define" stack marker. */

___EXP_FUNC(int,___make_cdef_stack_marker) ___P((___WORD *marker),(marker)
___WORD *marker;)
{
  ___WORD stack_marker = ___make_vector (1, ___TRU, ___STILL);

  if (stack_marker == ___FAL)
    return ___CDEF_HEAP_OVERFLOW_ERR;

  *marker = stack_marker;
  *___PSTATE->fp = stack_marker;

  return ___NO_ERR;
}


/* Invalidate a "c-define" stack marker.  Called when a "c-define" returns. */

___EXP_FUNC(void,___kill_cdef_stack_marker) ___P((___WORD marker),(marker)
___WORD marker;)
{
  ___FIELD(marker,0) = ___FAL; /* invalidate the C stack frame */
  ___still_obj_refcount_dec (marker); /* allow GC of stack marker */
}


/*---------------------------------------------------------------------------*/
