/*
 * X-mame config-file and commandline parsing
 */

#define __CONFIG_C_
#ifdef MAME_NET
#include "network.h"
#endif /* MAME_NET */
#include "xmame.h"
#include "sound.h"
#include "driver.h"

/* some prototypes */
int parse_command_line (int argc, char *argv[]);
int parse_xmamerc_file();
int check_and_create_dot_xmame(void);
void set_default_options (void);
int verify_options(void);

/* from ... */
extern int beam, flicker;
extern int antialias;
extern int translucency;
extern char *cheatfile;

/* OpenGL options */

#ifdef xgl
extern int doublebuffer;
extern int cabview;
extern char *cabname;
extern int cabspecified;
#endif

/* some local vars */
static int showconfig = 0;
static int showusage  = 0;
static int list       = 0;
static int showclones = 1;
static int scale      = 0;
static char *gamename = "";
static char *ident;
static float beam_f, flicker_f;
#ifndef MESS
static char *defaultgamename;
#endif

/* option-struct */
enum option_type { set_bool, clear_bool, string, integer, set_int,
   seperator, floater, file, inputlog };

typedef struct {
   char *name;
   int  type;
   void *dest;
   int  default_int; /* abused for the value to set with set_int, and for write flag with file */
   void *default_ptr;
} option;

static option opts[] = {
   { "### Input Related: ###", seperator, NULL,		0,	NULL},
   { "joy",		set_bool,	&use_joystick,	FALSE,	NULL},
   { "nojoy",		clear_bool,	&use_joystick,	FALSE,	NULL},
   { "pad",		set_bool,	&use_joystick,	FALSE,	NULL},
   { "nopad",		clear_bool,	&use_joystick,	FALSE,	NULL},
   { "mouse",		set_bool,	&use_mouse,	TRUE,	NULL},
   { "nomouse",		clear_bool,	&use_mouse,	TRUE,	NULL},
   { "swapjoyaxis",	set_bool,	&swapjoyaxis,	FALSE,	NULL},
   { "noswapjoyaxis",	clear_bool,	&swapjoyaxis,	FALSE,	NULL},
   { "analogstick",	set_bool,	&analogstick,	FALSE,	NULL},
   { "noanalogstick",	clear_bool,	&analogstick,	FALSE,	NULL},
#ifdef X11_JOYNAME
   { "x11joyname",	string,		&x11joyname,	0,	X11_JOYNAME},
#else
   { "x11joyname",	string,		&x11joyname,	0,	""},
#endif
   { "paddevname",	string,		&towns_pad_dev,	0,	"/dev/pad00"},
   { "### Video Related: ###", seperator, NULL,		0,	NULL},
   { "heightscale",	integer,	&heightscale,	1,	NULL},
   { "widthscale",	integer,	&widthscale,	1,	NULL},
   { "scale",		integer,	&scale,		0,	NULL},
   { "display",		string,		&displayname,	0,	DISPLAY},
   { "mitshm",		set_bool,	&mit_shm_avail,	TRUE,	NULL},
   { "nomitshm",	clear_bool,	&mit_shm_avail,	TRUE,	NULL},
   { "xsync",		set_bool,	&use_xsync,	TRUE,	NULL},
   { "noxsync",		clear_bool,	&use_xsync,	TRUE,	NULL},
   { "dirty",		set_bool,	&use_dirty,	TRUE,	NULL},
   { "nodirty",		clear_bool,	&use_dirty,	TRUE,	NULL},
   { "privatecmap",	set_bool,	&use_private_cmap, FALSE, NULL},
   { "noprivatecmap",	clear_bool,	&use_private_cmap, FALSE, NULL},
   { "forcetruecolor",  set_bool,	&force_truecolor, FALSE, NULL},
   { "noforcetruecolor",clear_bool,	&force_truecolor, FALSE, NULL},
   { "root_window_id",	integer,	&root_window_id,0,	NULL},
   { "videomode",	integer,	&video_mode,	0,	NULL},
   { "throttle",	set_bool,	&throttle,	TRUE,	NULL},
   { "nothrottle",	clear_bool,	&throttle,	TRUE,	NULL},
   { "frameskip",	integer,	&options.frameskip, 0,	NULL},
   { "norotate",	set_bool,	&options.norotate, FALSE, NULL},
   { "ror",		set_bool,	&options.ror,	FALSE,	NULL},
   { "rol",		set_bool,	&options.rol,	FALSE,	NULL},
   { "flipx",		set_bool,	&options.flipx,	FALSE,	NULL},
   { "flipy",		set_bool,	&options.flipy,	FALSE,	NULL},
   { "brightness",	integer,	&brightness,	100,	NULL},
   { "gamma-correction",floater,	&gamma_correction, 1,	NULL},
   { "### Vector Games: ###", seperator, NULL,		0,	NULL},
   { "beam",		floater,	&beam_f,	1,	NULL},
   { "flicker",		floater,	&flicker_f,	0,	NULL},
   { "antialias",	set_bool,	&antialias,	TRUE,	NULL},
   { "noantialias",	clear_bool,	&antialias,	TRUE,	NULL},
   { "translucency",	set_bool,	&translucency,	TRUE,	NULL},
   { "notranslucency",	clear_bool,	&translucency,	TRUE,	NULL},
#ifdef xgl
   { "dblbuffer",set_bool,&doublebuffer,TRUE,NULL},
   { "nodblbuffer",clear_bool,&doublebuffer,TRUE,NULL},
   { "cabview",set_bool,&cabview,TRUE,NULL},
   { "fullview",clear_bool,&cabview,TRUE,NULL},
   { "cabinet",string,&cabname,0,"glmame"},
#endif /* ifdef xgl */
   { "### Sound Related: ###", seperator, NULL,		0,	NULL},
   { "sound",		set_bool,	&play_sound,	FALSE,	NULL},
   { "nosound",		clear_bool,	&play_sound,	FALSE,	NULL},
   { "samplefreq",	integer,	&options.samplerate, AUDIO_SAMPLE_FREQ, NULL},
   { "timerfreq",	integer,	&audio_timer_freq, AUDIO_TIMER_FREQ,	NULL},
   { "fakesound",	set_bool,	&fake_sound,	FALSE,	NULL},
   { "fragsize",	integer,	&frag_size,	512,	NULL},
   { "numfrags",	integer,	&num_frags,	5,	NULL},
   { "audiodevice",	string,		&audiodevice,	0,	"/dev/dsp"},
   { "volume",		integer,	&attenuation,	-3,	NULL},
   { "### General: ###",seperator,	NULL,		0,	NULL},
   { "spooldir",	string,		&spooldir,	0,	SPOOLDIR},
   { "cheatfile",	string,		&cheatfile,	0,	CHEATFILE},
#ifndef MESS   
   { "defaultgame",	string,		&defaultgamename, 0,	"pacman"},
#endif
   { "record",		inputlog,	&options.record, 1,	NULL},
   { "playback",	inputlog,	&options.playback, 0,	NULL},
   { "stdout-file",	file,		&stdout_file,	1,	stdout},
   { "stderr-file",	file,		&stderr_file,	1,	stderr},
   { "log",		file,		&options.errorlog, 1,	NULL},
   { "profiler",	set_bool,	&use_profiler,	FALSE,	NULL},
   { "noprofiler",	clear_bool,	&use_profiler,	FALSE,	NULL},
   { "cheat",		set_bool,	&options.cheat,	FALSE,	NULL},
   { "nocheat",		clear_bool,	&options.cheat,	FALSE,	NULL},
   { "debug",		set_bool,	&options.mame_debug, FALSE, NULL},
   { "ident",		string,		&ident,		0,	NULL},
   { "list",		set_int,	&list,		LIST_LIST,	NULL},
   { "listfull",	set_int,	&list,		LIST_FULL,	NULL},
   { "listgames",	set_int,	&list,		LIST_GAMES,	NULL},
   { "listclones",	set_int,	&list,		LIST_CLONES,	NULL},
   { "listdetails",	set_int,	&list,		LIST_DETAILS,	NULL},
   { "listroms",	set_int,	&list,		LIST_ROMS,	NULL},
   { "listsamples",	set_int,	&list,		LIST_SAMPLES,	NULL},
   { "listsamdir",	set_int,	&list,		LIST_SAMDIR,	NULL},
   { "listcrc",		set_int,	&list,		LIST_CRC,	NULL},
   { "listdupcrc",	set_int,	&list,		LIST_DUPCRC,	NULL},
   { "listcolors",	set_int,	&list,		LIST_COLORS,	NULL},
   { "lmr",		set_int,	&list,		LIST_LMR,	NULL},
   { "wrongorientation",set_int,	&list,		LIST_WRONGORIENTATION,	NULL},
   { "verifyroms",	set_int,	&list,		VERIFY_ROMS,	NULL},
   { "verifysamples",	set_int,	&list,		VERIFY_SAMPLES,	NULL},
   { "noclones",	clear_bool,	&showclones,	TRUE,	NULL},
/* these are bools, but use set_int so they don't show up in -showconfig */
   { "showconfig",	set_int,	&showconfig,	TRUE,	NULL},
   { "h",		set_int,	&showusage,	TRUE,	NULL},
   { "?",		set_int,	&showusage,	TRUE,	NULL},
   { "help",		set_int,	&showusage,	TRUE,	NULL},
   { "-help",		set_int,	&showusage,	TRUE,	NULL},
   { NULL,		0,		NULL,		0,	NULL}
};

/*
 * get configuration from configfile and env.
 */
int get_config (int argc, char *argv[])
{
   char *pt;
   
   set_default_options();
   
   if (check_and_create_dot_xmame()==OSD_NOT_OK)
      return (OSD_NOT_OK);
   
   if (parse_xmamerc_file()==OSD_NOT_OK)
   {
      fprintf(stderr_file,"Error in parsing xmamerc file\n");
      return (OSD_NOT_OK);
   }
   
#ifdef MAME_NET
   netstate = 0;
   netkeymap = FALSE;
   players = 0;
#endif /* MAME_NET */

   /* get environment variables. This overrides xmamerc options */
   if ( (pt=getenv("DISPLAY")) ) displayname=pt;
   if ( (pt=getenv("ROMPATH"))  ) strncpy(rompath, pt, MAXPATHL-1);

   if (parse_command_line(argc, argv)==OSD_NOT_OK) return OSD_NOT_OK;
   
   init_rom_path();

   return verify_options();
}

/* 
 * check and if nescesarry create ${HOME}/.xmame (or .xmess).
 */
int check_and_create_dot_xmame(void)
{
   struct passwd   *pw;
   char name[MAXPATHL];
   struct stat stat_buffer;

   /* locate user's home directory */
   pw=getpwuid(getuid());
   if(!pw)
   { 
      fprintf(stderr_file, "Who are you? Not found in passwd database!!\n");
      return (OSD_NOT_OK);
   }
   home_dir = malloc(strlen(pw->pw_dir)+1);
   if (!home_dir)
   {
      fprintf(stderr_file, "Error allocating memory\n");
      return (OSD_NOT_OK);
   }
   strcpy(home_dir, pw->pw_dir);

   /* check for existence of ${HOME}/.xmame / ${HOME}/.xmess */
   sprintf(name, "%s/.%s", home_dir, NAME);
   if (stat(name, &stat_buffer))
   {
      /* error check if it doesn't exist or something else is wrong */
      if (errno == ENOENT)
      {
         /* doesn't exist letts create it ;) */
         if (mkdir(name, S_IRWXU|S_IRWXG|S_IROTH|S_IXOTH))
         {
            fprintf(stderr_file, "Error creating dir %s", name);
            perror(" ");
            return OSD_NOT_OK;
         }
      }
      else
      {
         /* something else went wrong yell about it */
         fprintf(stderr_file, "Error opening %s", name);
         perror(" ");
         return OSD_NOT_OK;
      }
   }
   else
   {
      /* file exists check it's a dir otherwise yell about it */
      if(!S_ISDIR(stat_buffer.st_mode))
      {
         fprintf(stderr_file,"Error %s exists but isn't a dir\n", name);
         return OSD_NOT_OK;
      }
   }
   return OSD_OK;
}

/* 
 * show help and exit
 */
void show_usage(void) 
{
char message[]="%s\n"
#ifdef MESS
  "Usage: xmess <system> [game] [options]\n"
#else
  "Usage: xmame [game] [options]\n"
#endif 
  "Options:\n"
  "\n"
  "*** Input ***\n"
  " -[no]joy                 Disable/enable joystick (if supported)\n"
  " -[no]mouse               Disable/enable mouse (if supported)\n"
  " -[no]pad                 Disable/enable FM-TOWNS game pad (if supported)\n"
  " -[no]swapjoyaxis         Swap X and Y joystick axis\n"
  " -[no]analogstick         Use Joystick as analog for analog controls\n"
  " -x11joyname <name>       Name of X-based joystick device (if compiled in)\n"
  " -paddevname <name>       Name of pad device (defaults to /dev/pad00)\n"
  " -mapkey <Xcode>,<Mcode>  Set an specific key mapping\n"
  "\n"
#ifdef MAME_NET
 "*** Network ***\n"
  " -master <no of players>     Enable master mode. Set number of players\n"
  " -slave  <hostname>          Enable slave mode. Set master hostname\n"
  " -netmapkey                  All players use the player 1 keys.\n"
  "                             To use with * real* multiplayer games\n"
  "\n"
#endif /* MAME_NET */
  "*** Video ***\n"
#ifdef x11
  " -heightscale<scale>      Set Y-Scale aspect ratio\n"
  " -widthscale	<scale>      Set X-Scale aspect ratio\n"
  " -scale 	<scale>      Set X-Y Scale to the same aspect ratio\n"
  " -display	<display>    To select display\n"
  " -[no]mitshm              Disable/enable MIT Shared Mem (if supported)\n"
  " -[no]xsync               Select XFlush/Xsync screen refresh method\n"
  " -[no]dirty               Disable/enable use of dirty rectangles\n"
  " -[no]privatecmap         Disable/enable use of private color maps\n"
  " -[no]forcetruecolor      Disable/enable try to use pseudocolor X-Visuals\n"
  " -root_window_id [id]     Create the xmame-window in an alternate\n"
  "                          root-window, only usefull for frontends!\n"
#endif /* ifdef x11 */
#ifdef svgalib
  " -videomode  <mode>       Select svgalib video mode\n"
#endif
  " -[no]throttle            Disable/enable throttle\n"
  " -frameskip	<#frames>    Skip <#frames> in video refresh\n"
  " -norotate                Disable rotation\n"
  " -ror                     Rotate display 90 degrees rigth\n"
  " -rol                     Rotate display 90 degrees left\n"
  " -flipx                   Flip X axis\n"
  " -flipy                   Flip Y axis\n"
  " -brightness <brightness> Set the brightness (0-100%%)\n"
  " -gamma-correction<gamma> Set the gamma-correction (0.5-2.0)\n"
  "\n"
  "*** Vector Games ***\n"
  " -beam                    Set the beam size for vector games\n"
  " -flicker                 Set the flicker for vector games\n"
  " -[no]antialias           Disable/enable antialiasing\n"
  " -[no]translucency        Disable/enable translucency\n"
  "\n"
#ifdef xgl
  "*** GLmame ***\n"
  " -[no]dblbuffer              Disable/enable double buffering\n"
  " -cabview                    Start in cabinet view mode\n"
  " -fullview                   Start in fullscreen view mode\n"
  " -cabinet <cabname>          Use the cabinet model <cabname>\n"
  "\n"
#endif /* ifdef xgl */
  "*** Sound ***\n"
  " -[no]sound               Disable/enable sound (if available)\n"
  " -audiodevice <device>    Use an alternative audiodevice instead of /dev/dsp\n"
  " -samplefreq <samplefreq> Set the playback sample-frequency/rate\n"
  " -timerfreq <timerfreq>   Set the timer frequency (if timer based audio)\n"
  " -fakesound               Generate sound even when sound is disabled, this is\n"
  "                          needed for some games which won't run without sound\n"
  " -fragsize <size> (*)     Set the size of the soundbuffers in bytes,\n"
  "                          Default=512, lower this if your sound seems lagged,\n"
  "                          raise it if your sound clicks. Note this defines the\n"
  "                          buffersize used with samplefreq=22050, the buffersize\n"
  "                          is automaticly scaled for other samplefrequencies\n"
  " -numfrags <num> (*)      Nummer of the above buffers to allocate, Default 5\n"
  "                          The default should be okay for everyone.\n"
  " (*)                      These options only applies to linux/freebsd, under\n"
  "                          other os's/archs these options only control xmame's\n"
  "                          own buffer, see doc/README.unix for details\n"
  " -volume <db>             Set volume to <db> db, (-24 (soft) - 0(loud) )\n"
  "\n"
  "*** General ***\n"
  " -rompath     <path>      Add <path> to the rom search path\n"
  " -spooldir    <dir>       Store high scores in <dir>\n"
  " -cheatfile   <filename>  Use <filename> as cheat database\n"
  " -record      <filename>  Record keypresses into <filename>\n"
  " -playback    <filename>  Playback keypresses from <filename>\n"
  " -stderr-file <filename>  Redirect stderr to <filename>\n"
  " -stdout-file <filename>  Redirect stdout to <filename>\n"
  " -log         <filename>  Log debug info to <filename>\n"
  " -[no]profiler            Disable/enable profiler\n"
  " -[no]cheat               Disable/enable cheat mode if game supports it\n"
  " -ident       <filename>  Identify unknown romdump <filename>, or unknown\n"
  "                          romdumps in dir/zip <filename>.\n"
  " -list          [regexp]  List supported games matching regexp, or all\n"
  " -listfull      [regexp]  List supported games with full description\n"
  " -listgames     [regexp]  List supported games with manufacturer and year\n"
  " -listclones    [regexp]  List clones of games matching regexp, or all\n"
  " -listdetails   [regexp]  Detailed list of supported games\n"
  " -listroms      [regexp]  List used ROMS\n"
  " -listsamples   [regexp]  List used audio samples\n"
  " -listsamdir    [regexp]  List dir where samples are taken from\n"
  " -listcrc       [regexp]  List used ROMS with crc\n" 
  " -listdupcrc    [regexp]  List ROMS with identical crc\n" 
  " -verifyroms    [regexp]  Verify ROMS for games matching regexp\n"
  " -verifysamples [regexp]  Verify audio samples for games matching regexp\n"
  " [regexp]                 Optional, May contain * and ? wildcards\n"
  " -noclones                Don't show bootlegs/clones in list commands\n"
  " -showconfig              Display Running parameters in rc style\n"
  " -help | -h | -?          Show this help\n"
  "\n"
  "Files: \n"
  " %s	Global configuration file\n"
  " ${HOME}/.%s/%src    User global configuration file\n"
  " ${HOME}/.%s/<game>.cnf Game-dependent user configuration file\n"
  "\n"
  "Environment variables:\n"
  "\n"
  " ROMPATH                  Rom search path\n"
  " DISPLAY                  X-Server to display in\n"
  "\n"
#ifdef MESS
  "M.E.S.S.  -  Multi-Emulator Super System\n"
  "Copyright (C) 1998  by the MESS team\n"
#else
  "Mame is an Original Idea of Nicola Salmoria and Mirko Buffoni\n"
#endif
  "%s port maintained by Hans de Goede\n"
  "";
  fprintf(stdout_file,message,title,MAMERC,NAME,NAME,NAME,NAME);
}

/*
 * show running parameters in xmamerc / xmessrc style format ( using stdout_file )
 * if exitflag is true exit program
 *
 */
void show_config(void)
{
   int i, section = 0;
   char blank[20];
   
   fprintf(stdout_file,"##### %s Running parameters: #####\n",title);
   fprintf(stdout_file,"#\n");
   
   for(i=0;opts[i].name;i++)
   {
      memset(blank, ' ', 20);
      switch(opts[i].type)
      {
         case set_bool:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%d\n", opts[i].name, blank,
               (*(int *)opts[i].dest)? 1:0);
            break;
         case string:
            if (*(char **)opts[i].dest==NULL) break;
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%s\n", opts[i].name, blank,
               *(char **)opts[i].dest);
            break;
         case integer:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%d\n", opts[i].name, blank,
               *(int *)opts[i].dest);
            break;
         case floater:
            blank[19 - strlen(opts[i].name)] = 0;
            fprintf(stdout_file, "%s%s%f\n", opts[i].name, blank,
               *(float *)opts[i].dest);
            break;
         case seperator:
            fprintf(stdout_file, "#\n%s\n#\n", opts[i].name);
            section++;
            switch(section) /* use this to print custom entries */
            {
               case 1: /* Input Related */
                  break;
               case 2: /* Video Related */
                  break;
               case 3: /* Vector Related */
                  break;
               case 4: /* Sound Related */
                  break;
               case 5: /* General */
                  fprintf(stdout_file,"rompath            %s\n",rompath);
                  break;
            }
            break;
      }
   }
}

/*
 * check for existence of ${HOME}/.xmame/xmamerc / ${HOME}/.xmess/xmessrc
 * if found parse it, else try global MAMERC #defined file
 */
int parse_xmamerc_file()
{
	FILE            *file;
	int             lineno;
	char            buffer[2048];

	sprintf(buffer, "%s/.%s/%src", home_dir, NAME, NAME);
	
	/* try to open file. If so parse it */
	if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
		fprintf(stderr_file,"File %s not found.\n",buffer);
		/* No user config found - trying global file */
		strcpy(buffer, MAMERC );
		if ( (file=fopen(buffer,"r")) == (FILE *) NULL) {
			fprintf(stderr_file,"File %s not found. Skipping ...\n",buffer);
			return (OSD_OK);
		}
	}
	lineno=0;
	while(fgets( buffer,2047,file) ) {
	    char *p;
	    char *(q[5]);
	    int i,j;
	    lineno++;
	    /* skip comments */
	    if ( ( p=strchr(buffer,'#') ) ) *p='\0';
	    /* scan for words */
	    for(i=0;i<5;i++) q[i]=(char *)NULL;
	    for(i=0,j=0,p=buffer; *p; p++ ) {
		if ( isspace(*p) ) { *p='\0'; j=0; }
		else               { if(!j++) q[i++]=p; }
	    }   /* end of stripping words ( for i=0 )*/
	    /* test for wrong number of args */
	    if ( i==0 ) continue; /* empty line */
	    if ( i!=2 ) { 
		    fprintf(stderr_file,"Line %d: wrong number of parameters \n",lineno);
		    fclose(file);
		    return OSD_NOT_OK;
	    }
	    
	    for (i=0; opts[i].name; i++)
	    {
	       if (strcasecmp(q[0], opts[i].name) == 0) break;
	    }
	    
	    if (opts[i].name == NULL) /* unkown option, or special case */
	    {
	       if (strcasecmp(q[0],"mapkey") == 0) {
	 	  int from,to;
#ifdef ultrix
		  /* ultrix sscanf() requires explicit leading of 0x for hex numbers */
		  if ( sscanf(q[1],"0x%x,0x%x",&from,&to) != 2 )
#else
		  if ( sscanf(q[1],"%x,%x",&from,&to) != 2 )
#endif
		     fprintf(stderr_file,"Line %d: Invalid keymapping %s. Ignoring...\n",lineno,q[1]);
		  else if (sysdep_mapkey(from,to) ) {
		     fprintf(stderr_file,"Line %d: invalid keymapping\n",lineno);
		     return OSD_NOT_OK;
		  }
		  continue;
	       } /* mapkey */
	       
	       if (strcasecmp(q[0], "rompath") == 0) {
	          strncpy(rompath, q[1], MAXPATHL - 1);
	          continue;
	       } /* rompath */
	    
	       fprintf(stderr_file,"Line %d: unknown command %s\n",lineno,q[0]);
	       fclose(file);
	       return OSD_NOT_OK;
	    }
	    
	    switch(opts[i].type)
	    {
	        case set_bool:
	        case integer:
	           *(int *)opts[i].dest = atoi(q[1]);
	           break;
	        case string:
	           *(char **)opts[i].dest = (char *)malloc(1+strlen(q[1]));
                   if(*(char **)opts[i].dest==NULL)
                   {
                        fprintf(stderr_file,"Malloc error: line %d\n",lineno);
                        return OSD_NOT_OK;
                   }
                   strcpy(*(char **)opts[i].dest, q[1]);
	           break;
	        case floater:
	           *(float *)opts[i].dest = atof(q[1]);
	           break;
	        default:
	           fprintf(stderr_file, "Line %d: unknown command %s\n",lineno,q[0]);
		   fclose(file);
		   return OSD_NOT_OK;
		   break;
	    }
	} /* while */
	fclose(file);
	return OSD_OK;
}

/*
 * Set all options to there defaults
 */
void set_default_options (void)
{
   int i;
   for(i=0;opts[i].name;i++)
   {
      switch(opts[i].type)
      {
         case set_bool:
         case clear_bool:
         case integer:
            *(int *)opts[i].dest = opts[i].default_int;
            break;
         case string:
            *(char **)opts[i].dest = (char *)opts[i].default_ptr;
            break;
         case floater:
            *(float *)opts[i].dest = opts[i].default_int;
            break;
         case file:
            *(FILE **)opts[i].dest = (FILE *)opts[i].default_ptr;
            break;
         case inputlog:
            *(void **)opts[i].dest = opts[i].default_ptr;
            break;
      }
   }
   /* initialise non option var's (mostly video subsystem */
   bitmap		= NULL;
   key			= NULL;
   No_FM		= TRUE;
   snapshot_no		= 0;
   use_layer		= FALSE;
   strncpy(rompath, ROMPATH, MAXPATHL - 1);
   options.samplebits	= 8;
#ifdef xf86_dga
   window = 0;
#endif
}

int verify_options(void)
{
   int power, j;
   unsigned char lsb_test[2]={0,1};
   
   /* Lett's see of the endians of this arch is correct otherwise
      YELL about it and bail out. */
#ifdef LSB_FIRST
   if(*((unsigned short*)lsb_test) != 0x0100)
#else	
   if(*((unsigned short*)lsb_test) != 0x0001)
#endif
   {
      fprintf(stderr_file, "Error: Compiled byte ordering doesn't match machine byte ordering\n"
         "Are you sure you choose the right arch?\n"
#ifdef LSB_FIRST
         "Compiled for lsb-first try removing -DLSB_FIRST from DEFS.$(ARCH)\nin makefile.unix\n");
#else
         "Compiled for msb-first try adding -DLSB_FIRST to DEFS.$(ARCH)\nin makefile.unix\n");
#endif
      return OSD_NOT_OK;
   }
   
   if (showusage)
   {
      show_usage();
      return OSD_OK;
   }
   
   /* handle frontend options */
   if (ident)
   {
      romident(ident, 1);
      return OSD_OK;
   }
   
   /* listclones is a special case since the strwildcmp */
   /* also has to be done on clone_of. */
   if (list == LIST_CLONES)
   {
      return frontend_list_clones(gamename);
   }
   
   if (list)
   {
      return frontend_list(list, gamename, showclones);
   }

#ifndef MESS
   if (gamename[0]==0) gamename = defaultgamename;
#endif

   /* do we have a drivers for this? */
   for (j = 0; drivers[j]; j++)
      if (strcasecmp(gamename,drivers[j]->name) == 0) break;

   if (drivers[j] == 0)
   {
      fprintf(stderr_file,"\"%s\" not supported\n", gamename);
      return OSD_NOT_OK;
   }
   else
   {
      game_index = j;
   }
   
   if (scale) widthscale = heightscale = scale;
   for (power=0; (1 << power) < frag_size; power++) {}
   if ((1 << power)!=frag_size)
   {
      fprintf(stderr_file, "illegal fragsize (%d)\n", frag_size);
      fprintf(stderr_file, "fragsize should be a power of 2 (eg 512,1024,2048,etc)\n");
      return (OSD_NOT_OK);
   }
   if (options.frameskip < 0) {
      fprintf (stderr_file,"illegal frameskip value (%d)\n", options.frameskip);
      return (OSD_NOT_OK);
   }
   if (options.frameskip > 3) options.frameskip = 3;
   if (options.samplerate < 8000 || options.samplerate > 48000) {
      fprintf (stderr_file,"illegal audio sample frequention (%d)\n", options.samplerate);
      return (OSD_NOT_OK);
   }
   if (video_mode < 0 || video_mode > 2) {
      fprintf (stderr_file,"illegal videomode (%d)\n", video_mode);
      return (OSD_NOT_OK);
   }
   
   beam = (int)(beam_f * 0x00010000);
   if      (beam < 0x00010000)
      beam = 0x00010000;
   else if (beam > 0x00100000)
      beam = 0x00100000;
   flicker = (int)(flicker_f * 2.55);
   if      (flicker < 0)
      flicker = 0;
   else if (flicker > 255)
      flicker = 255;

   if      (brightness < 0)
      brightness = 0;
   else if (brightness > 100)
      brightness = 100;
   if      (gamma_correction < 0.5)
      gamma_correction = 0.5;
   else if (gamma_correction > 2.0)
      gamma_correction = 2.0;
   if      (attenuation < -24)
      attenuation = -24;
   else if (attenuation > 0)
      attenuation = 0;
   
   if (showconfig)
   {
      show_config();
      return OSD_OK;
   }

   return 1234;
}

int parse_command_line (int argc, char *argv[])
{
   int i, j, got_gamename	= 0;
#ifdef MESS
   int num_roms		= 0;
   int num_floppies	= 0;
#endif
	
   /* parse argument invocation. */
   for (i = 1;i < argc;i++)
   {
      if (argv[i][0]=='-')
      {
         for(j=0;opts[j].name;j++)
         {
            if (strcasecmp(opts[j].name, argv[i]+1) == 0) break;
         }
         
         if(opts[j].name==NULL) /* unknown option, or special case */
         {
            if (strcasecmp(argv[i],"-mapkey") == 0) {
               if ( ! *argv[i+1] ) {
                  fprintf(stderr_file,"Error: -mapkey requires an extra argument\n");
                  return OSD_NOT_OK;
               } else {
                  int from,to;
                  if ( sscanf(argv[++i],"%x,%x",&from,&to) != 2 )
                     fprintf(stderr_file,"Invalid keymapping %s. Ignoring...\n",argv[i]);
                   else sysdep_mapkey(from,to);
               }
               continue;
	    }
#ifdef MAME_NET
                if (strcasecmp(argv[i], "-master") == 0) {
                        if (netstate == SLAVE)
                        {
                                fprintf(stderr, "Can't be Slave and Master\n");
                                return(OSD_NOT_OK);
                        }
                        netstate = MASTER;
                        players = atoi(argv[++i]);
                        if (players > 4)
                          {
                            fprintf(stderr, "4 players max.\n");
                            return(OSD_NOT_OK);
                          }
                        printf("XMame in network Master Mode\nWaiting for %d players.\n", players-1);
                        continue;
                }
                if (strcasecmp(argv[i], "-slave") == 0) {
                        if (netstate == MASTER)
                          {
                            fprintf(stderr, "Can't be Slave and Master\n");
                            return(OSD_NOT_OK);
                          }
                        netstate = SLAVE;
                        mastername = argv[++i];
                        printf("Slave Mode; Registering to Master %s\n", mastername);
                        continue;
                }
	if (strcasecmp(argv[i], "-netmapkey") == 0) {
                        netkeymap = TRUE;
                        continue;
                }
#endif /* MAME_NET */
            if (strcasecmp(argv[i], "-rompath") == 0)
            {
               i++;
               if (i==argc)
               {
                  fprintf(stderr_file, "Error: -rompath requires an extra argument\n");
                  return OSD_NOT_OK;
               }
               strncat(rompath, ":",     (MAXPATHL-1) - strlen(rompath));
               strncat(rompath, argv[i], (MAXPATHL-1) - strlen(rompath));
               continue;
            }
            fprintf(stderr_file,"Unknown option %s. Try %s -help\n", argv[i], argv[0]);
            return OSD_NOT_OK;
         }
         
         switch(opts[j].type)
         {
            case set_bool:
               *(int *)opts[j].dest = TRUE;
               break;
            case clear_bool:
               *(int *)opts[j].dest = FALSE;
               break;
            case set_int:
               *(int *)opts[j].dest = opts[j].default_int;
               break;
            case integer: /* these all require an argument */
            case string:
            case floater:
            case file:
            case inputlog:
               i++;
               if (i==argc)
               {
                  fprintf (stderr_file, "Error: -%s requires an extra argument.\n", opts[i].name);
                  return OSD_NOT_OK;
               }
               switch(opts[j].type)
               {
                  case integer:
                     *(int *)opts[j].dest = atoi(argv[i]);
                     break;
                  case string:
                     *(char **)opts[j].dest = argv[i];
                     break;
                  case floater:
                     *(float *)opts[j].dest = atof(argv[i]);
                     break;
                  case file:
                     *(FILE **)opts[j].dest = fopen(argv[i],
                        (opts[j].default_int)? "w":"r");
                     if (*(FILE **)opts[j].dest == NULL)
                     {
                        *(FILE **)opts[j].dest = (FILE *)opts[j].default_ptr;
                        fprintf(stderr_file, "Error: couldn't open %s\n", argv[i]);
                        return OSD_NOT_OK;
                     }
                     break;
                  case inputlog:
                     *(void **)opts[j].dest = osd_fopen(NULL, argv[i],
                        OSD_FILETYPE_INPUTLOG, opts[j].default_int);
                     if (*(void **)opts[j].dest == NULL)
                     {
                        *(void **)opts[j].dest = opts[j].default_ptr;
                        fprintf(stderr_file, "Error: couldn't open %s\n", argv[i]);
                        return OSD_NOT_OK;
                     }
                     break;
               }
               break;
         }
      }
      else
      {  /* if arrives here and not for the first time, 
            means syntax error */
         if (!got_gamename) /* notice: for MESS game means system */
         {
            gamename     = argv[i];
            got_gamename = 1;
         }
         else
#ifdef MESS
         {
            /* Is it a floppy or a rom? */
            if ((strstr(argv[i],".dsk")!=NULL) || (strstr(argv[i],".DSK")!=NULL))
            {
               if (num_floppies >= MAX_FLOPPY)
               {
                  fprintf(stderr_file, "Too many floppy names specified!\n");
                  return OSD_NOT_OK;
               }
               strcpy(options.floppy_name[num_floppies++],argv[i]);
            }
            else
            {
               if (num_roms >= MAX_ROM)
               {
                  fprintf(stderr_file, "Too many image names specified!\n");
                  return OSD_NOT_OK;
               }
               strcpy(options.rom_name[num_roms++],argv[i]);
            }
         }
#else
         {
            fprintf(stderr_file,"Duplicate gamename: %s\n",argv[i]);
            return OSD_NOT_OK;
         }
#endif
      }
   }

#ifdef MESS
   if (!got_gamename) showusage = TRUE;
   if (!num_floppies && !num_roms)
   {
      fprintf(stderr_file, "Warning:\n"
         "You didn't supply any rom/disk images on the commandline\n"
         "Most emulations of xmess will fail due to this, if xmess fails try:\n"
         "xmess <systemname> <rom/diskimage(s)> [options]\n"
         "Or try xmess -help / read README.unix for more info\n");
   }
#endif	
   return OSD_OK;
}
