/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/Form.h>
#include <Xm/Scale.h>
#include <Xm/TextF.h>
#include "xdir.h"

static struct {
	Widget w_dialog;
	Widget w_statusLabel;
	Widget w_status;
	Widget w_nameLabel;
	Widget w_name;
	Widget w_lengthLabel;
	Widget w_length;
	Widget w_dataRateLabel;
	Widget w_dataRate;
	Widget w_scale;
	Widget w_fileCount;
} xfermon;

extern Widget w_toplev;
extern Display *display;

char *cstring_to_text();
char *add_commas();
void cb_xfermon_close();
void cb_map_dialog();


/*
 * show_xfermon - Pop up file transfer monitor over directory window
 *                "dirwin".
 */
show_xfermon(dirwin)
struct dirwin_st *dirwin;
{
	/* Create file transfer monitor */
    create_xfermon_dialog(dirwin);

	/* Initialize widgets */
	set_xfermon_status("");
	set_xfermon_name("");
	set_xfermon_progress((long)-1, (long)-1, (double)0.0);

	/* Pop up dialog */
	XtManageChild(xfermon.w_dialog);
	add_dialog_to_list(xfermon.w_dialog);
	force_update(xfermon.w_dialog);
}


/*
 * create_xfermon_dialog - Create file transfer monitor.  Position over
 *                         directory window "dirwin".
 */
create_xfermon_dialog(dirwin)
struct dirwin_st *dirwin;
{
	XmString label;
	Arg args[3];
	int i;

    /* Create form dialog for connection to remote host */
	i = 0;
	XtSetArg(args[i], XmNmarginWidth, 10); i++;
	XtSetArg(args[i], XmNmarginHeight, 20); i++;
	XtSetArg(args[i], XmNresizePolicy, XmRESIZE_NONE); i++;
    xfermon.w_dialog = XmCreateFormDialog(dirwin->w_shell, "xfermon", args, i);
	XtAddCallback(xfermon.w_dialog, XmNmapCallback, cb_map_dialog,
		(XtPointer)XtWindow(dirwin->w_shell));

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(xfermon.w_dialog, cb_xfermon_close, NULL, False);

	/* Create status label */
	xfermon.w_statusLabel = XtVaCreateManagedWidget(
        "statusLabel",
        xmLabelWidgetClass,
        xfermon.w_dialog,
		XmNtopAttachment, 		XmATTACH_FORM,
		XmNleftAttachment, 		XmATTACH_FORM,
        NULL
    );

	/* Create file name label */
	xfermon.w_nameLabel = XtVaCreateManagedWidget(
        "nameLabel",
        xmLabelWidgetClass,
        xfermon.w_dialog,
		XmNtopAttachment, 	XmATTACH_WIDGET,
		XmNtopWidget,		xfermon.w_statusLabel,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create file length label */
	xfermon.w_lengthLabel = XtVaCreateManagedWidget(
        "lengthLabel",
        xmLabelWidgetClass,
        xfermon.w_dialog,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		xfermon.w_nameLabel,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create data rate label */
	xfermon.w_dataRateLabel = XtVaCreateManagedWidget(
        "dataRateLabel",
        xmLabelWidgetClass,
        xfermon.w_dialog,
		XmNtopAttachment, 	XmATTACH_WIDGET,
		XmNtopWidget,		xfermon.w_lengthLabel,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create scale for displaying percentage of file moved */
	label = XmStringCreateSimple("Percent Transferred Unavailable");
	xfermon.w_scale = XtVaCreateManagedWidget(
		"scale",
		xmScaleWidgetClass,
		xfermon.w_dialog,
		XmNtitleString,			label,
		XmNshowValue,			True,
		XmNorientation,			XmHORIZONTAL,
		XmNtopAttachment,		XmATTACH_WIDGET,
		XmNtopWidget,			xfermon.w_dataRateLabel,
		XmNtopOffset,			10,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
		NULL
	);
	XmStringFree(label);

	/* Create file count */
    xfermon.w_fileCount = XtVaCreateManagedWidget(
        "fileCount",
        xmTextFieldWidgetClass,
        xfermon.w_dialog,
		XmNshadowThickness,			0,
		XmNmarginHeight,			0,
		XmNmarginWidth,				0,
		XmNtraversalOn,				False,
		XmNeditable,				False,
		XmNcursorPositionVisible,	False,
		XmNresizeWidth,				True,
		XmNtopAttachment,			XmATTACH_WIDGET,
		XmNtopWidget,				xfermon.w_scale,
		XmNtopOffset,				15,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,			5,
		XmNleftAttachment,			XmATTACH_WIDGET,
		XmNleftWidget,				xfermon.w_dataRateLabel,
		XmNleftOffset,				10,
		XmNrightAttachment,			XmATTACH_FORM,
        NULL
    );

	/* Create data rate */
    xfermon.w_dataRate = XtVaCreateManagedWidget(
        "dataRate",
        xmTextFieldWidgetClass,
        xfermon.w_dialog,
		XmNshadowThickness,			0,
		XmNmarginHeight,			0,
		XmNmarginWidth,				0,
		XmNtraversalOn,				False,
		XmNeditable,				False,
		XmNcursorPositionVisible,	False,
		XmNtopAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,				xfermon.w_dataRateLabel,
		XmNtopOffset,				-1,
		XmNleftAttachment,			XmATTACH_WIDGET,
		XmNleftWidget,				xfermon.w_dataRateLabel,
		XmNleftOffset,				5,
		XmNrightAttachment,			XmATTACH_FORM,
        NULL
    );

	/* Create file length */
    xfermon.w_length = XtVaCreateManagedWidget(
        "length",
        xmTextFieldWidgetClass,
        xfermon.w_dialog,
		XmNshadowThickness,			0,
		XmNmarginHeight,			0,
		XmNmarginWidth,				0,
		XmNtraversalOn,				False,
		XmNeditable,				False,
		XmNcursorPositionVisible,	False,
		XmNtopAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,				xfermon.w_lengthLabel,
		XmNtopOffset,				-1,
		XmNleftAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,				xfermon.w_dataRate,
		XmNrightAttachment,			XmATTACH_FORM,
        NULL
    );

	/* Create file name */
    xfermon.w_name = XtVaCreateManagedWidget(
        "name",
        xmTextFieldWidgetClass,
        xfermon.w_dialog,
		XmNshadowThickness,			0,
		XmNmarginHeight,			0,
		XmNmarginWidth,				0,
		XmNtraversalOn,				False,
		XmNeditable,				False,
		XmNcursorPositionVisible,	False,
		XmNtopAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,				xfermon.w_nameLabel,
		XmNtopOffset,				-1,
		XmNleftAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,				xfermon.w_length,
		XmNrightAttachment,			XmATTACH_FORM,
        NULL
    );

	/* Create status */
    xfermon.w_status = XtVaCreateManagedWidget(
        "status",
        xmTextFieldWidgetClass,
        xfermon.w_dialog,
		XmNshadowThickness,			0,
		XmNmarginHeight,			0,
		XmNmarginWidth,				0,
		XmNtraversalOn,				False,
		XmNeditable,				False,
		XmNcursorPositionVisible,	False,
		XmNtopAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,				xfermon.w_statusLabel,
		XmNtopOffset,				-1,
		XmNleftAttachment,			XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,				xfermon.w_name,
		XmNrightAttachment,			XmATTACH_FORM,
        NULL
    );
}


/*
 * hide_xfermon - Hide file transfer monitor dialog.
 */
hide_xfermon()
{
	/* Remove dialog from cursor linked list */
	remove_dialog_from_list(xfermon.w_dialog);

	/* Get rid of file transfer monitor */
	XtDestroyWidget(XtParent(xfermon.w_dialog));
}


/*
 * cb_xfermon_close - Callback to handle closes from window manager.
 *                    A close is not allowed.
 */
void
cb_xfermon_close(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	beep();
}


/*
 * set_xfermon_status - Set the status field of the file transfer monitor
 *                      to the character string "status".
 */
set_xfermon_status(status)
char *status;
{
	set_textfield(xfermon.w_status, status);
}


/*
 * set_xfermon_name - Set the name field of the file transfer monitor
 *                    to the character string "name".
 */
set_xfermon_name(name)
char *name;
{
	set_textfield(xfermon.w_name, name);
}


/*
 * set_xfermon_name_label - Set the label of the name field of the file
 *                          transfer monitor to the character string
 *                          label.
 */
set_xfermon_name_label(label)
char *label;
{
	XmString string;

	string = XmStringCreateSimple(label);
	XtVaSetValues(xfermon.w_nameLabel, XmNlabelString, string, NULL);
	XmStringFree(string);
}


/*
 * set_xfermon_progress - Update the file length, bytes per second, and
 *                        percent complete widgets in the file transfer
 *                        monitor.  A value of -1 for "file_len" indicates
 *                        that the file length is unknown.  A value of -1
 *                        for "file_index" indicates that the number of
 *                        bytes transferred is unknown.  "elapsed_time" is
 *                        the number of microseconds that have elapsed
 *                        since the transfer began.  Care is taken not to
 *                        update an unchanged widget value to prevent
 *                        unnecessary blinking.
 */
set_xfermon_progress(file_len, file_index, elapsed_time)
long file_len;
long file_index;
double elapsed_time;
{
	char buf[20];
	XmString prev_title;
	XmString title;
	int prev_percent;
	int percent;

	/* Display file length */
	if (file_len == -1)
		set_textfield(xfermon.w_length, "Unavailable");
	else
		set_textfield(xfermon.w_length, add_commas(file_len));

	/* Display data rate */
	if (file_index == -1)
		set_textfield(xfermon.w_dataRate, "Unavailable");
	else if (elapsed_time <= 0)
		set_textfield(xfermon.w_dataRate, "(Very Fast)");
	else {
		sprintf(buf, "%3.1f",
			(double)file_index/(elapsed_time*1000.));
		set_textfield(xfermon.w_dataRate, buf);
	}

	/* Display percent transferred */
	XtVaGetValues(xfermon.w_scale,
		XmNvalue,		&prev_percent,
		XmNtitleString,	&prev_title,
		NULL
	);
	if ((file_len == -1) || (file_index == -1)) {
		percent = 0;
		title = XmStringCreateSimple("Percent Transferred Unavailable");
	} else {
		if (file_len > 0) {
			percent = ((100.*(float)file_index)/(float)file_len);
			percent = MIN(100, percent);
		} else
			percent = 100;
		title = XmStringCreateSimple("Percent Transferred");
	}
	if (percent != prev_percent)
		XtVaSetValues(xfermon.w_scale, XmNvalue, percent, NULL);
	if (!XmStringCompare(prev_title, title)) {
		XtVaSetValues(xfermon.w_scale, XmNtitleString, title, NULL);
		XmUpdateDisplay(w_toplev);
	}
	XmStringFree(prev_title);  /* Yes, this is necessary */
	XmStringFree(title);
}


/*
 * set_xfermon_file_count - Set the file count field of the file transfer
 *                          monitor to the values "count" and "total".  If
 *                          "total" is less than zero, then display a
 *                          question mark.
 */
set_xfermon_file_count(count, total)
int count;
int total;
{
	char buf[100];

	if (total >= 0)
		sprintf(buf, "Transferred %d of %d files", count, total);
	else
		 sprintf(buf, "Transferred %d of ? files", count);
	set_textfield(xfermon.w_fileCount, buf);
}

