//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This class provides a standard API for the screen.  This file contains
  the DOS VGA implementation of the API.

  @author  Keith Wilkins and Bradford W. Mott
  @version $Id: TermDOS.cxx,v 1.3 1997/06/07 00:46:00 bwmott Exp $
*/

#include <go32.h>
#include <dpmi.h>
#include <sys/farptr.h>
#include <sys/nearptr.h>
#include <dos.h>
#include <pc.h>

#include <assert.h>
#include <stdio.h>
#include <iostream.h>
#include <stdlib.h>
#include <time.h>

#include "Error.hxx"
#include "colors.h"
#include "Term.hxx"
#include "TIA.hxx"
#include "scandef.h"

class PCJoystick
{
  public:
    // Null constructor
    PCJoystick();

  public:
    // Answer true if joysticks are present
    bool present();

    // Answers state of buttons
    uByte buttons();

    // Answers state of positions
    uByte positions();

  private:
    // Calibrate "middle" position
    void calibrate();

    // Checks to see if a joystick is attached to the system
    void detect();

  private:
    uByte myPresent;
    uWord myGamePort;
    Long myMinimum[4];
    Long myMaximum[4];
    Long myTrigger[4];
    Long myZero[4];
};

// Pointer to the joystick object
PCJoystick* theJoystick;

// Indicates if an instance is alive
bool Terminal::ourInstanceCreated = false;

// Contains the state of each of the events based on the keyboard
static volatile uLong theKeyboardEventState[TERMINAL_NUMBER_OF_EVENTS];

// Contains the state of each of the events based on the Joystick
static volatile uLong theJoystickEventState[TERMINAL_NUMBER_OF_EVENTS];

// Mouse IRQ
#define MOUSE_BIOS           0x33

// VGA Card definitions
#define VGA_BIOS             0x10
#define VGA_PEL_ADDRESS      0x03c8
#define VGA_PEL_DATA         0x03c9

static word default_term_mode;
static uWord thePixelDataTable[256];

// Indicates the width and height based on the property information
static uLong theWidth;
static uLong theHeight;
static uLong theXStart;

// Keyboard Interrupt definitions
_go32_dpmi_seginfo theOldKeyboardHandler;
_go32_dpmi_seginfo theKeyboardHandler;
static void keyboardInterruptServiceRoutine(void);


//============================================================================
// Constructor
//============================================================================
Terminal::Terminal(const Properties& properties)
    :myProperties(&properties)
{
  union REGS regs;

  assert(!ourInstanceCreated);
  ourInstanceCreated = true;

  // Clear the terminal event state
  for(int i = 0;i < TERMINAL_NUMBER_OF_EVENTS; i++) 
  {
    theKeyboardEventState[i] = 0;
    theJoystickEventState[i] = 0;
  }

  // Lets save the current video mode
  regs.h.ah = 0x0f;
  int86(VGA_BIOS, &regs, &regs);
  default_term_mode = regs.h.al;

  // Plop into 320x200x256 mode 13
  regs.w.ax = 0x0013;
  regs.h.ah = 0x00;
  int86(VGA_BIOS, &regs, &regs);
    
  // Setup the pixel data table
  for(uLong j = 0; j < 256; ++j)
  {
    thePixelDataTable[j] = j | (j << 8);
  }

  // Get the desired width and height of the display
  theWidth = myProperties->integer("Display.Width");
  theXStart = myProperties->integer("Display.XStart");
  theHeight = myProperties->integer("Display.Height");

  // Set color pallete direct to the video card 
  outp(VGA_PEL_ADDRESS, 0);
  for(int index = 0; index < 256; index++)
  {
     outp(VGA_PEL_DATA, (theColorTable[index] & 0x00ff0000) >> 18);
     outp(VGA_PEL_DATA, (theColorTable[index] & 0x0000ff00) >> 10);
     outp(VGA_PEL_DATA, (theColorTable[index] & 0x000000ff) >> 2);
  }

  // Install keyboard interrupt handler
  disable();
  _go32_dpmi_get_protected_mode_interrupt_vector(0x09, &theOldKeyboardHandler);
  theKeyboardHandler.pm_selector = _my_cs();
  theKeyboardHandler.pm_offset = (int)keyboardInterruptServiceRoutine;
  _go32_dpmi_allocate_iret_wrapper(&theKeyboardHandler);
  _go32_dpmi_set_protected_mode_interrupt_vector(0x09, &theKeyboardHandler);
  enable();

  // Initialize mouse handler via DOS interrupt
  regs.w.ax = 0x0000;
  int86(MOUSE_BIOS, &regs, &regs);

  if(regs.w.ax == 0x0000)
  {
    cerr << "WARNING: Mouse initialization failed..." << endl;
  }
  else
  {
    // Set mouse bounding box to 0,0 to 511,511
    regs.w.ax = 0x0007;
    regs.w.cx = 0;
    regs.w.dx = 511;
    int86(MOUSE_BIOS, &regs, &regs);

    regs.w.ax = 0x0008;
    regs.w.cx = 0;
    regs.w.dx = 511;
    int86(MOUSE_BIOS, &regs, &regs);
  }

  // Set joystick pointer to null
  theJoystick = 0;
}

//============================================================================
// Destructor
//============================================================================
Terminal::~Terminal()
{
  union REGS regs;

  disable();
  _go32_dpmi_set_protected_mode_interrupt_vector(0x09, &theOldKeyboardHandler);
  _go32_dpmi_free_iret_wrapper(&theKeyboardHandler);
  enable();

  // Restore previous display mode
  regs.h.ah = 0x00;
  regs.h.al = default_term_mode;
  int86(VGA_BIOS, &regs, &regs);

  // Delete the joystick object if it exists
  delete theJoystick;

  ourInstanceCreated = false;
}

//============================================================================
// Update the screen with the TIA frame
//============================================================================
void Terminal::update(TIA& tia)
{
  uByte* currentFrame = tia.currentFrameBuffer();
  uByte* previousFrame = tia.previousFrameBuffer();

  uLong width = theWidth / 4;
  uLong height = (theHeight > 200) ? 200 : theHeight;

  int offset = ((200 - height) / 2) * 320;

  // See if we can enable near pointers for updating the screen
  if(__djgpp_nearptr_enable())
  {
    // We've got near pointers enabled so let's use them
    uWord* data = (uWord*)(0xA0000 + __djgpp_conventional_base + offset)
        + ((160 - theWidth) / 2);
 
    for(uLong y = 0; y < height; ++y)
    {
      uLong* current = (uLong*)(currentFrame + theXStart);
      uLong* previous = (uLong*)(previousFrame + theXStart);
      uWord* screen = data;

      for(uLong x = 0; x < width; ++x)
      {
        if(*current != *previous)
        {
          uByte* frame = (uByte*)current;

          *screen++ = thePixelDataTable[*frame++];
          *screen++ = thePixelDataTable[*frame++];
          *screen++ = thePixelDataTable[*frame++];
          *screen++ = thePixelDataTable[*frame++];
        }
        else
        {
          screen += 4;
        }
        current++;
        previous++;
      }
      currentFrame += 160;
      previousFrame += 160;
      data += 160;
    }

    // Disable the near pointers
    __djgpp_nearptr_disable();
  }
  else
  {
    // Counldn't enable near pointers so we'll use a slower methods :-(
    uWord* data = (uWord*)(0xA0000 + offset) + ((160 - theWidth) / 2);
 
    for(uLong y = 0; y < height; ++y)
    {
      uLong* current = (uLong*)(currentFrame + theXStart);
      uLong* previous = (uLong*)(previousFrame + theXStart);
      uWord* screen = data;

      for(uLong x = 0; x < width; ++x)
      {
        if(*current != *previous)
        {
          uByte* frame = (uByte*)current;

          _farpokew(_dos_ds, (uLong)screen++, thePixelDataTable[*frame++]);
          _farpokew(_dos_ds, (uLong)screen++, thePixelDataTable[*frame++]);
          _farpokew(_dos_ds, (uLong)screen++, thePixelDataTable[*frame++]);
          _farpokew(_dos_ds, (uLong)screen++, thePixelDataTable[*frame++]);
        }
        else
        {
          screen += 4;
        }
        current++;
        previous++;
      }
      currentFrame += 160;
      previousFrame += 160;
      data += 160;
    }
  }

  // Handle any events that have occured
  handleEvents();
}

//============================================================================
// Answer the state of the given event
//============================================================================
uLong Terminal::eventState(Event event)
{
  // If the value is not set by joystick then return the keyboard's value
  if(theKeyboardEventState[(int)event])
    return theKeyboardEventState[(int)event];
  else
    return theJoystickEventState[(int)event];
}

//============================================================================
// Process any events related to the terminal
//============================================================================
void Terminal::handleEvents()
{
  union REGS regs;

  // Update paddle events to current mouse location
  regs.w.ax = 0x0003;
  int86(MOUSE_BIOS, &regs, &regs);

  theJoystickEventState[Terminal::PaddleZeroPosition] = 512 - regs.w.cx;
  theJoystickEventState[Terminal::PaddleZeroFire] = (regs.w.bx & 0x07) ? 1 : 0;

  // If no joystick object is available create one
  if(theJoystick == 0)
  {
    theJoystick = new PCJoystick;
  }

  if(theJoystick->present())
  {
    uByte buttons = theJoystick->buttons();
    uByte positions = theJoystick->positions();

    theJoystickEventState[(int)Terminal::LeftJoystickFire] = buttons & 0x03;
    theJoystickEventState[(int)Terminal::LeftJoystickUp] = positions & 0x08;
    theJoystickEventState[(int)Terminal::LeftJoystickDown] = positions & 0x04;
    theJoystickEventState[(int)Terminal::LeftJoystickLeft] = positions & 0x02;
    theJoystickEventState[(int)Terminal::LeftJoystickRight] = positions & 0x01;

    theJoystickEventState[(int)Terminal::RightJoystickFire] = buttons & 0x0C;
    theJoystickEventState[(int)Terminal::RightJoystickUp] = positions & 0x80;
    theJoystickEventState[(int)Terminal::RightJoystickDown] = positions & 0x40;
    theJoystickEventState[(int)Terminal::RightJoystickLeft] = positions & 0x20;
    theJoystickEventState[(int)Terminal::RightJoystickRight] = positions & 0x10;
  }
}

//============================================================================
// Called whenever a keyboard interrupt occurs
//============================================================================
static void keyboardInterruptServiceRoutine(void)
{
  struct Switches
  {
    uByte scanCode;
    Terminal::Event eventCode;
  };

  static Switches list[] = {
    { SCAN_DOWN,        Terminal::LeftJoystickDown },
    { SCAN_UP,          Terminal::LeftJoystickUp },
    { SCAN_LEFT,        Terminal::LeftJoystickLeft },
    { SCAN_RIGHT,       Terminal::LeftJoystickRight },
    { SCAN_ENTER,       Terminal::LeftJoystickFire },
    { SCAN_SPACE,       Terminal::LeftJoystickFire },
    { SCAN_H,           Terminal::RightJoystickDown },
    { SCAN_Y,           Terminal::RightJoystickUp },
    { SCAN_G,           Terminal::RightJoystickLeft },
    { SCAN_J,           Terminal::RightJoystickRight },
    { SCAN_Z,           Terminal::RightJoystickFire },   
    { SCAN_C,           Terminal::Color },
    { SCAN_B,           Terminal::BlackAndWhite },
    { SCAN_1,           Terminal::LeftDifficultyA },
    { SCAN_2,           Terminal::LeftDifficultyB },
    { SCAN_3,           Terminal::RightDifficultyA },
    { SCAN_4,           Terminal::RightDifficultyB },
    { SCAN_S,           Terminal::Select },
    { SCAN_R,           Terminal::Reset },
    { SCAN_P,           Terminal::Pause },
    { SCAN_Q,           Terminal::Quit }
  };

  // Get the scan code of the key
  uByte scanCode = inportb(0x60);
  
  // If it is an 0xE0 or 0 then ignore it
  if((scanCode != 0xE0) && (scanCode != 0))
  {
    // Change the event state if needed
    for(unsigned int i = 0; i < sizeof(list) / sizeof(Switches); ++i)
    {
      if(list[i].scanCode == (scanCode & 0x7f))
      {
        theKeyboardEventState[(int)list[i].eventCode] =
           (scanCode & 0x80) ? 0 : 1;
      }
    }
  }

  // Ack the interrupt
  outp(0x20,0x20);
}


//============================================================================
// Null constructor
//============================================================================
PCJoystick::PCJoystick()
    : myGamePort(0x0201)
{
  for(int t = 0; t < 4; ++t)
  {
    myMinimum[t] = 0x000FFFFF;
    myMaximum[t] = 0;
    myTrigger[t] = 0;
    myZero[t] = 0;
  }
 
  // Determine which joysticks are present
  detect();

  // Calibrate the current position of the joystick as the "middle"
  calibrate();
}

//============================================================================
// Answers true if the specified joystick is present
//============================================================================
bool PCJoystick::present()
{
  return myPresent;
}

//============================================================================
// Answer the state of the buttons
//============================================================================
uByte PCJoystick::buttons()
{
  uByte i = ~(inportb(myGamePort));
  return (i >> 4) & 0x0F;
}

//============================================================================
// Answer the state of the positions
//============================================================================
uByte PCJoystick::positions()
{
  uByte i;
  Long count[4] = {0, 0, 0, 0};
  uLong counter = 0;

  disable();
  outportb(myGamePort, 0);
  do
  {
    i = inportb(myGamePort) & 0x0F;

    for(uWord t = 0, mask = 0x01; t < 4; ++t, mask <<= 1)
    {
      if(i & mask)
      {
        ++count[t];
      }
    }
  }
  while((counter++ < 0x000FFFFF) && (i & myPresent));
  enable();

  // See if a timeout occured
  if(counter >= 0x000FFFFF)
  {
    Error err;
    err.message() << "Joystick port read has timed out.";
    err.description()
      << "The operation to read the joystick could not be completed and\n"
      << "Stella has aborted. Your joystick port may not be working \n"
      << "correctly or the joystick may have become disconnected\n";
    Throw(err);
  }

  // Do axis checking and autocalibration
  for(uWord t = 0; t < 4; ++t)
  {
    if(myPresent & (1 << t))
    {
      if(count[t] > myMaximum[t])
      {
        myMaximum[t] = count[t];
      }

      if(count[t] < myMinimum[t])
      {
        myMinimum[t] = count[t];
      }

      myTrigger[t] = ((myMaximum[t] - myMinimum[t]) >> 3) + 32;
    }
  }

  uByte result = 0;

  if(myPresent & 0x03)
  {
    if(count[0] < (myZero[0] - myTrigger[0]))
      result |= 0x02;

    if(count[0] > (myZero[0] + myTrigger[0]))
      result |= 0x01;

    if(count[1] < (myZero[1] - myTrigger[1]))
      result |= 0x08;

    if(count[1] > (myZero[1] + myTrigger[1]))
      result |= 0x04;
  }

  if(myPresent & 0x0C)
  {
    if(count[2] < (myZero[2] - myTrigger[2]))
      result |= 0x20;

    if(count[2] > (myZero[2] + myTrigger[2]))
      result |= 0x10;

    if(count[3] < (myZero[3] - myTrigger[3]))
      result |= 0x80;

    if(count[3] > (myZero[3] + myTrigger[3]))
      result |= 0x40;
  }

  return result;
}

//============================================================================
// Calibrate "middle" position
//============================================================================
void PCJoystick::calibrate()
{
  // Make sure a joystick is present before we continue
  if(myPresent)
  {
    disable();
    outportb(myGamePort, 0);

    uByte i;
    uLong counter = 0;
    do
    {
      i = inportb(myGamePort) & 0x0F;

      for(uWord t = 0, mask = 0x01; t < 4; ++t, mask <<= 1)
      {
        if(i & mask)
        {
          ++myZero[t];
        }
      }
    }
    while((counter++ < 0x000FFFFF) && (i & myPresent));

    enable();

    // See if a timeout occured
    if(counter >= 0x000FFFFF)
    {
      Error err;
      err.message() << "Joystick port read has timed out.";
      err.description()
        << "The operation to read the joystick could not be completed and\n"
        << "Stella has aborted. Your joystick port may not be working \n"
        << "correctly or the joystick may have become disconnected\n";
      Throw(err);
    }
  }
}

//============================================================================
// Determines which joysticks are presents if any
//============================================================================
void PCJoystick::detect()
{
  // Reset the game port bits to 1's
  outportb(myGamePort, 0);

  clock_t start = clock();

  // Wait for low on all four joystick bits
  do
  {
    myPresent = inportb(myGamePort) & 0x0F;
  }
  while(((clock() - start) < (CLOCKS_PER_SEC / 2)) && myPresent);

  // Setup joystick present mask that tells which joysticks are present
  myPresent = (myPresent ^ 0x0F) & 0x0F;
  myPresent = myPresent & (((myPresent & 0x03) == 0x03) ? 0x0F : 0x0C);
  myPresent = myPresent & (((myPresent & 0x0C) == 0x0C) ? 0x0F : 0x03);
}

