/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  May 1995                               */
/*-----------------------------------------------------------------------*/
/*                 Key/value List Class source file                      */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include "EST_KV.h"

template <class K, class V> KVL<K, V>::KVL(const KVL<K, V> &kv)
{
    list = kv.list;
}

template <class K, class V> 
void KVL<K, V>::clear(void)
{
    list.clear();
}

template <class K, class V> 
EST_Litem *KVL<K, V>::find_pair_key(const K &key) const
{
    EST_Litem *ptr;

    for (ptr = list.head(); ptr != 0; ptr= next(ptr))
	if (list.item(ptr).k == key)
	    return ptr;
    return 0;
}


// look for pointer kptr in list. If found, change its value to rval and
// return true, otherwise return false.

template <class K, class V> 
int KVL<K, V>::change_val(EST_Litem *kptr, const V &rval)
{
    if (list.index(kptr) == -1)
	return 0;
    else
    {
	list.item(kptr).v = rval;
	return 1;
    }
}

template <class K, class V> 
int KVL<K, V>::change_key(EST_Litem *kptr, const K &rkey)
{
    if (list.index(kptr) == -1)
	return 0;
    else
    {
	list.item(kptr).k = rkey;
	return 1;
    }
}

// look for key rkey in list. If found, change its value to rval and
// return true, otherwise return false.
template <class K, class V> 
int KVL<K, V>::change_val(const K &rkey,const V &rval)
{
    EST_Litem *ptr=find_pair_key(rkey);
    if (ptr == 0)
	return 0;
    else
    {
	list.item(ptr).v = rval;
	return 1;
    }
}

template<class K, class V> 
const V &KVL<K, V>::val(const K &rkey, bool must) const
{ 
    EST_Litem *ptr = find_pair_key(rkey);
    static V dummyV;
    if (ptr == 0)
    {
	if (must)
	    cerr << "Error: No value set for " << rkey << endl;
	return dummyV;
    }
    else
	return list.item(ptr).v;
}

template<class K, class V> 
V &KVL<K, V>::val(const K &rkey, bool must)
{ 
    EST_Litem *ptr = find_pair_key(rkey);
    static V dummyV;
    if (ptr == 0)
    {
	if (must)
	    cerr << "Error: No value set for " << rkey << endl;
	return dummyV;
    }
    else
	return list.item(ptr).v;
}

template<class K, class V> 
const V &KVL<K, V>::val_def(const K &rkey, const V &def) const
{ 
    EST_Litem *ptr = find_pair_key(rkey);
    if (ptr == 0)
	return def;
    else
	return list.item(ptr).v;
}

template<class K, class V> 
const V &KVL<K, V>::val(EST_Litem *kptr, bool must) const
{ 
    if (must == 0)
	return list.item(kptr).v;
    /* check kptr is one of mine */
    if (list.index(kptr) == -1)
    {
	cerr << "Error: No value set in KVL \n";
	static V dummyV;
	return dummyV;
    }
    else
	return list.item(kptr).v;
}

template<class K, class V> 
const K &KVL<K, V>::key(EST_Litem *kptr, int must) const
{ 
    if (must == 0)
	return list.item(kptr).k;
    if (list.index(kptr) == -1)
    {
	cerr << "Error: No key set in KVL \n";
	static K dummyK;
	return dummyK;
    }
    else
	return list.item(kptr).k;
}

template<class K, class V> 
const int KVL<K, V>::present(const K &rkey) const
{ 
    if (find_pair_key(rkey) == 0)
	return 0;
    else
	return 1;
}

// map a function over the pairs

template<class K, class V>
void KVL<K,V>::map(void (*func)(K&, V&))
{
    EST_Litem *p;
    for(p=list.head(); p; p=next(p))
    {
	KVI<K,V> item = list.item(p);
	(*func)(item.k, item.v);
    }
}

// add item to list. By default, the list is searched to see if the
// item exists already. If so, its value is overwritten. This facility
// can be turned off by setting no_search = 1;

template<class K, class V> 
int KVL<K, V>::add_item(const K &rkey, const V &rval, int no_search)
{
    if (!no_search)
	if (change_val(rkey, rval)) // first see if key exists
	    return 1;
    
    KVI<K,V>  item;
    item.k = rkey;
    item.v = rval;
    
    list.append(item);
    return 1;
}

template<class K, class V> 
int KVL<K, V>::remove_item(const K &rkey, int quiet)
{
    EST_Litem *ptr = find_pair_key(rkey);
    if (ptr == 0)
    {
	if (!quiet)
	    cerr << "KVL: no item labelled \"" << rkey << "\"" << endl;
	return -1;
    }
    else 
    {
	list.remove(ptr);
	return 0;
    }
}

template<class K, class V> KVL<K, V> &KVL<K, V>::operator = 
(const KVL<K, V> &kv)
{
    list = kv.list;
    return *this;
}

template<class K, class V> KVL<K, V> &KVL<K, V>::operator += 
(const KVL<K, V> &kv)
{
    list += kv.list;
    return *this;
}

template<class K, class V> KVL<K, V> KVL<K, V>::operator + (const KVL<K, V> &kv)
{
    KVL<K, V> result;
    result = *this;
    result += kv;
    return result;
}

template<class K, class V> 
ostream &operator<<(ostream &s, KVI< K , V > const &i)
{  return s << i.k << "\t" << i.v << "\n"; }


template<class K, class V> 
ostream& operator << (ostream& s,  KVL< K , V > const &l) 
{EST_Litem *p; 
 for (p = l.list.head(); p ; p = next(p)) 
   s << l.list(p).k << "\t" << l.list(p).v << endl; 
 return s;
} 
