/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Alan W Black                             */
/*                    Date   :  March 1998                               */
/*-----------------------------------------------------------------------*/
/*   A class for feature value pairs                                     */
/*=======================================================================*/
#ifndef __EST_FEATURES_H__
#define __EST_FEATURES_H__

#include "EST_String.h"
#include "EST_TList.h"
#include "EST_KV.h"
#include "EST_Val.h"
#include "EST_types.h"
#include "EST_Token.h"

class EST_Features {
 protected:
    KVL<EST_String, EST_Val> *features;
 public:
    static EST_Val feature_default_value;
    EST_Features();
    EST_Features(const EST_Features &f);
    ~EST_Features();

    void clear() { features->clear(); }

    const EST_Val &val(const EST_String &name) const
       {return features->val_def(name,feature_default_value);}
    const EST_Val &operator() (const EST_String &name) const 
       {return val(name);}
    const float F(const EST_String &name) const
       {return features->val_def(name,feature_default_value).Float();}
    const EST_String S(const EST_String &name) const
       {return features->val_def(name,feature_default_value).string();}
    const int I(const EST_String &name) const
       {return features->val_def(name,feature_default_value).Int();}
    const void *P(const EST_String &name) const
       {return features->val_def(name,feature_default_value).ptr();}

    // Because so often in the code the fname is a literal we support
    // const char * access too
    const EST_Val &val(const char *name, bool m=0) const;
    const EST_Val &operator() (const char *name) const {return val(name);}
    const float F(const char* name) const {return val(name).Float(); }
    const int I(const char* name) const {return val(name).Int(); }
    const EST_String S(const char* name) const {return val(name).string(); }
    const void *P(const char* name) const {return val(name).ptr(); }

    int present(const EST_String &name) const
       {return features->present(name); }

    void set(const EST_String &name, int ival)
    { EST_Val pv(ival); features->add_item(name,pv); }
    void set(const EST_String &name, float fval)
    { EST_Val pv(fval); features->add_item(name,pv); }
    void set(const EST_String &name, double fval)
    { EST_Val pv((float)fval); features->add_item(name,pv); }
    void set(const EST_String &name, const EST_String &sval)
    { EST_Val pv(sval); features->add_item(name,pv); }
    void set(const EST_String &name, const char *cval)
    { EST_Val pv(cval); features->add_item(name,pv); }
    void set(const EST_String &name,void *c,void (*f)(void *))
    { EST_Val pv(c,f); features->add_item(name,pv); }
    void set(const EST_String &name,EST_feature_function f)
    { EST_Val pv(f); features->add_item(name,pv); }
    void set_val(const EST_String &name, EST_Val sval)
    { features->add_item(name,sval); }
    
    void remove(const EST_String &name)
    { features->remove_item(name,1); }

    // Iteration
    EST_Litem *head() { return features->list.head(); }
    EST_String &fname(EST_Litem *p) { return features->list(p).k; }
    EST_Val &fval(EST_Litem *p) { return features->list(p).v; }
    
    // I/O
    EST_read_status load(EST_TokenStream &ts);
    EST_write_status save(ostream &outf) const;
    
    EST_Features& operator = (const EST_Features& a);
    friend ostream& operator << (ostream &s, const EST_Features &f)
        { f.save(s); return s; }
};

void merge_features(EST_Features &to,EST_Features &from);

#endif

