 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1995,1996                         */
 /*                        All Rights Reserved.                           */
 /*                                                                       */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*                                                                       */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                   Author :  Paul Taylor                               */
 /*                Rewritten :  Richard Caley                             */
 /* -------------------------------------------------------------------   */
 /*                  EST_Track Class header file                          */
 /*                                                                       */
 /*************************************************************************/

class EST_Track;

#ifndef __Track_H__
#define __Track_H__

#include "EST_String.h"
#include "EST_Token.h"
#include "EST_FMatrix.h"
#include "EST_types.h"
#include "EST_ChannelType.h"
#include "EST_ContourType.h"
#include "EST_TrackMap.h"
#include "EST_Features.h"

typedef enum EST_TrackFileType {
  tff_none=0,
  tff_ascii,
  tff_esps,
  tff_htk,
  tff_htk_fbank,
  tff_htk_user,
  tff_htk_discrete,
  tff_xmg,
  tff_xgraph,
  tff_ema,
  tff_est_ascii,
  tff_est_binary,
  tff_snns
} EST_TrackFileType;

typedef enum EST_InterpType {
  it_nearest,			// nearest time point
  it_linear,			// linerar interpolation
  it_linear_nz			// .. unless one end near zero
} EST_InterpType;

/** A class for storing time aligned coefficients.

some stuff.
*/

class EST_Track {

protected:
  EST_FMatrix p_values;		// float x array 
  EST_FVector p_times;	        // float y array 
  EST_CVector p_is_val;		// for breaks and non-breaks
  float p_t_offset;			// time shift.
    
  EST_String p_name;			// name of whole thing
  EST_TrackMap::P p_map;
  EST_StrVector p_channel_names;	// name of each track

  bool p_equal_space;			// fixed or variable frame rate
  bool p_single_break;			// single break lots between data 

  EST_ContourType p_contour_type;	// eg F0, "default" etc.
  EST_TrackFileType p_file_type;        // orig file type (could be feature)

  void default_vals();
  void default_channel_names();
  void clear_arrays();
  void pad_breaks();		     // put in extra breaks 
    
  int interp_value(float x, float f);
  float interp_amp(float x, int c, float f);
  float estimate_shift(float x);
  void copy(const EST_Track& a);

public:
  static const float default_frame_shift;
  static const int default_sample_rate;

  EST_Track();
  EST_Track(const EST_Track &a);
  EST_Track(int num_frames, int num_channels);
  EST_Track(int num_frames, EST_TrackMap &map);

  void copy_setup(const EST_Track& a); // copy everything but data
  ~EST_Track();

  EST_String name() const {return p_name;}
  void set_name(EST_String n) {p_name = n;}

  EST_Features f;

  /**@name Functions for accessing parts of tracks
   */

  //@{

  /** Set {\tt fv} to frame {\tt n}.
   */

  void frame(EST_TVector<float> &fv, int n, int startf=0, int nf=EST_ALL)
    { p_values.row(fv, n, startf, nf); }
//  EST_FVector * frame(EST_FVector &fv, int n, int startf=0, int nf=EST_ALL)
//    { p_values.row(fv, n, startf, nf); 
//      return &fv;}

  void channel(EST_FVector &cv, int n, int startf=0, int nf=EST_ALL)
    { p_values.column(cv, n, startf, nf); }

  void sub_track(EST_Track &st,
		 int start_frame=0, int nframes=EST_ALL,
		 int start_chan=0, int nchans=EST_ALL);

  void sub_track(EST_Track &st,
		 int start_frame=0, int nframes=EST_ALL,
		 int start_chan=0, int nchans=EST_ALL) const
    { ((EST_Track *)this)->sub_track(st, start_frame, nframes, start_chan, nchans); } 

  void copy_sub_track(EST_Track &st,
		      int start_frame=0, int nframes=EST_ALL,
		      int start_chan=0, int nchans=EST_ALL) const;
  //@}

  void copy_channel(int n, float *buf, int offset=0, int num=EST_ALL) const
    { p_values.copy_column(n, buf, offset, num); } 
  void copy_frame(int n, float *buf, int offset=0, int num=EST_ALL) const
    {  p_values.copy_row(n, buf, offset, num); } 

  void set_channel(int n, const float *buf, int offset=0, int num=EST_ALL)
    { p_values.set_column(n, buf, offset, num); }
  void set_frame(int n, const float *buf, int offset=0, int num=EST_ALL)
    { p_values.set_row(n, buf, offset, num); }

  void set_channel(int c, 
		   const EST_Track &from, int from_c, int from_offset=0,
		   int offset=0, int num=EST_ALL)
    { p_values.set_column(c, from.p_values, from_c, from_offset, offset, num); }
  void set_frame(int i, 
		 const EST_Track &from, int from_f, int from_offset=0, 
		 int offset=0, int num=EST_ALL)
    { p_values.set_row(i, from.p_values, from_f, from_offset, offset, num); }

  /**@name TrackMap manipulation.
    */
  //@{
  // assign a known description to a track
  void assign_map(EST_TrackMap::P map);
  void assign_map(EST_TrackMap &map) { assign_map(&map); }

  // create a description for an unknown track
  void create_map(EST_ChannelNameMap &names);
  void create_map(void) { create_map(EST_default_channel_names); }

  EST_TrackMap::P map() const { return p_map; }
  //@}

  // find a channel, return position or -1 if it's not there.
  int channel_position(const char *name, int offset=0) const;
  int channel_position(EST_String name, int offset=0) const
    { return  channel_position((const char *)name, offset); }
  int channel_position(EST_ChannelType type, int offset=0) const;

  // just look to see if it's there
  bool has_channel(EST_ChannelType type) const 
    { int cp = channel_position(type);
      return cp>=0; }
  bool has_channel(const char *name) const
    { return channel_position(name) >=0; }
  bool has_channel(EST_String name) const 
    { return has_channel((const char *)name); }

  // Basic value access

  // return amplitude of point i, channel c
  float &a(int i, int c=0);
  float a(int i, int c=0) const;
  // return amplitude of point i, channel c with no bounds checking
  float &a_no_check(int i, int c=0) { return p_values.a_no_check(i,c); }
  float a_no_check(int i, int c=0) const { return p_values.a_no_check(i,c); }

  // return amplitude of point i, channel type c (plus offset)
  float &a(int i, EST_ChannelType c, int offset=0);
  float  a(int i, EST_ChannelType c, int offset=0) const
    { return ((EST_Track *)this)->a(i,c, offset); }

  // return amplitude of point i, channel named name (plus offset)
  float &a(int i, const char *name, int offset=0);
  float  a(int i, const char *name, int offset=0) const
    { return ((EST_Track *)this)->a(i, name, offset); }
  float &a(int i, EST_String name, int offset=0) 
    { return a(i, (const char *)name, offset); }
  float  a(int i, EST_String name, int offset=0) const
    { return ((EST_Track *)this)->a(i, (const char *)name, offset); }

  // return amplitude at time t, channel c
  float &a(float t, int c=0, EST_InterpType interp=it_nearest);	
  float  a(float t, int c=0, EST_InterpType interp=it_nearest) const
    { return ((EST_Track *)this)->a(t, c, interp); }	

  // return amplitude at time t, channel type c
  float &a(float t, EST_ChannelType c, EST_InterpType interp=it_nearest);
  float  a(float t, EST_ChannelType c, EST_InterpType interp=it_nearest) const
    { return ((EST_Track *)this)->a(t, c, interp); }

  // return amplitude at point i
  float &operator() (int i, int c)       { return a(i,c); }	
  float &operator() (int i)              { return a(i,0); }
  float  operator() (int i, int c) const { return a(i,c); }	
  float  operator() (int i) const        { return a(i,0); }	

  float &operator() (int i, EST_ChannelType c)       { return a(i,c); }
  float  operator() (int i, EST_ChannelType c) const { return a(i,c); }
  
  // return amplitude at time t
  float &operator() (float t, int c)       {return a(t,c); }	
  float &operator() (float t)              {return a(t,0); }
  float  operator() (float t, int c) const {return a(t,c); }	
  float  operator() (float t) const        {return a(t,0); }

  // return time position of point i
  float &t(int i)			   { return p_times[i]; }
  float  t(int i) const                    { return p_times(i); } 

  // return position at i in milli-seconds.
  float ms_t(int i) const		   { return p_times(i) * 1000.0; }

  float &t_offset()			   { return p_t_offset; }
  float t_offset() const		   { return p_t_offset; }

  EST_read_status load(const EST_String name, float ishift = 0.0);
  EST_read_status load(EST_TokenStream &ts, float ishift = 0.0);

  EST_read_status load(const EST_String name, const EST_String type, float ishift = 0.0);
  EST_write_status save(const EST_String name, 
			const EST_String EST_filetype = "");

  EST_write_status save(FILE *fp,
			const EST_String EST_filetype = "");

  EST_read_status load_channel_names(const EST_String name);
  EST_write_status save_channel_names(const EST_String name);

  void clear() {;}
  int empty() const;
    
  void set_break(int i);		// set point i to be a break
  void set_value(int i);		// set point i to be a value
  int val(int i) const;		// is point i a value?
  int track_break(int i) const;	// is point i a break?
  int prev_non_break(int i) const;
  int next_non_break(int i) const;
  int index(float x) const;		// return index of time value x in the time array
  int index_below(float x) const;	// return index below time value x in the time array
    
  int num_frames() const {return p_values.num_rows();}
  int length() const { return num_frames(); }
  int num_channels() const {return p_values.num_columns();}

  void set_num_channels(int n) { resize(EST_CURRENT, n); }
  void set_num_frames(int n)   { resize(n, EST_CURRENT); }

  const EST_String channel_name(int channel, const EST_ChannelNameMap &map, int strings_override=1) const;
  const EST_String channel_name(int channel, int strings_override=1) const 
    { return channel_name(channel, EST_default_channel_names, strings_override); }

  void set_channel_name(const EST_String &name, int channel);

  /** fill time points at regular intervals of time {\tt t}.
   */
  void fill_time(float t, int start =1);// Fill time axis at t intervals
  void fill(float v) { p_values.fill(v); }
  void fill() { p_values.fill(); }

  void sample(float shift);    // sample contour at "shift" spacing

  void change_type(float nshift, bool single_break);

  float shift() const;	//return frame spacing.
  float start() const;	//return time of first meaningful point.
  float end() const;		//return time value of last meaningful point.

  EST_Track& operator = (const EST_Track& a);
  EST_Track& operator+=(const EST_Track &a); // add to existing track
  EST_Track& operator|=(const EST_Track &a); // add to existing track in parallel
  friend ostream& operator << (ostream& s, const EST_Track &tr);

  void add_trailing_breaks();
  void rm_trailing_breaks();
  void rm_excess_breaks();	     // reduce to a single break

  void resize(int num_frames, int num_channels);
  void resize(int num_frames, EST_TrackMap &map);

  bool equal_space() const {return p_equal_space;}
  bool single_break() const {return p_single_break;}
  EST_ContourType contour_type() const {return p_contour_type;}
  EST_TrackFileType file_type() const {return p_file_type;}
  void set_equal_space(bool t) {p_equal_space = t;}
  void set_file_type(EST_TrackFileType t) {p_file_type = t;}
  void set_single_break(bool t) {p_single_break = t;}
  void set_contour_type(const EST_ContourType &t) {p_contour_type = t;}


public:
  // Generic itteration interface, used by e.g. EST_TTimeIndex  
  typedef int Index;
  Index first_index() const { return 0; }
  Index next_index(Index i) const { return i+1; }
  static Index bad_index() { return -1; }
  bool  valid_index(Index i) const { return i>=0 && i<num_frames(); }
  float time_of(Index i) const { return t(i); }
};

// list of tracks in serial
typedef EST_TList<EST_Track> EST_TrackList;


#endif /* __Track_H__ */
