/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Alan W Black                             */
/*                    Date   :  May 1996                                 */
/*-----------------------------------------------------------------------*/
/*            A class for representing ints floats and strings           */
/*                                                                       */
/*=======================================================================*/
#ifndef __EST_VAL_H__
#define __EST_VAL_H__

#include "EST_String.h"
#include "EST_Contents.h"

/* A class so ints floats and Strings may be happily used as one type */
enum EST_val_type {val_unset, val_int, val_float, val_string, val_ptr,
                   val_func };

class EST_Item;   //  have to know way beyond here
class EST_Val;
typedef EST_Val (*EST_feature_function)(EST_Item *);

class EST_Val {
  private:
    EST_val_type t;
    union 
    { int ival;
      float fval; 
      void *pval;} v;
    EST_String sval;
    const int to_int() const;
    const float to_flt() const;
    const EST_String &to_str() const;
  public:
    EST_Val() {t=val_unset;}
    EST_Val(const EST_Val &val);
    EST_Val(const int i) {t=val_int; v.ival=i;}
    EST_Val(const float f) {t=val_float; v.fval=f;}
    EST_Val(const double d) {t=val_float; v.fval=d;}
    EST_Val(const EST_String &s) {t=val_string; sval = s;}
    EST_Val(const char *s) {t=val_string; sval = s;}
    EST_Val(void *p, void (*f)(void *));
    EST_Val(EST_feature_function f);
    ~EST_Val(void);
    const EST_val_type type(void) const {return t;}
    const int Int(void) const {if (t==val_int) return v.ival; return to_int();}
    const float Float(void) const {if (t==val_float) return v.fval; return to_flt();}
    const void *ptr(void) const;
    const EST_feature_function function(void) const 
       {return (EST_feature_function)v.pval;}
    const EST_String &string(void) const
       {if (t!=val_string) to_str(); return sval;}
    const EST_String &string_only(void) const {return sval;}
    EST_Val &operator=(const int i) { t=val_int; v.ival=i; return *this;}
    EST_Val &operator=(const float f) { t=val_float; v.fval=f; return *this;}
    EST_Val &operator=(const double d) { t=val_float; v.fval=d; return *this;}
    EST_Val &operator=(const EST_String &s) { t=val_string; sval = s; return *this;}
    int operator ==(const EST_Val &a) const
    { if (t != a.t) return (1==0);
      else if (t == val_string) return (sval == a.sval);
      else if (t == val_int) return (v.ival == a.v.ival);
      else if (t == val_float) return (v.fval == a.v.fval);
      else return (1==0); }
    int operator ==(const EST_String &a) const { return (string() == a); }
    int operator ==(const char *a) const { return (string() == a); }
    int operator ==(const int &i) const { return (Int() == i); }
    int operator ==(const float &f) const { return (Float() == f); }
    int operator ==(const double &d) const { return (Float() == d); }
    int operator !=(const EST_Val &a) const { return (!(*this == a)); }
    int operator !=(const EST_String &a) const { return (string() != a); }
    int operator !=(const char *a) const { return (string() != a); }
    int operator !=(const int &i) const { return (Int() != i);}
    int operator !=(const float &f) const { return (Float() != f); }
    int operator !=(const double &d) const { return (Float() != d); }
    operator int() const { return Int(); }
    operator float() const { return Float(); }
    operator EST_String() const { return string(); }
    friend ostream& operator << (ostream &s, const EST_Val &a)
    { if (a.t == val_unset) s << "[PVAL unset]" ;
      else if (a.t == val_int)	  s << a.v.ival;
      else if (a.t == val_float) s << a.v.fval;
      else if (a.t == val_string) s << a.sval;
      else if (a.t == val_ptr) s << "[PVAL contents]";
      else if (a.t == val_func) s << "[PVAL function]";
      return s;
    }
    void set_ptr(void *p, void (*f)(void *));
    void set_function(EST_feature_function f);
    EST_Val &operator=(const EST_Val &c);
    EST_Val &operator=(const EST_feature_function f);
};

#endif 
