/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Simon King                              */
/*                     Date   :  July 1995                               */
/*-----------------------------------------------------------------------*/
/*                 Compute delta coefficients for                        */
/*                        Tracks and Tracks                                */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include "EST_Track.h"
#include "EST_Wave.h"
/// max. number of points on which the delta co-eff is based
# define MAX_DELTA_ORDER 2
# define MAX_REGRESSION_LENGTH 4

static float compute_gradient(const EST_FVector &x, int num_points);

void delta(EST_Track &tr, EST_Track &d, int regression_length)
{
    int reg_index, this_index;
    
    // need at least two points to compute gradient
    if ((regression_length < 2)||(regression_length > MAX_REGRESSION_LENGTH)){
	cerr << "delta(EST_Track&, int) : ERROR : regression_length is "
	    << regression_length << endl;
	exit(0);
    }
    
    // temp stores the points passed to compute_gradient
    EST_FVector temp(regression_length);
    
    for (int j = 0; j < tr.num_channels(); j++ )
	for (int i = 0; i < tr.num_frames(); i++)
	{
	    // copy values needed to compute gradient into temp[]
	    for (reg_index=0; reg_index<regression_length; reg_index++)
	    {
		// gradient is computed from points to left of current time
		// rather than points centred around the current time
		this_index = i - reg_index;
		if (this_index >= 0)
		    temp[reg_index] = tr.a(this_index, j);
	    }
	    
	    // gradient at frame 0 is defined as 0
	    if (i < 1)
		d.a(i, j) = 0.0;
	    else if (i < regression_length - 1)
		// enough data, but would prefer more
		// number of points available is only i+1
		d.a(i, j) = compute_gradient(temp, i + 1);
	    else
		// plenty of data, use the last regression_length points
		d.a(i, j) = compute_gradient(temp, regression_length);
	}
}

void delta(EST_Wave &tr, EST_Wave &d, int regression_length)
{
    int reg_index, this_index;
    
    // need at least two points to compute gradient
    if ((regression_length < 2)||(regression_length > MAX_REGRESSION_LENGTH)){
	cerr << "delta(EST_Track&, int) : ERROR : regression_length is "
	    << regression_length << endl;
	exit(0);
    }
    
    // temp stores the points passed to compute_gradient
    EST_FVector temp(regression_length);
    
    for (int j = 0; j < tr.num_channels(); j++ )
	for (int i = 0; i < tr.num_samples(); i++)
	{
	    // copy values needed to compute gradient into temp[]
	    for (reg_index=0; reg_index<regression_length; reg_index++)
	    {
		// gradient is computed from points to left of current time
		// rather than points centred around the current time
		this_index = i - reg_index;
		if (this_index >= 0)
		    temp[reg_index] = (float)tr.a(this_index, j);
	    }
	    
	    // gradient at frame 0 is defined as 0
	    if (i < 1)
		d.a(i, j) = 0;
	    else if (i < regression_length - 1)
		// enough data, but would prefer more
		// number of points available is only i+1
		d.a(i, j) = (short)compute_gradient(temp, i + 1);
	    else
		// plenty of data, use the last regression_length points
		d.a(i, j) = (short)compute_gradient(temp, regression_length);
	}
}

/*void delta(EST_Track &tr, int regression_length)
{
    // use regression_length points (including one at current time)
    // to estimate delta-coefficients
    
    int this_coeff, frame_index, reg_index, this_index;
    int i, j;
    float **x;
    
    // need at least two points to compute gradient
    if ((regression_length < 2)||(regression_length > MAX_REGRESSION_LENGTH)){
	cerr << "delta(EST_Track&, int) : ERROR : regression_length is "
	    << regression_length << endl;
	exit(0);
    }
    
    x = new float*[tr.num_frames()];
    for (i = 0; i < tr.num_frames(); ++i) x[i] = new float[tr.num_channels()];
    
    // temp stores the points passed to compute_gradient
    float *temp = new float[regression_length];
    
    // append _d to name of EST_tr
    tr.set_name( tr.name() + "_d" );
    
    // do one co-eff at a time
    for(this_coeff=0; this_coeff<tr.num_channels(); this_coeff++ ){
	
	// append _d to field name
	tr.set_channel_name( EST_String(tr.channel_name(this_coeff) + "_d"), 
			    this_coeff);
	
	// go through frame by frame
	for(frame_index=0; frame_index<tr.num_frames(); frame_index++){
	    
	    // copy values needed to compute gradient into temp[]
	    for(reg_index=0; reg_index<regression_length; reg_index++){
		
		// gradient is computed from points to left of current time
		// rather than points centred around the current time
		this_index = frame_index - reg_index;
		
		if(this_index >= 0)
		    temp[reg_index] = tr(this_index, this_coeff);
		
	    }			// reg_index
	    
	    // gradient at frame 0 is defined as 0
	    if(frame_index < 1){
		x[frame_index][this_coeff] = 0.0;
		
	    }else if(frame_index < regression_length-1){ 
		// enough data, but would prefer more
		
		// number of points available is only frame_index+1
		x[frame_index][this_coeff] = 
		    compute_gradient(temp,frame_index+1);
		
	    }else{
		// plenty of data, use the last regression_length points
		x[frame_index][this_coeff] = 
		    compute_gradient(temp,regression_length);
	    }
	}			// loop in frame_index
    }			// loop in this_coeff
    
    // copy data back into EST_Track
    for (i = 0; i < tr.num_frames(); ++i){
	
	for (j = 0; j < tr.num_channels(); ++j)
	    tr.a(i, j) = x[i][j];
	
	// free temporary storage
	delete [] x[i];
	
    }
    delete [] x;
    delete [] temp;
}
*/
// REORG - delete
/*
   void delta(EST_Track &track, int regression_length)
   {
   // use regression_length points (including one at current time)
   // to estimate delta-coefficients
   
   int frame_index, reg_index, this_index, i;
   
   // need at least two points to compute gradient
   if((regression_length < 2)|| (regression_length > MAX_REGRESSION_LENGTH) ){
   cerr << "delta(EST_Track, int) : ERROR : regression_length is " 
   << regression_length << endl;
   exit(0);
   }
   
   // temporary to hold new values
   float *a = new float[track.num_frames()];
   
   // temp stores the points passed to compute_gradient
   float *temp = new float[regression_length];
   
   // append _d to name of EST_Track
   track.set_contour_type( track.get_contour_type() + "_d" );
   
   // go through frame by frame
   for(frame_index=0; frame_index<track.num_frames(); frame_index++){
   
   // copy values needed to compute gradient into temp[]
   for(reg_index=0; reg_index<regression_length; reg_index++){
   
   // gradient is computed from points to left of current time
   // rather than points centred around the current time
   this_index = frame_index - reg_index;
   
   if(this_index >= 0)
   temp[reg_index] = track.a(this_index);
   
   }			// reg_index
   
   // gradient at frame 0 is defined as 0
   if(frame_index < 1){
   a[frame_index] = 0.0;
   
   }else if(frame_index < regression_length-1){ 
   // enough data, but would prefer more
   
   // number of points available is only frame_index+1
   a[frame_index] = compute_gradient(temp,frame_index+1);
   
   }else{			// plenty of data, use the last regression_length points
   a[frame_index] = compute_gradient(temp,regression_length);
   }
   }			// loop in frame_index
   
   // overwrite existing a with new values
   for (i = 0; i < track.num_frames(); ++i)
   track.a(i) = a[i];
   
   delete [] a;
   delete [] temp;
   }
   */

static float compute_gradient(const EST_FVector &x, int num_points)
{
    float gradient;
    
    // NB x[0] is the point LATEST in time
    // so x[1] is really x[t-1]
    
    // time between points is assumed to be one unit
    
    // these are solutions to least-sqaures fit of straight line
    // to num_points points
    switch (num_points){
	
    case 1:
	gradient = 0.0;
	break;
	
    case 2:
	gradient = x(0) - x(1);
	break;
	
    case 3:
	gradient = (x(0) -x(2)) / 2.0;
	break;
	
    case 4:
	gradient = ( (3.0*x(0)) + x(1) - x(2) - (3.0 * x(3)) ) / 10.0;
	break;
	
    default:
	
	cerr << "compute_gradient(float*, int) : ERROR : num_points is" 
	    << num_points << endl;
	
	exit(0);
	break;
    }
    
    return gradient;
}

