/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors: Paul Taylor and Simon King                   */
/*                 Date   :  March 1998                                  */
/*-----------------------------------------------------------------------*/
/*    Signal processing functions which operate on entire utterances     */
/*                                                                       */
/*=======================================================================*/


#include "EST_error.h"
#include "EST_track_aux.h"
#include "EST_inline_utils.h"
#include "sigpr/EST_fft.h"
#include "sigpr/EST_lpc.h"
#include "sigpr/EST_sigpr_frame.h"
#include "sigpr/EST_sigpr_utt.h"
#include "sigpr/EST_sig2fv.h"

int get_frame_size(EST_Track &pms, 
			 int i, int sample_rate, int prefer_prev)
{
    int prev = -1;
    int next = -1;
    
    if (i>0)
	prev = irint((pms.t(i) - pms.t(i-1))*sample_rate);
    if (i<pms.num_frames()-1)
	next = irint((pms.t(i+1) - pms.t(i))*sample_rate);
    
    if (prefer_prev)
	return prev>=0?prev:(next>=0?next:0);
    return next>=0?next:(prev>=0?prev:0);
}

float get_time_frame_size(EST_Track &pms, int i, int prefer_prev)
{
    float prev = -1;
    float next = -1;
    
    if (i > 0)
	prev = pms.t(i) - pms.t(i-1);
    if (i < pms.num_frames() -1)
	next = pms.t(i+1) - pms.t(i);
    
    if (prefer_prev)
	return prev>=0 ? prev: (next>=0 ? next : 0.0);
    return next>=0 ? next: (prev>=0 ? prev : 0.0);
}

/*void sig2lpc(EST_Wave &sig, EST_Track &lpc, EST_WindowFunc *wf, float factor)
{
    int order = lpc.num_channels() - 1;
    EST_FVector coefs(order + 1);
    int k;
    int window_start, window_size, length; // can be merged with window_size

    int sample_rate = sig.sample_rate();

    EST_FVector frame;
    
    for (k = 0; k < lpc.num_frames(); ++k)
    {
	int pos = irint(lpc.t(k) * sample_rate);
	
	length = get_local_frame_size(lpc, k, sig.sample_rate());
	window_size = irint(length * factor);
	window_start = pos - (window_size/2);

	EST_Window::window_signal(sig, wf, window_start, 
				  window_size, frame, 1);

	lpc.frame(coefs, k);
	sig2lpc(frame, coefs);
    }
    lpc.save("test.est", "est");
}   
*/

/*typedef void EST_FrameFunc(const EST_FVector &in_frame, 
			   EST_FVector &out_frame);

void sig2coef(EST_Wave &sig, EST_Track &lpc, EST_WindowFunc *wf, 
	      EST_FrameFunc *ff, float factor)
{
    EST_FVector coefs, frame;
    int start, size;
    
    for (int k = 0; k < lpc.num_frames(); ++k)
    {
	size = irint(get_local_frame_size(lpc, k, sig.sample_rate())* factor);
	start = (irint(lpc.t(k) * sig.sample_rate()) - (size/2));

	EST_Window::window_signal(sig, wf, start, size, frame, 1);

	lpc.frame(coefs, k);
	(*ff)(frame, coefs);
    }
} 
*/  

void sig2coef(EST_Wave &sig, EST_Track &tr, EST_String type, 
	      float factor, EST_WindowFunc *wf)
{
    EST_FVector coefs, frame;
    int start, size;

//    cout << "TYPE IS " << type << endl;

    for (int k = 0; k < tr.num_frames(); ++k)
    {
	size = irint(get_frame_size(tr, k, sig.sample_rate())* factor);
	start = (irint(tr.t(k) * sig.sample_rate()) - (size/2));

	EST_Window::window_signal(sig, wf, start, size, frame, 1);

	tr.frame(coefs, k);
	frame_convert(frame, "sig", coefs, type);
    }
}   

void power(EST_Wave &sig, EST_Track &pow, float factor)
{
    EST_FVector frame;
    int window_start, window_size, pos, k;

    EST_WindowFunc *wf =  EST_Window::creator("rectangular");

    for (k = 0; k < pow.num_frames(); ++k)
    {
	pos = irint(pow.t(k) * sig.sample_rate());
	window_size = irint(get_frame_size(pow, k, sig.sample_rate())
						 * factor);
	window_start = pos - window_size/2;
	EST_Window::window_signal(sig, wf, window_start, window_size,frame, 1);

	sig2pow(frame, pow.a(k));
    }
}

void energy(EST_Wave &sig, EST_Track &pow, float factor)
{
    EST_FVector frame;
    int window_start, window_size, pos, k;

    EST_WindowFunc *wf =  EST_Window::creator("rectangular");

    for (k = 0; k < pow.num_frames(); ++k)
    {
	pos = irint(pow.t(k) * sig.sample_rate());
	window_size = irint(get_frame_size(pow, k, sig.sample_rate())
						 * factor);
	window_start = pos - window_size/2;
	EST_Window::window_signal(sig, wf, window_start, window_size,frame,1);

	sig2rms(frame, pow.a(k));
    }
}

static EST_String determine_type(const EST_String &intype)
{
    return (intype.contains("_") ? intype.before("_"): intype);
}

void convert_track(EST_Track &in_track, EST_Track &out_track, 
		   const EST_String &out_type, const EST_String &in_type)
{
    if (in_track.num_frames() != out_track.num_frames())
	EST_error("In track has %d frames, out track has %d\n", 
		  in_track.num_frames(), out_track.num_frames());

    EST_String tmp;
    tmp = ((in_type == "") ? determine_type(in_track.channel_name(0)):in_type);

    EST_FVector in_frame(in_track.num_channels());
    EST_FVector out_frame(out_track.num_channels());

    for (int i = 0; i < in_track.num_frames(); ++i)
    {
	in_track.frame(in_frame, i);
	out_track.frame(out_frame, i);
	frame_convert(in_frame, tmp, out_frame, out_type);
    }
}



void fbank(EST_Wave &sig,
	   EST_Track &fbank_track,
	   const float factor,
	   EST_WindowFunc *wf,
	   const bool use_power_rather_than_energy,
	   const bool take_log)
{

    // still to add : high/low pass filtering

    int window_start, window_size, pos, k;
    EST_FVector frame,fbank_frame;

    // get_order(...) gives wrong answer ... Paul ?
    int fbank_order = fbank_track.num_channels();

    // sanity check
    if(fbank_order < 1)
    {
	EST_error("Filterbank order of %i makes no sense.\n",fbank_order);
	return;
    }

    for (k = 0; k < fbank_track.num_frames(); ++k)
    {
	window_size = irint(get_frame_size(fbank_track, k, sig.sample_rate())
			    * factor);
	pos = irint(fbank_track.t(k) * sig.sample_rate());
	window_start = pos - window_size/2;
	EST_Window::window_signal(sig, wf, window_start, window_size,frame, 1);

	fbank_track.frame(fbank_frame,k);
	sig2fbank(frame,fbank_frame,sig.sample_rate(),
		  use_power_rather_than_energy,take_log);

    }


}


void melcep(EST_Wave &sig, EST_Track &mfcc_track,
	    float factor,
	    int fbank_order,
	    float liftering_parameter,
	    EST_WindowFunc *wf,
	    const bool include_c0,
	    const bool use_power_rather_than_energy)
{

    EST_FVector frame,mfcc_frame,fbank_frame;
    int k;

    // first, do filterbank analysis
    // need a temporary track, with the same setup as mfcc_track
    EST_Track fbank_track;

    make_timed_track(mfcc_track, fbank_track, "filter", fbank_order, 0);

    // 'true' makes fbank(...) take logs
    fbank(sig,fbank_track,factor,wf,use_power_rather_than_energy,true);
    
    for (k = 0; k < mfcc_track.num_frames(); ++k)
    {

	mfcc_track.frame(mfcc_frame,k);
	fbank_track.frame(fbank_frame,k);

	fbank2melcep(fbank_frame, mfcc_frame,liftering_parameter,include_c0);
    }
}
