/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Paul Taylor and Alan Black                      */
/*             Date   :  June 1996                                       */
/*-----------------------------------------------------------------------*/
/*             more EST_Wave class methods                                   */
/*                                                                       */
/*=======================================================================*/

#include <string.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "EST_cutils.h"
#include "EST_string_aux.h"
#include "EST_Wave.h"
#include "EST_wave_utils.h"
#include "EST_wave_aux.h"
#include "EST_io_aux.h"

void extract(EST_Wave &sig, EST_Option &al);

void differentiate(EST_Wave &sig)
{
    for (int i = 0; i < sig.num_samples() -1; ++i)
	sig.a(i) = sig.a(i + 1) - sig.a(i);
    sig.resize(sig.num_samples()-1);
}

int wave_extract_channel(EST_Wave &single, const EST_Wave &multi, int channel)
{
  if (&single == &multi)
    {
      // some nasty person has passed us the same wave for output and input.
      EST_Wave tmp;
      int v = wave_extract_channel(tmp, multi, channel);
      if(v==0)
	single.copy(tmp);
      return v;
    }

    int c = multi.num_channels();
    
    if (channel < 0 || channel >= c)
    {
	cerr << "Can't extract channel " << channel << " from " << 
	    c << " channel waveform\n";
	return -1;
    }

    EST_Wave subwave;

    multi.sub_wave(subwave, 0, EST_ALL, channel, 1);

    single.copy(subwave);

    return 0;
}

void wave_combine_channels(EST_Wave &s,const EST_Wave &m)
{
    if (&s == &m)
    {
	// some nasty person has passed us the same wave for output and input.
	EST_Wave tmp;
	wave_combine_channels(tmp,m);
	s = tmp;
	return;
    }
    s.resize(m.num_samples(), 1, FALSE);
    s.set_sample_rate(m.sample_rate());
    
    for(int i=0; i<m.num_samples(); i++)
    {
	double sum=0.0;
	for(int j=0; j<m.num_channels(); j++)
	    sum += m.a(i,j);
	s.a(i,0) = (int)(sum/m.num_channels() + 0.5);
    }
}

void add_waves(EST_Wave &s, const EST_Wave &m)
{
    int new_samples = Gof(s.num_samples(), m.num_samples());
    int new_channels = Gof(s.num_channels(), m.num_channels());
    
    s.resize(new_samples, new_channels, 1);
    
    for (int i = 0; i < m.num_samples(); i++)
	for (int j = 0; j < m.num_channels(); j++)
	    s.a(i, j) += m.a(i, j);
}

void invert(EST_Wave &sig)
{
    sig.rescale(-1.0);
}

void reverse(EST_Wave &sig)
{
    int i, n;
    short t;
    n = (int)floor((float)(sig.num_samples())/2.0);
    
    for (i = 0; i < n; ++i)
    {
	t = sig.a(i);
	sig.a(i) = sig.a(sig.num_samples() - 1 -i);
	sig.a(sig.num_samples() - 1 -i) = t;
    }
}

void extract(EST_Wave &sig, EST_Option &al)
{
    int from, to;
    EST_Wave sub_wave, w2;
    
    if (al.present("-start"))
	from = (int)(sig.sample_rate() * al.fval("-start"));
    else if (al.present("-from"))
	from = al.ival("-from");
    else
	from = 0;
    
    if (al.present("-end"))
	to = (int)(sig.sample_rate() * al.fval("-end"));
    else if (al.present("-to"))
	to = al.ival("-to");
    else
	to = sig.num_samples() - 1;
    
    sig.sub_wave(sub_wave, from, to - from);
    w2 = sub_wave;
    sig = w2;
}

void gc_wave(void *w)
{
    delete (EST_Wave *)w;
}

void wave_info(EST_Wave &w)
{
    EST_String t;
    cout << "Duration: " << 
	ftoString((float)w.num_samples()/(float)w.sample_rate(),4,1) << endl;
    
    cout << "Sample rate: " << w.sample_rate() << endl;
    cout << "Number of samples: " << w.num_samples() << endl;
    cout << "Number of channels: " << w.num_channels() << endl;
    cout << "Header type: " << w.file_type() << endl;
    cout << "Data type: " << w.sample_type() << endl;
}

static EST_String options_wave_filetypes(void)
{
    // Returns list of currently support wave file types
    // Should be extracted from the list in EST_Wave (but that's
    // not very clear :-(
    
    return "nist, est, esps, snd, riff, aiff, audlab, raw, ascii";
}

EST_String options_subwave(void)
{
    return
	EST_String("")+
	    "-start <float>   Extract sub-wave starting at this time\n"+
		"-end   <float>   Extract sub-wave ending at this time\n"+
		    "-from  <int>     Extract sub-wave starting at this frame position\n"+
			"-to    <int>     Extract sub-wave ending at this frame position\n";
}
EST_String options_wave_input(void)
{
    // The standard waveform input options 
    return
	EST_String("")+
	    "-itype <string>  Input file type (optional).  If no type is\n"+
		"                 specified type is automatically derived from\n"+
		    "                 file's header, in unheaders and not itype file\n"+
			"                 is assumed to be raw (shorts).  Supported types\n"+
			    "                 are: "+options_wave_filetypes()+"\n"+
				"-c <int>         Select channel from input data (starts from 0)\n"+
				    "-n <int>         Number of channels of raw data\n"+
					"-f <int>         Input file sample rate in Hertz\n"+
					    "-ibo <string>    Input byte order: MSB, LSB, native, nonnative\n"+
						"                 Suns, HP, SGI Mips, M68000 are MSB (big endian)\n"+
						    "                 Intel, Alpha, DEC Mips, Vax are LSB (little endian)\n"+
							"-iswap           Swap bytes in input\n"+
							    "-istype <string> Sample type: short, mulaw, byte, ascii\n"+
								"-ulaw            Assume unheadered input is 8k ulaw\n"+
								    "-r*              Select subrange of file. (ESPS compatible)\n"+
									"-basic           HTML audio/basic format, if unheadered treat\n"+
									    "                 as ulaw 8K\n"+
										options_subwave();
}

EST_String options_wave_output(void)
{
    return 
	EST_String("")+
	    "-o <ofile>       Output filename, if not specified output is\n"+
		"                 to stdout\n"+
		    "-otype <string>  Output file type, (optional).  If no type is\n"+
			"                 Specified the type of the input file is assumed\n"+
			    "                 types are: "+options_wave_filetypes()+"\n"+
				"-F <int>         Output sample rate in Hertz, will cause resampling\n"+
				    "                 if value is different from input sample rate.\n"+
					"-obo <string>    Output byte order MSB, LSB, native, nonnative\n"+
					    "                 Suns, HP, SGI Mips, M68000 are MSB (big endian)\n"+
						"                 Intel, Alpha, DEC Mips, Vax are LSB (little endian)\n"+
						    "-oswap           Swap bytes when saving to output\n"+
							"-ostype <string> Output sample type: short, mulaw, byte, ascii\n";
    
}
