/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.source;

import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.excalibur.xml.XMLConsumer;
import org.apache.avalon.excalibur.xml.XMLizable;

import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import java.io.InputStream;
import java.io.IOException;

/**
 * Description of a source which is described by the resource protocol
 * which gets a resource from the classloader.
 *
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 * @version CVS $Revision: 1.2 $ $Date: 2002/01/08 12:55:51 $
 */

public final class ResourceSource
implements Composable, Source, XMLizable
{
    /** The system identifier */
    private String systemId;

    /** Location of the resource */
    private String location;

    /** The ComponentManager needed for streaming */
    private ComponentManager manager;

    /**
     * Constructor
     */
    public ResourceSource(String systemId)
    {
        this.systemId = systemId;
        final int pos = systemId.indexOf("://");
        this.location = systemId.substring(pos+3);
    }

    public void compose(ComponentManager manager )
    {
        this.manager = manager;
    }

    /**
     * Return an <code>InputStream</code> object to read from the source.
     */
    public InputStream getInputStream()
    throws IOException
    {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        if (loader == null)
        {
            loader = this.getClass().getClassLoader();
        }
        return loader.getResourceAsStream( this.location );
    }

    /**
     * Stream content to a content handler or to an XMLConsumer.
     *
     * @throws ResourceNotFoundException if file not found or
     *         HTTP location does not exist.
     * @throws SAXException if failed to parse source document.
     */
    public void toSAX(ContentHandler handler)
    throws SAXException
    {
        Parser parser = null;
        try {
            parser = (Parser)this.manager.lookup(Parser.ROLE);

            parser.parse( SourceUtil.getInputSource(this), handler);
        }
        catch (SAXException e)
        {
            // Preserve original exception
            throw e;
        } catch (Exception e)
        {
            throw new SAXException("Exception during processing of "
                                          + this.systemId, e);
        } finally
        {
            if (parser != null) this.manager.release(parser);
        }
    }

    /**
     * Return the unique identifer for this source
     */
    public String getSystemId()
    {
        return this.systemId;
    }

}
