/*
 * File    : TrackerWebDefaultPlugin.java
 * Created : 08-Dec-2003
 * By      : parg
 * 
 * Azureus - a Java Bittorrent client
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details ( see the LICENSE file ).
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.gudy.azureus2.ui.tracker;

/**
 * @author parg
 *
 */

import java.net.*;
import java.io.*;
import java.util.*;

import org.gudy.azureus2.plugins.*;
import org.gudy.azureus2.plugins.download.*;
import org.gudy.azureus2.plugins.torrent.*;
import org.gudy.azureus2.plugins.tracker.*;
import org.gudy.azureus2.plugins.tracker.web.*;
import org.gudy.azureus2.plugins.ui.menus.*;
import org.gudy.azureus2.plugins.ui.config.*;
import org.gudy.azureus2.plugins.ui.model.BasicPluginConfigModel;
import org.gudy.azureus2.plugins.ui.*;
import org.gudy.azureus2.plugins.ui.tables.*;


public class 
TrackerWebDefaultTrackerPlugin
	extends TrackerWeb
{
	protected DirectoryParameter	upload_dir;
	
	public void 
	initialize(
		PluginInterface _plugin_interface )
	{	
		super.initialize( _plugin_interface );
	
		plugin_interface.getUtilities().getLocaleUtilities().integrateLocalisedMessageBundle( "org.gudy.azureus2.ui.tracker.internat.Messages");
		
		UIManager	ui_manager = plugin_interface.getUIManager();
		
		BasicPluginConfigModel config_model = ui_manager.createBasicPluginConfigModel( "plugins", "plugins.TrackerWeb");
				
			// migrate from core to plugin
		
		if ( !plugin_config.getPluginBooleanParameter( CONFIG_MIGRATED, false )){
			
			plugin_config.setPluginParameter( CONFIG_MIGRATED, true );
						
			plugin_config.setPluginParameter(
					CONFIG_TRACKER_PUBLISH_ENABLE,
					plugin_config.getBooleanParameter(
							CONFIG_TRACKER_PUBLISH_ENABLE,
							CONFIG_TRACKER_PUBLISH_ENABLE_DEFAULT ));
			
			plugin_config.setPluginParameter(
					CONFIG_TRACKER_PUBLISH_ENABLE_DETAILS,
					plugin_config.getBooleanParameter(
							CONFIG_TRACKER_PUBLISH_ENABLE_DETAILS,
							CONFIG_TRACKER_PUBLISH_ENABLE_DETAILS_DEFAULT ));

			plugin_config.setPluginParameter(
					CONFIG_TRACKER_SKIP,
					plugin_config.getIntParameter(
							CONFIG_TRACKER_SKIP,
							CONFIG_TRACKER_SKIP_DEFAULT ));
		}
		

		BooleanParameter enable 		= config_model.addBooleanParameter2( CONFIG_TRACKER_PUBLISH_ENABLE, "ConfigView.section.tracker.publishenable", CONFIG_TRACKER_PUBLISH_ENABLE_DEFAULT );
		
		BooleanParameter enable_details	= config_model.addBooleanParameter2( CONFIG_TRACKER_PUBLISH_ENABLE_DETAILS, "ConfigView.section.tracker.publishenabledetails", CONFIG_TRACKER_PUBLISH_ENABLE_DETAILS_DEFAULT );
		
		BooleanParameter enable_peer_details	= config_model.addBooleanParameter2( CONFIG_TRACKER_PUBLISH_ENABLE_PEER_DETAILS, "ConfigView.section.tracker.publishenablepeerdetails", CONFIG_TRACKER_PUBLISH_ENABLE_PEER_DETAILS_DEFAULT );
		
		config_model.addIntParameter2( CONFIG_TRACKER_SKIP, "ConfigView.section.tracker.torrentsperpage", CONFIG_TRACKER_SKIP_DEFAULT );
		
		BooleanParameter enable_categories	= config_model.addBooleanParameter2( CONFIG_TRACKER_PUBLISH_ENABLE_CATEGORIES, "ConfigView.section.tracker.enablecategories", CONFIG_TRACKER_PUBLISH_ENABLE_CATEGORIES_DEFAULT );
		
		BooleanParameter enable_upload		= config_model.addBooleanParameter2( CONFIG_TRACKER_UPLOAD_DATA_ENABLE, "azplugins.tracker.showuploaddata", CONFIG_TRACKER_UPLOAD_DATA_ENABLE_DEFAULT );
		
		upload_dir		= config_model.addDirectoryParameter2( CONFIG_TRACKER_UPLOAD_DIR, "azplugins.tracker.uploaddir", "" );
		
		
		enable.addEnabledOnSelection( enable_details );
		enable.addEnabledOnSelection( enable_peer_details );
		enable.addEnabledOnSelection( enable_categories );
		enable.addEnabledOnSelection( enable_upload );
		enable.addEnabledOnSelection( upload_dir );
		
		TableContextMenuItem menu = plugin_interface.getUIManager().getTableManager().addContextMenuItem(TableManager.TABLE_MYTRACKER, "MyTrackerView.webui.contextmenu.copyurl" );
		
		menu.addListener(
			new MenuItemListener()
			{
				public void
				selected(
					MenuItem			_menu,
					Object	_target )
				{
				  TrackerTorrent trackerTorrent = (TrackerTorrent)((TableRow)_target).getDataSource();
				  if (trackerTorrent == null)
				    return;
					
					Torrent	torrent = trackerTorrent.getTorrent();
					
					String	hash_str = formatters.encodeBytesToString( torrent.getHash());
					
					String	torrent_name = torrent.getName();

					URL	url = torrent.getAnnounceURL();
					
					boolean https = url.getProtocol().toLowerCase().equals("https");
					
						// could be udp -> use http
					
					String url_str = https?"https":"http";
					
					url_str += "://" + url.getHost();
					
					if ( url.getPort() != -1 ){
						
						url_str += ":" + url.getPort();
					}
					
					
					try{
						url_str +=  "/torrents/" + URLEncoder.encode( torrent_name, torrent.getEncoding()) + ".torrent?" + hash_str;
						
						plugin_interface.getUIManager().copyToClipBoard( url_str );
						
					}catch( Throwable  e ){
						
						e.printStackTrace();
					}
				}
			});
	}
	
	public boolean
	generate(
		TrackerWebPageRequest		request,
		TrackerWebPageResponse		response )
	
		throws IOException
	{
		if ( !plugin_config.getPluginBooleanParameter( CONFIG_TRACKER_PUBLISH_ENABLE, CONFIG_TRACKER_PUBLISH_ENABLE_DEFAULT )){
			
			return( false );
		}
		
		String	url = request.getURL();
				
		TrackerTorrent[]	torrents = tracker.getTorrents();
		
		OutputStream	os = response.getOutputStream();
		
		// System.out.println( "TrackerWebDefaultTrackerPlugin: " + url);
		
		try{
			if ( url.startsWith( "/torrents/")){
				
				String	str = url.substring(10);
				
				int	pos = str.indexOf ( "?" );
				
				String	hash_str = str.substring(pos+1);
				
				byte[]	hash = formatters.decodeBytesFromString( hash_str );
									
				for (int i=0;i<torrents.length;i++){
						
					TrackerTorrent	tracker_torrent = torrents[i];
						
					Torrent	torrent = tracker_torrent.getTorrent();
						
					if ( Arrays.equals( hash, torrent.getHash())){
							
						response.writeTorrent( tracker_torrent );
							
						return( true );
					}
				}
				
				System.out.println( "Torrent not found at '" + url + "'" );
										
				response.setReplyStatus( 404 );
			
			}else if ( url.equalsIgnoreCase("/favicon.ico" )){
								
				response.setContentType( "image/x-icon" );
				
				response.setHeader( "Last Modified",
									"Fri,05 Sep 2003 01:01:01 GMT" );
				
				response.setHeader( "Expires",
									"Sun, 17 Jan 2038 01:01:01 GMT" );
				
				InputStream is = utilities.getImageAsStream( "favicon.ico" );
				
				if ( is == null ){
										
					response.setReplyStatus( 404 );
					
				}else{
					
					byte[] data = new byte[4096];
										
					while(true){
						
						int len = is.read(data, 0, 4096 );
						
						if ( len <= 0 ){
							
							break;
						}
						
						os.write( data, 0, len );
					}	
				}
			}else if ( 	url.equals( "/upload.cgi") &&
						plugin_config.getPluginBooleanParameter( CONFIG_TRACKER_UPLOAD_DATA_ENABLE, CONFIG_TRACKER_UPLOAD_DATA_ENABLE_DEFAULT )){

					// Content-Type: multipart/form-data; boundary=---------------------------7d437a28e70644

				PrintWriter	pw = new PrintWriter( new OutputStreamWriter( os ));
				
				pw.println( "<html>" );
				pw.println( "<head><link href=\"styles/default.css\" rel=\"stylesheet\" type=\"text/css\"></head>" );
				pw.println( "<body><pre>");
				
				pw.println( "Upload processing starts" );
				
				try{
					if ( !new File(upload_dir.getValue()).exists()){
						
						throw( new Exception( "Upload directory not configured" ));
					}
					
					String	header = request.getHeader();
					
					int	pos = header.toLowerCase().indexOf( "content-type:");
					
					if( pos == -1 ){
						
						throw( new Exception( "content-type missing" ));
					}
					
					int	p2 = header.indexOf( "\r", pos );
					
					String	content_type = header.substring(pos,p2);
					
					
					int bp = content_type.toLowerCase().indexOf( "boundary=" );
					
					if ( bp == -1 ){
						
						throw( new Exception( "boundary missing" ));
					}
					
					String	boundary = content_type.substring(bp+9).trim();
					
					int	ep = boundary.indexOf( ';' );
					
					if ( ep != -1 ){
						
						boundary = boundary.substring(0,ep).trim();
					}
					
					TrackerFormDecoder.formField[]	fields = 
						new TrackerFormDecoder().decode( boundary, request.getInputStream());
					
					String	torrent_category 		= "";	
					String	torrent_name			= null;
					String	torrent_comment			= null;
					String	torrent_hash			= null;
					boolean torrent_include_hashes	= false;
					boolean	force_start				= false;
					String  torrent_protocol		= "http";
					
					InputStream	torrent_data	= null;
					
					try{
						for (int i=0;i<fields.length;i++){
							
							TrackerFormDecoder.formField	field = fields[i];
							
							String	name = field.getName();
							
							if ( name.equalsIgnoreCase( "torrent_category" )){
								
								torrent_category	= field.getString();
															
							}else if ( name.equalsIgnoreCase( "torrent_comment" )){
								
								torrent_comment	= field.getString();
								
							}else if ( name.equalsIgnoreCase( "torrent_include_hashes" )){
								
								torrent_include_hashes	= true;
								
							}else if ( name.equalsIgnoreCase( "torrent_announce_protocol" )){
								
								torrent_protocol	= field.getString().trim().toLowerCase();
								
							}else if ( name.equalsIgnoreCase( "torrent_force_start" )){
								
								force_start	= true;
								
							}else if ( name.equalsIgnoreCase( "torrent_hash" )){
								
								torrent_hash	= field.getString();
								
							}else if ( name.equals( "upfile" )){
								
								torrent_data	= field.getInputStream();
								
								torrent_name 	= field.getAttribute( "filename");
								
								if ( torrent_name == null ){
									
									throw( new Exception( "'filename' attribute missing from upload" ));
								}
								
								int	ep1	= torrent_name.lastIndexOf("\\");
								int ep2	= torrent_name.lastIndexOf("/");
																				
								torrent_name = torrent_name.substring( Math.max( ep1, ep2 )+1 );
							}
						}
					
						if ( torrent_data == null ){
							
							throw( new Exception( "Failed to read upload data" ));
						}
						
						upload( pw, 
								torrent_data, 
								torrent_name, 
								torrent_category, 
								torrent_comment, 
								torrent_hash, 
								torrent_include_hashes,
								force_start,
								torrent_protocol );
						
						pw.println( "Upload successful" );
						
					}finally{
												
						for (int i=0;i<fields.length;i++){
							
							TrackerFormDecoder.formField	field = fields[i];
							
							field.destroy();
						}
					}
				}catch( Throwable e ){
					
					pw.println( "Upload failed" );
					
					e.printStackTrace( pw );
					
				}finally{
					
					pw.println( "</pre></body></html>");

					pw.flush();
				}
			}else{
				
				if ( url.equals("/")){
					
					url = "/index.tmpl";
				}
				
				Hashtable	params = null;
				
				int	p_pos = url.indexOf( '?' );
				
				if ( p_pos != -1 ){
					
					params = decodeParams( url.substring( p_pos+1 ));
					
					url = url.substring(0,p_pos);
				}	
				
				InputStream is = TrackerWebDefaultTrackerPlugin.class.getClassLoader().getResourceAsStream("org/gudy/azureus2/ui/tracker/templates" + url );
				
				if ( is == null ){
					
					return( false );
				}
				
				try{
					int	pos = url.lastIndexOf( "." );
					
					if ( pos == -1 ){
						
						return( false );
					}
					String	file_type = url.substring(pos+1);
					
					if ( file_type.equals("php") || file_type.equals("tmpl")){
						
						Hashtable	args = new Hashtable();
						
						args.put( "filehandle", new InputStreamReader( is ));

						return( handleTemplate( url, params, args, os ));
							
					}else{ 
													
						response.useStream( file_type, is );
						
						return( true );
					}	
				}finally{
													
					is.close();
				}			
			}
		}catch( Throwable e ){
						
			e.printStackTrace();
			
			os.write( e.toString().getBytes());
		}

		return( true );
	}
	
	protected void
	upload(
		final PrintWriter	pw,
		InputStream			is,
		String				torrent_name,
		String				torrent_category,
		String				torrent_comment,
		String				torrent_hash,
		boolean				torrent_include_hashes,
		boolean				force_start,
		String				torrent_protocol ) 		
	
		throws Exception
	{
		// System.out.println( "Update:" + torrent_name + "/" + torrent_category + "/" + torrent_comment );

		File		torrent_file;
		File		data_directory	= new File(upload_dir.getValue());
	
		Torrent		torrent;
				
		URL[]	defined_urls = plugin_interface.getTracker().getURLs();
		
		if ( defined_urls.length == 0 ){
			
			throw( new Exception( "Tracker not enabled" ));
		}
		
		URL	announce_url = null;
		
		for (int i=0;i<defined_urls.length;i++){
			
			URL	url	= defined_urls[i];
				
			if ( url.getProtocol().equalsIgnoreCase( torrent_protocol )){
				
				announce_url = url;
				
				break;
			}
		}
		
		if ( announce_url == null ){
			
			throw( new Exception("Tracker protocol '" + torrent_protocol + "' not enabled" ));
		}
			
		pw.println( "Using announce url '" + announce_url.toString() +"'" );
		
		if ( torrent_name.toLowerCase().endsWith( ".torrent" )){
		
			torrent_file 	= new File(upload_dir.getValue(), torrent_name );
		
			pw.println( "Saving torrent as '" + torrent_file.toString() + "'" );

			byte[]	buffer = new byte[65536];
			
			FileOutputStream	fos = new FileOutputStream( torrent_file );
			
			while(true){
				
				int	len = is.read( buffer );
				
				if ( len <= 0 ){
					
					break;
				}
				
				fos.write( buffer, 0, len );
			}
			
			fos.close();		
			
			torrent = plugin_interface.getTorrentManager().createFromBEncodedFile( torrent_file );

			torrent.setAnnounceURL( announce_url );
			
		}else{
					
			File	target_data		= new File( data_directory, torrent_name );

			torrent_file 	= new File( data_directory, torrent_name + ".torrent" );
			
			byte[]	buffer = new byte[65536];
			
			FileOutputStream	fos = new FileOutputStream( target_data );
			
			while(true){
				
				int	len = is.read( buffer );
				
				if ( len <= 0 ){
					
					break;
				}
				
				fos.write( buffer, 0, len );
			}
			
			fos.close();
			
			TorrentManager	tm = plugin_interface.getTorrentManager();
			
			TorrentManagerListener	tml = 
				new TorrentManagerListener()
				{
					public void
					event(
						TorrentManagerEvent	ev )
					{
						pw.println( ev.getData());
						
						pw.flush();
					}
				};
							
			try{
				
				pw.println( "Creating torrent from '" + target_data.toString() + "'" );
				pw.println( "Include other hashes = " + torrent_include_hashes );
				
				tm.addListener( tml );
				
				torrent = tm.createFromDataFile( 
							target_data,
							announce_url,
							torrent_include_hashes );

				byte[]	hash = torrent.getHash();
				
				String	hash_str = plugin_interface.getUtilities().getFormatters().formatByteArray( hash, true );
				
				pw.println( "Generated torrent hash = '" + hash_str + "'" );
				
				if( torrent_hash.length() > 0 ){
					
					pw.println( "Supplied hash = '" + torrent_hash + "'" );
					
					if ( !hash_str.equals( torrent_hash.trim())){
						
						// t.writeToFile(new File("D:\\temp\\failed.torrent"));
						
						throw(new Exception("Uploaded data hash mismatch"));
					}
				}
			}finally{
				
				tm.removeListener( tml );
			}
			
			torrent.setComplete( data_directory );
		}

		torrent.setComment( torrent_comment );
			
		torrent.writeToFile( torrent_file );
					
		Download dl = download_manager.addDownload( torrent, torrent_file, data_directory );
	
		pw.println( "Added download" );

		if ( torrent_category.length() > 0 ){
				
			dl.setCategory( torrent_category );

			pw.println( "Set category" );
		}
			
		if ( force_start ){

			dl.setForceStart( true );
			
			pw.println( "Force started it" );
		}

		plugin_interface.getTracker().host( dl.getTorrent(), true );
		
		pw.println( "Hosted it" );
	}
}
