//
// File:        CxxStubHeader.java
// Package:     gov.llnl.babel.backend.ucxx
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ucxx;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.ucxx.Cxx;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.LevelComparator;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.backend.Utilities;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.List;


/**
 * 
 */
public class CxxStubHeader {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private String d_self = null;

  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxStubHeader(Extendable ext) {
    d_ext = ext;
  }
  
    /**
   * Special constructor for generating super methods for IMPL files.
   * Since these can only exist for classes, d_self is just "self"
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxStubHeader(Extendable ext, LanguageWriterForCxx writer) {
    d_ext = ext;
    d_writer = writer;
    d_self = "self";
  }

  /**
   * This is a convenience utility function specifically for the generation
   * of super "Stub" functions in the Impl files. 
   * The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   *
   * @param methods is a collection of super methods to be generated.
   *
   * @param writer the output writer to which the stub source will
   *               be written. This will not be closed.
   *
   * @param cls The class in which these supers are to be generated      
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateSupers(Class             cls,
                                  LanguageWriterForCxx writer)
    throws CodeGenerationException
  {
    CxxStubHeader source = new CxxStubHeader(cls, writer);
    source.generateSupers();
  }  
  
  /**
   * This special function is only to be used when generating super
   * functions for the Super class in an Impl.  It may only be called though
   * the static method above, "generateSupers"
   *
   */
  private void generateSupers() 
    throws CodeGenerationException {
    Class cls = (Class) d_ext;
    SymbolID clsID = cls.getSymbolID();
    final String ext_name = IOR.getExternalName(clsID);
    d_writer.println("public:");
    d_writer.writeComment("Hold pointer to IOR functions.", false);

    d_writer.println("class Super {");
    d_writer.tab();
    d_writer.println();
    d_writer.println("private:");
    d_writer.tab();
    d_writer.writeComment("Hold pointer to Super EPV", false);
    d_writer.println(/*"const " + */IOR.getEPVName(cls.getParentClass().getSymbolID()) + 
                     "* superEPV;");
    d_writer.println(/*"const " + */IOR.getObjectName(clsID) + "* superSelf;");
    d_writer.println();
    d_writer.backTab();
    d_writer.println("public:"); 
    d_writer.tab();
    d_writer.println("Super() : superEPV(NULL), superSelf(NULL) {}");

    d_writer.println();
    d_writer.println("Super("+IOR.getObjectName(clsID)+"* loc_self, const "+IOR.getExternalName(cls.getSymbolID())+"* loc_ext) {");
    d_writer.tab();
    d_writer.println("superEPV = loc_ext->getSuperEPV();");
    d_writer.println("superSelf = loc_self;");
    d_writer.backTab();
    d_writer.println("}");
    Collection methods = cls.getOverwrittenClassMethods();
    for(Iterator mit = methods.iterator(); mit.hasNext();){
      Method method = (Method)mit.next();
      generateInlineMethodDispatch(method, null, true, true);
    }
    
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("};");

  }


  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    String filename = Cxx.generateFilename( d_ext.getSymbolID(), 
                                            Cxx.FILE_ROLE_STUB, 
                                            Cxx.FILE_TYPE_CXX_HEADER );
    //System.out.println("Create " + filename + "..." );
    
    if ( d_ext.isInterface() ) { 
      d_self = "loc_self->d_object";
    } else { 
      d_self = "loc_self";
    }

    try { 
      d_writer = Cxx.createHeader( d_ext, Cxx.FILE_ROLE_STUB, "STUBHDRS");
      if ( d_ext.getSymbolType() == Symbol.PACKAGE ) { 
	  d_writer.skipIncludeGuard();
      }
      d_writer.println();
      d_writer.openHeaderGuard( filename );

      writeDefineNullIORException();

      writeClassDeclaration();
      
      Cxx.generateDependencyIncludes( d_writer, d_ext, true );

      Cxx.nestPackagesInNamespaces( d_writer, d_ext );
      
      writeClassBegin();
      
      writeUserDefinedMethods();
      
      writeTypedefs();

      writeConstructors();

      writeCastingOperators();

      writeBindingSpecificMethods();

      writeClassEnd(); 
      
      Cxx.unnestPackagesInNamespaces( d_writer, d_ext );
      
      writeArrayDefinition();

      d_writer.closeHeaderGuard();
    } catch ( Exception ex ) { 
      throw new CodeGenerationException("Exception : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  /**
   * This method write out a #define if Null IOR Exceptions are defined in
   * this file.
   */
  private void writeDefineNullIORException() { 
    BabelConfiguration s_babel_config = BabelConfiguration.getInstance();
    String flagname = d_ext.getSymbolID().getFullName().replace('.', '_').toUpperCase() + 
      "_NULL_IOR_EXCEPTION";

    if(s_babel_config.makeCxxCheckNullIOR()) {
      d_writer.printlnUnformatted("#ifndef " + flagname);
      d_writer.printlnUnformatted("#define " + flagname + " 1");
      d_writer.printlnUnformatted("#endif ");
    }
  }

  private void writeClassDeclaration() throws CodeGenerationException { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    
    d_writer.writeCommentLine("declare class before #includes");
    d_writer.writeCommentLine("(this alleviates circular #include guard problems)[BUG#393]");
    Cxx.nestPackagesInNamespaces( d_writer, d_ext );
    d_writer.println("class " + name + ";");
    Cxx.unnestPackagesInNamespaces( d_writer, d_ext );

    

    d_writer.writeCommentLine("Some compilers need to define array template before the specializations");
    d_writer.generateInclude("sidl_ucxx.hh", true );
    Cxx.openUCxxNamespace(d_writer);
    d_writer.println("namespace sidl {");
    d_writer.tab();
    d_writer.println("template<>");
    d_writer.println("class array< " +  Cxx.getObjectName(id) + 
                     " >;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    Cxx.closeUCxxNamespace(d_writer);
    
        //Forward declarations for method dependencies

    Set decls = new HashSet();
    for( Iterator i = d_ext.getMethods(true).iterator(); i.hasNext(); ) { 
      Method method = (Method) i.next();
      decls.addAll(method.getSymbolReferences()); 
    }
    if (!decls.isEmpty()){
      d_writer.writeComment("Forward declarations for method dependencies.",false);
      
      List entries = Utilities.sort(decls);
      
      for (Iterator i = entries.iterator(); i.hasNext(); ) { 
        SymbolID s = (SymbolID) i.next();
        Symbol symbol = (Symbol) Utilities.lookupSymbol(s);
        //Enumerated types are included, and hence shouldn't be forward declared.
        if(symbol.getSymbolType() != Type.ENUM) {
          Cxx.nestPackagesInNamespaces( d_writer, s );
          d_writer.println("class " + s.getShortName() +";");
          Cxx.unnestPackagesInNamespaces( d_writer, s );
        }
      }
    }
    
  }

    private void writeClassBegin() { 
    /*    SymbolID id = d_ext.getSymbolID();
          String name = id.getShortName();

          d_writer.writeComment(d_ext, true);    
          d_writer.println("class " + name + " : public ::sidl::StubBase {");
          d_writer.tab();
    */
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    
    d_writer.writeComment(d_ext, true);    
    d_writer.print("class " + name);
    java.util.Collection parents = d_ext.getParents(false);
    boolean first = true;
    //Hack to see if there are no parents.  If no parents, parent is
    //stubbase 
    Iterator i2 = parents.iterator();
    if(!i2.hasNext()) {
      d_writer.print(": public virtual "+Cxx.prependGlobalUCxx()+"::sidl::StubBase");
 
    } else {
 
      for(Iterator i = parents.iterator(); i.hasNext(); ){
        Extendable parent = (Extendable) i.next();
        String parentName = Cxx.getSymbolName(parent.getSymbolID(), "");
        if(first)
          d_writer.print(": ");
        else
          d_writer.print(", ");
        first = false;
        d_writer.print("public virtual " + parentName);
      }
    }
    d_writer.println(" {");
    
  } 
  private void writeUserDefinedMethods() throws CodeGenerationException { 
    d_writer.beginBoldComment();
    d_writer.println("User Defined Methods");
    d_writer.endBoldComment();

    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    
    Iterator m = null;
    m = d_ext.getStaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      if(!Cxx.inlineStub(method)) {
        Cxx.generateMethodSignature( d_writer, method, 
                                     "user defined static method", 
                                     Cxx.FILE_ROLE_STUB, true );
        d_writer.println();
        if(method.hasRarray()) {
          Cxx.generateMethodSignature( d_writer, method, 
                                       "user defined static method", 
                                       Cxx.FILE_ROLE_STUB, false );
          d_writer.println();
        }
      } else {
        generateInlineMethodDispatch(method, 
                                     "user defined static method", false, true);
        d_writer.println();
      }

    }
    
    m = d_ext.getNonstaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      if(!Cxx.inlineStub(method)) {
        Cxx.generateMethodSignature( d_writer, method, 
                                     "user defined non-static method.", 
                                     Cxx.FILE_ROLE_STUB, true );
        d_writer.println();
        if(method.hasRarray()) {
          Cxx.generateMethodSignature( d_writer, method, 
                                       "user defined static method", 
                                       Cxx.FILE_ROLE_STUB, false );
          d_writer.println();
        }
      } else {
        generateInlineMethodDispatch(method, 
                                     "user defined non-static method", false, true);
        d_writer.println();
      }
    }
    d_writer.beginBoldComment();
    d_writer.println("End User Defined Methods");
    d_writer.println("(everything else in this file is specific to");
    d_writer.println(" Babel's C++ bindings)");
    d_writer.endBoldComment();
  }

  private void writeTypedefs() { 
    SymbolID id = d_ext.getSymbolID();

    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    d_writer.println("typedef " + IOR.getObjectName( id ) + " ior_t;");
    d_writer.println("typedef " + IOR.getExternalName( id ) + " ext_t;");
    d_writer.println("typedef " + IOR.getSEPVName( id ) + " sepv_t;");
    d_writer.println();
  }

  private void writeConstructors() { 
    SymbolID id = d_ext.getSymbolID();
    String ior_ptr = "ior_t*";
    String name = id.getShortName();

    d_writer.writeCommentLine("default constructor");
    d_writer.println( name + "() { }");
    d_writer.println();
    
    if( ! d_ext.isAbstract() ) {
      d_writer.writeCommentLine("static constructor");
      d_writer.println("static " + Cxx.getObjectName(id) + " _create();");
    
      d_writer.println();
    }

    d_writer.writeCommentLine("default destructor");
    d_writer.println("virtual ~" + name + " () { }");
    d_writer.println();
    
    d_writer.writeCommentLine("copy constructor");
    d_writer.println( name + " ( const " + name + "& original );");// : StubBase(original) { }");
    d_writer.println();

    d_writer.writeCommentLine("assignment operator");
    d_writer.println( name + "& operator= ( const " + name + "& rhs );");
    d_writer.println();

    //d_writer.writeCommentLine("assignment operator from StubBase");
    //d_writer.println( name + "& operator= ( const ::sidl::StubBase& rhs );");
    //d_writer.println();
  }
  
  private void writeCastingOperators() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    String ior_ptr = "ior_t*";

    d_writer.writeCommentLine("conversion from ior to C++ class");
    d_writer.println( name + " ( " + name + "::ior_t* ior );");
    d_writer.println();

    d_writer.writeCommentLine("Alternate constructor: does not call addRef()");
    d_writer.writeCommentLine("(sets d_weak_reference=isWeak)");
    d_writer.writeCommentLine("For internal use by Impls (fixes bug#275)");
    d_writer.println( name + " ( " + name + "::ior_t* ior, bool isWeak );");
    d_writer.println();

    //d_writer.writeCommentLine("conversion from a StubBase");
    //d_writer.println( name + " ( const ::sidl::StubBase& base );");
    //d_writer.println();
  }

  private void writeBindingSpecificMethods() { 
    SymbolID id = d_ext.getSymbolID();
    String name = id.getShortName();
    String ior_ptr = "ior_t*";

    d_writer.println( ior_ptr + " _get_ior() { return " + 
                      Cxx.reinterpretCast("ior_t*", "d_self") + "; }");
    d_writer.println();

    d_writer.println("const " + ior_ptr + " _get_ior() const { return " + 
                     Cxx.reinterpretCast("ior_t*", "d_self")+ "; }");
    d_writer.println();

    d_writer.println( "void _set_ior( " + ior_ptr + " ptr ) { d_self = " +
                      Cxx.reinterpretCast("void*", "ptr") + "; }");
    d_writer.println();

    d_writer.println("bool _is_nil() const { return (d_self==0); }");
    d_writer.println();

    d_writer.println("bool _not_nil() const { return (d_self!=0); }");
    d_writer.println();

    d_writer.println("bool operator !() const { return (d_self==0); }");
    d_writer.println();
    
    d_writer.println("static inline const char * type_name() { return \"" + 
                     id.getFullName()+"\";}");

    d_writer.println("virtual void* _cast(const char* type) const;");
    d_writer.println();

    d_writer.backTab();
    d_writer.println("protected:");
    d_writer.tab();
    //    d_writer.println("template<typename Target, typename Source>");
    //d_writer.println("friend Target babel_cast(Source& arg);");

    d_writer.tab();
    d_writer.writeCommentLine("Pointer to external (DLL loadable) symbols (shared among instances)");
    d_writer.println("static const ext_t * s_ext;");
    d_writer.println();

    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    d_writer.println("static const ext_t * _get_ext() throw ( "+Cxx.prependGlobalUCxx()+"::sidl::NullIORException );");
    d_writer.println();
    if ( d_ext.hasStaticMethod(true) ) { // if has static methods
      d_writer.println("static const sepv_t * _get_sepv() {");
      d_writer.tab();
      d_writer.println("return (*(_get_ext()->getStaticEPV))();");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }
  private void writeClassEnd() { 
    String name = d_ext.getSymbolID().getShortName();
    d_writer.backTab();
    d_writer.println("}; // end class " + name);
  } 

  private void writeArrayDefinition() { 
    SymbolID id = d_ext.getSymbolID();
    String cxx_item_t = Cxx.getObjectName(id);
    String cxx_array_t = "array< " + cxx_item_t + " >";
    String ior_item_t = IOR.getObjectName(id);
    String ior_array_t = IOR.getArrayName(id);
    String array_traits = "array_traits< " +cxx_item_t + " >";

    Cxx.openUCxxNamespace(d_writer);
    d_writer.println("namespace sidl {");
    d_writer.tab();
    
    d_writer.writeCommentLine("traits specialization");
    d_writer.println("template<>");
    d_writer.println("struct " + array_traits + " {");
    d_writer.tab();
    d_writer.println("typedef " + cxx_array_t + " cxx_array_t;");
    d_writer.println("typedef " + cxx_item_t + " cxx_item_t;");
    d_writer.println("typedef " + ior_array_t + " ior_array_t;");
    d_writer.println("typedef sidl_interface__array ior_array_internal_t;");
    d_writer.println("typedef " +  ior_item_t + " ior_item_t;");
    d_writer.println("typedef cxx_item_t value_type;");
    d_writer.println("typedef value_type reference;");
    d_writer.println("typedef value_type* pointer;");
    d_writer.println("typedef const value_type const_reference;");
    d_writer.println("typedef const value_type* const_pointer;");
    d_writer.println("typedef array_iter< " + array_traits + " > iterator;");
    d_writer.println("typedef const_array_iter< " + array_traits+ " > const_iterator;");
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println();

    d_writer.writeCommentLine("array specialization");
    d_writer.println("template<>");
    d_writer.println("class " + cxx_array_t +
                     ": public interface_array< " + array_traits + " > {");
    d_writer.println("public:");
    d_writer.tab();
    
    d_writer.println("typedef interface_array< " + array_traits + " > Base;");
    d_writer.println("typedef " + array_traits + "::cxx_array_t          cxx_array_t;");
    d_writer.println("typedef " + array_traits + "::cxx_item_t           cxx_item_t;");
    d_writer.println("typedef " + array_traits + "::ior_array_t          ior_array_t;");
    d_writer.println("typedef " + array_traits + "::ior_array_internal_t ior_array_internal_t;");
    d_writer.println("typedef " + array_traits + "::ior_item_t           ior_item_t;");
    d_writer.println();
    d_writer.beginBlockComment(true);
    d_writer.println("conversion from ior to C++ class");
    d_writer.println("(constructor/casting operator)");
    d_writer.endBlockComment(true);
    d_writer.println("array( " + ior_array_t + 
                     "* src = 0) : Base(src) {}");
    d_writer.println();
 

    d_writer.beginBlockComment(true);
    d_writer.println("copy constructor");
    d_writer.endBlockComment(true);
    d_writer.println("array( const " + cxx_array_t + "&src) : Base(src) {}");
    d_writer.println();

    if (BabelConfiguration.getBaseInterface().equals(id.getFullName())) {
      d_writer.beginBlockComment(true);
      d_writer.println("Assignment to promote a generic array to an");
      d_writer.println("array of sidl.BaseInterface references. This");
      d_writer.println("will produce a nil array if the generic array");
      d_writer.println("isn't an array of objects/interfaces.");
      d_writer.endBlockComment(true);
      d_writer.println(cxx_array_t + "&");
      d_writer.println("operator =(const basearray &rhs) throw() {");
      d_writer.tab();
      d_writer.println("if (this->d_array != rhs._get_baseior()) {");
      d_writer.tab();
      d_writer.println("deleteRef();");
      d_writer.println("this->d_array =");
      d_writer.tab();
      d_writer.println("(rhs._get_baseior() &&");
      d_writer.println(" (sidl_interface_array == rhs.arrayType()))");
      d_writer.println("? const_cast<sidl__array *>(rhs._get_baseior())");
      d_writer.println(": NULL;");
      d_writer.backTab();
      d_writer.println("addRef();");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("return *this;");
      d_writer.backTab();
      d_writer.println("}");
    }
 
    d_writer.beginBlockComment(true);
    d_writer.println("assignment");
    d_writer.endBlockComment(true);
    d_writer.println(cxx_array_t + "&");
    d_writer.println("operator =( const " + cxx_array_t + "&rhs ) { ");

    d_writer.tab();
    d_writer.println("if (d_array != rhs._get_baseior()) {");
    d_writer.tab();
    d_writer.println("if (d_array) deleteRef();");
    d_writer.println("d_array = const_cast<sidl__array *>(rhs._get_baseior());");
    d_writer.println("if (d_array) addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return *this;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
 
    d_writer.backTab();
    d_writer.println("};");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
    Cxx.closeUCxxNamespace(d_writer);
  }

  private void generateInlineMethodDispatch( Method m, String altcomment, boolean isSuper, boolean rarrays ) 
    throws CodeGenerationException { 
    if ( m == null ) { return; }
    final ArrayList vArgs = m.getArgumentList();
    final int nargs = vArgs.size();
    String extra_close_paren = "";
    String shortMethodName = m.getShortMethodName();
    String longMethodName = m.getLongMethodName();
    String className = d_ext.getSymbolID().getShortName();
    Comment comment = m.getComment();
    BabelConfiguration s_babel_config = BabelConfiguration.getInstance();
    Type return_type = m.getReturnType();
    SymbolID id = return_type.getSymbolID();

    //d_writer.writeComment( comment, altcomment );
    if ( shortMethodName.equals(className) ) { 
      System.out.println("WARNING: gov.llnl.babel.backend.UCxx.CxxStubSource: sidl / C++ conflict!");
      System.out.println("         methodName == className is not allowed in C++");
      System.out.println("         (this is restricted to constructors in C++)");
      System.out.println("         changing to " + className + "::f_" + shortMethodName + "()");
      shortMethodName = "f_" + shortMethodName;
    }

    //Output function deleration
    Cxx.generateInlineMethodSignature( d_writer, m, 
                                     "user defined static method", 
                                       Cxx.FILE_ROLE_STUB, isSuper, rarrays );
    //d_writer.println(Cxx.generateFunctionDeclaration(m, d_ext, d_self));

    //d_writer.println();
    //d_writer.println("{");
    d_writer.tab();

    if ( (! m.isStatic()) &&  s_babel_config.makeCxxCheckNullIOR()) { 

      d_writer.println("if ( d_self == 0 ) {");
      d_writer.tab();
      d_writer.println("throw "+Cxx.prependGlobalUCxx()+"::sidl::NullIORException( ::std::string (");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println( "\"" + 
                        "Null IOR Pointer in \\\"" + 
                        Cxx.getMethodStubName(d_ext.getSymbolID(),shortMethodName) +
                        "()\\\"\"");
      d_writer.enableLineBreak();
      d_writer.backTab();
      d_writer.println("));");
      d_writer.backTab();
      d_writer.println("}");
    }

    //Output almost nothing!
    d_writer.println(Cxx.generateInitialization(m, d_ext, d_self));

    if ( ! m.isStatic() ) { 
      if (shortMethodName.equals("addRef") ||
          shortMethodName.equals("deleteRef") ) { 
        d_writer.println("if ( !d_weak_reference ) {");
        d_writer.tab();
      }
    }

    //Prepare for IOR call!
    d_writer.println(Cxx.generatePreIORCall(m, d_ext, d_self, isSuper, rarrays));

    if(isSuper) {
      Class cls = (Class) d_ext;
      String tmp_self = Cxx.reinterpretCast(IOR.getObjectName(cls.getParentClass().getSymbolID())
                                            + "*", "superSelf");
      d_writer.println(Cxx.generateIORCall(m, d_ext, tmp_self, isSuper, rarrays));
    } else {
      d_writer.println(Cxx.generateIORCall(m, d_ext, d_self, isSuper, rarrays));
    }

    if (m.getThrows().size()>0) { // if throws exception
      d_writer.println("if (_exception != 0 ) {");      
      d_writer.tab();
      d_writer.println("void * _p = 0;");
      Object [] exceptions = m.getThrows().toArray();
      Arrays.sort(exceptions, new LevelComparator(SymbolTable.getInstance()));
      
      for( int i=0; i<exceptions.length; ++i) {
        SymbolID exid = (SymbolID)exceptions[i];
        d_writer.println("if ( (_p=(*(_exception->d_epv->f__cast))(_exception, \"" +
                         exid.getFullName() + "\")) != 0 ) {");
        d_writer.tab();
        d_writer.println(IOR.getObjectName(exid) + " * _realtype = " + 
                         Cxx.reinterpretCast(IOR.getObjectName(exid)+"*" ,
                                             "_p") + ";");
        d_writer.writeCommentLine("Note: alternate constructor does not increment refcount.");
        d_writer.println("throw " + Cxx.getObjectName(exid) + "( _realtype, false );");
        d_writer.backTab();
        d_writer.println("}");
      }
      d_writer.backTab();
      d_writer.println("}");
    }

    //Clean up from return from IOR
    d_writer.println(Cxx.generatePostIORCall(m, d_ext, d_self, rarrays));

    //d_writer.println( post_ior.toString().trim() );
    if ( ! m.isStatic() ) { 
      if (shortMethodName.equals("addRef") ||
          shortMethodName.equals("deleteRef") ) { 
        if ( shortMethodName.equals("deleteRef") ) { 
          d_writer.println("d_self = 0;");
        }
        d_writer.backTab();
        d_writer.println("}");
      }
    }

    if ( return_type.getType() != Type.VOID ) { 
      d_writer.println("return _result;");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

  }

}
