/*****************************************************************************
 * Copyright (C) The Apache Software Foundation. All rights reserved.        *
 * ------------------------------------------------------------------------- *
 * This software is published under the terms of the Apache Software License *
 * version 1.1, a copy of which has been included  with this distribution in *
 * the LICENSE file.                                                         *
 *****************************************************************************/

package org.apache.cocoon.components.language.programming.java;

import java.io.*;
import java.util.*;
import org.apache.avalon.*;

import org.apache.cocoon.components.language.programming.*;

import org.apache.log.Logger;

/**
 * This class implements the functionality common to all Java compilers.
 * @author <a href="mailto:stefano@apache.org">Stefano Mazzocchi</a>
 * @version $Revision: 1.1.2.10 $ $Date: 2001/01/22 21:56:38 $
 * @since 2.0
 */
public abstract class AbstractJavaCompiler implements LanguageCompiler, Recyclable, Loggable {
  /**
   * The logger
   */
  protected Logger log;

  /**
   * The source program filename
   */
  protected String    file;

  /**
   * The name of the directory containing the source program file
   */
  protected String    srcDir;

  /**
   * The name of the directory to contain the resulting object program file
   */
  protected String    destDir;

  /**
   * The classpath to be used for compilation
   */
  protected String    classpath;

  /**
   * The encoding of the source program or <code>null</code> to use the
   * platform's default encoding
   */
  protected String    encoding = null;

  /**
   * The input stream to output compilation errors
   */
  protected InputStream errors;

  /**
   * Set the name of the file containing the source program
   *
   * @param file The name of the file containing the source program
   */
  public void setFile(String file) {
    this.file = file;
  }

  /**
   * Set the name of the directory containing the source program file
   *
   * @param srcDir The name of the directory containing the source program file
   */
  public void setSource(String srcDir) {
    this.srcDir = srcDir;
  }

  /**
   * Set the name of the directory to contain the resulting object program file
   *
   * @param destDir The name of the directory to contain the resulting object
   * program file
   */
  public void setDestination(String destDir) {
      this.destDir = destDir;
  }

    public void setLogger(Logger logger) {
        if (this.log == null) {
            this.log = logger;
        }
    }

  /**
   * Set the classpath to be used for this compilation
   *
   * @param classpath The classpath to be used for this compilation
   */
  public void setClasspath(String classpath) {
    this.classpath = classpath;
  }

  /**
   * Set the encoding of the input source file or <code>null</code> to use the
   * platform's default encoding
   *
   * @param encoding The encoding of the input source file or <code>null</code>
   * to use the platform's default encoding
   */
  public void setEncoding(String encoding) {
    this.encoding = encoding;
  }

  /**
   * Return the list of errors generated by this compilation
   *
   * @return The list of errors generated by this compilation
   * @exception IOException If an error occurs during message collection
   */
  public List getErrors() throws IOException {
    return parseStream(new BufferedReader(new InputStreamReader(errors)));
  }

  /**
   * Parse the compiler error stream to produce a list of
   * <code>CompilerError</code>s
   *
   * @param errors The error stream
   * @return The list of compiler error messages
   * @exception IOException If an error occurs during message collection
   */
  protected abstract List parseStream(BufferedReader errors)
      throws IOException;

  /**
   * Fill the arguments taken by the Java compiler
   *
   * @param argument The list of compilation arguments
   * @return The prepared list of compilation arguments
   */
  protected List fillArguments(List arguments) {
    // destination directory
    arguments.add("-d");
    arguments.add(destDir);

    // classpath
    arguments.add("-classpath");
    arguments.add(classpath);

    // sourcepath
    arguments.add("-sourcepath");
    arguments.add(srcDir);

    // add optimization (for what is worth)
    arguments.add("-O");

    // add encoding if set
    if (encoding != null) {
      arguments.add("-encoding");
      arguments.add(encoding);
    }

    return arguments;
  }

  /**
   * Copy arguments to a string array
   *
   * @param arguments The compiler arguments
   * @return A string array containing compilation arguments
   */
  protected String[] toStringArray(List arguments) {
    int    i;
    String[] args = new String[arguments.size() + 1];

    for (i = 0; i < arguments.size(); i++) {
      args[i] = (String) arguments.get(i);
    }

    args[i] = file;

    return args;
  }

  /** Reset all internal state.
   * This method is called by the component manager before this
   * component is return to its pool.
   */
  public void recycle() {
    file = null;
    srcDir = null;
    destDir = null;
    classpath = null;
    encoding = null;
    errors = null;
  }
}
