package Xray::Fluo;
##  This program is copyright (c) 1998-2006 Bruce Ravel
##  <bravel@anl.gov>
##  http://cars9.uchicago.edu/~ravel/software/

require Exporter;

use Xray::Absorption;
use Statistics::Descriptive;
use strict;
use constant ETOK => 0.2624682917;
use vars qw($VERSION $atoms_dir $languages @available_languages
	    $cvs_info $module_version $messages @ISA @EXPORT @EXPORT_OK);


@ISA = qw(Exporter AutoLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
##@EXPORT_OK = qw();

$VERSION = '1';

Xray::Absorption -> load('Elam');

=head2 C<mcmaster>

This is called C<mcmaster> for historical reasons.  It calculates the
normalization correcion for a given central atom.

  $sigma_mm = &mcmaster($central, $edge);

It takes the central atoms tag and the alphanumeric edge symbol as
arguments and returns the normalization correction in units of
Angstrom squared.

Note that the values returned depend on the data resource used.  See
L<Xray::Absorption>.

=cut


sub load {
  shift;
  my $which = lc(shift) ;
  Xray::Absorption -> load($which);
};



## $span in the fortran version was 500eV and the regression was
## performed with a square term.
## Statistics::Descriptive::least_squares_fit only fits a line, so I
## drew the #span back to 300 volts.  This gives the "canonical"
## 0.00052 for copper.
sub mcmaster {
  shift;
  my ($central, $edge) = @_;
  $edge = lc($edge);
  my $emin = Xray::Absorption -> get_energy($central, $edge) + 10 ;
  my ($emax, $e_to_k, $span, $npost) = (0, 0.2624682917, 300, 20);

  ## get energy range for above edge fit
  my %next_e = ("l1"=>"k", "l2"=>"l1", "l3"=>"l2", "m"=>"l3");
  if (exists $next_e{$edge}) {
    $emax = Xray::Absorption -> get_energy($central, $next_e{$edge}) - 10;
    $emax = (($emax - $emin) > $span) ? $emin + $span : $emax;
  } else {
    $emax = $emin + $span;
  };

  ## need to show some care with the Chantler data
  (Xray::Absorption -> current_resource() =~ /chantler/i) and
    $emin += 50;
  ($emin >= $emax) and ($emin = $emax - 20); # whatever!

  my ($bpre, $slope) = mcmaster_pre_edge($central, $edge);
  my $delta  = ($emax - $emin)/$npost;
  my @i=(0..$npost-1);		# load the post edge energies and sigmas
  my @energy = map {$emin + $delta*$_} @i;
  ## and some more care...
  (Xray::Absorption -> current_resource() =~ /chantler/i) and do {
    shift @energy; shift @energy;
  };
  return 0 if ($bpre <= 0);
  my @sigma  = Xray::Absorption -> cross_section($central, \@energy);
  @sigma = map {$sigma[$_] - ($bpre+$energy[$_]*$slope)} (0 .. $#energy);
  ##map {printf "      %9.3f %9.3f\n", $energy[$_], $sigma[$_]} (0 .. $#energy);
  @energy    = map {$e_to_k * ($_-$emin)} @energy; # convert to k
  my $any_neg = grep {$_ <= 0} @sigma;
  return 0 if $any_neg;
  @sigma     = map {log($_)} @sigma;       # take logs of xsecs

  my $stat = Statistics::Descriptive::Full->new(); # fit the post edge
  $stat -> add_data(@sigma);
  my @a = $stat -> least_squares_fit(@energy);
  return ($a[1] < 0) ? -$a[1]/2 : 0;
};

sub mcmaster_pre_edge {
  my ($central, $edge) = @_;
  $edge = lc($edge);
  my $emin = Xray::Absorption -> get_energy($central, $edge) - 10;
  ## find the pre-edge line
  my %next_e = ("k"=>"l1", "l1"=>"l2", "l2"=>"l3", "l3"=>"m");
  my $ebelow;
  if (exists $next_e{$edge}) {
    $ebelow = Xray::Absorption -> get_energy($central, $next_e{$edge}) + 10;
    $ebelow = (($emin - $ebelow) > 100) ? $emin - 100 : $ebelow;
  } else {
    $ebelow = $emin - 100;
  };
  my $delta  = ($emin - $ebelow)/10;;
  my @i=(0..9);			# load the pre edge energies/sigmas
  my @energy = map {$ebelow + $delta*$_} @i;
  my @sigma  = Xray::Absorption -> cross_section($central, \@energy);
				#  and fit 'em
  my $pre_edge = Statistics::Descriptive::Full->new();
  $pre_edge -> add_data(@sigma);
  my ($bpre, $slope) = $pre_edge -> least_squares_fit(@energy);
  $bpre ||= 0; $slope ||= 0;
  return ($bpre, $slope);
};


=head2 C<i_zero>

This calculates the correcion due to the I0 fill gases in a
fluorescence experiment.

  $sigma_i0 = &i_zero($central, $edge, $nitrogen, $argon, $krypton);

It takes the central atoms tag, the alphanumeric edge symbol, and the
volume percentages of the three gases as arguments.  It assumes that
any remaining volume is filled with helium and it correctly accounts
for the fact that nitrogen is a diatom.  It returns the I0 correction
in units of Angstrom squared.

Note that the values returned depend on the data resource used.  See
L<Xray::Absorption>.

=cut

sub i_zero {
  shift;
  my ($central, $edge, $nitrogen, $argon, $krypton) = @_;

  ##   convert from pressure percentages to number of absorbers.
  ## nitrogen is diatomic
  my $helium = 1 - $nitrogen - $argon - $krypton;
  my $norm   = $helium + 2*$nitrogen + $argon + $krypton;
  $nitrogen  = 2*$nitrogen / $norm;
  $argon     = $argon      / $norm;
  $krypton   = $krypton    / $norm;

  my $emin = Xray::Absorption -> get_energy($central, $edge) ;
  my ($emax, $e_to_k, $span, $npost) = (0, 0.2624682917, 500, 20);
  ## careful not to run a gas edge
  my ($el, $ed, $en) = Xray::Absorption ->
    next_energy($central, $edge, "ar", "n", "kr");
  if (not defined $en) {
    $emax = $emin + $span;
  } else {
    $emax = ( ($en - 10) < ($emin + $span) ) ? ($en - 10) : ($emin + $span);
  };
  ## need to show some care with the Chantler data
  (Xray::Absorption -> current_resource() =~ /chantler/i) and
    $emin += 50;
  ($emin >= $emax) and ($emin = $emax - 20); # whatever!

  my @i=(0..$npost-1);		# load the post edge energies and sigmas
  my $delta  = ($emax - $emin)/$npost;
  my @energy = map {$emin + $delta*$_} @i;
  my @s_n = Xray::Absorption -> cross_section("n",  \@energy);
  my @s_a = Xray::Absorption -> cross_section("ar", \@energy);
  my @s_k = Xray::Absorption -> cross_section("kr", \@energy);
  my @sigma  = map
  {$nitrogen*$s_n[$_] + $argon*$s_a[$_] + $krypton*$s_k[$_]} (0 .. $#energy);
  @energy    = map {$e_to_k * ($_-$emin)} @energy; # convert to k
  @sigma     = map {log($_)} @sigma;       # take logs of xsecs

  my $stat = Statistics::Descriptive::Full->new(); # fit the post edge
  $stat -> add_data(@sigma);
  my @a = $stat -> least_squares_fit(@energy);
  return -$a[1]/2;
};

=head2 C<self>

This calculates the correcion due to self-absorption fluorescence
experiment.  It assumes that the sample is infinately thick and that
the entry and exit angles of the photons are the same.

  $sigma_i0 = &self($central, $edge, $cell);

It takes the central atoms tag, the alphanumeric edge symbol, and a
populated cell.  It returns a list whose zeroth element is the
multiplicative amplitude correction and whose first element is the a
correction in units of Angstrom squared.

Note that the values returned depend on the data resource used.  See
L<Xray::Absorption>.

=cut

sub self {
  shift;
  my ($central, $edge, $rcount) = @_;
  my @list = keys %$rcount;

  my $emin = Xray::Absorption -> get_energy($central, $edge) ;
  my ($emax, $span, $npost) = (0, 800, 20);
  my ($el, $ed, $en) = Xray::Absorption -> next_energy($central, $edge, @list);
  if (not defined $en) {
    $emax = $emin + $span;
  } else {
    $emax = ( ($en - 10) < ($emin + $span) ) ? ($en - 10) : ($emin + $span);
  };
  ## need to show some care with the Chantler data
  (Xray::Absorption -> current_resource() =~ /chantler/i) and
    $emin += 50;
  ($emin >= $emax) and ($emin = $emax - 20); # whatever!

  ## calculate total absorption at the fluorescence energy and 10
  ## volts below the edge
  my $xmuf = 0;
  my $fline   = substr($edge, 0, 1) . "alpha";
  my $e_fluor = Xray::Absorption -> get_energy($central, $fline);
  #my $e_below = $emin - 10;
  foreach my $atom (@list) {
    $xmuf += scalar Xray::Absorption -> cross_section($atom, $e_fluor) * $$rcount{$atom};
  };

  ## load the post edge energies and sigmas
  my @i=(0..$npost-1);
  my $delta  = ($emax - $emin)/$npost;
  my @energy = map {$emin + $delta*$_} @i;

  my @sigma = ();
  foreach my $j (@i) {
    my $xmu = 0;
    my $xmu_core = 0;
    foreach my $atom (@list) {
      if (lc($atom) eq lc($central)) {
	$xmu_core += $$rcount{$atom} * Xray::Absorption -> cross_section($atom, $energy[$j]);
      } else {
	$xmu += $$rcount{$atom} * Xray::Absorption -> cross_section($atom, $energy[$j]);
      };
    };
    $sigma[$j] = ($xmuf+$xmu+$xmu_core)/($xmuf+$xmu);
  };

  @energy = map {ETOK * ($_-$emin)} @energy; # convert to k
  @sigma  = map {log($_)} @sigma;

  my $stat = Statistics::Descriptive::Full->new(); # fit the post edge
  $stat -> add_data(@sigma);
  my @a = $stat -> least_squares_fit(@energy);
  return (exp($a[0]), -$a[1]/2);
};


1;
__END__
