/******************************************************************************

  Copyright(c) 2006 Intel Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110,
  USA

  The full GNU General Public License is included in this distribution
  in the file called LICENSE.

  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR
  97124-6497

******************************************************************************/

/******************************************************************************

  To build:

	% gcc -I/usr/src/linux/include -Wall -o ipwstats ipwstats.c

  To run:

	% ./ipwstats --help

  TODO:
  * Add ability to set the rtap_filter via sysfs from this program:
     --filter    Lists the filter bitmask values
     --filter=N  Sets the filter to N

******************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <linux/rtnetlink.h>
#include <linux/netlink.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <linux/if_arp.h>
#include <linux/if.h>
#include <linux/llc.h>
#include <linux/sockios.h>

#include <netinet/in.h>

#include <time.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <dirent.h>

typedef signed char s8;
typedef unsigned char u8;
typedef unsigned short u16;
typedef signed short s16;
typedef unsigned int u32;
typedef signed int s32;
typedef unsigned long long u64;
typedef signed long long s64;

#include <net/ieee80211_radiotap.h>

/*****************************************************************************
 *
 * IEEE 802.11 Declarations
 *
 * We define these here as including net/ieee80211.h has too many other
 * typedef dependencies.  Also, depending on the net/ieee80211.h on the system,
 * some of these types may not be defined.
 *
 */
#define IEEE80211_MAC_LEN       6
#define IEEE80211_3ADDR_LEN     24

#define IEEE80211_FCTL_FTYPE	0x000c
#define IEEE80211_FCTL_STYPE	0x00f0
#define IEEE80211_FCTL_TODS	0x0100

#define IEEE80211_FTYPE_MGMT		0x0000
#define IEEE80211_FTYPE_CTL		0x0004
#define IEEE80211_FTYPE_DATA		0x0008

#define IEEE80211_STYPE_PROBE_RESP 0x0050
#define IEEE80211_STYPE_BEACON	   0x0080
#define IEEE80211_STYPE_DEAUTH     0x00C0

struct ieee80211_hdr {
	__le16 frame_ctl;
	__le16 duration_id;
	u8 payload[0];
} __attribute__ ((packed));

struct ieee80211_hdr_3addr {
	__le16 frame_ctl;
	__le16 duration_id;
	u8 addr1[IEEE80211_MAC_LEN];
	u8 addr2[IEEE80211_MAC_LEN];
	u8 addr3[IEEE80211_MAC_LEN];
	__le16 seq_ctl;
	u8 payload[0];
} __attribute__ ((packed));

/* IEEE 802.11h Declarations */
#define IEEE80211_STYPE_ACTION 0x00D0

enum {
	IEEE80211_IE_CSA = 37,
};

enum {
	IEEE80211_ACTION_SPECTRUM_MGMT = 0,
};

enum {
	IEEE80211_ACTION_CATEGORY_CHANNEL_SWITCH = 4,
};

struct ieee80211_channel_switch {
	u8 id;
	u8 len;
	u8 mode;
	u8 channel;
	u8 count;
} __attribute__ ((packed));

struct ieee80211h_csa {
	struct ieee80211_hdr_3addr header;
	u8 category;
	u8 action;
	struct ieee80211_channel_switch channel_switch;
} __attribute__ ((packed));

struct ieee80211_deauth {
	struct ieee80211_hdr_3addr header;
	__le16 reason;
} __attribute__ ((packed));

/*****************************************************************************/

#define VERSION "0.1"
#define COPYRIGHT "Copyright (C) 2006, Intel Corporation."

struct ipw_rt_hdr {
	struct ieee80211_radiotap_header rt_hdr;
	u64 rt_tsf;
	u8 rt_flags;
	u8 rt_rate;
	u16 rt_channel;
	u16 rt_chbitmask;
	s8 rt_dbmsignal;
	s8 rt_dbmnoise;
	u8 rt_antenna;
	u8 payload[0];
} __attribute__ ((packed));

/* 3945 Simulation Sysfs Interface declarations */

#define SYSFS_DRIVER "/sys/class/net/eth1/device"

struct priv {
	char sysfs_path[PATH_MAX];
	u8 ifname[IFNAMSIZ];
	int ifindex;
	u8 channel;

	int set_channel;
	int set_bss;
	int set_dst;
	int set_src;
	int set_type;
	int set_rate;
	int set_ifindex;
	int set_ifname;
};

int usage(char *argv[], int ret)
{
	if (ret)
		fprintf(stderr, "Invalid parameter %d: %s.\n\n",
			ret, argv[ret]);

	fprintf(stdout, "usage: ipwstats [--OPTION]\n"
		"options:\n"
		"  --channel\t\tFilter results to one channel (eg. 9)\n"
		"  --iface\t\tInterface name (eg. rtap0) or index (eg. 12)\n"
		"  --version\t\tDisplay version information\n"
		"  --help\t\tView this help message\n");

	return -ret;
}

enum {
	PARAM_NONE = 0,
	PARAM_VALUE = 1,
};

inline int param_check(const char *param, const char *value, int type)
{
	int len = strlen(value);
	int comp = strncmp(param, value, len);

	if (comp)
		return 0;

	if (type == PARAM_VALUE)	/* = */
		return (param[len] != '=') ? -1 : len + 1;

	/* no = */

	return (param[len] != '\0') ? -1 : len;
}

int version()
{
	fprintf(stdout, "ipwstats - S/N vs. throughput statistics\n");
	fprintf(stdout, "%s\n", COPYRIGHT);
	fprintf(stdout, "version: " VERSION "\n");
	return 0;
}

#ifndef ETH_P_80211_RAW
#define ETH_P_80211_RAW (ETH_P_ECONET + 1)
#define ETH_P_80211_RADIOTAP (ETH_P_80211_RAW + 1)
#endif

enum {
	CSA = 0,
	DEAUTH = 1,
};

static int inline ieee80211mhz2chan(u16 freq)
{
	if (freq == 2484)
		return 14;
	if (freq <= 2472)
		return (freq - 2407) / 5;
	return (freq / 5) - 1000;
}

#define ieee80211_is_probe_response(fc) \
   ((fc & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_MGMT && \
    (fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_PROBE_RESP )

#define ieee80211_is_management(fc) \
   ((fc & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_MGMT)

#define ieee80211_is_control(fc) \
   ((fc & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_CTL)

#define ieee80211_is_data(fc) \
   ((fc & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_DATA)

#define ieee80211_is_assoc_request(fc) \
   ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_ASSOC_REQ)

#define ieee80211_is_reassoc_request(fc) \
   ((fc & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_REASSOC_REQ)

int main(int argc, char *argv[])
{
	struct priv priv;
	int pos, i = 0;
	u8 frame[3072];
	u8 *p;
	struct ieee80211_radiotap_header *rt_hdr;
	struct ieee80211_hdr *hdr;
	struct ipw_rt_hdr rt;
	char *ftype;
	struct sockaddr_ll sll_addr = {
		.sll_family = AF_PACKET,
		.sll_ifindex = -1,
		.sll_protocol = htons(ETH_P_80211_RAW),
		.sll_hatype = ARPHRD_IEEE80211_RADIOTAP,
		.sll_pkttype = PACKET_OTHERHOST,
	};
	struct ifreq ifr = {
		.ifr_name = "rtap0",
	};
	int err, err_len;

	int sock;

	memset(&priv, 0, sizeof(priv));
	while (i < argc - 1) {
		i++;

		if (argv[i][0] != '-' || argv[i][1] != '-')
			return usage(argv, i);

		pos = param_check(&argv[i][2], "iface", PARAM_VALUE);
		if (pos < 0)
			return usage(argv, i);
		if (pos) {
			priv.ifindex = strtol(&argv[i][2 + pos], NULL, 0);
			if (priv.ifindex == 0) {
				priv.set_ifname = 1;
				strncpy(priv.ifname, &argv[i][2 + pos],
					sizeof(priv.ifname));
			} else
				priv.set_ifindex = 1;

			continue;
		}

		pos = param_check(&argv[i][2], "channel", PARAM_VALUE);
		if (pos < 0)
			return usage(argv, i);
		if (pos) {
			priv.channel = strtol(&argv[i][2 + pos], NULL, 0);
			priv.set_channel = 1;
			continue;
		}

		pos = param_check(&argv[i][2], "help", PARAM_NONE);
		if (pos < 0)
			return usage(argv, i);
		if (pos)
			return usage(argv, 0);

		pos = param_check(&argv[i][2], "version", PARAM_NONE);
		if (pos < 0)
			return usage(argv, i);
		if (pos)
			return version();

		return usage(argv, i);
	}

	if (priv.set_ifindex)
		ifr.ifr_ifindex = priv.ifindex;
	else if (priv.set_ifname)
		strncpy(ifr.ifr_name, priv.ifname, sizeof(ifr.ifr_name));

	sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_80211_RAW));
	if (sock == -1) {
		fprintf(stderr, "Error: socket() failed reason (%d): %s\n",
			errno, strerror(errno));
		return 1;
	}

	/* Figure out the index (if given name) or name (if given an index)
	 * of the target interface */
	if (!priv.set_ifindex) {
		if (ioctl(sock, SIOCGIFINDEX, &ifr) == -1) {
			fprintf(stderr, "Error: ioctl() failed reason (%d): "
				"%s\n", errno, strerror(errno));
			return 1;
		}
	} else if (priv.set_ifname) {
		/* User provided device index */
		if (ioctl(sock, SIOCGIFNAME, &ifr) == -1) {
			fprintf(stderr, "Error: ioctl() failed reason (%d): "
				"%s\n", errno, strerror(errno));
			return 1;
		}
	}

	sll_addr.sll_ifindex = ifr.ifr_ifindex;

	fprintf(stdout, "using: %s (%d)\n", ifr.ifr_name, ifr.ifr_ifindex);

	/* Bind to the appropiate interface */
	if (bind(sock, (struct sockaddr *)&sll_addr, sizeof(sll_addr))) {
		fprintf(stderr, "Error: bind() failed reason (%d): %s\n",
			errno, strerror(errno));
		return 1;
	}

	/* Verify no errors on socket */
	if (getsockopt(sock, SOL_SOCKET, SO_ERROR, &err, &err_len)) {
		fprintf(stderr, "Error: getsockopt() reason (%d): %s\n",
			errno, strerror(errno));
		return 1;
	}

	while (1) {
		/*    err = poll(5000, sock);
		   if (err == -1) {
		   fprintf(stderr, "poll: failed %d: %s\n",
		   errno, strerror(errno));
		   break;
		   } */

		err = recv(sock, frame, sizeof(frame), MSG_TRUNC);
		if (err == -1) {
			fprintf(stderr, "recv failed: %s (%d)\n",
				strerror(errno), errno);
			err = -errno;
			close(sock);
			return err;
		}

		if (err < sizeof(*rt_hdr)) {
			fprintf(stderr, "Short packet received.\n");
			continue;
		}

		rt_hdr = (void *)frame;
		p = (void *)frame;

		/* Check radiotap header... */
		if (rt_hdr->it_version != PKTHDR_RADIOTAP_VERSION) {
			fprintf(stderr, "driver / app protocol version "
				"mismatch (v%d vs. v%d).\n",
				PKTHDR_RADIOTAP_VERSION, rt_hdr->it_version);
			continue;
		}

		/* Extract radiotap header information */
		memset(&rt, 0, sizeof(rt));
		memcpy(&rt.rt_hdr, rt_hdr, sizeof(rt.rt_hdr));
		p += sizeof(rt.rt_hdr);

		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_TSFT))
			p += sizeof(rt.rt_tsf);
		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_FLAGS)) {
			rt.rt_flags = *(u8 *) p;
			p += sizeof(rt.rt_flags);
		}
		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_RATE)) {
			rt.rt_rate = *(u8 *) p;
			p += sizeof(rt.rt_rate);
		}
		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_CHANNEL)) {
			rt.rt_channel = *(u16 *) p;
			p += sizeof(rt.rt_channel);
			rt.rt_channel = ieee80211mhz2chan(rt.rt_channel);
			rt.rt_chbitmask = *(u16 *) p;
			p += sizeof(rt.rt_channel);
		}
		if (rt_hdr->it_present &
		    (1 << IEEE80211_RADIOTAP_DBM_ANTSIGNAL)) {
			rt.rt_dbmsignal = *(u8 *) p;
			p += sizeof(rt.rt_dbmsignal);
		}
		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_DBM_ANTNOISE)) {
			rt.rt_dbmnoise = *(u8 *) p;
			p += sizeof(rt.rt_dbmnoise);
		}
		if (rt_hdr->it_present & (1 << IEEE80211_RADIOTAP_ANTENNA)) {
			rt.rt_antenna = *(u8 *) p;
			p += sizeof(rt.rt_antenna);
		}

		if (priv.set_channel && rt.rt_channel != priv.channel)
			continue;

		hdr = (void *)p;
		if (ieee80211_is_management(hdr->frame_ctl))
			ftype = "MGMT";
		else if (ieee80211_is_control(hdr->frame_ctl))
			ftype = "CTRL";
		else if (ieee80211_is_data(hdr->frame_ctl))
			ftype = "DATA";
		else
			ftype = "UNKN";

		fprintf(stdout,
			"%s: Ch.%3d S/N =%4d/%4ddBm, rate = %d%sMbps\n",
			ftype, rt.rt_channel,
			rt.rt_dbmsignal, rt.rt_dbmnoise,
			rt.rt_rate >> 1, (rt.rt_rate & 1) ? ".5" : "");
	}

	close(sock);

	return 0;
}
