/* 
 *   Creation Date: <1999/03/15 17:29:19 samuel>
 *   Time-stamp: <2001/04/15 13:46:59 samuel>
 *   
 *	<mmu_lvhash.c>
 *	
 *	This file implements the hash table
 *		lvptr 	-> 	pte
 *
 *	One key might correspond to many PTEs (possibly around 20)
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "compat.h"
#include <linux/config.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <asm/unistd.h>

#include "mmu.h"
#include "mmu_contexts.h"

#define PERFORMANCE_INFO
#include "performance.h"

#define	LV_HASH_SIZE	(256*1024)	// Must equal 2^X

typedef struct hash_entry hash_entry_t;
typedef struct lvhash lvhash_t;

static int	build_hash( lvhash_t *ht, ulong size );
static void	release_hash( lvhash_t *ht );
static void 	hash( lvhash_t *ht, ulong key, ulong *pte );
static void 	clear_key_range( lvhash_t *ht, ulong first_key, int num_keys, int do_tlbie );
static void 	clear_hash( lvhash_t *ht );
static void 	clear_key( lvhash_t *ht, ulong key, int do_tlbie );
static void 	do_clear_key( hash_entry_t *he, int do_tlbie );

struct hash_entry {
	ulong 		key;
	ulong 		*pte;		/* pte == NULL if unused */
};

struct lvhash {
	hash_entry_t	*table;

	int		num_keys;	/* # keys actually in hash */
	int		max_num_keys;	/* upper limit (table cleared exceeded) */
	ulong		size;
	ulong		mask;		/* size / sizeof(hash_entry_t) -1  */
};

/*
 * User keys do not use the twelve low-order bits (4K pages).
 *
 * We use bit 0 as a VALID bit (since 0 is a valid key).
 * The value UNUSED_KEY is used for released keys.
 */

#define VALID_KEY_BIT	0x1
#define UNUSED_KEY	0x2


#define MMU			(kv->mmu)
#define DECLARE_LV_HASH		lvhash_t *lv_hash = MMU.lvhash

/************************************************************************/
/*	World interface							*/
/************************************************************************/

int
init_lvhash( kernel_vars_t *kv )
{
	lvhash_t *lv_hash;

	if( !(lv_hash = kmalloc( sizeof(lvhash_t), GFP_KERNEL )) )
		return 1;
	memset( lv_hash, 0, sizeof(lvhash_t) );
	
	if( build_hash( lv_hash, LV_HASH_SIZE )) {
		printk("hash_table: Failed building hash\n");
		kfree( lv_hash );
		return 1;
	}
	MMU.lvhash = lv_hash;
	return 0;
}

void 
cleanup_lvhash( kernel_vars_t *kv )
{
	DECLARE_LV_HASH;

	if( !lv_hash )
		return;

	release_hash( lv_hash );
	kfree( lv_hash );
	MMU.lvhash = NULL;
}

/* this function might be called before we are initialized */
void
unmap_lv_range( kernel_vars_t *kv, ulong lv, size_t size )
{
	DECLARE_LV_HASH;
	BUMP( unmap_lv_range );

	if( !lv_hash )
		return;

	lv &= ~0xfffUL;
	clear_key_range( lv_hash, lv, (size>>12), 1 );
}

void 
hash_lv_to_pte( kernel_vars_t *kv, ulong *lv, ulong *pte )
{
	DECLARE_LV_HASH;
	BUMP(hash_lv_to_pte);
	
	lv = (ulong*)((ulong)lv & ~0xfffUL);
	hash( lv_hash, (ulong)lv, pte );
}


/***********************************************************************
 * Implementation of the hash table.
 *
 *  We use double hashing with
 *
 * 	h(k,i) = (h1(k) + i*h2(k)) & mask
 *		
 *  with
 *
 *	h1 = (key >> 12)
 *	h2 = ((key >> 12) ^ (key>>19) ) | 1
 ***********************************************************************/

/* these function shoud not use the low order 12 bits of key */
#define HASH_H1( key )		( (key) >> 12 )
#define HASH_H2( key, h1 )	( ((h1) ^ ((key) >> 19)) | 1 )

static int
build_hash( lvhash_t *ht, ulong size )
{
	ht->size = size;
	ht->mask = size / sizeof(hash_entry_t) -1;
	ht->table = (hash_entry_t*)vmalloc( ht->size );

	if( !ht->table )
		return -1;

	memset( ht->table, 0, ht->size );
	ht->num_keys = 0;

	/* allow 64% of table to be used */
	ht->max_num_keys = (ht->mask * 64)/100;
	return 0;
}

static void
release_hash( lvhash_t *ht )
{
	vfree( ht->table );
}


static void
clear_key_range( lvhash_t *ht, ulong first_key, int num_keys, int do_tlbie )
{
	int i;

	for(i=0; i<num_keys; i++ )
		clear_key( ht, first_key + (i<<12), do_tlbie );
}

static void
hash( lvhash_t *ht, ulong key, ulong *pte )
{
	ulong 		h, h2;
	hash_entry_t 	*el;

/*	printk("HASH: key %08lX, PTE %08lX\n", key,pte ); */

	if( ht->num_keys >= ht->max_num_keys )
		clear_hash( ht );

	key |= VALID_KEY_BIT;
	h = HASH_H1( key );
	h2 = HASH_H2( key, h );

	for( ;; h += h2 ){
		h &= ht->mask;
		el = &ht->table[h];

		if( !(el->key & VALID_KEY_BIT) ){
			el->key = key;
			el->pte = pte;
			ht->num_keys++;
			return;
		}
		/* Guard against multiple addition of the same key */
		if( el->key == key && el->pte == pte )
			return;
	}
}

static void
clear_hash( lvhash_t *ht )
{
	int i;
	hash_entry_t *he = ht->table;
	
	BUMP(clear_hash);

	for(i=0; i<=ht->mask; i++, he++ )
		if( he->key & VALID_KEY_BIT )
			do_clear_key( he, 1 );

	memset( ht->table, 0, ht->size );
	ht->num_keys = 0;
}

static void
clear_key( lvhash_t *ht, ulong key, int do_tlbie )
{
	ulong 		h, h2;
	hash_entry_t 	*el;

	key |= VALID_KEY_BIT;
	h = HASH_H1( key );
	h2 = HASH_H2( key, h );

	for( ;; h += h2 ){
		h &= ht->mask;
		el = &ht->table[h];

		if( el->key == key ) {
			do_clear_key( el, do_tlbie );
			continue;
		}
		if( !el->key )
			break;
	}

}


/* This function clears the entry in the hash table,
 * and optionally looks up the effective address in the 
 * PTE hash table and perform a tlbie instruction.
 */
static void
do_clear_key( hash_entry_t *he, int do_tlbie )
{
	ulong ea;

	if( !(he->key & VALID_KEY_BIT ))
		return;

/*	printk("clear_key, KEY %08lX, PTE %08lX\n",he->key, he->pte ); */

#ifdef DEBUG_SA
	printk("clear_key, KEY %08lX, PTE %08lX\n",he->key, he->pte );
	return;
#endif

	he->key = UNUSED_KEY;
	if( do_tlbie ){
		/* at most BITs 13-19 are used for
		 * the tlbie instruction
		 */
		ea = (ulong)he->pte;
		
		/* Secondary hash? */
		if( (*he->pte & BIT(25) ))
			ea = ~ea;

		/* primary hash function */
		ea = ea ^ (*he->pte << 5);

		/* now bits 13-19 should be valid */
		*he->pte = 0;
		_tlbie(ea);
	} else {
		*he->pte = 0;
	}
}
