/**************************************************************
*   
*   Creation Date: <1998-11-20 16:18:20 samuel>
*   Time-stamp: <2001/10/09 00:15:28 samuel>
*   
*	<mmu_vsid.c>
*	
*   Conversion mac VSID -> linux VSID
*
*   Copyright (C) 1998, 1999, 2000, 2001 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <asm/mmu_context.h>
#include <asm/mmu.h>
#include <asm/unistd.h>
#include <asm/io.h>

#include "mmu.h"
#include "mmu_contexts.h"
#include "misc.h"
#include "asmfuncs.h"
#include "emu.h"

#define PERFORMANCE_INFO
#include "performance.h"

// #define DISABLE_CACHE

#define NUM_ROWS	512
#define NUM_COLS	32

static int s_next_mol_context;

/************************************************************************/
/*	Linux context allocation					*/
/************************************************************************/

static int vsids_initialized;

int
flush_all_PTEs( void )
{
	ulong v, sdr1 = _get_sdr1();
	ulong ea;
	ulong *pte = phys_to_virt( sdr1 & ~0xffff );
	int npte = ((((sdr1 & 0x1ff) << 16) | 0xffff) + 1) / 8;
	int i, count=0;

	/* SDR1 might not be initialized (yet) on the 603 */
	if( !vsids_initialized || !sdr1 )
		return 0;
	for(i=0; i<npte; i++, pte+=2 ) {
		v = *pte;
		if( !(v & BIT(0)) )	/* test V-bit */
			continue;
		v = (v & ~BIT(0)) >> 7;
		v = (v - ((v & 0xf) * MUNGE_ESID_ADD)) * MUNGE_MUL_INVERSE;
		v = (v>>4) & CTX_MASK;

		// What is the inverse of 
		if( v >= FIRST_MOL_CONTEXT && v <= LAST_MOL_CONTEXT ) {
 			*pte = 0;
			count++;
		}
	}

	/* 601 (7 bit), 603 (5 bit), 604,750,7400 (6 bit) */
	for( ea=0; ea <= (0x7f << 12); ea += 0x1000 )
		_tlbie(ea);
	/* SMP: _tlbsync() */
	
	/* printk("Flush_all_PTEs: %d\n", count ); */
	return count;
}

int
setup_mol_contexts( void )
{
	/* Remember: vsids live in the session-global namespace */
	if( vsids_initialized )
		return 0;

	s_next_mol_context = FIRST_MOL_CONTEXT;
	
#ifndef NEW_LINUX_MM
	if( atomic_read( mc_next_mmu_context ) > LAST_LINUX_CONTEXT ) {
		printk("Context overflow\n");
		/* let the real overflow occur as soon as possible */
		atomic_set( mc_next_mmu_context, LAST_LINUX_CONTEXT );

		/* mostly unnecessary */
		flush_all_PTEs();
		vsids_initialized = 0;
		return 1;
	}
#endif
	vsids_initialized = 1;
	flush_all_PTEs();
	return 0;
}

void
release_mol_contexts( void )
{
	/* To be called when the last session exits! */
	flush_all_PTEs();
	vsids_initialized = 0;
}

#ifndef NEW_LINUX_MM
void
linux_context_overflow( void )
{
	int v = atomic_read( mc_next_mmu_context );
	
	if( v >= LAST_CONTEXT )
		return;
	if( v >= FIRST_MOL_CONTEXT )
		printk("MOL internal error: contexts overwritten\n");
	// Linux will notice this and wrap around
	atomic_set( mc_next_mmu_context, LAST_CONTEXT );
}
#endif

static int
alloc_contexts( int n )
{
	int i, ret;

	if( s_next_mol_context + n > LAST_MOL_CONTEXT ) {
		s_next_mol_context = FIRST_MOL_CONTEXT;
		printk("MOL context overflow\n");
		flush_all_PTEs();
		
		for( i=0; i<MAX_NUM_SESSIONS; i++ ) {
			kernel_vars_t *kv;
			if( !(kv=g_sesstab->kvars[i]) )
				continue;
			clear_vsids( kv, 0 );
		}
	}
	ret = s_next_mol_context;
	s_next_mol_context += n;
	
	return ret;
}


/************************************************************************/
/*	mac_vsid <-> linux_vsid conversion				*/
/************************************************************************/

/*
 * VSID is 24 bits, the top 20 bits are translated mac -> linux
 * table dimensions (*warning* hardcoded in vsid_lookup)
 * both should have sizes of the form 2^x
 *
 * The translation table contains NUM_ROWS*NUM_COLS entries;
 * bits 11-19 of the mac-vsid are used to index the table.
 */ 

typedef union vsid_entry {
	struct 	{
	 ulong 		lvsid:20;	/* top 20 bits of linux vsid */
	 ulong		v:1;		/* valid bit */
	 ulong 		mvsid:11;	/* top 11 bits of mac-vsid */
	} s;
	ulong		word;
} vsid_entry_t;

typedef struct vsid_head_ {
	int		cur;		/* index of last inserted */
	vsid_entry_t	*entries;
} vsid_head_t;

typedef struct vsid_data {
	vsid_head_t	*vtable;
	ulong		cache[3];
} vsid_data_t;

#define VSID_TABLE_SIZE	(sizeof(vsid_head_t)*NUM_ROWS + NUM_ROWS*NUM_COLS*sizeof(vsid_entry_t))
#define MMU		(kv->mmu)
#define DECLARE_VD	vsid_data_t *vd = MMU.vsid_data


/************** IMPLEMENTATION *************************/

int
init_vsid_table( kernel_vars_t *kv )
{
	vsid_data_t *vd;

	if( !(vd = kmalloc( sizeof(vsid_data_t), GFP_KERNEL )) )
		return 1;
	memset( vd, 0, sizeof(vsid_data_t) );

	if( !(vd->vtable = (vsid_head_t*) vmalloc( VSID_TABLE_SIZE )) ) {
		kfree( vd );
		return 1;
	}
	MMU.vsid_data = vd;

	flush_vsid_table( kv );
	return 0;
}


void
cleanup_vsid_table( kernel_vars_t *kv )
{
	DECLARE_VD;
	if( !vd )
		return;

	if( vd->vtable )
		vfree( vd->vtable );

	kfree( vd );
	MMU.vsid_data = NULL;
}


/* Initialize/flush the vsid table */
void
flush_vsid_table( kernel_vars_t *kv )
{
	DECLARE_VD;
	vsid_entry_t	*se;
	int 		i;
       
	BUMP(flush_vsid_table);
	
	memset( (void*)vd->vtable, 0, VSID_TABLE_SIZE );

	/* clear cache entry */
	vd->cache[0] = -1;

	se = (vsid_entry_t*)(vd->vtable + NUM_ROWS);

	for(i=0; i<NUM_ROWS; i++ ){
		vd->vtable[i].entries = se;
		vd->vtable[i].cur = 0;
		se += NUM_COLS;
	}
}


/*
 * mac vsid -> linux vsid. Note that this function depends on
 * the algorithm of MUNGE_CONTEXT (to calculate su_ret)
 */
void 
vsid_lookup( kernel_vars_t *kv, ulong mac_vsid, ulong *user_ret, ulong *su_ret )
{
	DECLARE_VD;
	vsid_head_t	*head;	
	vsid_entry_t 	*se, x, cword;
	ulong	vsid;
	int i;

	mac_vsid &= 0xfffff0;

#ifndef DISABLE_CACHE
	/* we cache last access */
	if( vd->cache[0]==mac_vsid ){
		*user_ret=vd->cache[1];
		*su_ret=vd->cache[2];
		return;
	}
	vd->cache[0]=mac_vsid;
#endif

	head = &vd->vtable[ (mac_vsid >> 4) & (NUM_ROWS-1) ];
	
	se = head->entries;

	cword.word = (1<<11) + (mac_vsid>>13);

	for(i=1; i<=NUM_COLS; i++){
		/* Search the table backwards, starting with the
		 * last inserted entry.
		 */
		x = se[ (head->cur - i) & (NUM_COLS-1) ];
		if( (x.word & 0xfff) != cword.word )
			continue;
		vd->cache[1]=*user_ret = x.s.lvsid << 4;
		vd->cache[2]=*su_ret = ((x.s.lvsid + MUNGE_ADD_NEXT) & CTX_MASK) << 4;
		return;
	}

	/* Allocate new VSIDs.
	 * We use _two_ context numbers, one for user access,
	 * one for supervisor access. This is necessary for the 
	 * implementation of the Vs/Vu bits of the BATS.
	 */
	vsid = MUNGE_CONTEXT( alloc_contexts( 2 ) );
	vd->cache[1]=*user_ret = vsid << 4;
	vd->cache[2]=*su_ret = ((vsid + MUNGE_ADD_NEXT) & CTX_MASK) << 4; 
	
	cword.s.lvsid = vsid;

	se[head->cur] = cword;
	head->cur = (head->cur+1) & (NUM_COLS-1);

	BUMP( vsid_add );
}


/*
 * Discard all virtual segment IDs allocated/used previously. 
 * This has the same effect as removing all of our entries from the 
 * TLB hash table. Also called upon initialization to setup the
 * segment registers.
 */
void
clear_vsids( kernel_vars_t *kv, int is_init )
{
	ulong sr_unmapped;
	int i;
	int ctx;
	
	/* printk("clear_vsids %d\n", is_init ); */

	/* IMPORANT: alloc_contexts() will call us when there is an overflow
	 * and all contexts are invalidated. Thus, we must be careful not to
	 * store contexts in local variables...
	 */

	/* allocate new VSID for unmapped MMU mode, see <asm/mmu_context.h>
	 * set Ku, but not Ks.
	 */
	ctx = alloc_contexts( 2 );
	
	sr_unmapped = MUNGE_CONTEXT(ctx) << 4;
	MMU.illegal_sr = MUNGE_CONTEXT(ctx+1) << 4;
	
	/* setup unmapped context */
	for(i=0; i<16; i++) {
		/* MMU.split_sr is for the 601 only */
		MMU.split_sr[i] = ((MMU.illegal_sr + i * MUNGE_ESID_ADD) & VSID_MASK) | BIT(2);
		MMU.unmapped_sr[i] = ((sr_unmapped + i * MUNGE_ESID_ADD) & VSID_MASK) | BIT(2);
	}

	/* Invalidate all VSIDs in order to make sure
	 * no stale PTEs are left.
	 */
	flush_vsid_table( kv );

	/* and make sure new ones are allocated */
	for( i=0; i<16; i++ )
		do_mtsr( kv, i, is_init ? 0 : kv->mregs.segr[i] );

	/* XXX WHAT ABOUT SPLITMODE? DOES WE CACHE VSIDS? */
}
