#include <glib.h>
#include <string.h>
#include <stdlib.h>

#include "gprop.h"

static struct {
	char	*id;
	int	type;
} typetab[] = {
	{"str.", GPROP_STR} ,
	{"int.", GPROP_INT} ,
	{"bool." , GPROP_BOOL} ,
};

GSList *gprop_list = NULL;

gprop *gprop_parse(str)
char *str;
{
	gprop *gp;
	int i;
	char *p = NULL;

	gp = g_malloc(sizeof(gprop));

	for (i = 0; i < (sizeof(typetab)/sizeof(typetab[0])) ; i++)
	{
		if (!strncmp(str, typetab[i].id , strlen(typetab[i].id)))
		{
			gp->type = typetab[i].type;
			p = str + strlen(typetab[i].id);
		}
	}

	if (!p)
	{
		g_free(gp);
		return NULL;
	}

	i = strcspn(p, "=");

	if (!i)
	{
		g_free(gp);
		return NULL;
	}

	gp->name = g_strndup(p, i);

	p += i;
	p += strspn(p, "=");

	switch (gp->type)
	{
		case GPROP_STR:
			gp->value = g_strdup(p);
		break;
		case GPROP_INT:
			gp->value = (void *)atoi(p);
		break;
		case GPROP_BOOL:
			gp->value = (void *)(!strcmp(p, "true"));
		break;
	}

	return gp;
}

char *gprop_dump(gp)
gprop *gp;
{
	static char pom[256];
	char *p = pom;

	switch (gp->type)
	{
		case GPROP_STR:
			sprintf(pom, "%s%s=%s", typetab[gp->type].id,
				gp->name, (char *)gp->value);
		break;
		case GPROP_INT:
			sprintf(pom, "%s%s=%d", typetab[gp->type].id,
				gp->name, (int)gp->value);
		break;
		case GPROP_BOOL:
			sprintf(pom, "%s%s=%s", typetab[gp->type].id,
				gp->name, gp->value ? "true" : "false");
		break;
	}

	return p;
}

void gprop_add(gp)
gprop *gp;
{
	gprop_list = g_slist_append(gprop_list, gp);
}


static int gprop_find(gp, name)
gprop *gp;
char *name;
{
	return strcmp(name, gp->name);
}

void gprop_set_str(name, value)
char *name;
char *value;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		g_free(gp->value);
		gp->type = GPROP_STR;
		gp->value = g_strdup(value);
	}
	else
	{
		gp = g_malloc(sizeof(gprop));

		gp->type = GPROP_STR;
		gp->name = g_strdup(name);
		gp->value = g_strdup(value);

		gprop_list = g_slist_append(gprop_list, gp);
	}
}

void gprop_set_int(name, value)
char *name;
int value;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		gp->type = GPROP_INT;
		gp->value = (void *)value;
	}
	else
	{
		gp = g_malloc(sizeof(gprop));

		gp->type = GPROP_INT;
		gp->name = g_strdup(name);
		gp->value = (void *)value;

		gprop_list = g_slist_append(gprop_list, gp);
	}
}

void gprop_set_bool(name, value)
char *name;
int value;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		gp->type = GPROP_BOOL;
		gp->value = (void *)value;
	}
	else
	{
		gp = g_malloc(sizeof(gprop));

		gp->type = GPROP_BOOL;
		gp->name = g_strdup(name);
		gp->value = (void *)value;

		gprop_list = g_slist_append(gprop_list, gp);
	}
}

int gprop_get_str(name, valp)
char *name;
char **valp;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		*valp = (char *)gp->value;
	}

	return (node != NULL);
}

int gprop_get_int(name, valp)
char *name;
int *valp;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		*valp = (int)gp->value;
	}

	return (node != NULL);
}

int gprop_get_bool(name, valp)
char *name;
int *valp;
{
	GSList *node;
	gprop *gp;

	node = g_slist_find_custom(gprop_list, name, (GCompareFunc)gprop_find);

	if (node)
	{
		gp = (gprop *)node->data;

		*valp = (int)gp->value;
	}

	return (node != NULL);
}

void gprop_save(f)
FILE *f;
{
	GSList *lst = gprop_list;
	char *p;

	while(lst)
	{
		p = (char *)gprop_dump((gprop *)lst->data);
		if (p)
		{
			fprintf(f , "Property: %s\n" , p);
		}
		lst= lst->next;
	}
}

