/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


//---------------------------------------------------------------------------

#ifndef PLATFORM_UNIX
#include <vcl.h>
#pragma hdrstop
#else
#include "linux_inc.h"
#endif

#include "art.h"
#include "console.h"

#define ReadWord(x) (*((short *)&(x)))
#define ReadLong(x) (*((long *)&(x)))

Art::Art()
{
    fileCount=0;
    for (int i=0;i<MAX_TILES;i++)
    {
        data[i]=NULL;
        for (int j=0;j<MAX_PAL;j++)
            tex[j][i]=NULL;
        sizeX[i]=origSizeX[i]=0;
        sizeY[i]=origSizeY[i]=0;
    }
}

Art::~Art()
{
    for (int i=0;i<MAX_TILES;i++)
    {
        for (int j=0;j<MAX_PAL;j++)
        {
            if (tex[j][i]) { delete tex[j][i]; tex[j][i]=NULL; }
        }
        if (data[i]) { delete[] data[i]; data[i]=NULL; }
    }
}

void Art::LoadPalette(int p,GroupFile *group,char *name)
{
    // Load the palette file into a buffer
#ifdef PLATFORM_UNIX
    char *buf=new char[group->FileSize(name)];
#else
    unsigned char *buf=new unsigned char[group->FileSize(name)];
#endif
    group->LoadFile(name,(char*)buf);
    // Convert the VGA colors in the palette file
    // (6 bits per component) into 16-bit colors
    // (5 bits per component and one bit for alpha)
    for (int i=0;i<256;i++)
    {
        palR[p][i]=buf[i*3]<<2;
        palG[p][i]=buf[i*3+1]<<2;
        palB[p][i]=buf[i*3+2]<<2;
        pal[p][i]=((buf[i*3]>>1)<<10)|((buf[i*3+1]>>1)<<5)|
            (buf[i*3+2]>>1)|0x8000;
    }
    pal[p][255]=0;
    delete[] buf;
    render->LoadPalette(p,pal[p]);
    palUsed[p]=1;
}

void Art::LoadPalette(int p,unsigned char *buf)
{
    // Convert the VGA colors in the palette data
    // (6 bits per component) into 16-bit colors
    // (5 bits per component and one bit for alpha)
    for (int i=0;i<256;i++)
    {
        palR[p][i]=buf[i*3]<<2;
        palG[p][i]=buf[i*3+1]<<2;
        palB[p][i]=buf[i*3+2]<<2;
        pal[p][i]=((buf[i*3]>>1)<<10)|((buf[i*3+1]>>1)<<5)|
            (buf[i*3+2]>>1)|0x8000;
    }
    pal[p][255]=0;
    render->LoadPalette(p,pal[p]);
    palUsed[p]=1;
}

void Art::FreePalette(int p)
{
    palUsed[p]=0;
}

int Art::FindFreePalette()
{
    for (int i=MAX_PAL-1;i>=0;i--)
    {
        if (!palUsed[i])
            return i;
    }
    cprintf("Warning: Palette count exceeded maximum\n");
    return MAX_PAL-1;
}

void Art::Import(GroupFile *group,char *name)
{
    // Load the art file header into a buffer and read
    // the info
    char *buf=new char[16];
    group->LoadFileB(name,buf,0,16);
    if (ReadLong(buf[0])!=1)
        throw "Incorrect art version";
    int start=ReadLong(buf[8]);
    int end=ReadLong(buf[12]);
    delete[] buf;
    // Load the size and animation information into
    // a buffer and copy it into the correct arrays
    buf=new char[8*(end-start+1)];
    group->LoadFileB(name,buf,16,8*(end-start+1));
    int index=0;
    memcpy(&sizeX[start],&buf[index],2*(end-start+1));
    index+=2*(end-start+1);
    memcpy(&sizeY[start],&buf[index],2*(end-start+1));
    index+=2*(end-start+1);
    memcpy(&anim[start],&buf[index],4*(end-start+1));
    index+=4*(end-start+1);
    delete[] buf;
    // Store the texture offset information for
    // dynamic loading later
    for (int tile=start;tile<=end;tile++)
    {
        fileNum[tile]=fileCount;
        fileOfs[tile]=index+16;
        index+=sizeX[tile]*sizeY[tile];
        data[tile]=NULL;
        for (int i=0;i<MAX_PAL;i++)
            tex[i][tile]=NULL;
        // Save the original size of the texture for
        // sprites (the size will later be converted
        // to the nearest power of 2)
        origSizeX[tile]=sizeX[tile];
        origSizeY[tile]=sizeY[tile];
    }
    // Save the filename for dynamic loading later
    fileGroup[fileCount]=group;
    fileNames[fileCount]=new char[strlen(name)+1];
    strcpy(fileNames[fileCount++],name);
}

void Art::ImportAll(GroupFile *group)
{
    // Get the list of all file names in all
    // open group files
    char *files=new char[group->NumFiles()*13];
    group->FileList(files);
    // Loop though the names and import all
    // art files
    for (int i=0;i<group->NumFiles();i++)
    {
        if ((!strncmp(&files[i*13],"tiles",5))&&
            (!strncmp(&files[i*13+8],".art",4)))
            Import(group,&files[i*13]);
    }
    delete[] files;
}

void Art::RequestTexture(int tile,int pn,int type)
{
    // Return if the texture is already allocated
    if (tex[pn][tile])
        return;
    // Return if the texture is blank
    if (sizeX[tile]==0||sizeY[tile]==0)
        return;
    // Calculate the nearest power of 2 for the width
    // and height, always rounding up
    int newW=1;
    int newH=1;
    if (type==TEXTYPE_REPEAT)
    {
        while (newW<sizeX[tile])
            newW<<=1;
        while (newH<sizeY[tile])
            newH<<=1;
    }
    else
    {
        while (newW<(sizeX[tile]+2))
            newW<<=1;
        while (newH<(sizeY[tile]+2))
            newH<<=1;
    }
    // Load the texture data from the art file if it
    // hasn't already been loaded
    if (!data[tile])
    {
#ifdef PLATFORM_UNIX
        char *buf=new char[sizeX[tile]*sizeY[tile]];
#else
        unsigned char *buf=new unsigned char[sizeX[tile]*sizeY[tile]];
#endif
        data[tile]=new unsigned char[newW*newH];
        fileGroup[fileNum[tile]]->LoadFileB(fileNames[fileNum[tile]],buf,fileOfs[tile],sizeX[tile]*sizeY[tile]);
        // Convert the data from the top to bottom format
        // to the normal left to right format
        if (type==TEXTYPE_REPEAT)
        {
            // Repeat the texture if it was not an even
            // power of 2 in width or height
            int oldY=0;
            for (int y=0;y<newH;y++)
            {
                int oldX=0;
                for (int x=0;x<newW;x++)
                {
                    data[tile][y*newW+x]=buf[oldX*origSizeY[tile]+oldY];
                    oldX=(oldX+1)%origSizeX[tile];
                }
                oldY=(oldY+1)%origSizeY[tile];
            }
        }
        else if (type==TEXTYPE_BITMAP)
        {
            // Duplicate the edges by one pixel to prevent
            // incorrect texturing of "bitmaps" by hardware
            // accelerated renderers
            int oldY=0;
            for (int y=0;y<newH;y++)
            {
                int oldX=0;
                for (int x=0;x<newW;x++)
                {
                    data[tile][y*newW+x]=buf[oldX*origSizeY[tile]+oldY];
                    if ((x>=1)&&(oldX<(origSizeX[tile]-1)))
                        oldX++;
                }
                if ((y>=1)&&(oldY<(origSizeY[tile]-1)))
                    oldY++;
            }
        }
        else if (type==TEXTYPE_BITMAP_NOBORDER)
        {
            // Make a one-pixel transparent border around
            // the edge of the texture for correct rendering
            // of such textures as letters and numbers
            int oldY=-1;
            for (int y=0;y<newH;y++)
            {
                int oldX=-1;
                for (int x=0;x<newW;x++)
                {
                    if ((oldX<0)||(oldX>=origSizeX[tile])||
                        (oldY<0)||(oldY>=origSizeY[tile]))
                        data[tile][y*newW+x]=255;
                    else
                        data[tile][y*newW+x]=buf[oldX*origSizeY[tile]+oldY];
                    oldX++;
                }
                oldY++;
            }
        }
        delete[] buf;
    }
    // Create the texture object
    tex[pn][tile]=new Texture;
    tex[pn][tile]->location=TLOC_MEMORY;
    tex[pn][tile]->memTex.width=newW;
    tex[pn][tile]->memTex.height=newH;
    tex[pn][tile]->memTex.data=new unsigned short[newW*newH];
    tex[pn][tile]->pal=pn;
    tex[pn][tile]->type=type;
    tex[pn][tile]->data=data[tile];
    // Copy the texture data, converting from 8-bit
    // to 16-bit in the process.
    for (int y=0;y<newH;y++)
    {
        for (int x=0;x<newW;x++)
            tex[pn][tile]->memTex.data[y*newW+x]=pal[pn][data[tile][y*newW+x]];
    }
    // Save the new width and new height values
    sizeX[tile]=newW;
    sizeY[tile]=newH;
    // Load the texture into the video card's texture memory
    render->CreateMemoryTexture(tex[pn][tile]);
}

void Art::RequestTextureData(int tile)
{
    // Return if the texture is blank
    if (sizeX[tile]==0||sizeY[tile]==0)
        return;
    // Calculate the nearest power of 2 for the width
    // and height, always rounding up
    int newW=1;
    int newH=1;
    while (newW<sizeX[tile])
        newW<<=1;
    while (newH<sizeY[tile])
        newH<<=1;
    // Load the texture data from the art file if it
    // hasn't already been loaded
    if (!data[tile])
    {
#ifdef PLATFORM_UNIX
        char *buf=new char[sizeX[tile]*sizeY[tile]];
#else
        unsigned char *buf=new unsigned char[sizeX[tile]*sizeY[tile]];
#endif
        data[tile]=new unsigned char[newW*newH];
        fileGroup[fileNum[tile]]->LoadFileB(fileNames[fileNum[tile]],buf,fileOfs[tile],sizeX[tile]*sizeY[tile]);
        // Convert the data from the top to bottom format
        // to the normal left to right format
        // Repeat the texture if it was not an even
        // power of 2 in width or height
        int oldY=0;
        for (int y=0;y<newH;y++)
        {
            int oldX=0;
            for (int x=0;x<newW;x++)
            {
                data[tile][y*newW+x]=buf[oldX*origSizeY[tile]+oldY];
                oldX=(oldX+1)%origSizeX[tile];
            }
            oldY=(oldY+1)%origSizeY[tile];
        }
        delete[] buf;
    }
    // Save the new width and new height values
    sizeX[tile]=newW;
    sizeY[tile]=newH;
}

void Art::ReloadTextures()
{
    // Go though all textures and reload them into the
    // video card's texture memory.  This function
    // only works if the renderer object that created
    // the texture still exists
    for (int i=0;i<MAX_TILES;i++)
    {
        for (int j=0;j<MAX_PAL;j++)
        {
            if (tex[j][i])
                render->ReloadTexture(tex[j][i]);
        }
    }
}
//---------------------------------------------------------------------------

void Art::RecreateTextures()
{
    // Go though all textures and reload them into the
    // video card's texture memory.  This function is
    // used when the renderer object that created the
    // texture has been deleted and replaced
    for (int i=0;i<MAX_TILES;i++)
    {
        for (int j=0;j<MAX_PAL;j++)
        {
            if (tex[j][i])
            {
                tex[j][i]->data=data[i];
                render->CreateMemoryTexture(tex[j][i]);
            }
        }
    }
}

void Art::FreeTextureMem()
{
    // Free all texture memory (but not the original
    // data in system memory)
    for (int i=0;i<MAX_TILES;i++)
    {
        for (int j=0;j<MAX_PAL;j++)
        {
            if (tex[j][i])
                render->FreeTexture(tex[j][i]);
        }
    }
}

void Art::ReloadPalettes()
{
    for (int i=0;i<MAX_PAL;i++)
    {
        if (palUsed[i])
            render->LoadPalette(i,pal[i]);
    }
}

int Art::GetTextureCount()
{
    int count=0;
    for (int i=0;i<MAX_TILES;i++)
    {
        for (int j=0;j<MAX_PAL;j++)
        {
            if (tex[j][i])
                count++;
        }
    }
    return count;
}

int Art::GetTextureCount(int pal)
{
    int count=0;
    for (int i=0;i<MAX_TILES;i++)
    {
        if (tex[pal][i])
            count++;
    }
    return count;
}

int Art::GetMaxTextureID()
{
    int max=0;
    for (int i=0;i<MAX_TILES;i++)
    {
        if (sizeX[i])
            max=i;
    }
    return max;
}
#pragma package(smart_init)
