// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/cg_shift.cpp,v 1.2 2001/08/13 09:59:51 xhshi Exp $
//


#include "defines.h"
#define INLINE_LONG_SHIFT 1

#include "jit_intf.h"
#include "jit_runtime_support.h"
#include <iostream.h>
#include "code_emitter.h"
#include "stack.h"
#include "operand.h"
#include "lazy_code_selector.h"
#include "cg_shift.h"

extern void make_dst_killable(Mem_Manager& mem_manager,Code_Emitter& emitter, Stack& stack, 
							  Operand*& dst, Operand*& src,int is_commutable);

void gen_shift(Mem_Manager& mem_manager,
			   Code_Emitter& emitter, Stack& stack,
			   X86_Shift_Opcode opc) {
	//
	//	mov  ecx, [esp + off_1]  --- loadin shift amount (if it is in a variable)
	//  mov  eax, [esp + off_2]  --- loadin the value that is to be shifted
    //  shl  eax, CL
	//
	Operand *shift = stack.pop();
	if ( shift->kind != Operand::Imm && // imm can be folded into x86 shl/sal
		(!shift->is_reg() ||            // mem cannot be folded into shl/sal
		 !shift->contain(ecx_reg))) {   // shift amount must be in ecx_reg
		//
		// Because we need to load the number of bit positions to be shifted into ecx,
		// we free up ecx.
		//
		stack.spill_opnds_contain(ecx_reg);
		shift->free_opnd(&stack.reg_manager);
		Reg_Operand *sh_reg = stack.reg_manager.get_reg(ecx_reg);
		shift->emit_mov_to_reg(emitter,&sh_reg->opnd);
		shift = sh_reg;
	}
	Operand *dst = stack.pop();

	make_dst_killable(mem_manager,emitter,stack,dst,shift,0); // not commutable
	shift->free_opnd(&stack.reg_manager);
	dst->emit_shift(emitter,opc,shift);
	stack.push(dst);
}

#ifdef INLINE_LONG_SHIFT
void gen_lshl(Mem_Manager& mem_manager,
			   Code_Emitter& emitter, Stack& stack,
			   Pre_Alloc_Operand_Pool& op_pool,
			   Reg_Operand *src_lo,
			   Reg_Operand *src_hi,
			   int val,
			   Reg_Operand *scratch_reg) {

	src_lo->emit_mov_to_reg(emitter, &scratch_reg->opnd);
	emitter.emit_shift(shl_opc, &src_hi->opnd, &Imm_Opnd(val));
	emitter.emit_shift(shr_opc, &scratch_reg->opnd, &Imm_Opnd(32 - val));
	emitter.emit_shift(shl_opc, &src_lo->opnd, &Imm_Opnd(val));
	emitter.emit_alu(or_opc, &src_hi->opnd, &scratch_reg->opnd);
} //gen_lshl

void gen_lshr(Mem_Manager& mem_manager,
			   Code_Emitter& emitter, Stack& stack,
			   Pre_Alloc_Operand_Pool& op_pool,
			   Reg_Operand *src_lo,
			   Reg_Operand *src_hi,
			   int val,
			   Reg_Operand *scratch_reg) {
	src_hi->emit_mov_to_reg(emitter, &scratch_reg->opnd);
	emitter.emit_shift(sar_opc, &src_hi->opnd, &Imm_Opnd(val));
	emitter.emit_shift(shl_opc, &scratch_reg->opnd, &Imm_Opnd(32 - val));
	emitter.emit_shift(shr_opc, &src_lo->opnd, &Imm_Opnd(val));
	emitter.emit_alu(or_opc, &src_lo->opnd, &scratch_reg->opnd);
} //gen_lshr

void gen_lushr(Mem_Manager& mem_manager,
			   Code_Emitter& emitter, Stack& stack,
			   Pre_Alloc_Operand_Pool& op_pool,
			   Reg_Operand *src_lo,
			   Reg_Operand *src_hi,
			   int val,
			   Reg_Operand *scratch_reg) {
	if(val == 32) {
		src_hi->emit_mov_to_reg(emitter, &src_lo->opnd);
		src_hi->emit_alu_inst(emitter, &src_hi->opnd, xor_opc);
	} else if(val < 32) {
		src_hi->emit_mov_to_reg(emitter, &scratch_reg->opnd);
		emitter.emit_shift(shr_opc, &src_hi->opnd, &Imm_Opnd(val));
		emitter.emit_shift(shl_opc, &scratch_reg->opnd, &Imm_Opnd(32 - val));
		emitter.emit_shift(shr_opc, &src_lo->opnd, &Imm_Opnd(val));
		emitter.emit_alu(or_opc, &src_lo->opnd, &scratch_reg->opnd);
	} else { // val in 33..63
		assert(0);
	}


} //gen_lushr
#endif // INLINE_LONG_SHIFT

//
//	mov  ecx, [esp + off_1]  -- loadin shift amount (if it is in a variable)
//  mov  eax, [esp + off_2]  -- loadin the value (low 32 bit) that is to be shifted
//  mov  edx, [esp + off_2+4]  -- high 32 bit
//	call    _allshl
//          in:  EDX:EAX,CL
//          out: EDX:EAX
//          use: ECX
//
void gen_long_shift(Mem_Manager& mem_manager,
					Code_Emitter& emitter, Stack& stack,
					Code_Patch *& code_patch_list,
					Pre_Alloc_Operand_Pool& op_pool,
					ORP_RT_SUPPORT help_func) {
	Operand *src_lo, *src_hi;
	Operand *shift = stack.pop();
	Stack_Operand *stk_shift = op_pool.nth_stack(stack.depth());
	stack.pop64(src_lo,src_hi);
	Stack_Operand *stk_hi = op_pool.nth_stack(stack.depth());
	//
	// loadin the value that is to be shifted into eax and edx
	// If shift uses either eax or edx, we then free up shift operand. 
	//


#ifdef INLINE_LONG_SHIFT
	if(shift->kind == Operand::Imm) {
		int val = ((Imm_Operand*)shift)->imm_opnd.value;
		val &= 0x3f;
		if(stack.reg_manager.has_available_reg()) {
			bool code_generated = false;
			Reg_Operand *scratch_reg = stack.reg_manager.get_reg();
			Reg_Operand *src_lo_reg;
			if (src_lo->is_mem() && stack.reg_manager.has_available_reg()) {
				src_lo_reg = stack.reg_manager.get_reg();
			} else {
				src_lo_reg = NULL;
			}
			Reg_Operand *src_hi_reg;
			if (src_hi->is_mem() && stack.reg_manager.has_available_reg()) {
				src_hi_reg = stack.reg_manager.get_reg();
			} else if(src_hi->is_reg()) {
				src_hi_reg = (Reg_Operand *)src_hi;
			} else {
				src_hi_reg = NULL;
			}
			if(src_lo_reg && src_hi_reg) {
				src_lo->emit_mov_to_reg(emitter, &src_lo_reg->opnd);
				src_hi->emit_mov_to_reg(emitter, &src_hi_reg->opnd);
				switch(help_func) {
				case ORP_RT_LSHL:
					if(val < 32) {
						gen_lshl(mem_manager, emitter, stack, op_pool, 
							src_lo_reg, src_hi_reg,	val, scratch_reg);
						code_generated = true;
					}
					break;
				case ORP_RT_LSHR:
					if(val < 32) {
						gen_lshr(mem_manager, emitter, stack, op_pool,
							src_lo_reg, src_hi_reg,	val, scratch_reg);
						code_generated = true;
					}
					break;
				case ORP_RT_LUSHR:
					if(val <= 32) {
						gen_lushr(mem_manager, emitter, stack, op_pool, 
							src_lo_reg, src_hi_reg,	val, scratch_reg);
						code_generated = true;
					}
					break;
				default:
					break;
				}
			}
			scratch_reg->free_opnd(&stack.reg_manager);

			if(code_generated) {
				if (src_lo_reg != src_lo) src_lo->free_opnd(&stack.reg_manager);
				if (src_hi_reg != src_hi) src_hi->free_opnd(&stack.reg_manager);
				stack.push64(src_lo_reg, src_hi_reg);
				return;
			} else {
				if(src_lo_reg && src_lo_reg != src_lo)
					src_lo_reg->free_opnd(&stack.reg_manager);
				if(src_hi_reg && src_hi_reg != src_hi)
					src_hi_reg->free_opnd(&stack.reg_manager);
			}
		}
	}
#endif // INLINE_LONG_SHIFT

	if (shift->contain(eax_reg) || shift->contain(edx_reg)) {
		gen_store32(emitter,stack,&stk_shift->opnd,shift);
		shift = stk_shift;
	}
	// spill operands that use eax and edx
	if (!src_lo->contain(eax_reg) && !src_hi->contain(eax_reg))
		stack.spill_opnds_contain(eax_reg);
	if (!src_lo->contain(edx_reg) && !src_hi->contain(edx_reg))
		stack.spill_opnds_contain(edx_reg);

	if (src_lo->is_mem()) { // src_hi must also be mem as well
		if(src_lo->contain(eax_reg)) {
			// load effective address into edx so that we can load low 32 bit
			// value into eax
			src_lo->free_opnd(&stack.reg_manager);
			Reg_Operand *reg = stack.reg_manager.get_reg(edx_reg);
			((Mem_Operand*)src_lo)->emit_lea(emitter,&reg->opnd);
			// load low 32 bit to eax and high 32 bit to edx
			emitter.emit_mov(&eax_opnd,&M_Base_Opnd(edx_reg,0));
			emitter.emit_mov(&edx_opnd,&M_Base_Opnd(edx_reg,4));
		} else {
			src_lo->emit_mov_to_reg(emitter,&eax_opnd);
			src_hi->emit_mov_to_reg(emitter,&edx_opnd);
		}

	} else {
		if (src_hi->contain(eax_reg)) { // src_hi is eax_reg
			gen_store32(emitter,stack,&stk_hi->opnd,src_hi);
			src_hi = stk_hi;
		} 
		// if src_lo is already in eax, then we don't need to do anything
		if (!src_lo->contain(eax_reg))
			src_lo->emit_mov_to_reg(emitter,&eax_opnd);
		if (!src_hi->contain(edx_reg))
			src_hi->emit_mov_to_reg(emitter,&edx_opnd);
	}
	// make sure that eax, edx are booked
	src_lo->free_opnd(&stack.reg_manager);
	src_hi->free_opnd(&stack.reg_manager);
	stack.reg_manager.free_reg(eax_reg);
	stack.reg_manager.free_reg(edx_reg);
	stack.reg_manager.get_reg(eax_reg);
	stack.reg_manager.get_reg(edx_reg);
	//
	// if shift is already in ecx, then we don't need to do anything
	//
	if (!shift->is_scratch_reg() || !shift->contain(ecx_reg)) {
        //if (!shift->contain(ecx_reg) && !stack.reg_manager.is_free(ecx_reg))
        // JMS: The original condition didn't work if "shift" is something like [ecx+12].
        if ((!shift->contain(ecx_reg) || shift->kind != Operand::Reg) &&
            !stack.reg_manager.is_free(ecx_reg))
        {
            stack.push(shift); // JMS HACK
			stack.spill_opnds_contain(ecx_reg);
            shift = stack.pop(); // JMS HACK
        }
		shift->emit_mov_to_reg(emitter,&stack.reg_manager.get_reg(ecx_reg)->opnd);
	}
	void *addr = orp_get_rt_support_addr(help_func);
	unsigned patch_offset = emitter.get_offset()+1;
    emitter.emit_call((char *)addr);
	code_patch_list = 
		new(mem_manager) Call_Patch(code_patch_list,patch_offset,(char*)addr);
	stack.reg_manager.reset_after_call(); // free eax, edx, ecx
	stack.push64(stack.reg_manager.get_reg(eax_reg),stack.reg_manager.get_reg(edx_reg));
}
