// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/expression.h,v 1.6 2002/01/10 03:12:09 xhshi Exp $
//

#ifndef _EXPRESSION_H_
#define _EXPRESSION_H_

#include "jit_types.h"
#include "template.h"

#define MAX_REGID_SIZE 24
class RegID_Link {
public:
    RegID_Link(unsigned i,char t, unsigned rid, RegID_Link* n) : 
      id(i), ty(t), reg_id(rid), next(n) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}

    const unsigned id;
    const char ty;
    const unsigned reg_id: MAX_REGID_SIZE;
    RegID_Link *const next;
};

#define MAX_JSTACK 8
//
//  physical reg  :  0 --> n_reg (eax,ecx,edx,ebx,ebp,esp,esi,edi)
//                   start_reg_id >= n_reg
//  virtual reg id:  start_reg_id ----------> start_reg_id + max_locals - 1 
//  java stack:      start_reg_id + max_locals ---->start_reg_id + max_locals + max_stack - 1
//  temp reg:        start_reg_id + max_locals + max_stack --->
//
class RegID_Map {
public:
    RegID_Map(Mem_Manager& mm,unsigned locals, unsigned depth, unsigned start_id) :
      _mem(mm), _max_locals(locals), _max_stack(depth), _start_reg_id(start_id) {
    _next_temp_id = start_id + locals + depth;
    for (int i = 0; i < MAX_JSTACK; i++) _table[i] = 0;
    }
	unsigned virtual_reg_id(unsigned var_no, O3_Jit_Type ty) { 
		assert(var_no < _max_locals && _start_reg_id >= n_reg);
		unsigned id = _start_reg_id + var_no;
        return lookup_reg_id(id,ty);
	}
	unsigned stack_reg_id(unsigned depth, O3_Jit_Type ty) {
		assert(depth < _max_stack);
        unsigned id = _start_reg_id + _max_locals + depth;
        return lookup_reg_id(id,ty);
    }
	// a new reg id is returned
	unsigned get_tmp_reg_id(int need64) {
        unsigned id = _next_temp_id;
        _next_temp_id += (need64) ? 2 : 1;
        return id;
    }
    unsigned curr_tmp_reg_id() {return _next_temp_id;}
    //
    // reset context for inlining
    //
    void set_context(unsigned start, unsigned max_locals, unsigned max_stk) {
        _start_reg_id = start;
        _max_locals = max_locals;
        _max_stack = max_stk;
        _next_temp_id = start + max_locals + max_stk;
    }
    // checkpoint the original context state
    void save_context(unsigned& start, unsigned& max_locals, unsigned& max_stk) {
        start = _start_reg_id;
        max_locals = _max_locals;
        max_stk = _max_stack;
    }
    // restore original context, except don't restore _next_temp_id
    void restore_context(unsigned start, unsigned max_locals, unsigned max_stk) {
        _start_reg_id = start;
        _max_locals = max_locals;
        _max_stack = max_stk;
    }
private:
    unsigned lookup_reg_id(unsigned id, O3_Jit_Type ty);
    //
    // private fields
    //
	unsigned _next_temp_id;
	unsigned _start_reg_id; // starting reg id (origin)
	unsigned _max_locals;   // var_no cannot exceed max_locals
	unsigned _max_stack;    // Java stack depth cannot exceed max_stack
    Mem_Manager& _mem;
    RegID_Link *_table[MAX_JSTACK];
};

class ID_Link {
public:
    ID_Link(void *o,unsigned i,ID_Link* n) : opnd(o), next(n), id(i) {}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}

    void *const opnd;
    ID_Link *const next;
    const unsigned id;
};

#define INVALID_KILL_ID -1
#define IS_VALID_KILL_ID(kill_id) ((kill_id) != (unsigned short)INVALID_KILL_ID)
#define MAX_KILL_HASH 256

class Kill_ID_Map {
public:
    Kill_ID_Map() : _next_kill_id(0) {
        int i;
        for (i = 0; i < MAX_KILL_HASH; i++) _table[i] = NULL;
    }
    unsigned lookup(Mem_Manager& m, void *o);
    unsigned get_next_kill_id() {return _next_kill_id++;}
    unsigned get_curr_kill_id() {return _next_kill_id;}
private:
    unsigned _next_kill_id;
    ID_Link *_table[MAX_KILL_HASH];
};

class Kill_Set;  // forward declaration
class Live_LCSE; // forward declaration
class LCSE_Pool; // forward declaration

class Exp {
public:
    enum Kind {
        Opnd,    // operand
        Array,   // array access
        Field,   // field access
        Length,  // array length
        Widen,   // widen instruction (array load of char)
        Assign,  // x = y
        Add,     // integer add
        Adc,     // integer add with carry
        Sub,     // integer sub
        Sbb,     // integer sub with carry
        Fadd,    // floating point add (single and double)
        Fsub,    // floating point sub (single and double)
        Mul,     // integer mutiplication
        Fmul,    // floating point mul (single and double)
//#ifdef INLINE_NATIVE
		Smul,    // special mul: Long = INT * INT
//#endif
        Div,     // integer division
        Fdiv,    // floating point div (single and double)
        Rem,     // remainder (div)
        Frem,    // floating point remainder (div)
        Neg,     // negate int/long
        Fneg,    // negate float/double
        Convt,   // conversion (i2l,i2d,...)
        And,     // bitwise and
        Or,      // bitwise or
        Xor,     // bitwise xor
        Shl,     // shift left
        Sar,     // arithmetic shift right
        Shr,     // shift right
        Compare, // compare
        Cmp_LT,  // compare less than
        Cmp_GT,  // compare greater than
        Test,    // test 
        Beq,    // branch if equal 
        Bne,    // branch if not equal
        Blt,    // branch if less than 
        Bge,    // branch if greater than or equal
        Bgt,    // branch if greater than
        Ble,    // branch if less than or equal
		Call,    // method call (virtual/special)
        New,     // new 
        Newarray,// newarray 
        Anewarray,//anewarray
        Multinew,// multianewarray call  
		Vtable,  // vtable of a class
        Return,  // return
        Next_PC, // the next pc of a jsr
        Jump,    // jump 
        Tableswitch, // table switch 
        Lookupswitch,// lookup switch
        String,  // get string from a constant pool  
        Instanceof, // is instance of 
        Checkcast,  // check cast
        Monitorenter, // monitor enter
        Monitorexit,  // monitor exit
        Monitorenter_static, // monitor enter for static methods
        Monitorexit_static,  // monitor exit  for static methods
        Athrow,  // throw exception
        InterfaceVtable, // resolve interface - get vtable at runtime
        Push,    // push value onto stack frame
        Pop,     // pop value out of stack frame
        ClassInit, // class initialization
        WriteBarrier, // write barrier
		ReadBarrier , //:: read barrier
#ifdef STAT_INDIRECT_CALL
		StatIndirectCall, // For statistics of indirect branch
#endif
#ifdef O3_VTune_Support
		VTuneMethodCall,
#endif
        Intr_Sin,
        Intr_Cos,
        Intr_Sqrt,
        Intr_Rndint,
        Intr_Abs,
        n_exp
    };
    const unsigned id;
    const O3_Jit_Type type;
    const unsigned char op;
    const static char* symbol[n_exp];


    Exp(const Kind o, O3_Jit_Type t,unsigned i) 
		: op((unsigned char)o), id(i), type(t), _inst(0), _kill(0), _cse_cand(1) {}
    void set_inst(Inst *i) {_inst = i;}
    Inst *inst() {return _inst;}
	Kill_Set *kill_set() {return _kill;}
    bool is_in_kill_set(unsigned kill_id); //return true, if kill_id is in kill_set
    bool contain_of_array(O3_Jit_Type ty); // return true, ty is set in kill_set
    bool is_same_exp(Exp *e) {return this == e;}
    bool is_alias_across_call(); // return true, if the value may be killed by calls
    void clr_cse_cand() {_cse_cand = 0;}
    bool is_cse_cand()  {return _cse_cand == 1;}
    bool is_branch_exp() {return op >= Beq && op <= Ble;}
    //
    // virtual functions
    //
    virtual unsigned hash() = 0; // return key for hashing
    virtual unsigned kill_id() {return (unsigned)INVALID_KILL_ID;}
    virtual bool is_imm_exp()   {return false;}
    virtual bool is_opnd_exp()  {return false;}
    virtual bool is_inst_exp()  {return false;}
    virtual int  is_same_reg(unsigned reg_id, O3_Jit_Type ty) {return 0;}
    virtual int  is_same_array(Exp *array,Exp *index,unsigned shift, unsigned off,O3_Jit_Type ty){return 0;}
    virtual int  is_same_field(Exp *base, unsigned offset, O3_Jit_Type type) {return 0;}
    virtual int  is_same_imm(unsigned imm, O3_Jit_Type type) {return 0;}
    virtual int  is_same_addr(void *addr)  {return 0;}
    virtual int  is_same_const(Value *val,O3_Jit_Type ty) {return 0;}
    virtual int  is_same_arg(unsigned i,O3_Jit_Type ty){return 0;}
    virtual int  is_same_ret(O3_Jit_Type ty) {return 0;}
    virtual int  is_same_inst(Kind op, Exp *left, Exp *rght, O3_Jit_Type ty){return 0;}
    virtual void compute_kill(Mem_Manager& mem, unsigned max_bv_size) = 0;
    virtual void asgn_kill_local_cse(Live_LCSE *live_lcse) = 0;
    virtual Inst *build_inst(Expressions& exprs, Inst *inst_head) = 0;
    virtual void print(ostream& cout) = 0;
protected:
    char _cse_cand; // set if this exp can be a cse candidate
    Inst *_inst;  // the most recent inst that holds the value of the expression
	Kill_Set *_kill;
};

class Operand_Exp : public Exp {
public:
    Operand *const opnd;
    Operand_Exp(Operand *o,unsigned id) 
        : opnd(o), _kill_id((unsigned short)INVALID_KILL_ID), Exp(Opnd,o->type,id) {}
    unsigned kill_id() {return _kill_id;}
    void set_kill_id(unsigned i) {assert(i < (unsigned short)-1);_kill_id = (unsigned short)i;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    unsigned short _kill_id;

    unsigned hash() {
        switch(opnd->kind) {
        case Operand::Immediate: 
            return ((Imm_Operand*)opnd)->imm();   break;
        case Operand::GCTrack:
            return ((GCTrack_Operand*)opnd)->id;      break;
        case Operand::Static:
            return (unsigned)((Static_Operand*)opnd)->addr; break;
        case Operand::Const:
            return (unsigned)((Const_Operand*)opnd)->val.i;  break;
        default: assert(0); return 0;
        }
    }
    bool is_imm_exp()  {return opnd->kind == Operand::Immediate;}
    bool is_opnd_exp() {return true;}
    int  is_same_reg(unsigned reg_id, O3_Jit_Type ty) {
        return opnd->is_reg() && opnd->type == ty &&
               ((Reg_Operand*)opnd)->id == reg_id;
    }
    int  is_same_imm(unsigned imm, O3_Jit_Type ty) {
        return opnd->type == ty &&
               opnd->kind == Operand::Immediate &&
               ((Imm_Operand*)opnd)->imm() == imm;
    }
    int  is_same_addr(void *addr) {
        return (opnd->kind == Operand::Static && ((Static_Operand*)opnd)->addr == addr);
    }
    int  is_same_const(Value *val, O3_Jit_Type ty);
    int  is_same_arg(unsigned i, O3_Jit_Type ty) {
        return opnd->is_arg() && opnd->type == ty && 
               ((Arg_Operand*)opnd)->num == i;
    }
    int  is_same_ret(O3_Jit_Type ty) {
        return opnd->is_ret() && opnd->type == ty;
    }
    void compute_kill(Mem_Manager& mem, unsigned max_bv_size);
    void asgn_kill_local_cse(Live_LCSE *live_lcse);
    Inst *build_inst(Expressions& exprs, Inst *inst_head);
    void print(ostream& cout) {opnd->print(cout);}
};

extern unsigned array_hash(Exp *base, Exp *indx, unsigned off);
extern unsigned field_hash(Exp *base, unsigned off);
extern unsigned inst_hash(Exp::Kind op, Exp *left, Exp *rght, O3_Jit_Type ty);

class Array_Exp : public Exp {
public:
    Exp *const base;
    Exp *const indx;
    const unsigned shift;
    const unsigned offset;

    Array_Exp(Exp *b, Exp *i, unsigned s, unsigned off, O3_Jit_Type t,unsigned id) : 
      base(b), indx(i), shift(s), offset(off), Exp(Array,t,id){}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    unsigned hash() {return array_hash(base,indx,offset);}
    int  is_same_array(Exp *array,Exp *index,unsigned shift, unsigned off,O3_Jit_Type ty){
        return base == array && indx == index && off == offset && ty == type;
    }
    void compute_kill(Mem_Manager& mem, unsigned max_bv_size);
    Inst *build_inst(Expressions& exprs, Inst *inst_head);
    void asgn_kill_local_cse(Live_LCSE *live_lcse);
    void print(ostream& cout) { // print [b+i*s+o]
        cout << "["; base->print(cout); cout << " + "; indx->print(cout);
        cout << " * " << (1<<shift) << " + " << offset << "]";
    }

	Array_Operand *_array_opnd;
};

class Field_Exp : public Exp {
public:
    Exp *const base;
    const unsigned offset;

    Field_Exp(Exp *b, unsigned off, O3_Jit_Type t,unsigned id, FIELD_UID f) : 
      base(b), offset(off), _kill_id((unsigned short)INVALID_KILL_ID), 
      Exp(Field,t,id), _fid(f) {}
    unsigned kill_id() {return _kill_id;}
    void set_kill_id(unsigned i) {assert(i < (unsigned short)-1);_kill_id = (unsigned short)i;}
    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
private:
    unsigned short _kill_id;
	FIELD_UID _fid;

    unsigned hash() {return field_hash(base,offset);}
    int  is_same_field(Exp *b, unsigned off, O3_Jit_Type ty) {
        return b == base && off == offset && ty == type;
    }
    void compute_kill(Mem_Manager& mem, unsigned max_bv_size);
    void asgn_kill_local_cse(Live_LCSE *live_lcse);
    Inst *build_inst(Expressions& exprs, Inst *inst_head);
    void print(ostream& cout) { // print [b+o]
        cout << "["; base->print(cout); cout << " + " << offset << "]";
    }
};

class Inst_Exp : public Exp {
public:
    const unsigned char n_srcs;

    void *operator new(size_t sz,Mem_Manager& m) {return m.alloc(sz);}
    // no child
    Inst_Exp(Kind op, O3_Jit_Type t,unsigned id) : 
    n_srcs(0), Exp(op,t,id) {_child[0] = _child[1] = NULL;}
    // one child
    Inst_Exp(Kind op, Exp *left, O3_Jit_Type t,unsigned id) : 
    n_srcs(1), Exp(op,t,id) { _child[0] = left;  _child[1] = NULL;}
    // two children
    Inst_Exp(Kind op, Exp *left, Exp *right, O3_Jit_Type t,unsigned id) : 
    n_srcs(2), Exp(op,t,id) {_child[0] = left;  _child[1] = right;}

    Exp *left_child() {return _child[0];}
    Exp *rght_child() {return _child[1];}
    Inst *build_inst(Expressions& exprs, Inst *inst_head);
private:
    Exp *_child[MAX_SRCS];

    unsigned hash() {return inst_hash((Kind)op,_child[0],_child[1],type);}
    int  is_same_inst(Kind o, Exp *left, Exp *rght, O3_Jit_Type ty) {
        return o == (Kind)op && left == _child[0] && rght == _child[1] && ty == type;
    }
    void compute_kill(Mem_Manager& mem, unsigned max_bv_size);
    void asgn_kill_local_cse(Live_LCSE *live_lcse) {assert(0);}
    bool is_inst_exp() {return true;}
    void print(ostream& cout) {
        if (!_child[0])
            cout << symbol[op];
        else if (!_child[1]) {
            cout << "(" << symbol[op] << " ";
            _child[0]->print(cout); cout << " )";
        } else {
            cout << "("; _child[0]->print(cout);
            cout << " " << symbol[op] << " "; 
            _child[1]->print(cout); cout << ")";
        }
    }
};


#define MAX_HASH_ENTRY   (1<<10)
#define MAX_FP_STK_ENTRY 8

class Expressions {
public:
    Expressions(Mem_Manager& mem_manager, RegID_Map& rm, Kill_ID_Map& km, LCSE_Pool& pool) : 
     mem(mem_manager),  kill_id_map(km), _live_lcse(NULL), lcse_pool(pool), reg_map(rm) {
         _next_exp_id = 0;
         _data_space = 0;
         // init hash table entries
        unsigned i;
        for (i = 0; i < MAX_HASH_ENTRY; i++) _table[i] = NULL;
        for (i = 0; i < MAX_FP_STK_ENTRY; i++)
            _fp_stk[i] = new (mem) Fp_Stk(i,reg_map.get_tmp_reg_id(0),JIT_TYPE_DOUBLE);

    }
    void set_live_lcse(Live_LCSE *l) {_live_lcse = l;}
    Live_LCSE *live_lcse() {return _live_lcse;}
    Temp_Reg *create_new_temp_reg(O3_Jit_Type type) { // create a new temp reg of type
        unsigned reg_id = reg_map.get_tmp_reg_id(IS_64BIT(type));
		Temp_Reg *reg = new (mem) Temp_Reg(reg_id,type);
        if (IS_64BIT(type))
            reg->set_hi_opnd(new (mem) Temp_Reg(reg_id+1,type));
        return reg;
	}
    void gen_local_cse(Exp *e,Inst *i, bool for_assgn = false);

    //
    // generate T = ...
    //
    Inst *lookup_imm(unsigned imm, O3_Jit_Type ty, Inst *inst_head);
    Inst *lookup_vreg(unsigned var_no, O3_Jit_Type ty,Inst *inst_head);
    Inst *lookup_preg(unsigned reg_id, O3_Jit_Type ty,Inst *inst_head);
    Inst *lookup_arg(unsigned i, O3_Jit_Type ty, Inst *inst_head);
    Inst *lookup_stack(unsigned stk_entry, O3_Jit_Type ty,Inst *inst_head);
    Inst *lookup_static(void *addr,O3_Jit_Type ty,Inst *inst_head, Field_Handle fh);
    Inst *lookup_const(Value *val,O3_Jit_Type ty,Inst *inst_head);
    Inst *lookup_array(Inst *array,Inst *index,Inst *bnd,unsigned shift,
                      unsigned offset,O3_Jit_Type ty,Inst *inst_head);
    Inst *lookup_ret(O3_Jit_Type ty, Inst *inst_head);
    Inst *lookup_field(Inst *base, unsigned offset, O3_Jit_Type ty,
                       Inst *inst_head, FIELD_UID unique_id);
    Inst *lookup_inst(Exp::Kind op, Inst *left, Inst *rght, O3_Jit_Type ty,Inst *inst_head);
	// 
    // look up if expressions exist or not, if not, create the expression
    //
    Operand_Exp *lookup_const_exp(Value *val,O3_Jit_Type ty);
    Operand_Exp *lookup_imm_exp(unsigned imm, O3_Jit_Type ty);
    Operand_Exp *lookup_reg_exp(unsigned reg_id, O3_Jit_Type ty,int is_vreg);
    Operand_Exp *lookup_temp_reg_exp(Reg_Operand *treg);
    Operand_Exp *lookup_arg_exp(unsigned i,O3_Jit_Type ty);
    Operand_Exp *lookup_ret_exp(O3_Jit_Type ty);
    Operand_Exp *lookup_static_exp(void *addr, O3_Jit_Type ty, void *unique_id, Field_Handle fh);
    Field_Exp   *lookup_field_exp(Exp *base,unsigned off,O3_Jit_Type ty,FIELD_UID unique_id);
    Inst_Exp    *lookup_inst_exp(Exp::Kind,Exp *left,Exp *rght,O3_Jit_Type ty);
    Inst_Exp    *lookup_array_cmp_exp(Exp *left, Exp *rght,O3_Jit_Type ty);
    Array_Exp   *lookup_array_exp(Exp *base,Exp *index,Exp *bnd,unsigned s,
                                  unsigned off,O3_Jit_Type ty);
    //
    // generate T = ...
    //
	Inst *gen_opnd_tuple(Inst *inst_head,Operand_Exp *exp);
	Inst *gen_array_tuple(Inst *inst_head,Array_Exp *exp,Inst *base,Inst *index,Inst *bnd);
	Inst *gen_field_tuple(Inst *inst_head,Field_Exp *exp,Inst *base, FIELD_UID fid);
	Inst *gen_inst_tuple(Inst *inst_head,Inst_Exp *exp,
		                 Exp::Kind op, Inst *left, Inst *rght);

    int is_local_cse(Exp *exp) { return exp->inst() != NULL; }
    Inst *local_cse(Exp *exp);
    Fp_Stk *fp_stk_opnd(unsigned i) { 
        assert(i < MAX_FP_STK_ENTRY);
        return _fp_stk[i];
    }
    unsigned data_space() {return _data_space;}
    void inc_data_space(unsigned i) {_data_space += i;}
    unsigned curr_exp_id() {return _next_exp_id;}

    Mem_Manager& mem;    // for memory allocation
    RegID_Map& reg_map;  // register id mapping 
    Kill_ID_Map& kill_id_map;
    LCSE_Pool& lcse_pool; // maintaining free list of LCSE
private:
    //
    // hash table for expressions
    //
    Link<Exp> *_table[MAX_HASH_ENTRY];
    //
    // the starting id for tuple representation in the current (or extended) block 
    //
    unsigned _next_exp_id;
    Live_LCSE *_live_lcse;
    Fp_Stk *_fp_stk[MAX_FP_STK_ENTRY];
    //
    // we need to allocate space in data block for constants.  total_const_size
    // is the size of the data block.
    //
    unsigned _data_space;
};


#endif // _EXPRESSION_H_
