// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_net_InetAddress.cpp,v 1.5 2001/12/12 06:30:03 xli18 Exp $
//

#include "platform.h"
#include <assert.h>
#include <errno.h>

#ifdef ORP_NT
#include <winsock2.h>
#endif

#ifdef ORP_POSIX
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <string.h>
#endif

#include <jni.h>
#include <gnu_classpath_jni_utils.h>

#include "java_net_InetAddress.h"

#ifdef OBJECT_LOCK_V2
#include "platform_utils_olv2.h"  //used to replace strerror(error) with socket_strerror(error)
#else
#include "platform_utils.h"
#endif

#ifdef ORP_POSIX
extern int  WSAGetLastError (); 
#endif

/* This criticalsection is for mutual exclusive use of gethostbyXXX, 
 * because there're little chances gethostbyaddr and gethostbyname will 
 * be invoked simultaneously and frequently, so they share one global
 * criticalsection object.
 */
#include "orp_utils.h"
CriticalSection g_csForGetByX;

/*
 * Class:     java_net_InetAddress
 * Method:    getLocalHostName
 * Signature: ()Ljava/lang/String;
 */

JNIEXPORT jstring JNICALL Java_java_net_InetAddress_getLocalHostName
  (JNIEnv *jenv, jclass clazz)
{
	// Allocate space for the name.
	char * name = (char *) malloc(sizeof(char) * 1024);
	assert(name);
	
	if (gethostname(name, 1023) == -1) {
		// Failed due to some reason...
		int error = WSAGetLastError (); 
#ifdef _DEBUG
      	printf("Java_java_net_InetAddress_getLocalHostName():  %s\n", socket_strerror(error));
#endif
		// We will simply return "localhost"
		strcpy(name, "localhost");
	}
	jstring ret_string = jenv->NewStringUTF(name);
	assert(ret_string);
	free(name);
	return ret_string;;
} // Java_java_net_InetAddress_getLocalHostName


/*
 * Class:     java_net_InetAddress
 * Method:    lookupInaddrAny
 * Signature: ()[I
 */


JNIEXPORT jintArray JNICALL Java_java_net_InetAddress_lookupInaddrAny
  (JNIEnv *jenv, jclass clazz)
{
	// Create array to store the IP address derived from INADDR_ANY
	jintArray ipaddr_array = jenv->NewIntArray(4);
	assert(ipaddr_array);
	
	// Copy in the values 
	jboolean is_copy;
	jint *ints = jenv->GetIntArrayElements(ipaddr_array, &is_copy);
	assert(ints);

	ints[3] = (INADDR_ANY & 0x000000FF);
	ints[2] = (INADDR_ANY & 0x0000FF00) >> 8;
	ints[1] = (INADDR_ANY & 0x00FF0000) >> 16;
	ints[0] = (INADDR_ANY & 0xFF000000) >> 24;

	jenv->ReleaseIntArrayElements(ipaddr_array, ints, 0);
	return(ipaddr_array);

} // Java_java_net_InetAddress_lookupInaddrAny



/*
 * Class:     java_net_InetAddress
 * Method:    getHostByAddr
 * Signature: ([I)Ljava/lang/String;
 */

JNIEXPORT jstring JNICALL Java_java_net_InetAddress_getHostByAddr
  (JNIEnv *jenv, jclass clazz, jintArray array)
{
	if (!array) { 
#ifdef _DEBUG
        printf("Java_java_net_InetAddress_getHostByAddr:: null address passed in");
#endif
        throw_exception_from_jni(jenv, "java/net/UnknownHostException", "Null address passed in");
        return 0;
	}

	// Extract the ints out of the array passed in.
	jsize len = jenv->GetArrayLength(array);
	
	// Because an IP address is a quartet of ints.
	if (len != 4) { 
#ifdef _DEBUG
        printf("Java_java_net_InetAddress_getHostByAddr:: Wrong IP address");
#endif
        throw_exception_from_jni(jenv, "java/net/UnknownHostException", "Wrong IP address");
        return 0;
	}

	jboolean is_copy;
	jint *ints = jenv->GetIntArrayElements(array, &is_copy);
	assert(ints);

	// Now transform to a 32-bit representation and use "gethostbyaddr"
	int address = (ints[0] << 24) + (ints[1] << 16) + (ints[2] << 8) + ints[3];
	// Convert to network byte order.
	address = htonl(address); 
	
	struct hostent *h;
	jstring ret_string;
	//The following is a CriticalSection block
	{
		CriticalSectionHelper csp((LPCRITICAL_SECTION)g_csForGetByX);
		if ((h = gethostbyaddr((const char *) &address, sizeof(address), AF_INET)) == NULL) { 
			// Failed due to some reason...
   			int error = WSAGetLastError ();
#ifdef _DEBUG
			printf("Java_java_net_InetAddress_getHostByAddr(): %s\n", socket_strerror(error));
#endif
			throw_exception_from_jni(jenv, "java/net/UnknownHostException", socket_strerror(error));
			return 0;
		}
		// Package return string into JNI and return it.
		assert(h->h_name);
		ret_string = jenv->NewStringUTF(h->h_name);
		assert(ret_string);
	} //CriticalSection block

	jenv->ReleaseIntArrayElements(array, ints, JNI_ABORT);
	return ret_string;

} // Java_java_net_InetAddress_getHostByAddr



/*
 * Class:     java_net_InetAddress
 * Method:    getHostByName
 * Signature: (Ljava/lang/String;)[[I
 */

JNIEXPORT jobjectArray JNICALL Java_java_net_InetAddress_getHostByName
  (JNIEnv *jenv, jclass clazz, jstring host)
{
	jboolean is_copy;
	const char *host_name = jenv->GetStringUTFChars(host, &is_copy);
	assert(host_name);

	struct hostent *h;
	jobjectArray addrs;
	//The following is a CriticalSection block
	{
		CriticalSectionHelper csp((LPCRITICAL_SECTION)g_csForGetByX);
		// Look up the host by name 
		h = gethostbyname(host_name);
		if (!h) { 
			// Throw UnknownHostException.
			int error = WSAGetLastError ();
#ifdef _DEBUG
			printf("Java_java_net_InetAddress_getHostByName(): %s\n", socket_strerror(error));
#endif
			throw_exception_from_jni(jenv, "java/net/UnknownHostException", socket_strerror(error));
			return 0;
		}

		jenv->ReleaseStringUTFChars(host, host_name);

		// Count the number of addresses to be returned.
		int num = 0;
		while (h->h_addr_list[num])
			num++;
 
		assert(num > 0);

		// Create an array of objects with each element of it being an array of int.
		jclass int_arr_class = jenv->FindClass("[I");
		assert(int_arr_class);
  		addrs = jenv->NewObjectArray(num, int_arr_class, &is_copy);
		assert(addrs);

		// Loop and copy for each individual address. 
		for (int i = 0; i < num; i++) {
			jarray addr_arr = jenv->NewIntArray(4);
			assert(addr_arr);
			jint *ints = jenv->GetIntArrayElements(addr_arr, &is_copy);
			assert(ints);

			// Convert from network order and copy in.
			uint32 ip = ntohl(*(uint32 *)(h->h_addr_list[i]));
			ints[3] = (ip & 0x000000FF);
			ints[2] = (ip & 0x0000FF00) >> 8;
			ints[1] = (ip & 0x00FF0000) >> 16;
			ints[0] = (ip & 0xFF000000) >> 24;

			// Copy back the array into JNI.
			jenv->ReleaseIntArrayElements(addr_arr, ints, 0);
			// Set the index i to point to this array containing the address.
			jenv->SetObjectArrayElement(addrs, i, addr_arr);
		}
	}//CriticalSection block
    
	return(addrs);

} // Java_java_net_InetAddress_getHostByName





