// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_net_PlainSocketImpl_common.cpp,v 1.10 2002/01/14 01:27:48 gwu2 Exp $
//



#include "platform.h"
#include <assert.h>
#include <errno.h>

#ifdef ORP_NT
#include <winsock2.h>
#endif

#ifdef ORP_POSIX
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <string.h>
#endif

#include <jni.h>
#include "jni_direct.h"
#include "gnu_classpath_jni_utils.h"
#include "java_net_PlainSocketImpl_common.h"

#include "root_set_enum.h" //for Object_Handle

#ifdef ORP_POSIX
extern int WSAGetLastError();
#define SOCKET_ERROR -1
#define INVALID_SOCKET -1
#define SOCKET int
#endif

#ifdef OBJECT_LOCK_V2
#include "platform_utils_olv2.h"  //used to replace strerror(error) with socket_strerror(error)
#else
#include "platform_utils.h"
#endif

/* 
 * Generic shared code between socket stream and datagram implementations.
 */


void create_socket(JNIEnv *jenv, jobject jobj, jboolean is_stream)
{
	uint32 sock_Type;

	if (is_stream) 
		sock_Type = SOCK_STREAM;
    else 
		sock_Type = SOCK_DGRAM;

	SOCKET sock = socket(AF_INET, sock_Type, 0);
	
	if (sock == INVALID_SOCKET) { 
		int error = WSAGetLastError();
#ifdef _DEBUG
		printf("Create socket failed: %s\n", socket_strerror(error));
#endif
		throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
		return;
	}
    
 	jclass clazz;
	// Set the "native_fd" field depending on stream or datagram object.
	if (sock_Type == SOCK_STREAM)
		clazz = get_plainSocketImpl_class(jenv);
    else 
		clazz = get_plainDatagramSocketImpl_class(jenv);
	
	set_socket_native_fd(jenv, clazz, jobj, sock);
} // create_socket



void close_socket(JNIEnv *jenv, jobject jobj)
{
	//jclass clazz = get_plainSocketImpl_class(jenv);
	jclass clazz = GetObjectClass(jenv, jobj);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
	//Socket type will never less than zero, so use (int) cast
	if ((int)sock < 0)		// Default value -- nothing to be done.
		return ;
#ifdef ORP_NT		
	if (closesocket(sock) == SOCKET_ERROR) { 
#else
	if (close(sock) == SOCKET_ERROR) { 
#endif
		int error = WSAGetLastError (); 
#ifdef _DEBUG
       	printf("java_net_PlainSocketImpl_close(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return;
	}
	
    // Reset "native_fd" to -1
	set_socket_native_fd(jenv, clazz, jobj, -1);

} // close_socket


void  bind_socket(JNIEnv *jenv, jobject jobj, jobject INET_address, jint localPortNumber, jboolean is_stream)
{
    struct sockaddr socketStruct;
	socketStruct.sa_family = AF_INET;

    uint16 *p16 = (uint16 *)&(socketStruct.sa_data[0]);
	// Check for valid port number value.
	if (localPortNumber == -1)
		*p16 = 0;
    else 
		*p16 = htons((unsigned short)localPortNumber);

    uint32 *p32 = (uint32 *)&(socketStruct.sa_data[2]);
   	
	// Get the address to be connected to using the "getAddress()" method.
	jclass InetAddress_clazz = jenv->GetObjectClass(INET_address);
	jmethodID meth_id  = jenv->GetMethodID(InetAddress_clazz, "getAddress", "()[B");
	assert(meth_id);
	jarray arr = jenv->CallObjectMethod(INET_address, meth_id);
	assert(arr);
	jboolean is_copy;
	jbyte *bytes = jenv->GetByteArrayElements(arr, &is_copy);   
	assert(bytes);
	
    // Now copy the address into the sockaddr structure.
	*p32 = *(uint32 *)bytes;
	jenv->ReleaseByteArrayElements(arr, bytes, JNI_ABORT);
	
	jclass clazz;
	if (is_stream)
		clazz = get_plainSocketImpl_class(jenv);
	else 
		clazz = get_plainDatagramSocketImpl_class(jenv);

	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
	if (sock < 0) { 
#ifdef _DEBUG
		printf("Socket.bind(): passed wrong socket descriptor.\n");
#endif
		assert(0);
	}

    int yesxx = 1;
	uint32 status;

	// Set the socket options needed.

#ifdef ORP_POSIX 
	socklen_t len = sizeof(yesxx);
#else
	int len = sizeof(yesxx);
#endif

    status = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const char *)&yesxx, len );

	if (status == SOCKET_ERROR) {
       	int error = WSAGetLastError ();     
#ifdef _DEBUG
     	printf("java_net_PlainSocketImpl_socketBind(): %s\n", socket_strerror(error));
#endif
       	throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
       	return;
   	}  
	
	// DO NOT LINGER....

	struct linger linger_struct;
	memset(&linger_struct, 0, sizeof(linger_struct));
	linger_struct.l_onoff = 0;	//  Do not linger.
	len = sizeof(linger_struct);

    status = setsockopt(sock, SOL_SOCKET, SO_LINGER, (const char *) &linger_struct, len);

	if (status == SOCKET_ERROR) {
       	int error = WSAGetLastError ();
		if(error != 10042){ //Bad protocol option
#ifdef _DEBUG
       		printf("java_net_PlainSocketImpl_socketBind(): %s\n", socket_strerror(error));
#endif
       		throw_exception_from_jni(jenv, "java/io/IOException",  socket_strerror(error));
       		return;
		}
    }

	// Now, go ahead and actually bind.
    status = bind(sock, &socketStruct, sizeof(socketStruct));

	if (status == SOCKET_ERROR)  {
	    int error = WSAGetLastError ();
#ifdef _DEBUG
        printf("java_net_PlainSocketImpl_socketBind(): %s\n", socket_strerror(error));
#endif
        throw_exception_from_jni(jenv, "java/io/IOException",  socket_strerror(error));
        return;
   	 }

#ifdef ORP_POSIX
	socklen_t nameLen = sizeof(socketStruct);
#else
	int nameLen = sizeof(socketStruct);
#endif	
	// Retrieve the local name for the socket
	status = getsockname(sock, &socketStruct, &nameLen);
	
	if (status == SOCKET_ERROR) {
	    int error = WSAGetLastError ();
#ifdef _DEBUG
	    printf("java_net_PlainSocketImpl_bind(): %s", socket_strerror(error));
#endif
	    throw_exception_from_jni(jenv, "java/io/IOException", socket_strerror(error));
	    return;
	} 
	// Conver to host byte order.
	localPortNumber = ntohs(*p16);
	// Now, set the "localport" to localPortNumber.
	set_socket_localport(jenv, clazz, jobj, localPortNumber);

	if(is_stream){
		jfieldID address_id = GetFieldID(jenv, clazz, "address", "Ljava/net/InetAddress;");
		assert(address_id);
		SetObjectField(jenv, jobj, address_id, INET_address);
	}

} // bind_socket



// Return an InetAddress object corresponding to address passed in.

jobject create_inet_addr(JNIEnv *jenv, int net_address)
{
	jclass inet_class = get_InetAddress_class(jenv);
	assert(inet_class);
	jclass clazz = get_plainSocketImpl_class(jenv);
	assert(clazz);
////Method I:
	// Use the static method "getByName" to create a new InetAddress object.
	jmethodID meth_id = 
        jenv->GetStaticMethodID(inet_class, "getByName", "(Ljava/lang/String;)Ljava/net/InetAddress;");
	assert(meth_id);
	char addr_buf[16];	
	// Inscribe an IP address into it.
	/*sprintf(addr_buf, "%d.%d.%d.%d",((net_address & 0xFF000000) >> 24), 
									((net_address & 0x00FF0000) >> 16), 
									((net_address &0x0000FF00) >> 8),
									(net_address & 0x000000FF));
	*/
	sprintf(addr_buf, "%d.%d.%d.%d",(net_address & 0x000000FF),
									((net_address &0x0000FF00) >> 8),
									((net_address & 0x00FF0000) >> 16),
									((net_address & 0xFF000000) >> 24)
									);

	jstring jstr = jenv->NewStringUTF(addr_buf);
	assert(jstr);
	jobject inet_object = jenv->CallStaticObjectMethod(inet_class, meth_id, jstr);
	assert(inet_object);
////Method II:
/*
	jintArray ipa = jenv->NewIntArray(4);
	jint *ip = new jint[4];
	ip[0] = (net_address & 0xFF000000) >> 24;
	ip[1] = (net_address & 0x00FF0000) >> 16;
	ip[2] = (net_address &0x0000FF00) >> 8;
	ip[3] = net_address & 0x000000FF;
	
	jenv->SetIntArrayRegion(ipa, 0, 4, ip);
	
	jmethodID init_ID = jenv->GetMethodID(inet_class, "<init>", "([I)V");
	assert(init_ID);
	jvalue jargs[1];
	jargs[0].l = ipa;
	jobject inet_object = jenv->NewObjectA(inet_class, init_ID, jargs);
	assert(inet_object);
	delete ip;
	*/

	return inet_object;
}


// Utility function to cull static options


int get_socket_option(JNIEnv *jenv, jclass clazz, const char *opt, const char *type)
{
	jfieldID field = jenv->GetStaticFieldID(clazz, opt, type);
	assert(field);
	int option = jenv->GetStaticIntField(clazz, field);
	assert(option >= 0);
	return option;
} // get_socket_option



/* 
 * Utility functions for getOption
 */


void handle_getsockopt_exception(JNIEnv *jenv)
{
	int  error = WSAGetLastError();
#ifdef _DEBUG
	printf("java_net_PlainSocketImpl_getOption(): %s", socket_strerror(error));
#endif
	throw_exception_from_jni(jenv, "java/io/SocketException", socket_strerror(error));
	return;
} // handle_getsockopt_exception

jobject create_boolean_object(JNIEnv *jenv, jboolean value)
{
	jclass bool_class = jenv->FindClass("java/lang/Boolean");
	assert(bool_class);
	jmethodID meth_id = jenv->GetMethodID(bool_class, "<init>", "(Z)V");
	assert(meth_id);
	jobject ret_obj = jenv->NewObject(bool_class, meth_id, value);
	assert(ret_obj);
	return(ret_obj);
} // create_boolean_object


jobject create_integer_object(JNIEnv *jenv, jint value)
{
	jclass int_class = jenv->FindClass("java/lang/Integer");
	assert(int_class);
	jmethodID meth_id = jenv->GetMethodID(int_class, "<init>", "(I)V");
	assert(meth_id);
	jobject ret_obj = jenv->NewObject(int_class, meth_id, value);
	assert(ret_obj);
	return(ret_obj);
} // create_integer_object


// COmmon code to retrieve value for socket options across Datagram and
// SOCK_STREAM implementations.

jobject socketGetOption(JNIEnv *jenv, jobject jobj, jint option_type)
{
	//jclass clazz = get_plainSocketImpl_class(jenv);
	jclass clazz = GetObjectClass(jenv, jobj);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
	assert(sock >= 0);

	// Extract the static socket option values from "SocketOptions"
	jclass SocketOptions_class = jenv->FindClass("java/net/SocketOptions");
	assert(SocketOptions_class);
	
	int _SO_LINGER_ = get_socket_option(jenv, SocketOptions_class, "SO_LINGER", "I");
	int _SO_TIMEOUT_ = get_socket_option(jenv, SocketOptions_class, "SO_TIMEOUT", "I");
	int _SO_BINDADDR_ = get_socket_option(jenv, SocketOptions_class, "SO_BINDADDR", "I");
	int _SO_SNDBUF_ = get_socket_option(jenv, SocketOptions_class, "SO_SNDBUF", "I");
	int _SO_RCVBUF_ = get_socket_option(jenv, SocketOptions_class, "SO_RCVBUF", "I");
	int _SO_REUSEADDR_ = get_socket_option(jenv, SocketOptions_class, "SO_REUSEADDR", "I");
	int _TCP_NODELAY_ = get_socket_option(jenv, SocketOptions_class, "TCP_NODELAY", "I");
	int _IP_MULTICAST_IF_ = get_socket_option(jenv, SocketOptions_class,"IP_MULTICAST_IF", "I");

#ifdef ORP_NT
	int size;
#else
	socklen_t size;
#endif
	jint ret ;

	if (option_type == _TCP_NODELAY_) { 
		
		// Disables the Nagle algorithm for send coalescing.
		int value = 0;
		size = sizeof(value);
		ret = getsockopt(sock, IPPROTO_TCP, TCP_NODELAY,(char *) &value,  &size);
		if (ret == SOCKET_ERROR) 
			handle_getsockopt_exception(jenv);
		// Return 0 or 1 depending on return value from getsockopt().
		return create_boolean_object(jenv, value);
		
	} else if (option_type == _SO_LINGER_) { 
		
		// This option controls the action taken when unsent data is queued 
		// on a socket and a "closesocket()" is performed -- to linger or not to.
		struct linger linger_struct;
		memset(&linger_struct, 0, sizeof(linger_struct));
		size = sizeof(linger_struct);
       	ret = getsockopt(sock, SOL_SOCKET, SO_LINGER, (char *) &linger_struct,  &size);

        if (ret == SOCKET_ERROR) 			
			handle_getsockopt_exception(jenv);
        if (linger_struct.l_onoff)
          	return(create_integer_object(jenv, linger_struct.l_linger));	// Return linger value
        else
        	return(create_boolean_object(jenv, 0)); // NO linger -- so return boolean
	
	} else if ((option_type == _SO_SNDBUF_) || (option_type == _SO_RCVBUF_)) {
		
		// Being able to specify the total per-socket buffer space 
		// reserved for sends/receives	
		int value;
		size = sizeof(value);
		ret = getsockopt(sock, SOL_SOCKET, 
						(option_type == _SO_SNDBUF_) ? SO_SNDBUF : SO_RCVBUF,
						(char *) &value,  &size);
		if (ret == SOCKET_ERROR) 
			handle_getsockopt_exception(jenv);
		return create_integer_object(jenv, value);

	} else if (option_type == _SO_REUSEADDR_) { 

		// Allow the socket to be bound to an address that is already in use. 
		// This is boolean.; use "booleanValue()" to extract the value
		int value;
		size = sizeof(value);
		ret = getsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
						(char *) &value,  &size);
		if (ret == SOCKET_ERROR) 
			handle_getsockopt_exception(jenv);
		// Return 0 or 1 depending on return value from getsockopt().
		assert((value == 0) || (value == 1));
		return create_boolean_object(jenv, value);

	} else if (option_type == _SO_BINDADDR_) { 

#ifdef ORP_NT
		// This option is not supported in winsock
		orp_cout << " Unsupported socket option SO_BINDADDR in WIN32 " << endl;
		assert(0);
#else
		// Doesn't exist on Linux
		assert(0);
#endif
	} else if (option_type == _SO_TIMEOUT_) { 
	
#ifdef ORP_NT
		//The following is only work on Winsock, we have to urge Classpath to help 
		// VM realize this function on Linux
		int timeout; 
		int size = sizeof(timeout);
		ret = getsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, 
			(char *)&timeout, &size);
		if (ret == SOCKET_ERROR) 
			handle_getsockopt_exception(jenv);
		return create_integer_object(jenv, timeout);
		//assert(0);
#else
		// Doesn't exist on Linux
//		assert(0);
#endif

	} else if (option_type == _IP_MULTICAST_IF_) { 

#ifdef ORP_NT 
		// This option is not supported in winsock
		orp_cout << " Unsupported socket option IP_MULTICAST_IF in WIN32 " << endl;
		assert(0);
#else
		// This option I believe exists on Linux but is not supported by us. 
		orp_cout << " Unimplemented socket option IP_MULTICAST_IF" << endl;
		assert(0);
#endif

	} else { 
		// Unknown option.
		// Throw SocketException anyway.
		orp_cout << "PlainSocketImpl.getOption(): " << 
					" getsockopt failed -- Unknown option";
		throw_exception_from_jni(jenv, "java/io/SocketException", "");
		return 0;
	}
	
	// Can't come here.
	assert(0);
	return 0;
} // socketGetOption


#include "environment.h"
#include "orp_utils.h"
void socketSetOption(JNIEnv *jenv, jobject jobj, jint option_type, jobject value)
{
	jmethodID meth_id = NULL; 
	jint ret = 0;

	assert(value);

	//jclass clazz = get_plainSocketImpl_class(jenv);
	jclass clazz = GetObjectClass(jenv, jobj);
	assert(clazz);
	SOCKET sock = get_socket_native_fd(jenv, clazz, jobj);
	assert(sock >= 0);

	// Now extract the type of the "value" passed in.
	jclass value_class = jenv->GetObjectClass(value); 
	assert(value_class);
	
	// Extract the static socket option values from "SocketOptions"
	jclass SocketOptions_class = jenv->FindClass("java/net/SocketOptions");
	assert(SocketOptions_class);
	
	int _SO_LINGER_ = get_socket_option(jenv, SocketOptions_class, "SO_LINGER", "I");
	int _SO_TIMEOUT_ = get_socket_option(jenv, SocketOptions_class, "SO_TIMEOUT", "I");
	int _SO_BINDADDR_ = get_socket_option(jenv, SocketOptions_class, "SO_BINDADDR", "I");
	int _SO_SNDBUF_ = get_socket_option(jenv, SocketOptions_class, "SO_SNDBUF", "I");
	int _SO_RCVBUF_ = get_socket_option(jenv, SocketOptions_class, "SO_RCVBUF", "I");
	int _SO_REUSEADDR_ = get_socket_option(jenv, SocketOptions_class, "SO_REUSEADDR", "I");
	int _TCP_NODELAY_ = get_socket_option(jenv, SocketOptions_class, "TCP_NODELAY", "I");
	int _IP_MULTICAST_IF_ = get_socket_option(jenv, SocketOptions_class,"IP_MULTICAST_IF", "I");

#ifdef ORP_POSIX 
	socklen_t len;
#else // NT
	int len;
#endif

	if (option_type == _TCP_NODELAY_) { 
		
		// Disables the Nagle algorithm for send coalescing.
		// Boolean "value" is expected.
		// Use "booleanValue()" to extract the value
		meth_id = jenv->GetMethodID(value_class, "booleanValue", "()Z");
		assert(meth_id);
		const int val = jenv->CallBooleanMethod(value, meth_id); 
		assert((val == 0) || (val == 1)); // Only two permissible values.
        len = sizeof(int);

        ret = setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (const char *) &val, len);

	} else if (option_type == _SO_LINGER_) { 
		
		// This option controls the action taken when unsent data is queued 
		// on a socket and a "closesocket()" is performed -- to linger or not to.
		
        struct linger linger_struct;
		// There are two cases here. Either boolean value of 0 is passed in 
		// or integer value representing linger value is passed in.
	
        memset(&linger_struct, 0, sizeof(linger_struct));
		Class *clss = (Class*)((Object_Handle)value_class)->java_reference;
		if(!strcmp(clss->name->bytes, "java/lang/Boolean"))
			meth_id = jenv->GetMethodID(value_class, "booleanValue", "()Z");
		
        if (meth_id) { 
			// No need to linger on close.
			linger_struct.l_onoff = 0;
		} else { 
			// Extract integer linger value.
			meth_id = jenv->GetMethodID(value_class, "intValue", "()I");
			assert(meth_id);	// Has to have such a method.
			linger_struct.l_onoff = 1;	// Have to linger.
			linger_struct.l_linger = (unsigned short) jenv->CallIntMethod(value, meth_id);
		}
		len = sizeof(linger_struct);
        ret = setsockopt(sock, SOL_SOCKET, SO_LINGER,	(const char *) &linger_struct, len);
	
	} else if ((option_type == _SO_SNDBUF_) || (option_type == _SO_RCVBUF_)) {
		
		// Being able to specify the total per-socket buffer space 
		// reserved for sends/receives	
		meth_id = jenv->GetMethodID(value_class, "intValue", "()I");
		assert(meth_id);	// Has to have such a method.
		
        int val = jenv->CallIntMethod(value, meth_id);
		int sock_option = (option_type == _SO_SNDBUF_) ? SO_SNDBUF : SO_RCVBUF;

        len = sizeof(int);
		ret = setsockopt(sock, SOL_SOCKET, sock_option, (const char *) &val, len);

	} else if (option_type == _SO_REUSEADDR_) { 

		// Allow the socket to be bound to an address that is already in use. 
		// This is boolean.; use "booleanValue()" to extract the value

		meth_id = jenv->GetMethodID(value_class, "booleanValue", "()Z");
		assert(meth_id);
		const int val = jenv->CallBooleanMethod(value, meth_id); 
		assert((val == 0) || (val == 1)); // Only two permissible values.

		int len = sizeof(int);
		ret = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const char *) &val, len);

	} else if (option_type == _SO_BINDADDR_) { 

#ifdef ORP_NT 
		// This option is not supported in winsock
		orp_cout << " Unsupported socket option SO_BINDADDR in WIN32 " << endl;
       		return;
#else 
		// Doesn't exist on Linux
		assert(0);
#endif
	} else if (option_type == _SO_TIMEOUT_) { 
	
#ifdef ORP_NT
		//The following is only work on Winsock, we have to urge Classpath to help 
		// VM realize this function on Linux
		meth_id = jenv->GetMethodID(value_class, "intValue", "()I");
		assert(meth_id);	// Has to have such a method.
		
        int val = jenv->CallIntMethod(value, meth_id);
		ret = setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, 
				(char *)&val, sizeof(val));
        return;
#else
		// Doesn't exist on Linux
//		assert(0);
#endif

	} else if (option_type == _IP_MULTICAST_IF_) { 

#ifdef ORP_NT
		// This option is not supported in winsock
		orp_cout << " Unsupported socket option IP_MULTICAST_IF in WIN32 " << endl;
        	return;
#else
		// This option I believe exists on Linux but is not supported by us. 
		orp_cout << " Unimplemented socket option IP_MULTICAST_IF" << endl;
		assert(0);
#endif

	} else { 
		// Unknown option.
		assert(0);
	}

	// Throw SocketException if needed.
    if (ret == SOCKET_ERROR) { 
		int  error = WSAGetLastError();
#ifdef _DEBUG
		printf("java_net_PlainSocketImpl_setOption(): %s\n", socket_strerror(error));
#endif
		throw_exception_from_jni(jenv, "java/net/SocketException", socket_strerror(error));
		return;
	}
	return;	
} // socketSetOption
