//******************************************************************************
//
// File:    DnaSequenceTreeList.java
// Package: edu.rit.phyl.pars
// Unit:    Class edu.rit.phyl.pars.DnaSequenceTreeList
//
// This Java source file is copyright (C) 2007 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.phyl.pars;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Class DnaSequenceTreeList provides a list of {@linkplain DnaSequenceTree}
 * objects. It also keeps track of the best score of any DNA sequence tree added
 * to the list. Only DNA sequence trees with scores less than or equal to the
 * best score are retained in the list.
 * <P>
 * Class DnaSequenceTreeList obeys the Conservation of Objects Principle. When
 * an item is added to the list, the item being added is swapped with an
 * existing unused item. This can reduce the amount of object allocation and
 * garbage collection that has to happen, which in turn can reduce the running
 * time of a high performance program.
 * <P>
 * <B><I>Note:</I></B> Class DnaSequenceTreeList is not multiple thread safe.
 *
 * @author  Alan Kaminsky
 * @version 05-May-2007
 */
public class DnaSequenceTreeList
	implements Iterable<DnaSequenceTree>
	{

// Hidden constants.

	private static final int ALLOC = 20;

// Hidden data members.

	// The array holding the list elements.
	private DnaSequenceTree[] myElement = new DnaSequenceTree [0];

	// Number of elements in the list.
	private int mySize = 0;

	// Best score.
	private int myBestScore = Integer.MAX_VALUE;

	// Capacity of DNA sequence trees stored in this list.
	private int myTreeCapacity;

// Exported constructors.

	/**
	 * Construct a new DNA sequence tree list. Initially, the list is empty and
	 * the best score is <TT>Integer.MAX_VALUE</TT>. The capacity of the DNA
	 * sequence trees to be stored in this list is <TT>N</TT>.
	 *
	 * @param  N  Capacity (maximum number of tip nodes) of the DNA sequence
	 *            trees to be stored.
	 *
	 * @exception  NegativeArraySizeException
	 *     (unchecked exception) Thrown if <TT>N</TT> &lt; 1.
	 */
	public DnaSequenceTreeList
		(int N)
		{
		if (N < 1)
			{
			throw new NegativeArraySizeException
				("DnaSequenceTreeList(): N = " + N + " illegal");
			}
		myTreeCapacity = N;
		}

// Exported operations.

	/**
	 * Determine if this DNA sequence tree list is empty.
	 *
	 * @return  True if this list is empty, false otherwise.
	 */
	public boolean isEmpty()
		{
		return mySize == 0;
		}

	/**
	 * Get this DNA sequence tree list's size.
	 *
	 * @return  Size (number of elements).
	 */
	public int size()
		{
		return mySize;
		}

	/**
	 * Clear this DNA sequence tree list. The list becomes empty and the best
	 * score becomes <TT>Integer.MAX_VALUE</TT>.
	 */
	public void clear()
		{
		mySize = 0;
		myBestScore = Integer.MAX_VALUE;
		}

	/**
	 * Get this DNA sequence tree list's best score.
	 *
	 * @return  Best score.
	 */
	public int bestScore()
		{
		return myBestScore;
		}

	/**
	 * Set this DNA sequence tree list's best score. The best score is set to
	 * the smaller of its current value and the given <TT>score</TT>. Any DNA
	 * sequence trees whose scores exceed the resulting best score are removed
	 * from this list.
	 *
	 * @param  score  Best score.
	 */
	public void bestScore
		(int score)
		{
		if (score < myBestScore)
			{
			mySize = 0;
			myBestScore = score;
			}
		}

	/**
	 * Add the given DNA sequence tree to this list. If the given tree's score
	 * is better than this list's best score, then the best score is set to the
	 * given tree's score, and any trees whose scores exceed the resulting best
	 * scores are removed from this list.
	 * <P>
	 * An unused DNA sequence tree object is returned to take the place of the
	 * given tree. However, if the given tree's score exceeds this list's best
	 * score, then the given tree is not added, and <TT>tree</TT> is returned.
	 *
	 * @param  tree  DNA sequence tree.
	 *
	 * @return  DNA sequence tree object to take the place of <TT>tree</TT>, or
	 *          <TT>tree</TT> itself if the given tree was not added to this
	 *          list.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>tree</TT> is null.
	 */
	public DnaSequenceTree add
		(DnaSequenceTree tree)
		{
		int score = tree.score();
		if (score > myBestScore)
			{
			return tree;
			}
		else
			{
			bestScore (score);
			ensureAllocation (1);
			DnaSequenceTree swap = myElement[mySize];
			myElement[mySize] = tree;
			++ mySize;
			return swap;
			}
		}

	/**
	 * Obtain an iterator for traversing the DNA sequence trees in this list.
	 * The returned iterator does not support the <TT>remove()</TT> method. If
	 * this list's contents are altered while an iteration is in progress, the
	 * iteration's behavior is not specified.
	 *
	 * @return  Iterator.
	 */
	public Iterator<DnaSequenceTree> iterator()
		{
		return new Iterator<DnaSequenceTree>()
			{
			private int i = 0;
			public boolean hasNext()
				{
				return i < mySize;
				}
			public DnaSequenceTree next()
				{
				if (i >= mySize)
					{
					throw new NoSuchElementException();
					}
				return myElement[i++];
				}
			public void remove()
				{
				throw new UnsupportedOperationException();
				}
			};
		}

// Hidden operations.

	/**
	 * Ensure that the myElement array has enough storage allocated to add the
	 * given number of elements.
	 *
	 * @param  count  Number of elements to add.
	 */
	private void ensureAllocation
		(int count)
		{
		int n = myElement.length;
		if (mySize + count > n)
			{
			n += Math.max (count, ALLOC);
			DnaSequenceTree[] newElement = new DnaSequenceTree [n];
			System.arraycopy (myElement, 0, newElement, 0, mySize);
			for (int i = mySize; i < n; ++ i)
				{
				newElement[i] = new DnaSequenceTree (myTreeCapacity);
				}
			myElement = newElement;
			}
		}

	}
