/* ------------------------------------------------------------------------
 * WSInterface.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _WSInterface_hh_
#define _WSInterface_hh_

// -- System Includes
#include <string>
#include <list>

// -- 3Dwm Includes
#include "Celsius/Types.hh"
#include "Celsius/Runnable.hh"
#include "Nobel/Image.hh"

// -- Forward Declarations
class Mutex;

// -- Class Declarations

/**
 * Windowing system interface class. Provides functionality for
 * retrieving a virtual framebuffer (that can be used as a texture in
 * 3Dwm), receiving events, and sending input events. The class has
 * been designed to run in a separate thread, leaving the rest of the
 * system free to do other stuff.
 **/ 
class WindowingSystemInterface : public Runnable {

public:

    // Window system framebuffer pixel format
    struct PixelFormat {
	uint32 depth;		// color depth (in bits)
	uint32 size;		// pixel element size (in bits)
	uint32 red_bits;	// number of red bits
	uint32 red_mask;	// bitmask for red component
	uint32 red_shift;	// red component shift 
	uint32 green_bits;	// number of green bits
	uint32 green_mask;	// bitmask for green component
	uint32 green_shift;	// green component shift 
	uint32 blue_bits;	// number of blue bits
	uint32 blue_mask;	// bitmask for blue component
	uint32 blue_shift;	// blue component shift 
    };

    /**
     * Run the windowing system interface. If the interface is run in
     * a Celsius thread, this is the function that will be called when
     * the thread is started.
     **/
    virtual void run() = 0;

    /**
     * Update the framebuffer. Garbo will talk to the underlying
     * windowing system and update the framebuffer. Please note that
     * since this may be an asynchronous operation, there is no
     * guarantee that the buffer is immediately updated.
     *
     * @param x the x position of the upper left rectangle to update.
     * @param y the y position of the upper left rectangle to update.
     * @param w width of the rectangle to update.
     * @param h height of the rectangle to update.
     **/ 
    virtual void updateFramebuffer(int x, int y, int w, int h) = 0;
    
    /**
     * Send a pointer event to the windowing system.
     * 
     * @param x horizontal position of the pointer.
     * @param y vertical position of the pointer.
     * @param button_mask buttons depressed (bits 0-7 are buttons 1-8).
     **/
    virtual void pointerEvent(int x, int y, int button_mask) = 0;

    /**
     * Send a key event to the windowing system.
     * 
     * @param key affected key (X keysym).
     * @param pressed key pressed (=down) or not?
     **/
    virtual void keyEvent(unsigned int key, bool pressed = true) = 0;
    
    /**
     * Retrieve the frame buffer pixel format.
     **/
    const PixelFormat &getPixelFormat() const { return _pf; }

    /**
     * Retrieve the frame buffer width.
     **/
    int getWidth() const { return _width; }

    /**
     * Retrieve the frame buffer height.
     **/
    int getHeight() const { return _height; }

    /**
     * Retrieve the size, in bits, of each pixel.
     **/
    int getPixelSize() const { return _pf.size; }

    /**
     * Retrieve the pixel type. For VNC it is 32-bit RGBA
     * although alpha is currently unused.
     **/
    Nobel::PixelType getPixelType() const { return Nobel::RGBA8888; }

    /**
     * Retrieve the frame buffer pointer.
     **/
    const char *getFramebuffer() const { return _framebuffer; }

    /**
     * Retrieve the desktop name.
     **/ 
    const std::string &getName() const { return _name; }

    /**
     * Rectangle struct used for framebuffer updates. (I'd like to add
     * the merge() and isMergeable() members to this class, but this
     * will be impossible if it is to be CORBA-exposed.)
     **/
    struct Rectangle {
	int x, y;		// upper left corner of rectangle
	int width, height;	// dimensions of rectangle
    };

    /**
     * Retrieve a frame buffer update.
     *
     * @param r a rectangle that will be filled in if the call succeeds.
     * @return true if there is an update, false otherwise.
     **/
    bool getFramebufferUpdate(Rectangle &r);
    
protected:
    
    // Add a rectangle update to the update queue
    void addUpdate(int x, int y, int w, int h);
    
    // Should two rectangles be merged into one?
    bool isMergeable(const Rectangle &r1, const Rectangle &r2);

    // Merge two rectangles, producing a new rectangle as a result
    Rectangle merge(const Rectangle &r1, const Rectangle &r2);
    
    // Copy a rectangle buffer to the framebuffer
    void copyRectToFramebuffer(char *buf, int x, int y, int w, int h);

    WindowingSystemInterface();
    virtual ~WindowingSystemInterface();
    
    // Member variables
    PixelFormat _pf;
    int _width, _height;
    char *_framebuffer;
    std::string _name;

    std::list<Rectangle> _rectQueue;

    // Access mutex (for protecting shared data)
    Mutex _mutex;
};

#endif /* WSInterface.hh */
