/* ------------------------------------------------------------------------
 * $Id: Platform.hh,v 1.9 2001/07/18 07:47:49 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-11-17 by Antony Suter.
 *
 * Copyright (c) 2000 Antony Suter <antony@mira.net>.
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Platform_hh_
#define _Platform_hh_

// -- System Includes
#include <vector>

// -- 3Dwm Includes
#include "Celsius/ThreadQueue.hh"
#include "Zorn/Value.hh"

// Uncomment one of the following to get the corresponding platform
// implementation. Note that this will be automated in the configure
// script later on! (The platforms might even be turned into plugins.)

//#define PLATFORM_GGI
//#define PLATFORM_JUGGLER
//#define PLATFORM_SDL_FBCON
#define PLATFORM_SDL

// -- Forward Declarations
class InputDevice;
class RendererImpl;

// -- Class Declarations

/**
 * Input event class.
 **/
class InputEvent : public std::vector<Zorn::NamedValue> {
public:
    void add(const std::string &id) {
	push_back(Zorn::NamedValue(id, Zorn::Value(true)));
    }
    template <class T>
    void add(const std::string &id, const T &t) {
	push_back(Zorn::NamedValue(id, Zorn::Value(t)));
    }
};

/**
 * Input event queue type definition. This queue is implemented using
 * a thread queue, since it will be used in a producer-consumer
 * fashion by the input devices and the event manager.
 **/
typedef ThreadQueue<InputEvent> InputEventQueue;

/*
// Display parameter struct
struct DisplayInfo {

    /// Desired 
    int width;

    /// 
    int height;


    int depth;
};
*/

/**
 * Platform tie class.
 **/
template <typename T>
class PlatformTie {

private:

    struct Reaper { ~Reaper() { delete PlatformTie::t; } };
    friend struct Reaper;

public:

    /**
     * Initialize the platform. Must be called before any other
     * operations are called on the platform.
     *
     * @param argc the argument count. 
     * @param argv the argument string array.
     * @param w desired display device width.
     * @param h desired display device height.
     * @param d desired display device color depth.
     **/
    static void init(int &argc, char **argv, int w, int h, int d);
    
    /**
     * Retrieve the next input event from the platform subsystem.
     * Will block on this call until an event is received.
     *
     * @return the next input event. 
     **/
    static InputEvent getNextInputEvent();

    /**
     * Clear the display buffers.
     **/
    static void clearDisplay();
    
    /**
     * Swap the display back/front buffers (if double-buffered).
     **/
    static void swapDisplay();

    /**
     * 
     **/
    // TODO: when we have a choice of multiple renderers per platform
    // refactor this to enumerate the renderers and let the user choose
    // which to create.
    static RendererImpl *createRenderer();

    //static RendererImpl *createRenderer(const RendererDescription &rd);

    //static std::vector<RendererDescription> enumerateRenderers() const;

    /**
     * Retrieve display device width.
     *
     * @return width of the screen or window.
     **/
    int getDisplayWidth() const;

    /**
     * Retrieve display device height.
     *
     * @return height of screen or window.
     **/
    int getDisplayHeight() const;

    /**
     * Retrieve display device color depth.
     *
     * @return color depth of screen or window.     
     **/
    int getDisplayDepth() const;

    /**
     * Add an input device to the platform. The input device will be
     * launched in its own thread and given access to the event queue
     * that will allow it to work as a producer of input events (with
     * the event manager as the consumer).
     *
     * @param device input device to add to the platform.
     **/
    void addInputDevice(InputDevice *device);
    
private:

    PlatformTie();
    ~PlatformTie();

    // Platform implementation
    static T *t;

    // Reaper class (takes care of deallocating when necessary)
    static Reaper reaper;
};

// -- Inline Functions

template <typename T>
inline PlatformTie<T>::PlatformTie() { }

template <typename T>
inline PlatformTie<T>::~PlatformTie() { }

template <typename T>
inline void PlatformTie<T>::init(int &argc, char **argv, int w, int h, int d) {
    if (t == 0) t = new T(argc, argv, w, h, d);
}

template <typename T>
inline RendererImpl *PlatformTie<T>::createRenderer() {
    return t->createRenderer();
}

template <typename T>
inline InputEvent PlatformTie<T>::getNextInputEvent() {
    return t->getNextInputEvent();
}

template <typename T>
inline void PlatformTie<T>::clearDisplay() {
    t->clearDisplay();
}

template <typename T>
inline void PlatformTie<T>::swapDisplay() {
    t->swapDisplay();
}

template <typename T>
inline int PlatformTie<T>::getDisplayWidth() const {
    return t->getDisplayWidth();
}

template <typename T>
inline int PlatformTie<T>::getDisplayHeight() const {
    return t->getDisplayHeight();
}

template <typename T>
inline int PlatformTie<T>::getDisplayDepth() const {
    return t->getDisplayDepth();
}

template <typename T>
inline void PlatformTie<T>::addInputDevice(InputDevice *device) {
    t->addInputDevice(device);
}

// -- Platform Implementation Headers

#if defined(PLATFORM_GGI)
#   include <Polhem/GGI.hh>

typedef PlatformTie<GGIPlatform> Platform;

#elif defined(PLATFORM_SDL)
#     include <Polhem/SDLPlatform.hh>

typedef PlatformTie<SDLPlatform> Platform;

#elif defined(PLATFORM_SDL_FBCON)
#     include <Polhem/SDLfbconPlatform.hh>

typedef PlatformTie<SDLfbconPlatform> Platform;

#elif defined(PLATFORM_JUGGLER)
#     include <Polhem/JugglerPlatform.hh>

typedef PlatformTie<JugglerPlatform> Platform;

#else
#  warning "no platform type defined"
#endif

#endif /* Platform.hh */
