/* ------------------------------------------------------------------------
 * $Id: PickerImpl.cc,v 1.6 2001/08/22 12:23:20 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <steve@2dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Nobel/Node.hh"
#include "Celsius/Math.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Logger.hh"
#include "Celsius/Matrix3D.hh"
#include "Celsius/Vector3D.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/View.hh"
#include "Polhem/PickerImpl.hh"
#include "Polhem/LogGroup.hh"

using namespace Nobel;

// -- Code Segment

PickerImpl::PickerImpl() 
{
    // empty
}

PickerImpl::~PickerImpl()
{
    // empty
}

void PickerImpl::visit(Nobel::Node_ptr n)
{
    // Perform picking of the node 
    n->pick(Picker_var(_this()));
}

////
// RayPicker constructor. 
//
// Takes a 2D point and projects a ray from that point into the scene.
// Stores the 3D origin and direction of the ray.
////

RayPicker::RayPicker(const Vector2D &point, View *view)
{
    // Retrieve view parameters
    View::ViewParameters params = view->getViewParameters();
    
    // Convert viewport pixel coordinates to normalized coordinates,
    // with the origin at the centre of the viewport and values on
    // each axis ranging from -1.0 to 1.0
    float nx = ( (point.x() / ((float) params.width / 2)) - 1.0f) *
	params.aspect;
    float ny = 1.0f - (point.y() / ((float) params.height / 2));

    // To calculate the slope of the ray take the tangent of FOV*0.5
    // (which gives you a ratio of X/Z or Y/Z).
    float fov_radians = (params.fov / 2) * Math::pi / 180;
    float slope = tanf(fov_radians);

    // Multiply this by the normalized coordinates to get the slope at
    // that point.
    float slope_nx = nx * slope; 
    float slope_ny = ny * slope;
    
    // Calculate the points on the near and far clipping planes.
    Vector3D p1(slope_nx * params.near, slope_ny * params.near, -params.near);
    Vector3D p2(slope_nx * params.far, slope_ny * params.far, -params.far);

    // Get the current view transform
    Matrix3D viewTrans = view->computeViewTransform();
    viewTrans.invert();

    // Convert ray to world coordinates
    Vector3D wp1 = p1 * viewTrans;
    Vector3D wp2 = p2 * viewTrans;
    
    _dir = wp2 - wp1;
    _dir.normalize();
    
    _origin = wp1; 
}

RayPicker::~RayPicker()
{
    // empty
}

////
// RayPicker::intersectsVolume
//
// Takes a bounding volume (AABB) and tests if the ray intersects it. If so,
// store a pointer to the node the AABB bounds and the distance from the 
// origin to the point of intersection.
//
// Uses the Fast Ray-Box Intersection algorithm by Andrew Woo
// from "Graphics Gems", Academic Press, 1990
////

#define NUMDIM	3
#define RIGHT	0
#define LEFT	1
#define MIDDLE	2

void RayPicker::intersectsVolume(Nobel::Volume_ptr bounds, 
				 Nobel::Node_ptr n)
{
    Vertex3D min, max;
    bounds->getBounds(min, max);

    // AABB min and max in model coords
    Vector3D mmin(min.x, min.y, min.z, 1.0f);
    Vector3D mmax(max.x, max.y, max.z, 1.0f);

    // Transform the model coords into world coords
    Matrix3D transform = _trafos.top()->matrix();

    Vector3D wmin = mmin * transform;
    Vector3D wmax = mmax * transform;

    // Extract coords into float arrays for faster access.
    float minB[NUMDIM] = { wmin.x(), wmin.y(), wmin.z() };
    float maxB[NUMDIM] = { wmax.x(), wmax.y(), wmax.z() };
    float origin[NUMDIM] = { _origin.x(), _origin.y(), _origin.z() };
    float dir[NUMDIM] = { _dir.x(), _dir.y(), _dir.z() };
    float coord[NUMDIM];

    bool inside = true;
    char quadrant[NUMDIM];
    register int i;
    int whichPlane;
    double maxT[NUMDIM];
    double candidatePlane[NUMDIM];

    // Find candidate planes; this loop can be avoided if rays cast
    // all from the eye(assume perpsective view)
    for (i = 0; i < NUMDIM; i++)
	if (origin[i] < minB[i]) {
	    quadrant[i] = LEFT;
	    candidatePlane[i] = minB[i];
	    inside = false;
	} else if (origin[i] > maxB[i]) {
	    quadrant[i] = RIGHT;
	    candidatePlane[i] = maxB[i];
	    inside = false;
	} else {
	    quadrant[i] = MIDDLE;
	}

    // Ray origin inside bounding box
    if (inside)	{
	coord = origin;
	Nobel::Vertex3D v;
	v.x = coord[0];	v.y = coord[1];	v.z = coord[2];
	HitNode hitnode = { Node::_duplicate(n), v, 0.0f };
	_hitList.push_back(hitnode);
	return;
    }

    // Calculate T distances to candidate planes
    for (i = 0; i < NUMDIM; i++)
	if (quadrant[i] != MIDDLE && dir[i] !=0.)
	    maxT[i] = (candidatePlane[i]-origin[i]) / dir[i];
	else
	    maxT[i] = -1.;
    
    // Get largest of the maxT's for final choice of intersection
    whichPlane = 0;
    for (i = 1; i < NUMDIM; i++)
	if (maxT[whichPlane] < maxT[i])
	    whichPlane = i;
    
    // Check final candidate actually inside box
    if (maxT[whichPlane] < 0.) {
	return;
    }
    for (i = 0; i < NUMDIM; i++) {
	if (whichPlane != i) {
	    coord[i] = origin[i] + maxT[whichPlane] * dir[i];
	    if (coord[i] < minB[i] || coord[i] > maxB[i]) {
		return;
	    }
	} else {
	    coord[i] = candidatePlane[i];
	}
    }

    Vector3D hitPoint(coord[0], coord[1], coord[2], 1.0);
    Vector3D hitFromOrigin = hitPoint - _origin;
    float distance = hitFromOrigin.magnitude();
    
    Nobel::Vertex3D v;
    v.x = coord[0]; v.y = coord[1], v.z = coord[2];

    HitNode hitnode = { Node::_duplicate(n), v, distance };
    _hitList.push_back(hitnode);
}
