/* ------------------------------------------------------------------------
 * $Id: PrimitiveKitImpl.cc,v 1.2 2001/08/02 14:22:10 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-08-01 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Solid/SolidBuilder.hh"
#include "Polhem/PrimitiveImpl.hh"
#include "Polhem/PrimitiveKitImpl.hh"

using namespace Nobel;

// -- Code Segment

TriangleMesh *buildMeshFromSolid(const SolidMesh &solid) 
{
    // Create the triangle mesh 
    TriangleMesh *mesh = new TriangleMesh();
    
    // Start with the vertices
    mesh->vertexList.length(solid.vertices.size());
    for (unsigned int i = 0; i < solid.vertices.size(); i++) {
	mesh->vertexList[i].x = solid.vertices[i].x();
	mesh->vertexList[i].y = solid.vertices[i].y();
	mesh->vertexList[i].z = solid.vertices[i].z();
    }

    // Now do the normals
    mesh->normalList.length(solid.normals.size());
    for (unsigned int i = 0; i < solid.normals.size(); i++) {
	mesh->normalList[i].x = solid.normals[i].x();
	mesh->normalList[i].y = solid.normals[i].y();
	mesh->normalList[i].z = solid.normals[i].z();
    }

    // Then do the texture coordinates
    mesh->texCoordList.length(solid.texCoords.size());
    for (unsigned int i = 0; i < solid.texCoords.size(); i++) {
	mesh->texCoordList[i].u = solid.texCoords[i].x();
	mesh->texCoordList[i].v = solid.texCoords[i].y();
    }

    // All right, now we do the triangle faces
    mesh->vertexIndexList.length(solid.faces.size() * 3);
    mesh->normalIndexList.length(solid.faces.size() * 3);
    mesh->texCoordIndexList.length(solid.faces.size() * 3);
    for (unsigned int i = 0; i < solid.faces.size(); i++) {

	// Retrieve the triangle face
	const TriangleFace &t = solid.faces[i];

	// Define the indices for the faces
	for (int j = 0; j < 3; j++) {
	    mesh->vertexIndexList[i * 3 + j] = t.v[j];
	    mesh->normalIndexList[i * 3 + j] = t.n[j];
	    mesh->texCoordIndexList[i * 3 + j] = t.tc[j];
	}	
    }
    
    return mesh;
}

Primitive_ptr PrimitiveKitImpl::createCuboid()
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildCuboid();
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;

    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}

Primitive_ptr PrimitiveKitImpl::createCone(CORBA::Long segments)
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildCone(segments);
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;
    
    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}

Primitive_ptr PrimitiveKitImpl::createSphere(CORBA::Long latSegments,
					     CORBA::Long longSegments)
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildSphere(latSegments, longSegments);
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;
    
    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}

Primitive_ptr PrimitiveKitImpl::createCylinder(CORBA::Long segments)
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildCylinder(segments);
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;
    
    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}

Primitive_ptr PrimitiveKitImpl::createPlane()
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildPlane();
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;
    
    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}

Primitive_ptr PrimitiveKitImpl::createTorus(CORBA::Long slices,
					    CORBA::Long segments)
{
    // Create the triangle mesh, then create the primitive, activate
    // it, and return it.
    SolidMesh *s = SolidBuilder::buildTorus(slices, segments);
    TriangleMesh *mesh = buildMeshFromSolid(*s);
    delete s;
    
    PrimitiveImpl *primitive = activate(new PrimitiveImpl(mesh));
    return primitive->_this();
}
