/* ------------------------------------------------------------------------
 * $Id: SolidCache.cc,v 1.1 2001/08/09 14:19:34 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-08-03 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <map>

// -- 3Dwm Includes
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"
#include "Polhem/SolidCache.hh"

using namespace Nobel;

// -- Code Segment

SolidCache::SolidCache()
{
    // empty
}

SolidCache::~SolidCache()
{
    // Deallocate the cached data
    clear();
}

BSPTree *SolidCache::lookup(Nobel::Solid::Node_ptr key) const
{
    // Look it up in the cache map
    std::hash_map<cache_key_t, BSPTree *>::const_iterator i = 
	_cache.find(cache_key_t(key));

    // Now, what we did we get back? 
    if (i == _cache.end())
	// Didn't find it, return the zero pointer
	return 0;
    else 
	// Oh, we did find it, return the pointer!
	return i->second;
}

bool SolidCache::exists(Nobel::Solid::Node_ptr key) const
{
    // The lookup call will return 0 if not found
    return lookup(key) != 0;
}

void SolidCache::invalidate(Nobel::Solid::Node_ptr key)
{
    // First look it up in the cache map
    std::hash_map<cache_key_t, BSPTree *>::iterator i = _cache.find(key);
    
    // Did we find it? 
    if (i == _cache.end()) return;
    
    // Yes, so first deallocate the BSP tree and then erase the entry
    delete i->second;
    _cache.erase(i);
}
    
void SolidCache::cache(Nobel::Solid::Node_ptr key, BSPTree *tree)
{
    // Just to be sure, we start off by invalidating the key. This
    // will deallocate any existing cached version of the tree.
    invalidate(key);

    // Finally, add the node to the cache
    _cache[cache_key_t(key)] = tree;
}

void SolidCache::clear()
{
    // Step through the map and deallocate all BSP trees
    for (std::hash_map<cache_key_t, BSPTree *>::iterator i = _cache.begin();
	 i != _cache.end(); i++) 
	delete i->second;
    
    // Clear the map, too
    _cache.clear();
}
