/* ------------------------------------------------------------------------
 * $Id: EvalVisitor.cc,v 1.2 2001/06/25 12:28:36 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- Zorn Includes
#include "AST.hh"
#include "Store.hh"
#include "EvalFunctors.hh"
#include "FunctionSet.hh"
#include "FunctionObject.hh"
#include "EvalVisitor.hh"

// -- Code Segment

Zorn::EvalVisitor::EvalVisitor(Store *store)
    : _store(store)
{
    _binary["+"] = new AddFunctor();
    _binary["-"] = new SubFunctor();
    _binary["*"] = new MulFunctor();
    _binary["/"] = new DivFunctor();
    _binary["=="] = new EqFunctor();
    _binary["!="] = new NeqFunctor();
    _binary["<="] = new LeqFunctor();
    _binary[">="] = new GeqFunctor();
    _binary[">"] = new GreatFunctor();
    _binary["<"] = new LessFunctor();
    _binary["&&"] = new AndFunctor();
    _binary["||"] = new OrFunctor();

    _unary["-"] = new NegFunctor();
    _unary["!"] = new NotFunctor();
}

Zorn::EvalVisitor::~EvalVisitor()
{
    // Deallocate all unary functors
    for (std::map<std::string, UnaryFunctor *>::iterator i = _unary.begin();
	 i != _unary.end(); i++) 
	delete i->second;
    // ...as well as the binary ones
    for (std::map<std::string, BinaryFunctor *>::iterator i = _binary.begin();
	 i != _binary.end(); i++) 
	delete i->second;
}   

void Zorn::EvalVisitor::beginVisit()
{
    // Clear evaluation stacks prior to evaluation
    _evalStack.clear();
    _paramCountStack.clear();
}

void Zorn::EvalVisitor::endVisit()
{
    // empty
}

void Zorn::EvalVisitor::visitRule(ASTRule *node) 
{
    // Evaluate the condition branch
    node->getFirstChild()->accept(this);
    Value t = popState();
    
    // If the condition is true, execute the actions
    if (t.getBool()) {
	node->getFirstChild()->getNextSibling()->accept(this);
    }

    // Move to the next rule in the list (if there is one). This is
    // used in nested rules.
    if (node->getNextSibling() != 0)
	node->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitActionCall(ASTActionCall *node)
{
    // Evaluate the function call and remove the state from the stack
    node->getFirstChild()->accept(this);
    popState();

    // Move to the next action in the list (if there is one)
    if (node->getNextSibling() != 0)
	node->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitAssign(ASTAssign *node) 
{
    // Evaluate the expression
    node->getFirstChild()->accept(this);
    Value t = popState();

    // Store the state for the given value
    _store->set(node->getText(), t);
    
    // Move to the next action in the list
    if (node->getNextSibling() != 0)
	node->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitCall(ASTCall *node) 
{
    // Evaluate the parameter list 
    beginParams();
    if (node->getFirstChild() != 0) 
	node->getFirstChild()->accept(this);
    int count = endParams();
    
    // Retrieve the parameter list state and put it in a vector
    std::vector<Value> params;
    for (int i = 0; i < count; i++) {
	Value v = popState();
	params.insert(params.begin(), v);
    }

    // Retrieve the function object (if it exists)
    FunctionObject *fun = FunctionSet::instance()->find(node->getText());
    if (fun == 0)
	pushState(Value(false));
    else {
	// Make sure the parameter count matches
	if (fun->getParamCount(node->getText()) == count) {

	    // It does, execute the call!
	    Value result = fun->execute(node->getText(), params, _store);
	    
	    // Store the result on the stack
	    pushState(result);
	}
    }
}

void Zorn::EvalVisitor::visitParam(ASTParam *node) 
{
    // Evaluate the expression (leave the state on the stack)
    incrementParamCount();
    node->getFirstChild()->accept(this);
    
    // Move to the next action in the list
    if (node->getNextSibling() != 0)
	node->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitIf(ASTIf *node)
{
    // Evaluate the condition branch
    node->getFirstChild()->accept(this);
    Value t = popState();
    
    // Now, depending on the result, evaluate first or second branch
    if (t.getBool() == true)
	node->getFirstChild()->getNextSibling()->accept(this);
    else
	node->getFirstChild()->getNextSibling()->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitExists(ASTExists *node)
{
    pushState(_store->exists(node->getText()));
}

void Zorn::EvalVisitor::visitBinOp(ASTBinOp *node)
{
    // Visit the two branches for the operands
    node->getFirstChild()->accept(this);
    node->getFirstChild()->getNextSibling()->accept(this);

    // Retrieve state (in the correct order)
    Value t2 = popState();
    Value t1 = popState();

    // Find the relevant functor and execute it on the operands
    Value r = _binary[node->getName()]->execute(t1, t2);
    pushState(r);
}

void Zorn::EvalVisitor::visitAnd(ASTAnd *node)
{
    // For efficiency, we will short-circuit the AND operation if
    // possible.
    
    // Visit the first branch
    node->getFirstChild()->accept(this);

    // Retrieve state and check it
    Value t1 = popState();
    if (t1.getBool() == false)
	pushState(Value(false));
    else
	// Visit the second branch, the value of the AND operation
	// will be left on the stack
	node->getFirstChild()->getNextSibling()->accept(this);
}

void Zorn::EvalVisitor::visitUnOp(ASTUnOp *node)
{
    // Visit the operand branch and retrieve state
    node->getFirstChild()->accept(this);
    Value t = popState();
    
    // Execute the relevant functor on the operand
    Value r = _unary[node->getName()]->execute(t);
    pushState(r);
}

void Zorn::EvalVisitor::visitInt(ASTInt *node)
{
    pushState(Value(node->getValue()));
}

void Zorn::EvalVisitor::visitDouble(ASTDouble *node)
{
    pushState(Value(node->getValue()));
}

void Zorn::EvalVisitor::visitString(ASTString *node)
{
    pushState(Value(node->getText()));
}

void Zorn::EvalVisitor::visitBool(ASTBool *node) 
{
    pushState(Value(node->getValue()));
}

void Zorn::EvalVisitor::visitId(ASTId *node) 
{
    pushState(_store->get(node->getText()));
}
