/* ------------------------------------------------------------------------
 * ImageLoader.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-18 by Steve Houston.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */


#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <Nobel++/ImageLoader.hh>
#include <Celsius/debug.hh>

using namespace Nobel;

/*
 * ImageLoader - utility class to help with loading of image files.
 * Compiled into Nobel library which should be linked by client.
 */

Nobelxx::ImageLoader::ImageLoader()
    : _imageData(0)
{

}

Nobelxx::ImageLoader::~ImageLoader()
{
    if (_imageData != 0)
	free(_imageData);
}

bool Nobelxx::ImageLoader::loadPNG(const char *filename)
{
    const int     BYTES_TO_CHECK = 8;
    png_uint_32   width, height, row;
    int           bit_depth, color_type, interface_type;
    int           compression_type, filter_type;
    Nobel::Image::ImgInfo img_info;
    FILE          *fp;
    struct stat   st;
    unsigned char header[BYTES_TO_CHECK];

    // Get file stats
    if (stat(filename, &st))
	return false;

    // Open file for reading
    if (!(fp = fopen(filename, "rb")))
	return false;
    
    // Make sure we have a png file
    fread(header, 1, BYTES_TO_CHECK, fp);
    
    if (png_sig_cmp(header, 0, BYTES_TO_CHECK)) {
	fclose(fp);
	return false;
    }
    
    // Allocate read struct using default error handlers
    png_structp png_ptr = png_create_read_struct(
        PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    png_infop info_ptr = png_create_info_struct(png_ptr);
    png_infop end_info = png_create_info_struct(png_ptr);

    png_init_io(png_ptr, fp);
    png_set_sig_bytes(png_ptr, BYTES_TO_CHECK);
    
    // Set the read and error callbacks
    //png_set_read_fn(png_ptr, buf, &png_read_data);
    // TODO: set error function when 3Dwm logging service is complete

    // Read the image info
    png_read_info(png_ptr, info_ptr);
    
    // Extract the info
    png_get_IHDR(png_ptr, info_ptr, &width, &height,
		 &bit_depth, &color_type, &interface_type,
		 &compression_type, &filter_type);
    DPRINTF("Image bit_depth: %d\n", bit_depth);

    // Update member variables
    _width = (CORBA::Long) width;
    _height = (CORBA::Long) height;

    // If the PNG has 16-bits per channel convert 
    // down to 8-bits.
    if (bit_depth == 16)
	png_set_strip_16(png_ptr);

    switch(color_type) {
    case PNG_COLOR_TYPE_GRAY:
	png_set_gray_to_rgb(png_ptr);
	_pixelType = Nobel::RGB888;
	break;
    case PNG_COLOR_TYPE_GRAY_ALPHA:	
	png_set_gray_to_rgb(png_ptr);
	_pixelType = Nobel::RGBA8888;
	break;
    case PNG_COLOR_TYPE_RGB:
	_pixelType = Nobel::RGB888;
	break;
    case PNG_COLOR_TYPE_RGB_ALPHA:
	_pixelType = Nobel::RGBA8888;
	break;
    default:
	// Sorry, we only support RGB and RGBA formats
	png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
	return false;
    }	

    png_read_update_info(png_ptr, info_ptr);

    // If this ImageLoader was previously used to load another image
    // discard the old image buffer.
    if (_imageData != 0)
	free(_imageData);
    // Allocate space for the image data
    png_bytep row_pointers[height];
    int pitch = png_get_rowbytes(png_ptr, info_ptr);
    _bufferSize = (pitch * height);
    _imageData = malloc(_bufferSize);

    // Set pointer to last row of image
    png_bytep ptr = (png_bytep)_imageData + ((height-1) * pitch);

    // Initialise row pointers
    for (row = 0; row < height; row++) {
	row_pointers[row] = ptr;
	ptr -= pitch;
    }

    // Read the image!
    png_read_image(png_ptr, row_pointers);
    png_read_end(png_ptr, end_info);
    png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);

    // Create a CORBA binary sequence reference to the data
    // We retain control of the buffer as it wasn't allocated by allocbuf
    _binData = BinaryData(_bufferSize, _bufferSize, 
			  (CORBA::Octet*) _imageData, 0);

    return true;
}
    
const BinaryData& Nobelxx::ImageLoader::rawData(void)
{
    return _binData;
}






