/*
 * Copyright (C) 2003-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* TODO LIST:
 * - if possible (in other words, if documentation can be found): Make the
 *   FlashROM actually flashable.
 * - implement CRC
 * - fix (? does it need to be fixed?) flexible mode
 * - (?) implement pci_reset - depends on: PCI
 */

/* How much Flash ROM we have. (64k on 82557) */ /* Should be 128k - FIXME */
#define SZ_E100FLASHROM	(64*1024)

/* How much space we REQUEST for ROM (64kB on 82557) */
#define SZ_E100ROMREQUEST (64*1024)

#define NUMCONFBYTES		22      /* We have 22 Configuration Bytes */

#define MAXPACKETSIZE 1518	/* Developer manual states 2600, but that */
				/* is impossible over ethernet. Because we */
				/* simulate a card attached to standard 100 */
				/* MBit Ethernet, this value is more sane. */

#ifdef STATE

struct {
	// struct process process;

	unsigned char flashrom[SZ_E100FLASHROM];/* The flashrom of the card */

	/* Config Space Registers */
	uint8_t io_space;
	uint8_t memory_space;
	uint8_t bus_master;
	uint8_t memory_write_and_invalidate_enable;
	uint8_t parity_error_response;
	uint8_t serr_disable;
	uint8_t data_parity_reported;
	uint8_t received_target_abort;
	uint8_t received_master_abort;
	uint8_t signaled_system_error;
	uint8_t detected_parity_error;
	uint8_t cache_line_size;
	uint8_t latency_timer;
	uint32_t memaddr;
	uint32_t ioaddr;
	uint32_t flashaddr;
	uint8_t romenabled;
	uint32_t romaddr;
	uint8_t interrupt_line;
		
	/* I/O Space Registers */
	/* Byte 0 */
	uint8_t cu_status;
	uint8_t ru_status;
	/* Byte 1 */
	uint8_t stat_ack;
	/* Byte 2 */
	uint8_t cu_command;
	uint8_t ru_command;
	/* Byte 3 */
	uint8_t ic;
	/* Byte 4-7 */
	uint32_t param;
	/* Byte 8-11 */
	uint32_t port;

	/* Byte 14 */
	uint8_t eeprom_cr;
	/* Byte 15 */
	/* Reserved */
	/* Byte 16-19 */
	uint16_t mdi_data;
	uint16_t mdi;

#ifdef __CARDBUS__
	uint32_t reg30;
	uint32_t reg34;
#endif
	/* Internal Registers */
	uint32_t header;	  /* Last header read */
	uint32_t cu_base;	  /* Offset added to all Pointers given
				   * to the CU */
	uint32_t cu_pos;          /* Current pointer for the command unit */
	uint32_t cu_addr;         /* Address of last header */
	uint32_t ru_base;	  /* Offset added to all Pointers given
				   * to the RU */
	uint32_t ru_pos;          /* Current pointer for the receive unit */
	uint32_t StatDumpAddress; /* Where we shall dump our stats */
	uint32_t statcounters[17];/* Statistic counters */

	uint8_t confbytes[NUMCONFBYTES];	/* Configuration Bytes */
	uint8_t mac[6];				/* MAC Address of the device */
	uint8_t mcasthash[8];			/* Multicast Hash */

	uint8_t tmprpacket[MAXPACKETSIZE];	/* temporary space for
						 * receiving packet */
	uint8_t tmpspacket[MAXPACKETSIZE];	/* temporary space for
						 * sending packet */
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

/* Happy little defines... because, in our world, an eepro100 has a
 * lot of features, it needs a lot of defines too.
 */

/* How many bytes of our mapped memory are actually useful (i.e. used by
   registers) */
#ifdef __CARDBUS__
#define SZ_E100MEMORY	0x40
#else
#define SZ_E100MEMORY	0x20
#endif

/* SCB System Control Block, Offset 0x00 - 0x07
 * |-SCB Status Word,        Offset 0x00 - 0x01
 * | |-SCB Status Byte,      Offset 0x00
 * | \-STAT/ACK byte,        Offset 0x01
 * |-SCB Command Word,       Offset 0x02 - 0x03
 * | |-Command Byte,         Offset 0x02
 * | \-Interrupt Control B., Offset 0x03
 * \-SCB General Pointer,    Offset 0x04 - 0x07
 */
#define SCB_STATACK_CX	0x80	/* The CU has finished executing a command */
#define SCB_STATACK_FR	0x40	/* The RU has finished receiving a frame */
#define SCB_STATACK_CNA	0x20	/* The CU has left the active state */
#define SCB_STATACK_RNR	0x10	/* The RU has left the ready state */
#define SCB_STATACK_MDI	0x08	/* MDI R or W cycle has completed */
#define SCB_STATACK_SWI	0x04	/* Software generated interrupt */

#define SCB_ICB_SI	0x02	/* Software generated Interrupt */
#define SCB_ICB_M	0x01	/* Interrupt Mask Bit */

/* Status Codes for RUS */
#define RUS_IDLE	0	/* Idle */
#define RUS_SUSPENDED	1	/* Suspended */
#define RUS_NORES	2	/* No resources (EVIL!) */
#define RUS_READY	4	/* Ready */
/* Status Codes for CUS */
#define CUS_IDLE	0	/* Idle */
#define CUS_SUSPENDED	1	/* Suspended */
#define CUS_LPQACT	2	/* LPQ Active */
#define CUS_HQPACT	3	/* HQP Active */

/* the PORT register,  Offset 0x08 - 0x0B */
#define PORT_OPCODEMASK 0x0000000F /* which bits of the port register are */
				   /* the opcode (rest is address) */

/* Reserved, Offset 0x0C - 0x0D */

/* EEPROM Interface, Offset 0x0E - 0x0F */
#define EEPROM_EEDO		(1 << 3)	/* EEPROM Serial Data Out */
/* Macros for getting the values of some signal (0 or 1) */
#define EEPROM_DISIGNAL(x)	((x >> 2) & 1)
#define EEPROM_CSSIGNAL(x)	((x >> 1) & 1)
#define EEPROM_SKSIGNAL(x)	((x >> 0) & 1)

#define SZ_E100EEPROM	64

/* MDI Management Data Interface, Offset 0x10 - 0x13 */
#define MDI_REGAD	0x001F	/* PHY Register Address */
#define MDI_REGAD_SHIFT	     0
#define MDI_PHYAD	0x03E0	/* PHY Address */
#define MDI_PHYAD_SHIFT	     5
#define MDI_OPCODE	0x0C00	/* MDI Opcode */
#define MDI_OPCODE_SHIFT    10	/* How Far we have to shift the Opcode */
#define MDI_READY	0x1000	/* Ready */
#define MDI_IE		0x2000	/* Interrupt Enable */

#define MDI_PHY_ADDR	1	/* which address on the MDI does our */
				/* PHY listen to */

#define MDIREG(x) ((x & MDI_REGAD) >> MDI_REGAD_SHIFT)
#define MDIPHY(x) ((x & MDI_PHYAD) >> MDI_PHYAD_SHIFT)
#define MDICMD(x) ((x & MDI_OPCODE) >> MDI_OPCODE_SHIFT)

/* The stats area: Array-Index-Defines. */
#define STA_TRANSGOOD		 0	/* properly transmitted frames */
#define STA_TRANSEXCESSCOLL	 1	/* frames not transmitted due to */
					/* excessive collissions */
#define STA_TRANSLATECOLL	 2	/* frames not transmitted due to late collissions */
#define STA_TRANSUNDERRUN	 3	/* transmit underrun errors */
#define STA_TRANSCARRIERLOST	 4	/* carrier sense lost errors */
#define STA_TRANSDEFERRED	 5	/* deferred transmits due to activity on the link */
#define STA_TRANSSINGCOLL	 6	/* single collissions */
#define STA_TRANSMULTCOLL	 7	/* multiple collissions */
#define STA_TRANSTOTALCOLL	 8	/* total number of collissions */
#define STA_RECVGOOD		 9	/* properly received frames */
#define STA_RECVCRCERRS		10	/* receive CRC errors */
#define STA_RECVALIGNERR	11	/* receive alignment errors */
#define STA_RECVRESERR		12	/* receive resource errors */
#define STA_RECVOVERRUN		13	/* receive overruns */
#define STA_RECVCOLLDET		14	/* receive collission detect errors */
#define STA_RECVSHORTFRAME	15	/* receive short frames */
#define STA_SIGNATURE		16	/* signature dword (0xA005 or 0xA007) */

/* Structure of a Command Block Header */
/* General Structure of a Command Block Header (or, mostly identical, a
 * Receive Frame Header):
 * Bytes 0-3: Command, and status bits
 * Bytes 4-7: Pointer to next CB
 * Byte  8+:  Additional data for specific commands */
/* these are set by the driver */
#define CB_EL		0x80000000	/* End of List (this is the last element) */
#define CB_S		0x40000000	/* Suspend after this */
#define CB_I		0x20000000	/* Generate Interrupt after this */
#define CB_NC		0x00100000	/* Don't automatically insert SrcAddr */
#define CB_SF		0x00080000	/* Device operating in flexible mode */
/* and these are modified by the NIC */
#define CB_C		0x00008000	/* DMA has completed */
#define CB_OK		0x00002000	/* Command was executed w/o error */
#define CB_U		0x00001000	/* Underrun(s) occured */

/* Receive Frame Header - bitmasks are mostly the same as in the command
 * blocks, so this is mostly just a list of aliases */
#define RF_EL	CB_EL
#define RF_S	CB_S
#define RF_SF	CB_SF
#define RF_C	CB_C
#define RF_OK	CB_OK
#define RF_NORES	0x00000100	/* Ran out of Buffer space */
#define RF_TYPELEN	0x00000020	/* Received Frame was Type Frame */

/* defines for the Command Block */
#define CB_CMDMASK	0x00070000	/* Mask for the Command */
#define CB_CMDSHIFT	16		/* and how far we have to shift it */
#define CBCMD(x)	((x & CB_CMDMASK) >> CB_CMDSHIFT)

/* TBD Transport Buffer Descriptor */
#define TBD_NUMBERMASK	0xFF000000	/* Mask for the TBD number */
#define TBD_NUMBERMASK_SHIFT	24
#define TBD_TRATHRMASK	0x00FF0000	/* Transmit Threshold */
#define TBD_TRATHRMASK_SHIFT	16
#define TBD_BYTECNTMASK	0x00003FFF	/* Transmit Control Block Byte Count */

#define TBDNUM(x)	((x & TBD_NUMBERMASK) >> TBD_NUMBERMASK_SHIFT)

/* RFD Receive Frame Descriptor */
#define RFD_SIZEMASK	0x3FFF0000	/* Size of the Buffer */
#define RFD_SIZEMASK_SHIFT	16
#define RFD_COUNTMASK	0x00003FFF	/* used bytes in the buffer */
#define RFD_EOF		0x00008000	/* set by nic when buffer is filled */
#define RFD_F		0x00004000	/* set by nic when count is updated */

#define RFD_GETSIZE(x)	((x & RFD_SIZEMASK) >> RFD_SIZEMASK_SHIFT)

/* Defines concerning the Configuration Bytes */
/* Byte 0 */
#define CONFBYTECOUNTMASK	0x3F	/* Mask for number of configuration bytes */
/* Byte 6 */
#define COBY_SAVEBADFRAMES	0x80	/* Save Bad Frames */
#define COBY_DISCARDOVERRUN	0x40	/* do not Discard Overrun Frames */
#define COBY_CIINTERRUPT	0x08	/* don't generate interrupt if CU goes */
					/* to suspend, only when it goes to idle */
/* Byte 10 */
#define COBY_LOOPBACKMODE	0xC0	/* Loopback enable / mode */
#define COBY_NOSOURCEADINS	0x08	/* No source address insertion */
/* Byte 15 */
#define COBY_BROADDISABLE	0x02	/* Broadcast Disable */
#define COBY_PROMISC		0x01	/* Promiscuous Mode */
/* Byte 18 */
#define COBY_RECEIVECRC		0x04	/* CRC gets transferred to memory */
#define COBY_TRANSMITPADDING	0x02	/* Pad short frames to 60 Bytes */
#define COBY_STRIPPING		0x01	/* Strip everything that exceeds the */
					/* length field of a incoming packet */
/* Byte 21 */
#define COBY_MULTICASTALL	0x08	/* Listen to ANY multicast address */

/* General note (for reference): format of a ethernet packet.
 * -------------------------------------------------------------------
 * | Destination | Source    | Length/Type | Data          | CRC     |
 * -------------------------------------------------------------------
 *   Byte 0-5      Byte 6-11   Byte 12-13    46-1500 Bytes   4 Bytes
 * This brings us to a total maximum frame length of
 * 1500+6+6+2+4 = 1518 Bytes. Minimum Length is 64 Bytes.
 * If the actual data is less than 46 bytes it has to be padded.
 * The eepro100 offers a feature to automatically do that. The length
 * field can be used to mark the number of actually used bytes in the
 * data field. However, it also has a second meaning: if length is
 * greater than 1500 (0x05DC), then it is not a length but instead a
 * type, marking higher level protocols. Some often seen types:
 *       0x0800  IPv(4|6)                  0x0806  ARP
 * Multi-Byte-Values (like length/type) are stored Big-Endian (network
 * byte order).
 * Note that the CRC field is not / not usually stored in memory when the
 * NIC transmits / receives a packet. It can be explicitly requested
 * on reception, but not on transmission (so you cannot send
 * artificially corrupted packets). The sig_eth layer we use for sending/
 * receiving also expects the packets without CRC.
 */

static void
eeprom_do_set(struct cpssp *cpssp, unsigned int val)
{
	cpssp->eeprom_do = val;
}

/*
 * Selective reset: (page 29)
 * Maintains:
 * - PCI configuration
 * - RU/CU base registers
 * - HDS size
 * - error counters
 * - configure
 * - IA setup
 * - multicast setup
 * Resets
 * - RU/CU are set to IDLE state
 * All other setup info is lost.
 */
static void
NAME_(selective_reset)(struct cpssp *cpssp)
{
	cpssp->NAME.cu_command = 0;
	cpssp->NAME.cu_status = CUS_IDLE;
	cpssp->NAME.ru_command = 0;
	cpssp->NAME.ru_status = RUS_IDLE;
	cpssp->NAME.stat_ack = 0;
	cpssp->NAME.ic = 0; /* Correct? FIXME */
	cpssp->NAME.param = 0;
	cpssp->NAME.port = 0;
	cpssp->NAME.eeprom_cr = 0;
	cpssp->NAME.mdi_data = 0;
	cpssp->NAME.mdi = 0;
}

/*
 * Software reset: (page 29)
 * Maintains
 * - PCI configuration
 * Resets
 * - all other setup info
 */
static void
NAME_(software_reset)(struct cpssp *cpssp)
{
	NAME_(selective_reset)(cpssp);

	/* Internal Registers */
	cpssp->NAME.cu_base = 0;
	cpssp->NAME.ru_base = 0;
	cpssp->NAME.cu_pos = 0;
	cpssp->NAME.ru_pos = 0;
	cpssp->NAME.StatDumpAddress = 0;
	memset(&cpssp->NAME.statcounters[0], 0x00, 17*4);
	
	memset(&cpssp->NAME.mac[0], 0xff, 6);
	memset(&cpssp->NAME.mcasthash[0], 0x00, 8);

	/* Initialize *all* configuration bytes - FIXME */
	memset(cpssp->NAME.confbytes, 0x00, sizeof(cpssp->NAME.confbytes));
	cpssp->NAME.confbytes[ 0] = NUMCONFBYTES;	/* Byte Count */
	cpssp->NAME.confbytes[ 6] = 0x32; /* Save bad frames, discard overrun
				      * recv, CI int and others */
	cpssp->NAME.confbytes[15] = 0xC8; /* Broadcast Disable, Promisc. Mode */
	cpssp->NAME.confbytes[18] = 0x02; /* padding, stripping, receive crc */
	cpssp->NAME.confbytes[21] = 0x05; /* Multicast All */
}

/*
 * Hardware Reset
 */
static void
NAME_(hardware_reset)(struct cpssp *cpssp)
{
	/* Initialize/reset PCI config space. */
	cpssp->NAME.io_space = 0;
	cpssp->NAME.memory_space = 0;
	cpssp->NAME.bus_master = 1;
	cpssp->NAME.memory_write_and_invalidate_enable = 0;
	cpssp->NAME.parity_error_response = 0;
	cpssp->NAME.serr_disable = 0;
	cpssp->NAME.data_parity_reported = 0;
	cpssp->NAME.received_target_abort = 0;
	cpssp->NAME.received_master_abort = 0;
	cpssp->NAME.signaled_system_error = 0;
	cpssp->NAME.detected_parity_error = 0;
	cpssp->NAME.cache_line_size = 0;
	cpssp->NAME.latency_timer = 32;
	cpssp->NAME.memaddr = 0x00000000;
	cpssp->NAME.ioaddr = 0x00000000;
	cpssp->NAME.flashaddr = 0x00000000;
	cpssp->NAME.romenabled = 0;
	cpssp->NAME.romaddr = 0x00000000;
	cpssp->NAME.interrupt_line = 0x00;

	/* Initialize rest. */
	NAME_(software_reset)(cpssp);
}

static void
NAME_(read)(
	struct cpssp *cpssp,
	uint32_t addr,
	void *_to,
	unsigned int len
)
{
	uint8_t *to = (uint8_t *) _to;

	while (0 < len) {
		unsigned int count;
		unsigned int bs;
		uint32_t val;

		count = len;
		if (4 < (addr & 3) + count) {
			count = 4 - (addr & 3);
		}
		bs = ((1 << count) - 1) << (addr & 3);

		if (sig_pci_bus_mr(cpssp->port_bus, cpssp, addr & ~3, bs, &val) != 0) {
			sig_pci_bus_addr_type(cpssp->port_bus, cpssp,
					addr & ~3, SIG_PCI_BUS_MR);
			/* delay... */
			sig_pci_bus_read_data(cpssp->port_bus, cpssp,
					bs, &val);
		}

		if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
				__FUNCTION__, addr & ~3, bs, val);

		memcpy(to, (uint8_t *) &val + (addr & 3), count);

		addr += count;
		len -= count;
		to += count;
	}
}

static void
NAME_(write)(
	struct cpssp *cpssp,
	uint32_t addr,
	const void *_from,
	unsigned int len
)
{
	const uint8_t *from = (const uint8_t *) _from;

	while (0 < len) {
		unsigned int count;
		unsigned int bs;
		uint32_t val;

		count = len;
		if (4 < (addr & 3) + count) {
			count = 4 - (addr & 3);
		}
		bs = ((1 << count) - 1) << (addr & 3);

		memcpy((uint8_t *) &val + (addr & 3), from, count);

		if (sig_pci_bus_mw(cpssp->port_bus, cpssp, addr & ~3, bs, val) != 0) {
			sig_pci_bus_addr_type(cpssp->port_bus, cpssp,
					addr & ~3, SIG_PCI_BUS_MW);
			/* delay... */
			sig_pci_bus_write_data(cpssp->port_bus, cpssp,
					bs, val);
		}

		if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
				__FUNCTION__, addr & ~3, bs, val);

		addr += count;
		len -= count;
		from += count;
	}
}

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int irq;

	irq = ! (cpssp->NAME.ic & SCB_ICB_M) && cpssp->NAME.stat_ack;

	if (DEBUG) fprintf(stderr, "%s: irq=%d\n", __FUNCTION__, irq);

	sig_std_logic_or_set(cpssp->port_int, cpssp, irq);
}

/* Checks if the E100 is in any loopback mode. Returns 1 if it is. */
static int
NAME_(checkloopbackmode)(struct cpssp *cpssp)
{
	if (cpssp->NAME.confbytes[10] & COBY_LOOPBACKMODE) {
		return 1;
	}
	return 0;
}

/* return the 6-bit index into the multicast
 * table. Taken from umne2000.c, which took it from Bochs mcast_index()
 */
static unsigned char
NAME_(mcasthash)(unsigned char *dst)
{
#define POLYNOMIAL 0x04c11db6
	unsigned long crc = 0xffffffffL;
	int carry, i, j;
	unsigned char b;

	for (i = 6; --i >= 0;) {
		b = *dst++;
		for (j = 8; --j >= 0;) {
			carry = ((crc & 0x80000000L) ? 1 : 0) ^ (b & 0x01);
			crc <<= 1;
			b >>= 1;
			if (carry) {
				crc = ((crc ^ POLYNOMIAL) | carry);
			}
		}
	}
	return (crc >> 25) & 0x3F; /* Selects Bits 2 to 7 */
#undef POLYNOMIAL
}

/*
 * Checks if a packet is addressed for us.
 * This can be the case if it is
 * a) a broadcast
 * b) unicast for our mac
 * c) multicast for a multicast address we listen to
 */
static int
NAME_(ispacketforme)(struct cpssp *cpssp, unsigned char *tpacket)
{
	if (memcmp(tpacket, cpssp->NAME.mac, 6) == 0) {
		return 1; /* This is for our unicast mac-address */
	}
	if (cpssp->NAME.confbytes[15] & COBY_PROMISC) {
		/* We are in promiscuous mode, so everything is for us */
		return 1;
	}
	if (tpacket[0] == 0xFF
	 && tpacket[1] == 0xFF
	 && tpacket[2] == 0xFF
	 && tpacket[3] == 0xFF
	 && tpacket[4] == 0xFF
	 && tpacket[5] == 0xFF) {
	   	/* A broadcast always is for us too */
		if (cpssp->NAME.confbytes[15] & COBY_BROADDISABLE) {
			/* ...Unless we are set to ignore them */
			return 0;
		} else {
			return 1;
		}
	}
	if (tpacket[0] & 0x01) { /* Multicast Bit is set */
		if (cpssp->NAME.confbytes[21] & COBY_MULTICASTALL) {
			/* We accept any multicast address */
			return 1;
		} else {
			/* The E100 uses a hash table to check if the packet
			 * COULD be for any of its multicast addresses.
			 * Of course, that means it will receive some packets
			 * that weren't meant for it.
			 * The hash function it uses is not documented, so we
			 * just use the same as the NE2000. */
			/* FIXME fox: check / Handle multicast correctly! */
			unsigned int idx;

			idx = NAME_(mcasthash)(&tpacket[0]);
			assert(/* 0 <= (idx >> 3) && */ (idx >> 3) <= 7);
			return (cpssp->NAME.mcasthash[idx >> 3] & (1 << (idx & 0x7)));
		}
	}
	return 0;
}

/*
 * This function handles packets received from the network.
 * It feeds them to the driver, writes headers, generates ints etc.
 */
static void
NAME_(handlereceivedpacket)(struct cpssp *cpssp, int len)
{
	struct {
		uint32_t val0;
		uint32_t val4;
		uint32_t val8;
		uint32_t valc;
	} rfd;
	unsigned char * tmppacket;

	if (DEBUG) fprintf(stderr, "%s\n", __FUNCTION__);

	tmppacket = &cpssp->NAME.tmprpacket[0];
	if (len < 6) {
		/* No valid packet */
		return;
	}

	if (cpssp->NAME.ru_base == 0
	 && cpssp->NAME.ru_pos == 0) {
		/* no receive area */
		cpssp->NAME.statcounters[STA_RECVRESERR]++;
		return;
	}
	if (cpssp->NAME.ru_status != RUS_READY) {
		/* We are not ready to receive. Abort. */
		if (cpssp->NAME.ru_status == RUS_NORES) {
			/* Count the overrun */
			cpssp->NAME.statcounters[STA_RECVOVERRUN]++;
		}
		return;
	}

	if (NAME_(ispacketforme)(cpssp, tmppacket) <= 0) {
		return;
	}

	/* Time to read the Receive Frame Descriptor */
	NAME_(read)(cpssp, cpssp->NAME.ru_base + cpssp->NAME.ru_pos, &rfd, sizeof(rfd));

	if ((rfd.valc & RFD_EOF) == RFD_EOF
	 || (rfd.valc & RFD_F) == RFD_F) {
		fprintf(stderr, "Pre-Used RFD detected @%08x\n", cpssp->NAME.ru_pos);
		cpssp->NAME.statcounters[STA_RECVRESERR]++;
		cpssp->NAME.ru_status = RUS_NORES;
		cpssp->NAME.stat_ack |= SCB_STATACK_RNR;
	}

	if (cpssp->NAME.confbytes[18] & COBY_STRIPPING) {
		int leninpacket;

		leninpacket = be16_to_cpu(*(unsigned short *) &tmppacket[12]);
		leninpacket += 14;
		if (leninpacket < len && leninpacket <= 1514) {
			len = leninpacket;
		}
	}
	if (0   == be16_to_cpu(*(unsigned short *)&tmppacket[12])
	 || 1518 < be16_to_cpu(*(unsigned short *)&tmppacket[12])) {
		rfd.val0 |= RF_TYPELEN;
	} 
	if (cpssp->NAME.confbytes[18] & COBY_RECEIVECRC) {
		/* FIXME fox: calculate CRC and put it into the packet. */
	}
	if (rfd.val0 & RF_SF) {
		/*
		 * Flexible Mode
		 */
		uint32_t rbdstart;
		int lenremain;
		
		lenremain = len;
		fprintf(stderr, "RFD @%08x is not in simplified mode!\n",
				cpssp->NAME.ru_pos);
		rbdstart = rfd.val8;

		while (0 < lenremain) {
			uint32_t bufpnt;
			uint32_t bufsiz;
			uint32_t bufstat;
			
			NAME_(read)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x00,
					&bufstat, 4);
			NAME_(read)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x08,
					&bufpnt, 4);
			NAME_(read)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x0C,
					&bufsiz, 4);
			if (bufpnt == 0 || bufpnt == 0xffffffffUL) {
				/* Nonsense values, don't destroy data */
				break;
			}
			if (bufsiz == 0 || 0x0000ffffUL < bufsiz) {
				/* Nonsense blocksize */
				break;
			}
			if (bufsiz < lenremain) {
				NAME_(write)(cpssp,
						cpssp->NAME.ru_base + bufpnt,
						tmppacket, bufsiz);
				tmppacket += bufsiz;
				lenremain -= bufsiz;
			} else {
				NAME_(write)(cpssp,
						cpssp->NAME.ru_base + bufpnt,
						tmppacket, lenremain);
				tmppacket += lenremain;
				lenremain = 0;
			}
			/* Now we need to update bufstat - but I have no idea
			 * HOW */
			/* bufstat |= 0x10101010; */
			NAME_(write)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x00,
					&bufstat, 4);
			NAME_(write)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x0C,
					&bufsiz, 4);
			NAME_(read)(cpssp,
					cpssp->NAME.ru_base + rbdstart + 0x04,
					&rbdstart, 4);
		}
		if (lenremain == 0) {
			rfd.val0 |= RF_OK;  /* Reception OK */
			rfd.valc &= RFD_COUNTMASK;  /* write length to header */
			rfd.valc |= len;
			rfd.valc |= RFD_EOF | RFD_F;
			cpssp->NAME.statcounters[STA_RECVGOOD]++;
		} else {
			rfd.val0 |= RF_NORES;
			cpssp->NAME.statcounters[STA_RECVRESERR]++;
		}

	} else {
		/*
		 * Simplified Mode
		 */
		/* This is not mentioned in the documentation, but win
		 * seems to use it: The special size "0" seems to mean
		 * "unlimited". */
		if (RFD_GETSIZE(rfd.valc) < len
		 && RFD_GETSIZE(rfd.valc) != 0) {
			/* That frame is too large for this RF */
			fprintf(stderr, "Dropping %d byte frame,"
				" too large for RFD with %d bytes space!\n",
				len, RFD_GETSIZE(rfd.valc));
			rfd.val0 |= RF_NORES;
			cpssp->NAME.statcounters[STA_RECVRESERR]++;
			cpssp->NAME.ru_status = RUS_NORES;
			cpssp->NAME.stat_ack |= SCB_STATACK_RNR;
		} else {
			NAME_(write)(cpssp,
					cpssp->NAME.ru_base + cpssp->NAME.ru_pos + 0x10,
					&tmppacket[0], len);
			rfd.val0 |= RF_OK;  /* Reception OK */
			rfd.valc &= RFD_COUNTMASK;  /* write length to header */
			rfd.valc |= len;
			rfd.valc |= RFD_EOF | RFD_F;
			cpssp->NAME.statcounters[STA_RECVGOOD]++;
		}
	}

	rfd.val0 |= RF_C; /* Reception complete (successful or not) */

	/* Write back headers (containing the status) */
	NAME_(write)(cpssp, cpssp->NAME.ru_base + cpssp->NAME.ru_pos, &rfd, sizeof(rfd));
	if ((rfd.val0 & RF_EL) == RF_EL) {
		/* End of List, no more memory */
		fprintf(stderr, "%s\n", "Receive Buffers full, next packet"
			" will be lost");
		cpssp->NAME.stat_ack |= SCB_STATACK_RNR;
		cpssp->NAME.ru_status = RUS_NORES;
	} else if ((rfd.val0 & RF_S) == RF_S) {
		/* Suspend after this */
		cpssp->NAME.stat_ack |= SCB_STATACK_RNR;
		cpssp->NAME.ru_status = RUS_SUSPENDED;
	}

	/* Read next chain entry - this happens even at the End of List! */
	cpssp->NAME.ru_pos = rfd.val4;

	cpssp->NAME.stat_ack |= SCB_STATACK_FR;
	NAME_(irq_update)(cpssp);
}

/*
 * This parses a packet as given to the TRANSMIT Command-Unit Command,
 * and then sends it out to the network.
 */
static void
NAME_(parseandsendpacket)(
	struct cpssp *cpssp,
	uint32_t offset,
	uint32_t CBHeader
)
{
	int len;
	int pos;
	uint32_t tmps[2];
	int i;
	uint32_t tmptbd[2];
	unsigned char * tmppacket;
	
	if (DEBUG) fprintf(stderr, "%s\n", __FUNCTION__);

	if (NAME_(checkloopbackmode)(cpssp)) {
		/* Loopbackmode, so put it into the RECEIVE buffer */
		tmppacket = cpssp->NAME.tmprpacket;
	} else {
		tmppacket = cpssp->NAME.tmpspacket;
	}

	NAME_(read)(cpssp, cpssp->NAME.cu_base + offset + 8, &tmps[0], 8);
	len = 0;
	pos = 0;
	if (0 < (tmps[1] & TBD_BYTECNTMASK)) {
		len += tmps[1] & TBD_BYTECNTMASK;
		if (MAXPACKETSIZE < len) {
			fprintf(stderr, "Packet too large to send! (%d > %d)\n",
				len, MAXPACKETSIZE);
			return;
		}
		NAME_(read)(cpssp, cpssp->NAME.cu_base + offset + 0x10,
			&tmppacket[pos], tmps[1] & TBD_BYTECNTMASK);
		pos += tmps[1] & TBD_BYTECNTMASK;
	}
	if (CBHeader & CB_SF) {
		/* Flexible Mode */
		for (i = 0; i < TBDNUM(tmps[1]); i++) {
			NAME_(read)(cpssp,
				cpssp->NAME.cu_base + tmps[0] + i * 8, &tmptbd[0], 8);
			len += tmptbd[1] & TBD_BYTECNTMASK;
			if (MAXPACKETSIZE < len) {
				fprintf(stderr, "Packet too large to send! (%d > %d)\n",
					len, MAXPACKETSIZE);
				return;
			}
			NAME_(read)(cpssp, tmptbd[0], &tmppacket[pos],
					tmptbd[1] & TBD_BYTECNTMASK);
			pos += tmptbd[1] & TBD_BYTECNTMASK;
		}
	}
	if (len == 0) {
		fprintf(stderr, "%s\n", "Transmit for 0 Byte packet!");
		return; /* 0 Bytes to send?! */
	}
	if ((CBHeader & CB_NC) == 0) { /* Fill in SRC if requested */
		memcpy(&tmppacket[6], &cpssp->NAME.mac[0], 6);
	}
	if ((cpssp->NAME.confbytes[18] & COBY_TRANSMITPADDING)) {
		for (i = len; i < 60; i++) {
			tmppacket[i] = 0x7F;
			len++;
		}
	}

	/* this would be the place to fill in crc... however, this is not
	   needed, as port_eth expects the packet without crc. */
	if (NAME_(checkloopbackmode)(cpssp)) {
		NAME_(handlereceivedpacket)(cpssp, len);
	} else {
		NAME_(send)(cpssp, tmppacket, len);
	}
}

/*
 * The following function takes the next Command Block, pointed to
 * by cu_base+cu_pos, and handles it. It then updates the result bits in
 * the CB, and sets cu_pos to the next position in the chained Command
 * Block List.
 * It returns 0 when it reaches the end of the Command Block List,
 * or 1 if there are more commands to handle.
 */
static int
NAME_(handlenextcb)(struct cpssp *cpssp)
{
	uint32_t tmps[2];
	int res;
	unsigned int idx;
	uint16_t mccount;
	uint16_t mcoffs;
	
	if (cpssp->NAME.cu_pos == 0 && cpssp->NAME.cu_base == 0) {
		cpssp->NAME.cu_status = CUS_IDLE;
		return 0;
	}
	cpssp->NAME.cu_addr = cpssp->NAME.cu_base + cpssp->NAME.cu_pos;
	NAME_(read)(cpssp, cpssp->NAME.cu_addr, &cpssp->NAME.header, sizeof(cpssp->NAME.header));

	if (DEBUG) fprintf(stderr, "%s: cmd=%d\n", __FUNCTION__, CBCMD(cpssp->NAME.header));

	/* The default is "all OK" - individual handlers may change that
	   if they encounter an error */
	cpssp->NAME.header |= CB_C | CB_OK;
	switch (CBCMD(cpssp->NAME.header)) {
	case 0: /* NOOP */
		break;
	case 1: /* Individual Address Setup */
		NAME_(read)(cpssp,
			cpssp->NAME.cu_base + cpssp->NAME.cu_pos + 8, &tmps[0], 8);
		memcpy(&cpssp->NAME.mac[0], &tmps[0], 6);
		break;
	case 2: /* Configure */
		NAME_(read)(cpssp,
			cpssp->NAME.cu_base + cpssp->NAME.cu_pos + 8, &tmps[0], 4);
		tmps[0] &= CONFBYTECOUNTMASK;
		if (tmps[0] > NUMCONFBYTES) {
			tmps[0] = NUMCONFBYTES;
		}
		NAME_(read)(cpssp,
			cpssp->NAME.cu_base + cpssp->NAME.cu_pos + 8,
			cpssp->NAME.confbytes, tmps[0]);
		break;
	case 3: /* Multicast Address Setup */
		/* First reset the hash list, then read new one supplied by
		 * the driver. */
		memset(&cpssp->NAME.mcasthash[0], 0x00, 8);
		NAME_(read)(cpssp,
			cpssp->NAME.cu_base + cpssp->NAME.cu_pos + 8, &mccount, 2);
		mccount &= 0x3FFF; /* Only 14 Bits are valid */
		mcoffs = 10; /* no, not 0x10! */
		while (6 <= mccount) {
			NAME_(read)(cpssp,
				cpssp->NAME.cu_base + cpssp->NAME.cu_pos + mcoffs, &tmps[0], 6);
			idx = NAME_(mcasthash)((char *)&tmps[0]);
			assert(/* 0 <= (idx >> 3) && */ (idx >> 3) <= 7);
			cpssp->NAME.mcasthash[idx >> 3] |= (1 << (idx & 0x7));
			mccount -= 6;
			mcoffs += 6;
		}
		break;
	case 4: /* Transmit */
		cpssp->NAME.header &= ~CB_U; /* No Underrun */
		NAME_(parseandsendpacket)(cpssp, cpssp->NAME.cu_pos, cpssp->NAME.header);
		cpssp->NAME.statcounters[STA_TRANSGOOD]++;
		break;
	case 5: /* Load Microcode */
		/* I'm a simulator damnit, I have no changeable microcode */
		fprintf(stderr, "%s%s\n","Driver is trying to update my Microcode! -> ",
			"I'll ignore it and continue, that should work.");
		break;
	case 6: /* Dump */
		fprintf(stderr, "%s\n","Driver is trying to dump my internal status!");
		break;
	case 7: /* Diagnose */
		/* Note: This already is the implementation. There is just
		 *       nothing to do since we have no physical hardware
		 *       to check. */
		break;
	};

	sched_delay(TIME_HZ / (16*1024));

	/* Write back header (containing the status) */
	NAME_(write)(cpssp, cpssp->NAME.cu_base + cpssp->NAME.cu_pos, &cpssp->NAME.header, 4);

	if (cpssp->NAME.header & CB_EL) {
		cpssp->NAME.stat_ack |= SCB_STATACK_CNA;
		cpssp->NAME.cu_status = CUS_IDLE;
		res = 0;
	} else if (cpssp->NAME.header & CB_S) {
		cpssp->NAME.stat_ack |= SCB_STATACK_CNA;
		cpssp->NAME.cu_status = CUS_SUSPENDED;
		res = 0;
	} else {
		res = 1;
	}

	/* Read next chain entry - this happens even at the End of List! */
	NAME_(read)(cpssp, cpssp->NAME.cu_base + cpssp->NAME.cu_pos + 4, &cpssp->NAME.cu_pos, 4);
	if (cpssp->NAME.header & CB_I) {
		cpssp->NAME.stat_ack |= SCB_STATACK_CX;
	}
	NAME_(irq_update)(cpssp);

	return res;
}

static void // __attribute__((__noreturn__))
NAME_(process)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->NAME.cu_status = CUS_IDLE;
	/* for (;;) */ {
		uint8_t cu;

		assert(cpssp->NAME.cu_status == CUS_IDLE
		    || cpssp->NAME.cu_status == CUS_SUSPENDED);
		while (cpssp->NAME.cu_command == 0) {
			sched_sleep();
		}
		cpssp->NAME.cu_status = CUS_LPQACT; /* FIXME */

		cu = cpssp->NAME.cu_command;
		cpssp->NAME.cu_command = 0;

		if (DEBUG) fprintf(stderr, "%s: cu=0x%x base=0x%x\n", __FUNCTION__, cu, cpssp->NAME.cu_base);

		switch (cu) {
		case 0: /* NOOP */
			cpssp->NAME.cu_status = CUS_IDLE;
			break;

		case 1: /* CU Start */
			cpssp->NAME.cu_pos = cpssp->NAME.param;
			cpssp->NAME.cu_addr = cpssp->NAME.cu_base + cpssp->NAME.cu_pos;
			cpssp->NAME.header = 0x00000000;
			goto work;

		case 2: /* CU Resume */
			/*
			 * Re-read last header.
			 * If suspend bit still set -> suspend.
			 * See page 105.
			 */
			NAME_(read)(cpssp, cpssp->NAME.cu_addr, &cpssp->NAME.header,
					sizeof(cpssp->NAME.header));
#if 0
			if (cpssp->NAME.header & CB_S) {
				fprintf(stderr, "Resuspended!\n");
				cpssp->NAME.cu_status = CUS_SUSPENDED;
				break;
			}
#endif
		work:	;
			while (NAME_(handlenextcb)(cpssp)) {
				assert(cpssp->NAME.cu_status == CUS_LPQACT);
			}
			assert(cpssp->NAME.cu_status == CUS_IDLE
			    || cpssp->NAME.cu_status == CUS_SUSPENDED);
			break;

		case 4: /* Load Dump Counters Address */
			cpssp->NAME.StatDumpAddress = cpssp->NAME.param;
			cpssp->NAME.cu_status = CUS_IDLE;
			break;

		/* case 5 is below 6 */
		case 6: /* Load CU Base */
			cpssp->NAME.cu_base = cpssp->NAME.param;
			cpssp->NAME.cu_status = CUS_IDLE;
			break;

		case 5: /* Dump Stat Counters */
		case 7: /* Dump and Reset Stat Counters */
			cpssp->NAME.statcounters[STA_SIGNATURE] = 0xA000 + cu;
			NAME_(write)(cpssp, cpssp->NAME.StatDumpAddress,
					&cpssp->NAME.statcounters[0], 17 * 4);
			if (cu == 7) {
				/* Reset stat counters */
				memset(&cpssp->NAME.statcounters[0], 0x00, 17 * 4);
			}
			cpssp->NAME.cu_status = CUS_IDLE;
			break;

		default:
			/* FIXME */
			cpssp->NAME.cu_status = CUS_IDLE;
			break;
		};

		// sched_to_scheduler();
	}
}

static void
NAME_(command)(struct cpssp *cpssp)
{
	int ru;

	/*
	 * Execute CU command.
	 */
	if (cpssp->NAME.cu_command) {
#if 0
		sched_wakeup(&cpssp->NAME.process);
#else
		NAME_(process)(cpssp);
#endif
	}

	/*
	 * Execute RU command.
	 */
	ru = cpssp->NAME.ru_command;
	cpssp->NAME.ru_command = 0; /* Set to 0 in order to signal acceptance. */

	switch (ru) {
	case 0: /* NOOP */
		break;

	case 1: /* RU Start */
		cpssp->NAME.ru_pos = cpssp->NAME.param;
		/* Fall through to resume! */

	case 2: /* RU Resume */
		cpssp->NAME.ru_status = RUS_READY;
		break;
		
	case 4: /* RU Abort */
		cpssp->NAME.ru_status = RUS_IDLE;
		break;

	case 5:	/* Load Header Data Size */
		break;

	case 6: /* Load RU Base */
		cpssp->NAME.ru_base = cpssp->NAME.param;
		break;

	default:
		/* FIXME */
		break;
	};
}

static void
NAME_(eeprom)(struct cpssp *cpssp)
{
	eeprom_cs_set(cpssp, EEPROM_CSSIGNAL(cpssp->NAME.eeprom_cr));
	eeprom_di_set(cpssp, EEPROM_DISIGNAL(cpssp->NAME.eeprom_cr));
	eeprom_clock_set(cpssp, EEPROM_SKSIGNAL(cpssp->NAME.eeprom_cr));
}

static void
NAME_(port)(struct cpssp *cpssp)
{
	unsigned long selftestres;

	switch (cpssp->NAME.port & PORT_OPCODEMASK) {
	/* Note: Sequence is 0x01 - 0x00 - 0x02 because the self test
	 * is followed by a full reset, and a full reset is a
	 * selective reset plus something more */
	case 0x01: /* Self Test */
		/* We write the self test results to the supplied
		 * memory address. */
		/* Signature must not be zero */
		selftestres = 0xffffffff;
		NAME_(write)(cpssp,
			cpssp->NAME.port & ~PORT_OPCODEMASK,
			&selftestres, 4);
		/* Result of all zeros means everything is fine. */
		selftestres = 0;
		NAME_(write)(cpssp,
			(cpssp->NAME.port & ~PORT_OPCODEMASK) + 4,
			&selftestres, 4);
		/*FALLTHROUGH*/

	case 0x00: /* Software Reset */
		NAME_(software_reset)(cpssp);
		/*FALLTHROUGH*/

	case 0x02: /* Selective Reset */
		NAME_(selective_reset)(cpssp);
		break;

	case 0x03: /* Dump */
		break;

	default: /* We don't know what to do with this */
		break;
	}
}

static void
NAME_(mdi)(struct cpssp *cpssp)
{
	if (MDIPHY(cpssp->NAME.mdi) == MDI_PHY_ADDR) {
		/* It really is a command for us. */
		switch (MDICMD(cpssp->NAME.mdi)) {
		case 0x1:
			mac_phy_write(cpssp, MDIREG(cpssp->NAME.mdi),
					cpssp->NAME.mdi_data);
			break;

		case 0x2:
			mac_phy_read(cpssp, MDIREG(cpssp->NAME.mdi),
					&cpssp->NAME.mdi_data);
			break;
		case 0x0:
		case 0x3:
			/*
			 * Reserved
			 */
			fprintf(stderr, "Illegal MDI Command (%ld) for PHY %ld\n",
				(unsigned long) MDICMD(cpssp->NAME.mdi),
				(unsigned long) MDIPHY(cpssp->NAME.mdi));
			cpssp->NAME.mdi_data = 0x0000;
			break;
		}
	} else {
		cpssp->NAME.mdi_data = 0x0000;
	}

	if (cpssp->NAME.mdi & MDI_IE) {
		cpssp->NAME.stat_ack |= SCB_STATACK_MDI;
		NAME_(irq_update)(cpssp);
	}
}

static void
NAME_(in)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	uint32_t *valp
)
{
	assert(! (port & 3));
	assert(port < SZ_E100MEMORY);

	*valp = 0;

	switch (port) {
	case 0x00:
		if ((bs >> 0) & 1) {
			/* CU/RU Status Register */
			*valp |= (cpssp->NAME.cu_status << 6)
				| (cpssp->NAME.ru_status << 2);
		}
		if ((bs >> 1) & 1) {
			/* Interrupt Status / Interrupt Ack Register */
			*valp |= cpssp->NAME.stat_ack << 8;
		}
		if ((bs >> 2) & 1) {
			/* CU/RU Command Register */
			*valp |= (cpssp->NAME.cu_command << 20)
				| (cpssp->NAME.ru_command << 16);
		}
		if ((bs >> 3) & 1) {
			/* Interrupt Control Register */
			*valp |= cpssp->NAME.ic << 24;
		}
		break;

	case 0x04:
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.param & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.param & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->NAME.param & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.param & (0xff << 24);
		}
		break;

	case 0x08:
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.port & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.port & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->NAME.port & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.port & (0xff << 24);
		}
		break;

	case 0x0c:
		if ((bs >> 0) & 1) {
			/* FLASH ROM Access? */
			/* FIXME */
			*valp |= 0x00 << 0;
		}
		if ((bs >> 1) & 1) {
			/* FLASH ROM Access? */
			/* FIXME */
			*valp |= 0x00 << 8;
		}
		if ((bs >> 2) & 1) {
			/* EEPROM Access */
			*valp |= cpssp->NAME.eeprom_cr << 16;
			*valp |= cpssp->eeprom_do << 19;
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
			*valp |= 0x00 << 24;
		}
		break;

	case 0x10:
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.mdi_data & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.mdi_data & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			*valp |= (cpssp->NAME.mdi << 16) & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			*valp |= (cpssp->NAME.mdi << 16) & (0xff << 24);
			*valp |= MDI_READY << 16;
		}
		break;

	case 0x14:
		/* Early Receive Count Register */
		*valp = 0x00000000; /* FIXME */
		break;

	case 0x18:
		/* Reserved */
		*valp |= 0x00 << 0;

		/* Flow Control Register */
		/* Not supported by 82557. */
		*valp |= 0x0000 << 8;

		/* Power Management Driver Register */
		/* Not supported by 82557. */
		*valp |= 0x00 << 24;
		break;

	case 0x1c:
		/* General Control Register */
		/* Not supported by 82557. */
		*valp |= 0x00 << 0;

		/* General Status Register */
		/* Not supported by 82557. */
		*valp |= 0x00 << 8;

		/* Reserved */
		*valp |= 0x0000 << 16;
		break;
#ifdef __CARDBUS__
	case 0x20 ... 0x2c:
		/* Reserved */
		*valp = 0x00000000;
		break;

	case 0x30:
		*valp = cpssp->NAME.reg30;
		break;

	case 0x34:
		*valp = cpssp->NAME.reg34;
		break;

	case 0x38 ... 0x3c:
		/* Reserved/Read-only */
		*valp = 0x00000000;
		break;
#endif /* __CARDBUS__ */
	default:
		assert(0);
		break;
	}

	if (DEBUG) fprintf(stderr, "%s: port=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, port, bs, *valp);
}

static void
NAME_(out)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	uint32_t val
)
{
	assert(! (port & 3));
	assert(port < SZ_E100MEMORY);

	if (DEBUG) fprintf(stderr, "%s: port=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, port, bs, val);

	switch (port) {
	case 0x00: /* Access to System Control Block */
		if ((bs >> 0) & 1) {
			/* CU/RU Status Register */
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Interrupt Status / Interrupt Ack Register */
			/*
			 * Bits set in "val" will reset interrupts
			 * in Interrupt Status Register.
			 */
			cpssp->NAME.stat_ack &= ~((val >> 8) & 0xff);
			NAME_(irq_update)(cpssp);
		}
		if ((bs >> 2) & 1) {
			/* CU/RU Command Register */
			cpssp->NAME.cu_command = (val >> 20) & 0xf;
			cpssp->NAME.ru_command = (val >> 16) & 0x7;
			NAME_(command)(cpssp);
		}
		if ((bs >> 3) & 1) {
			/* Interrupt Control Byte */
			cpssp->NAME.ic = ((val >> 24) & 0xff) & SCB_ICB_M;
			if (((val >> 24) & 0xff) & SCB_ICB_SI) {
				/* Request for Interrupt */
				cpssp->NAME.stat_ack |= SCB_STATACK_SWI;
			}
			NAME_(irq_update)(cpssp);
		}
		break;

	case 0x04: /* Parameter Register */
		if ((bs >> 0) & 1) {
			cpssp->NAME.param &= ~(0xff << 0);
			cpssp->NAME.param |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.param &= ~(0xff << 8);
			cpssp->NAME.param |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.param &= ~(0xff << 16);
			cpssp->NAME.param |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.param &= ~(0xff << 24);
			cpssp->NAME.param |= val & (0xff << 24);
		}
		break;
		
	case 0x08: /* Port Register */
		if ((bs >> 0) & 1) {
			cpssp->NAME.port &= ~(0xff << 0);
			cpssp->NAME.port |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.port &= ~(0xff << 8);
			cpssp->NAME.port |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.port &= ~(0xff << 16);
			cpssp->NAME.port |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.port &= ~(0xff << 24);
			cpssp->NAME.port |= val & (0xff << 24);
		}
		if ((bs >> 3) & 1) {
			NAME_(port)(cpssp);
		}
		break;

	case 0x0c:
		if ((bs >> 0) & 1) {
			/* FLASH ROM Access? */
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FLASH ROM Access? */
			/* FIXME */
		}
		if ((bs >> 2) & 1) {
			/* EEPROM Access */
			cpssp->NAME.eeprom_cr = (val >> 16) & 0x07;
			NAME_(eeprom)(cpssp);
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x10:
		if ((bs >> 0) & 1) {
			cpssp->NAME.mdi_data &= ~(0xff << 0);
			cpssp->NAME.mdi_data |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.mdi_data &= ~(0xff << 8);
			cpssp->NAME.mdi_data |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.mdi &= ~(0xff << 0);
			cpssp->NAME.mdi |= (val >> 16) & (0xff << 0);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.mdi &= ~(0xff << 8);
			cpssp->NAME.mdi |= (val >> 16) & (0xff << 8);
		}
		if ((bs >> 3) & 1) {
			NAME_(mdi)(cpssp);
		}
		break;

	case 0x14:
		/* Early Receive Count Register */
		/* Read-only */
		break;

	case 0x18:
		/* Reserved */

		/* Flow Control Register */
		/* Not supported by 82557. */

		/* Power Management Driver Register */
		/* Not supported by 82557. */
		break;

	case 0x1c:
		/* General Control Register */
		/* Not supported by 82557. */

		/* General Status Register */
		/* Not supported by 82557. */

		/* Reserved */
		break;
#ifdef __CARDBUS__
	case 0x20 ... 0x2c:
		/* Reserved */
		break;

		/* FIXME cardbus */
	case 0x30:
		cpssp->NAME.reg30 &= ~(val & (1 << 15));
		break;
	case 0x34:
		cpssp->NAME.reg34 |= val & (1 << 15);
		break;
	case 0x38:
		/* read only */
		break;
	case 0x3C:
		cpssp->NAME.reg30 |= val & (1 << 15);
		/* FIXME generate interrupt */
		break;
#endif /* __CARDBUS__ */
	default:
		assert(0);
	}
}

static int
NAME_(ior)(void *_cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->NAME.io_space) {
		return -1;

	} else if ((port & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.ioaddr) {
		port &= SZ_E100MEMORY - 1;
		NAME_(in)(cpssp, port, bs, valp);

	} else {
		return -1;
	}

	return 0;
}

static int
NAME_(iow)(void *_cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->NAME.io_space) {
		return -1;

	} else if ((port & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.ioaddr) {
		port &= SZ_E100MEMORY - 1;
		NAME_(out)(cpssp, port, bs, val);

	} else {
		return -1;
	}

	return 0;
}

static int
NAME_(mr)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->NAME.memory_space) {
		return -1;

	} else if ((addr & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.memaddr) {
		addr &= SZ_E100MEMORY - 1;
		NAME_(in)(cpssp, addr, bs, valp);

	} else if ((addr & ~(SZ_E100FLASHROM - 1)) == cpssp->NAME.flashaddr) {
		addr &= SZ_E100FLASHROM - 1;
		*valp = 0;
		*valp |= cpssp->NAME.flashrom[addr + 0] << 0;
		*valp |= cpssp->NAME.flashrom[addr + 1] << 8;
		*valp |= cpssp->NAME.flashrom[addr + 2] << 16;
		*valp |= cpssp->NAME.flashrom[addr + 3] << 24;

	} else if (cpssp->NAME.romenabled
		&& (addr & 0xfff00000) == cpssp->NAME.romaddr) {
		uint8_t val8;

		addr &= SZ_E100ROMREQUEST - 1;
		*valp = 0;
		if ((bs >> 0) & 1) {
			NAME_(rom_read)(cpssp, addr + 0, &val8);
			*valp |= val8 << 0;
		}
		if ((bs >> 1) & 1) {
			NAME_(rom_read)(cpssp, addr + 1, &val8);
			*valp |= val8 << 8;
		}
		if ((bs >> 2) & 1) {
			NAME_(rom_read)(cpssp, addr + 2, &val8);
			*valp |= val8 << 16;
		}
		if ((bs >> 3) & 1) {
			NAME_(rom_read)(cpssp, addr + 3, &val8);
			*valp |= val8 << 24;
		}

	} else {
		return -1;
	}

	return 0;
}

static int
NAME_(mw)(
	void *_cpssp,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (! cpssp->NAME.memory_space) {
		return -1;

	} else if ((addr & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.memaddr) {
		addr &= SZ_E100MEMORY - 1;
		NAME_(out)(cpssp, addr, bs, val);

	} else if ((addr & ~(SZ_E100FLASHROM - 1)) == cpssp->NAME.flashaddr) {
		/* FIXME fox: needs to be implemented -
		 * if documentation can be found, or someone
		 * figures it out. */
		addr &= SZ_E100FLASHROM - 1;
		/* FIXME */

	} else if (cpssp->NAME.romenabled
		&& (addr & 0xfff00000) == cpssp->NAME.romaddr) {
		/*
		 * This is and always will be write protected.
		 * Just ignore the write...
		 */
		/* addr &= SZ_E100ROMREQUEST - 1; */

	} else {
		return -1;
	}

	return 0;
}

static int
NAME_(map_r)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->NAME.memory_space) {
		return -1;
	}

	if ((pa & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.memaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mr);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	if ((pa & ~(SZ_E100FLASHROM - 1)) == cpssp->NAME.flashaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mr);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	if (cpssp->NAME.romenabled
	 && (pa & 0xfff00000) == cpssp->NAME.romaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mr);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	return 1;
}

static int
NAME_(map_w)(
	void *_cpssp,
	uint32_t pa,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->NAME.memory_space) {
		return -1;
	}

	if ((pa & ~(SZ_E100MEMORY - 1)) == cpssp->NAME.memaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mw);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	if ((pa & ~(SZ_E100FLASHROM - 1)) == cpssp->NAME.flashaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mw);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	if (cpssp->NAME.romenabled
	 && (pa & 0xfff00000) == cpssp->NAME.romaddr) {
		/* don't map but simulate access instead. */
		*cfp = NAME_(mw);
		*csp = cpssp;
		*haddr_p = NULL;
		return 0;
	}
	return 1;
}

static int
NAME_(c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *const cpssp = (struct cpssp *) _cpssp;

	assert (! (addr & 3));

	addr &= 0x7ff;
	if ((addr >> 8) & 7) {
		return -1;
	}
	addr &= 0xff;

	if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, addr, bs, val);

	switch (addr) {
	case 0x00:
		/* Vendor ID Register */
		/* Read-only */

		/* Device ID Register */
		/* Read-only */
		break;

	case 0x04:
		/* Command Register */
		if ((bs >> 0) & 1) {
			/* Just filter the hardwired-to-0 bits */
			cpssp->NAME.io_space = (val >> 0) & 1;
			cpssp->NAME.memory_space = (val >> 1) & 1;
			cpssp->NAME.bus_master = (val >> 2) & 1;
			cpssp->NAME.memory_write_and_invalidate_enable = (val >> 4) & 1;
			cpssp->NAME.parity_error_response = (val >> 6) & 1;

			/* Re-map memory regions. */
			sig_pci_bus_unmap(cpssp->port_bus, cpssp,
					cpssp->NAME.memaddr, 4096);
			sig_pci_bus_unmap(cpssp->port_bus, cpssp,
					cpssp->NAME.flashaddr, SZ_E100ROMREQUEST);
			sig_pci_bus_unmap(cpssp->port_bus, cpssp,
					cpssp->NAME.romaddr, SZ_E100ROMREQUEST);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.serr_disable = (val >> 8) & 1;
			/*
			 * I'm not entirely sure whether Bit 9 ("Enable fast
			 * back-to-back" should get filtered here. The
			 * documentation says so, but at the same time it
			 * says we announce to be FB2B capable, so I don't
			 * think we should.
			 */
		}

		/* Status Register */
		if ((bs >> 2) & 1) {
			/* PCI Status */
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.data_parity_reported &= ~((val >> (16+8)) & 1);
			cpssp->NAME.received_target_abort &= ~((val >> (16+12)) & 1);
			cpssp->NAME.received_master_abort &= ~((val >> (16+13)) & 1);
			cpssp->NAME.signaled_system_error &= ~((val >> (16+14)) & 1);
			cpssp->NAME.detected_parity_error &= ~((val >> (16+15)) & 1);
		}
		break;

	case 0x08:
		/* Revision ID Register */
		/* Read-only */
		
		/* Class ID Register */
		/* Read-only */
		break;

	case 0x0c:
		/* Cache Line Size Register */
		val &= 0x18;
		if ((bs >> 0) & 1) {
			cpssp->NAME.cache_line_size = (val >> 0) & 0xff;
		}

		/* Latency Timer Register */
		if ((bs >> 1) & 1) {
			cpssp->NAME.latency_timer = (val >> 8) & 0xff;
		}

		/* Header Type Register */
		/* Read-only */

		/* BIST Register */
		/* Read-only */
		break;

	case 0x10: {
		/* Base Address 0 Register */
		/* Request 4 KB Memory space */
		uint32_t oaddr;
		uint32_t naddr;

		val &= ~0xfff;
		oaddr = cpssp->NAME.memaddr;
		if ((bs >> 0) & 1) {
			cpssp->NAME.memaddr &= ~(0xff << 0);
			cpssp->NAME.memaddr |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.memaddr &= ~(0xff << 8);
			cpssp->NAME.memaddr |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.memaddr &= ~(0xff << 16);
			cpssp->NAME.memaddr |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.memaddr &= ~(0xff << 24);
			cpssp->NAME.memaddr |= val & (0xff << 24);
		}
		naddr = cpssp->NAME.memaddr;
		if (oaddr != naddr) {
			/* Re-map old/new region. */
			sig_pci_bus_unmap(cpssp->port_bus, cpssp, oaddr, 4096);
			sig_pci_bus_unmap(cpssp->port_bus, cpssp, naddr, 4096);
		}
		break;
	    }

	case 0x14:
		/* Base Address 1 Register */
		/* Request SZ_E100MEMORY IO Ports */
		val &= ~(SZ_E100MEMORY - 1);
		if ((bs >> 0) & 1) {
			cpssp->NAME.ioaddr &= ~(0xff << 0);
			cpssp->NAME.ioaddr |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ioaddr &= ~(0xff << 8);
			cpssp->NAME.ioaddr |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.ioaddr &= ~(0xff << 16);
			cpssp->NAME.ioaddr |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.ioaddr &= ~(0xff << 24);
			cpssp->NAME.ioaddr |= val & (0xff << 24);
		}
		break;

	case 0x18: {
		/* Base Address 2 Register */
		/* used by flash rom (request 1 MB) */
		uint32_t oaddr;
		uint32_t naddr;

		val &= ~(SZ_E100ROMREQUEST - 1);
		oaddr = cpssp->NAME.flashaddr;
		if ((bs >> 0) & 1) {
			cpssp->NAME.flashaddr &= ~(0xff << 0);
			cpssp->NAME.flashaddr |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.flashaddr &= ~(0xff << 8);
			cpssp->NAME.flashaddr |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.flashaddr &= ~(0xff << 16);
			cpssp->NAME.flashaddr |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.flashaddr &= ~(0xff << 24);
			cpssp->NAME.flashaddr |= val & (0xff << 24);
		}
		naddr = cpssp->NAME.flashaddr;
		if (oaddr != naddr) {
			/* Re-map old/new region. */
			sig_pci_bus_unmap(cpssp->port_bus, cpssp,
					oaddr, SZ_E100ROMREQUEST);
			sig_pci_bus_unmap(cpssp->port_bus, cpssp,
					naddr, SZ_E100ROMREQUEST);
		}
		break;
	    }
	case 0x1c:
		/* Base Address 3 Register */
		/* Unused */
		break;

	case 0x20:
		/* Base Address 4 Register */
		/* Unused */
		break;

	case 0x24:
		/* Base Address 5 Register */
		/* Unused */
		break;

	case 0x28:
		/* Cardbus CIS Pointer Register */
#ifdef __CARDBUS__
		/* Read-only */
#else
		/* Unused */
#endif
		break;

	case 0x2c:
		/* Subsystem Vendor ID Register */
		/* Read-only */

		/* Subsystem Device ID Register */
		/* Read-only */
		break;

	case 0x30: {
		/* Expansion ROM Base Address */
		uint32_t oaddr;
		uint32_t naddr;

		oaddr = cpssp->NAME.romaddr;
		if ((bs >> 0) & 1) {
			cpssp->NAME.romenabled = (val >> 0) & 1;
			/* Bit 1-7: Hardwired to '0'. */
		}
		if ((bs >> 1) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.romaddr &= ~(0xf0 << 16);
			cpssp->NAME.romaddr |= val & (0xf0 << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.romaddr &= ~(0xff << 24);
			cpssp->NAME.romaddr |= val & (0xff << 24);
		}
		naddr = cpssp->NAME.romaddr;

		/* Re-map region. */
		sig_pci_bus_unmap(cpssp->port_bus, cpssp,
				oaddr, SZ_E100ROMREQUEST);
		sig_pci_bus_unmap(cpssp->port_bus, cpssp,
				naddr, SZ_E100ROMREQUEST);
		break;
	    }
	case 0x34:
		/* Capability List Register */
		/* Hard-wired to 0. */
		break;

	case 0x38:
		/* Reserved */
		break;

	case 0x3c:
		/* Interrupt Line Register */
		if ((bs >> 0) & 1) {
			cpssp->NAME.interrupt_line = (val >> 0) & 0xff;
		}

		/* Interrupt Pin Register */
		/* Read-only */

		/* Min Gnt Register */
		/* Read-only */

		/* Max Lat Register */
		/* Read-only */
		break;

	case 0x40 ... 0xfc:
#ifdef __CARDBUS__
		/* Read-only */
#else
		/* Unused/Reserved */
#endif
		break;

	default:
		assert(0);
	}

	return 0;
}

static int
NAME_(c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *const cpssp = (struct cpssp *) _cpssp;
	
	assert(! (addr & 3));

	addr &= 0x7ff;
	if ((addr >> 8) & 7) {
		return -1;
	}
	addr &= 0xff;

	*valp = 0;
	switch (addr) {
	case 0x00:
		/* Vendor ID Register (Intel) */
		*valp |= 0x8086 << 0;

		/* Device ID Register (82557) */
		*valp |= 0x1229 << 16;
		break;

	case 0x04:
		/* Command Register */
		*valp |= cpssp->NAME.io_space << 0;
		*valp |= cpssp->NAME.memory_space << 1;
		*valp |= cpssp->NAME.bus_master << 2;
		*valp |= cpssp->NAME.memory_write_and_invalidate_enable << 4;
		*valp |= cpssp->NAME.parity_error_response << 6;

		*valp |= cpssp->NAME.serr_disable << 8;

		/* Status Register */
		*valp |= 1 << (16+7); /* Fast Back-to-Back Capable */

		*valp |= cpssp->NAME.data_parity_reported << (16+8);
		*valp |= 0x1 << (16+10); /* DEVSEL Timing */
		*valp |= cpssp->NAME.received_target_abort << (16+12);
		*valp |= cpssp->NAME.received_master_abort << (16+13);
		*valp |= cpssp->NAME.signaled_system_error << (16+14);
		*valp |= cpssp->NAME.detected_parity_error << (16+15);
		break;

	case 0x08:
		/* Revision ID Register */
		*valp |= 0x01 << 0;
		
		/* Class Prog Register */
		*valp |= 0x00 << 8;

		/* Class Device Register (Network/Ethernet) */
		*valp |= 0x0200 << 16;
		break;

	case 0x0c:
		/* Cache Line Size Register */
		*valp |= cpssp->NAME.cache_line_size << 0;

		/* Latency Timer Register */
		*valp |= cpssp->NAME.latency_timer << 8;

		/* Header Type Register */
		*valp |= 0x00 << 16;

		/* BIST Register */
		*valp |= 0x00 << 24;
		break;

	case 0x10:
		/* Base Address 0 Register */
		*valp = cpssp->NAME.memaddr
			| 0 /* Memory */
			| 0; /* 32-Bit Addresses */
		break;

	case 0x14:
		/* Base Address 1 Register */
		*valp = cpssp->NAME.ioaddr
			| 1; /* I/O */
		break;

	case 0x18:
		/* Base Address 2 Register */
		*valp = cpssp->NAME.flashaddr
			| 0 /* Memory */
			| 0; /* 32-Bit Addresses */
		break;

	case 0x1c:
		/* Base Address 3 Register */
		/* Unused */
		*valp = 0x00000000;
		break;

	case 0x20:
		/* Base Address 4 Register */
		/* Unused */
		*valp = 0x00000000;
		break;

	case 0x24:
		/* Base Address 5 Register */
		/* Unused */
		*valp = 0x00000000;
		break;

	case 0x28:
		/* Cardbus CIS Pointer Register */
#ifdef __CARDBUS__
		*valp = 0x40 << 3;
#else
		/* Unused */
		*valp = 0x00000000;
#endif
		break;

	case 0x2c:
		/* Subsystem Vendor ID Register */
		*valp |= 0x0000 << 0;

		/* Subsystem Device ID Register */
		*valp |= 0x0000 << 16;
		break;

	case 0x30:
		/* Expansion ROM Base Address */
		*valp = (cpssp->NAME.romenabled << 0)
			| cpssp->NAME.romaddr;
		break;

	case 0x34:
	case 0x38:
		/* Reserved */
		*valp = 0x00000000;
		break;

	case 0x3c:
		/* Interrupt Line Register */
		*valp |= cpssp->NAME.interrupt_line << 0;

		/* Interrupt Pin Register */
		*valp |= 1 << 8; /* INT A */

		/* Min Gnt Register */
		*valp |= 0x08 << 16;

		/* Max Lat Register */
		*valp |= 0x18 << 24;
		break;

	case 0x40 ... 0xfc:
#ifdef __CARDBUS__
	    {
		/* 
		 * Naive approach of a Card Information Structure.
		 * Unfortunately i have no real CardBus EEPro100 so some
		 * entries should be considered as wild speculations.
		 */
		/* FIXME waldi: explain all tuple byte values */
		static const uint8_t cis_eepro100[0x100 - 0x40] = {
			0x13, /* CISTPL_LINKTARGET */
			0x03, /* link to next tuple, always follows the tuple type
				 except for null tuples CISTPL_NULL */
			0x43, /* "C" */
			0x49, /* "I" */
			0x53, /* "S" */

			0x04, /* CISTPL_CONFIG_CB */
			0x06,
			0x03,
			0x40,
			0x00,
			0x00,
			0x00,
			0x00,

			0x05, /* CISTPL_CFTABLE_ENTRY_CB */
			0x05,
			0x40,
			0x38,
			0x02,
			0xA0,
			0x02,

			0x07, /* CISTPL_BAR */
			0x06,
			0x01,
			0x00,
			0x00,
			0x00,
			0x10,
			0x00,

			0x07, /* CISTPL_BAR */
			0x06,
			0x12,
			0x00,
			0x00,
			0x00,
			0x00,
			0x20,

			0x07, /* CISTPL_BAR */
			0x06,
			0x07,
			0x00,
			0x00,
			0x10,
			0x00,
			0x00,
			
			0x15, /* CISTPL_VERS_1 */
			0x0b,
			0x07,
			0x00,
			0x49, /* "I" */
			0x6E, /* "n" */
			0x74, /* "t" */
			0x65, /* "e" */
			0x6C, /* "l" */
			0x00, 
			0x00, /* skip the rest */
			0x00, /* and see what happens */
			0x00,

			0x20, /* CISTPL_MANFID */
			0x04,
			0x89, /* I found this somewhere and hope it's correct */
			0x00, /* Intel Manf. Id 0089, Prod. Id 0102 */
			0x02,
			0x01,

			0x1c, /* CISTPL_DEVICE_OC */
			0x03,
			0x03, /* other conditions info */
			0x47, /* EEPROM */
			0x0D, /* 2 * 512 KB */

			0xff, /* CISTPL_END */
		};

		*valp |= cis_eepro100[addr - 0x40 + 0] << 0;
		*valp |= cis_eepro100[addr - 0x40 + 1] << 8;
		*valp |= cis_eepro100[addr - 0x40 + 2] << 16;
		*valp |= cis_eepro100[addr - 0x40 + 3] << 24;
	    }
#else
		/* Reserved */
		*valp = 0x00000000;
#endif
		break;
	default:
		assert(0);
	}

	if (DEBUG) fprintf(stderr, "%s: addr=0x%x, bs=0x%x, val=0x%x\n",
			__FUNCTION__, addr, bs, *valp);

	return 0;
}

static void
NAME_(recv)(void *_cpssp, const void *buf, unsigned int buflen)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_power) {
		return;
	}
	if (cpssp->NAME.confbytes[10] & COBY_LOOPBACKMODE) {
		/* Loopback Mode */
		return;
	}

	assert(buflen <= sizeof(cpssp->NAME.tmprpacket));
	memcpy(cpssp->NAME.tmprpacket, buf, buflen);
	NAME_(handlereceivedpacket)(cpssp, buflen);
}

static void
NAME_(fixeepromchecksum)(struct cpssp *cpssp)
{
	unsigned short sum;
	int i;

	sum = 0;
	for (i = 0; i < 63; i++) {
		sum += cpssp->eeprom.data[i];
	}
	cpssp->eeprom.data[63] = 0xbaba - sum;
}

static void
NAME_(create)(
	struct cpssp *cpssp,
	const char *name,
	const char *mac,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_int,
	struct sig_mdi *port_mdi
)
{
	char fn[1024];
	int fd;
	int ret;

	/* initialize eeprom */
	memset(cpssp->eeprom.data, 0, SZ_E100EEPROM);
	sprintf(cpssp->eeprom.filename, "%s.eeprom", system_path());
	fd = open(cpssp->eeprom.filename, O_RDONLY);
	if (0 <= fd) { /* Load file */
		ret = read(fd, cpssp->eeprom.data, SZ_E100EEPROM * 2);
		assert(ret == SZ_E100EEPROM * 2);
		ret = close(fd);
		assert(0 <= ret);
	} else {
		int m[6];
		int i;

		/* Word 0-2: MAC */
		ret = sscanf(mac, "%x:%x:%x:%x:%x:%x",
				&m[0], &m[1], &m[2], &m[3], &m[4], &m[5]);
		cpssp->eeprom.data[0] = (m[1] << 8) | m[0];
		cpssp->eeprom.data[1] = (m[3] << 8) | m[2];
		cpssp->eeprom.data[2] = (m[5] << 8) | m[4];

		/* Word 3: FIXME */
		/* eepro100.c uses that bits to detect a "receiver
		 * lock-up bug" - well lets make it happy by telling
		 * it that we are not buggy. */
		cpssp->eeprom.data[3] = 0x0003;

		/* Word 4: FIXME */
		cpssp->eeprom.data[4] = 0;

		/*
		 * Word 5: Connector Type:
		 * Bit 0: RJ45 
		 * Bit 1: BNC
		 * Bit 2: AUI
		 * Bit 3: MII
		 */
		cpssp->eeprom.data[5] = 0x0001; /* only a RJ45 */

		/*
		 * Word 6: Primary interface chip
		 * Byte 0: Address
		 * Byte 1: Type:
		 *	0: None
		 *	1: Intel 82553-A/B
		 *	2: Intel 82553-C
		 *	3: Intel 82503
		 *	4: National Semiconductor DP83840
		 *	5: 80c240
		 *	6: 80c24
		 *	7: Intel 82555
		 */
		cpssp->eeprom.data[6]
			= (7 << 8)	/* Intel 82555 */
			| MDI_PHY_ADDR;	/* Address */

		/* Word 7: Secondary interface chip */
		/* secondary int. chip i82555 */
		cpssp->eeprom.data[7]
			= (0 << 8)	/* None */
			| 0;

		/* Word 8, 9: Board assembly */
		cpssp->eeprom.data[8] = 0x1234; /* FIXME */
		cpssp->eeprom.data[9] = 0x4321; /* FIXME */

		/* Word 10: Sleepmode */
		cpssp->eeprom.data[10] = 0;

		/* Word 11-62: FIXME */
		for (i = 11; i < 63; i++) {
			cpssp->eeprom.data[i] = 0x0000;
		}

		/* Word 63: Checksum */
		NAME_(fixeepromchecksum)(cpssp);
	}
	/* initialize flash rom */
	sprintf(fn, "%s.flashrom", system_path());
	fd = open(fn, O_RDONLY);
	if (0 <= fd) { /* Load file */
		ret = read(fd, cpssp->NAME.flashrom, SZ_E100FLASHROM);
		assert(ret == SZ_E100FLASHROM);
		ret = close(fd);
		assert(0 <= ret);
	} else {
		memset(cpssp->NAME.flashrom, 0, SZ_E100FLASHROM);
	}

	// cpssp->NAME.process.inst_hz = 33*1024*1024;
	// sched_process_init(&cpssp->NAME.process, NAME_(process), cpssp);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
