/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"

static FeriteClass *__ferite_error_class = NULL;

/*!
 * \fn void __ferite_raise_script_error( FeriteScript *script, int err, char *fmt, ... )
 * \brief Raise an exception within the ferite engine.
 * \param script The running script
 * \param err    The error code
 * \param fmt    The format of the error string
 */
void __ferite_raise_script_error( FeriteScript *script, int err, char *fmt, ... )
{
   FeriteNamespaceBucket *nsb;
   FeriteVariable *gerr, *newError;
   FeriteVariable *errstr, *erno;
   char *msg;
   va_list ap;
   
   FE_ENTER_FUNCTION;
   
   msg = fmalloc( 8096 );
   va_start( ap, fmt );
   vsprintf( msg, fmt, ap );
   FUD(("ERROR RAISED: %s %d\n", msg, err ));
   
   nsb = __ferite_namespace_element_exists( script, script->mainns, "err" );
   FE_ASSERT( nsb && nsb->type == FENS_VAR );
   gerr = nsb->data;
   script->error_state = ERROR_THROWN;
 
   if( VAO(gerr) == NULL )
   {
      nsb = __ferite_namespace_element_exists( script, script->mainns, "Error" );
      __ferite_error_class = nsb->data;
      FUD(( "ERROR RAISED: %s\n", __ferite_error_class->name ));
      FUD(( "ERROR RAISED: script error called, no eror object yet. rectifying this 'issue'\n" ));
      newError = (FeriteVariable *)__ferite_new_object( script, __ferite_error_class, NULL );
      VAO(gerr) = VAO(newError);
      VAO(gerr)->refcount++;
      __ferite_variable_destroy( script, newError );
   }
   
   errstr = __ferite_get_variable_from_hash( script, VAO(gerr)->variables, "errstr" );
   ffree( VAS(errstr) );
   VAS(errstr) = fstrdup( msg );
   ffree( msg );
   
   erno = __ferite_get_variable_from_hash( script, VAO(gerr)->variables, "errno" );
   VAI(erno) = err;
   
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_error( FeriteScript *script, char *errormsg, ... )
 * \brief Throw an exception.
 * \param script   The scipt we are using
 * \param errormsg The error information
 */
void ferite_error( FeriteScript *script, char *errormsg, ... )
{
   va_list ap;
   char *buf;
   
   FE_ENTER_FUNCTION;
   buf = fmalloc( 8096 );
   va_start( ap, errormsg );
   vsprintf( buf, errormsg, ap );
   if( script == NULL || !__ferite_is_executing( script ) )
   {
      fprintf( stderr, "[ferite]: Error: %s", buf );
   }
   else
   {
      if( script->error_state != ERROR_THROWN ) {
		 __ferite_raise_script_error( script, 0, buf );
      } else {
		 fprintf( stderr, "[ferite]: Error: %s", buf );
		 ferite_warning( script, "RECURSIVE ERROR FOUND (PROBABLY AN INTERNAL ERROR, PLEASE REPORT):\n" );
		 exit(-1); /* force an exit on recursive errors. as they are evil (TM). */
      }
   }
   ffree( buf );
   va_end( ap );
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_warning( FeriteScript *script, char *errormsg, ... )
 * \brief Display a warning message. This does not cause an exception
 * \param script   The current script
 * \param errormsg The warning to be displayed
 */
void ferite_warning( FeriteScript *script, char *errormsg, ... )
{
   va_list ap;
   char *buf;
   
   FE_ENTER_FUNCTION;
   va_start( ap, errormsg );
   if( script == NULL || script->warning_cb == NULL ){
      fprintf( stderr, "[ferite]: Warning: " );
      vfprintf( stderr, errormsg, ap );
	  if( script != NULL && __ferite_is_executing( script ) ){
		 fprintf( stderr, "       On Line %d in file '%s'\n", script->current_op_line, script->current_op_file );
	  }
   } else {
      buf = fcalloc( 8096, sizeof(char) );
      vsprintf( buf, errormsg, ap );
      (script->warning_cb)( script, buf );
      ffree( buf );
   }
   va_end( ap );   
   FE_LEAVE_FUNCTION(NOWT);
}

/*!
 * \fn void __ferite_init_error_system( FeriteScript *script, FeriteNamespace *ns )
 * \brief Setup the special error handling class on a script
 * \param script The current script
 * \param ns     The namespace in which the class should be registered
 */
void __ferite_init_error_system( FeriteScript *script, FeriteNamespace *ns )
{
   FeriteNamespaceBucket *nsb;
   FeriteVariable *gerr, *newError;
   
   FE_ENTER_FUNCTION;
   __ferite_error_class = ferite_register_class( script, ns, "Error" );
   ferite_register_class_variable( script, __ferite_error_class, fe_new_lng( "errno", 0 ), 0 );
   ferite_register_class_variable( script, __ferite_error_class, fe_new_str( "errstr", "" ), 0 );
   
   nsb = __ferite_namespace_element_exists( script, script->mainns, "err" );
   FE_ASSERT( nsb && nsb->type == FENS_VAR );
   gerr = nsb->data;
   
   nsb = __ferite_namespace_element_exists( script, script->mainns, "Error" );
   __ferite_error_class = nsb->data;
   newError = (FeriteVariable *)__ferite_new_object( script, __ferite_error_class, NULL );
   VAO(gerr) = VAO(newError);
   VAO(gerr)->refcount++;
   __ferite_variable_destroy( script, newError );
   
   FE_LEAVE_FUNCTION(NOWT);
}

void __ferite_set_error( FeriteScript *script, int num, char *fmt, ... )
{
   FeriteNamespaceBucket *nsb;
   FeriteVariable *gerr, *newError;
   FeriteVariable *errstr, *erno;
   va_list ap;
   char *buf;
   
   FE_ENTER_FUNCTION;
   buf = fmalloc( 8096 );
   va_start( ap, fmt );
   vsprintf( buf, fmt, ap );
   
   nsb = __ferite_namespace_element_exists( script, script->mainns, "err" );
   FE_ASSERT( nsb && nsb->type == FENS_VAR );
   gerr = nsb->data;
 
   if( VAO(gerr) == NULL )
   {
      nsb = __ferite_namespace_element_exists( script, script->mainns, "Error" );
      __ferite_error_class = nsb->data;
      FUD(( "ERROR RAISED: %s\n", __ferite_error_class->name ));
      FUD(( "ERROR RAISED: script error called, no eror object yet. rectifying this 'issue'\n" ));
      newError = (FeriteVariable *)__ferite_new_object( script, __ferite_error_class, NULL );
      VAO(gerr) = VAO(newError);
      VAO(gerr)->refcount++;
      __ferite_variable_destroy( script, newError );
   }
   
   errstr = __ferite_get_variable_from_hash( script, VAO(gerr)->variables, "errstr" );
   ffree( VAS(errstr) );
   VAS(errstr) = buf;
   
   erno = __ferite_get_variable_from_hash( script, VAO(gerr)->variables, "errno" );
   VAI(erno) = num;   
   
   va_end( ap );
   FE_LEAVE_FUNCTION( NOWT );
}
