package freenet.fs.acct;

import java.io.DataInput;
import java.io.IOException;

/**
 * Adds a transactional layer to an AccountingTable.
 * The actual implementation is expected to provide a method
 * for flushing/freezing -- i.e., causing the changes to be
 * written to disk.
 * @author tavin
 */
public interface AccountingProcess {

    /**
     * @return  the maximum length of the user data that
     *          may be stored in the accounting blocks
     */
    int getDataWidth();

    /**
     * @return  the total number of slots in the accounting table
     */
    int getBlockCount();

    /**
     * @return  the number of (logically) filled blocks
     */
    int getUseCount();

    /**
     * @return  the number of empty slots available;
     *          i.e., the number of times that create/modify
     *          may be called before a flush is required
     */
    int getFreeCount();

    /**
     * Resumes a BlockTransaction obtained from the
     * AccountingInitialization scan, as if create/modify
     * had been called normally.
     * @return  the annotations that had been collected
     */
    DataInput resume(BlockTransaction btx, BlockEntry be);

    /**
     * Allocates a new block.
     */
    BlockTransaction create(BlockEntry be);
    
    /**
     * Modifies an existing block.
     */
    BlockTransaction modify(int bnum, BlockEntry be);

    /**
     * Reverts the block back to its original state
     * (after the next freeze).
     */
    void abandon(BlockTransaction btx);
    
    /**
     * Schedule the block to be retired in the next freeze.
     * Don't do this for a block that has been scheduled for
     * modification..
     */
    void retire(int bnum);
    
    /**
     * Reads the contents of the block.  It is inadvisable to do this
     * for a block that is scheduled for modification or retiring..
     */
    DataInput load(int bnum) throws IOException;
}


